/*
 * Copyright 2013 Goldman Sachs.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.gs.collections.impl.map.mutable.primitive;

import java.io.Externalizable;
import java.io.IOException;
import java.io.ObjectInput;
import java.io.ObjectOutput;
import java.lang.reflect.Array;
import java.util.Arrays;
import java.util.Collection;
import java.util.Iterator;
import java.util.NoSuchElementException;
import java.util.Set;

import com.gs.collections.api.ShortIterable;
import com.gs.collections.api.LazyShortIterable;
import com.gs.collections.api.LazyIterable;
import com.gs.collections.api.RichIterable;
import com.gs.collections.api.bag.primitive.MutableShortBag;
import com.gs.collections.api.block.function.primitive.ShortFunction;
import com.gs.collections.api.block.function.primitive.ShortFunction0;
import com.gs.collections.api.block.function.primitive.ShortToShortFunction;
import com.gs.collections.api.block.function.primitive.ShortToObjectFunction;
import com.gs.collections.api.block.function.primitive.ObjectShortToObjectFunction;
import com.gs.collections.api.block.predicate.primitive.ShortPredicate;
import com.gs.collections.api.block.predicate.primitive.ObjectShortPredicate;
import com.gs.collections.api.block.procedure.Procedure;
import com.gs.collections.api.block.procedure.Procedure2;
import com.gs.collections.api.block.procedure.primitive.ShortProcedure;
import com.gs.collections.api.block.procedure.primitive.ObjectShortProcedure;
import com.gs.collections.api.collection.MutableCollection;
import com.gs.collections.api.collection.primitive.ImmutableShortCollection;
import com.gs.collections.api.collection.primitive.MutableShortCollection;
import com.gs.collections.api.iterator.ShortIterator;
import com.gs.collections.api.list.MutableList;
import com.gs.collections.api.list.primitive.MutableShortList;
import com.gs.collections.api.map.primitive.ImmutableObjectShortMap;
import com.gs.collections.api.map.primitive.MutableObjectShortMap;
import com.gs.collections.api.map.primitive.ObjectShortMap;
import com.gs.collections.api.set.primitive.MutableShortSet;
import com.gs.collections.api.tuple.primitive.ObjectShortPair;
import com.gs.collections.impl.bag.mutable.primitive.ShortHashBag;
import com.gs.collections.impl.collection.mutable.primitive.SynchronizedShortCollection;
import com.gs.collections.impl.collection.mutable.primitive.UnmodifiableShortCollection;
import com.gs.collections.impl.lazy.AbstractLazyIterable;
import com.gs.collections.impl.factory.primitive.ShortLists;
import com.gs.collections.impl.factory.primitive.ObjectShortMaps;
import com.gs.collections.impl.lazy.primitive.LazyShortIterableAdapter;
import com.gs.collections.impl.list.mutable.FastList;
import com.gs.collections.impl.list.mutable.primitive.ShortArrayList;
import com.gs.collections.impl.set.mutable.primitive.ShortHashSet;
import com.gs.collections.api.block.procedure.primitive.ObjectIntProcedure;
import com.gs.collections.impl.tuple.primitive.PrimitiveTuples;
/**
 * This file was automatically generated from template file objectPrimitiveHashMap.stg.
 *
 * @since 3.0.
 */
public class ObjectShortHashMap<K> implements MutableObjectShortMap<K>, Externalizable
{
    public static final short EMPTY_VALUE = (short) 0;

    private static final long serialVersionUID = 1L;
    private static final int OCCUPIED_DATA_RATIO = 2;
    private static final int OCCUPIED_SENTINEL_RATIO = 4;
    private static final int DEFAULT_INITIAL_CAPACITY = 8;

    private static final Object NULL_KEY = new Object()
    {
        @Override
        public boolean equals(Object obj)
        {
            throw new RuntimeException("Possible corruption through unsynchronized concurrent modification.");
        }

        @Override
        public int hashCode()
        {
            throw new RuntimeException("Possible corruption through unsynchronized concurrent modification.");
        }

        @Override
        public String toString()
        {
            return "ObjectShortHashMap.NULL_KEY";
        }
    };

    private static final Object REMOVED_KEY = new Object()
    {
        @Override
        public boolean equals(Object obj)
        {
            throw new RuntimeException("Possible corruption through unsynchronized concurrent modification.");
        }

        @Override
        public int hashCode()
        {
            throw new RuntimeException("Possible corruption through unsynchronized concurrent modification.");
        }

        @Override
        public String toString()
        {
            return "ObjectShortHashMap.REMOVED_KEY";
        }
    };

    private Object[] keys;
    private short[] values;

    private int occupiedWithData;
    private int occupiedWithSentinels;

    public ObjectShortHashMap()
    {
        this.allocateTable(DEFAULT_INITIAL_CAPACITY << 1);
    }

    public ObjectShortHashMap(int initialCapacity)
    {
        if (initialCapacity < 0)
        {
            throw new IllegalArgumentException("initial capacity cannot be less than 0");
        }
        int capacity = this.smallestPowerOfTwoGreaterThan(this.fastCeil(initialCapacity * OCCUPIED_DATA_RATIO));
        this.allocateTable(capacity);
    }

    private int smallestPowerOfTwoGreaterThan(int n)
    {
        return n > 1 ? Integer.highestOneBit(n - 1) << 1 : 1;
    }

    private int fastCeil(float v)
    {
        int possibleResult = (int) v;
        if (v - possibleResult > 0.0F)
        {
            possibleResult++;
        }
        return possibleResult;
    }

    public ObjectShortHashMap(ObjectShortMap<? extends K> map)
    {
        this(Math.max(map.size(), DEFAULT_INITIAL_CAPACITY));
        this.putAll(map);
    }

    public static <K> ObjectShortHashMap<K> newMap()
    {
        return new ObjectShortHashMap<K>();
    }

    public static <K> ObjectShortHashMap<K> newWithKeysValues(K key1, short value1)
    {
        return new ObjectShortHashMap<K>(1).withKeyValue(key1, value1);
    }

    public static <K> ObjectShortHashMap<K> newWithKeysValues(K key1, short value1, K key2, short value2)
    {
        return new ObjectShortHashMap<K>(2).withKeysValues(key1, value1, key2, value2);
    }

    public static <K> ObjectShortHashMap<K> newWithKeysValues(K key1, short value1, K key2, short value2, K key3, short value3)
    {
        return new ObjectShortHashMap<K>(3).withKeysValues(key1, value1, key2, value2, key3, value3);
    }

    public static <K> ObjectShortHashMap<K> newWithKeysValues(K key1, short value1, K key2, short value2, K key3, short value3, K key4, short value4)
    {
        return new ObjectShortHashMap<K>(4).withKeysValues(key1, value1, key2, value2, key3, value3, key4, value4);
    }

    @Override
    public boolean equals(Object obj)
    {
        if (this == obj)
        {
            return true;
        }

        if (!(obj instanceof ObjectShortMap))
        {
            return false;
        }

        ObjectShortMap<K> other = (ObjectShortMap<K>) obj;

        if (this.size() != other.size())
        {
            return false;
        }

        for (int i = 0; i < this.keys.length; i++)
        {
            if (isNonSentinel(this.keys[i]) && (!other.containsKey(this.toNonSentinel(this.keys[i])) || this.values[i] != other.getOrThrow(this.toNonSentinel(this.keys[i]))))
            {
                return false;
            }
        }
        return true;
    }

    @Override
    public int hashCode()
    {
        int result = 0;

        for (int i = 0; i < this.keys.length; i++)
        {
            if (isNonSentinel(this.keys[i]))
            {
                result += (this.toNonSentinel(this.keys[i]) == null ? 0 : this.keys[i].hashCode()) ^ (int) this.values[i];
            }
        }
        return result;
    }

    @Override
    public String toString()
    {
        return this.makeString("{", ", ", "}");
    }

    public int size()
    {
        return this.occupiedWithData;
    }

    public boolean isEmpty()
    {
        return this.size() == 0;
    }

    public boolean notEmpty()
    {
        return this.size() != 0;
    }

    public String makeString()
    {
        return this.makeString(", ");
    }

    public String makeString(String separator)
    {
        return this.makeString("", separator, "");
    }

    public String makeString(String start, String separator, String end)
    {
        Appendable stringBuilder = new StringBuilder();
        this.appendString(stringBuilder, start, separator, end);
        return stringBuilder.toString();
    }

    public void appendString(Appendable appendable)
    {
        this.appendString(appendable, ", ");
    }

    public void appendString(Appendable appendable, String separator)
    {
        this.appendString(appendable, "", separator, "");
    }

    public void appendString(Appendable appendable, String start, String separator, String end)
    {
        try
        {
            appendable.append(start);

            boolean first = true;

            for (int i = 0; i < this.keys.length; i++)
            {
                Object key = this.keys[i];
                if (isNonSentinel(key))
                {
                    if (!first)
                    {
                        appendable.append(separator);
                    }
                    appendable.append(String.valueOf(this.toNonSentinel(key))).append("=").append(String.valueOf(this.values[i]));
                    first = false;
                }
            }
            appendable.append(end);
        }
        catch (IOException e)
        {
            throw new RuntimeException(e);
        }
    }

    public ShortIterator shortIterator()
    {
        return new InternalShortIterator();
    }

    public short[] toArray()
    {
        short[] result = new short[this.size()];
        int index = 0;

        for (int i = 0; i < this.keys.length; i++)
        {
            if (isNonSentinel(this.keys[i]))
            {
                result[index] = this.values[i];
                index++;
            }
        }
        return result;
    }

    public boolean contains(short value)
    {
        return this.containsValue(value);
    }

    public boolean containsAll(short... source)
    {
        for (short item : source)
        {
            if (!this.containsValue(item))
            {
                return false;
            }
        }
        return true;
    }

    public boolean containsAll(ShortIterable source)
    {
        return this.containsAll(source.toArray());
    }

    public void clear()
    {
        this.occupiedWithData = 0;
        this.occupiedWithSentinels = 0;
        Arrays.fill(this.keys, null);
        Arrays.fill(this.values, EMPTY_VALUE);
    }

    public void put(K key, short value)
    {
        int index = this.probe(key);

        if (isNonSentinel(this.keys[index]) && nullSafeEquals(this.toNonSentinel(this.keys[index]), key))
        {
            // key already present in map
            this.values[index] = value;
            return;
        }

        if (this.keys[index] == REMOVED_KEY)
        {
            --this.occupiedWithSentinels;
        }
        this.keys[index] = toSentinelIfNull(key);
        this.values[index] = value;
        ++this.occupiedWithData;
        if (this.occupiedWithData > this.maxOccupiedWithData())
        {
            this.rehashAndGrow();
        }
    }

    public void putAll(ObjectShortMap<? extends K> map)
    {
        map.forEachKeyValue(new ObjectShortProcedure<K>()
        {
            public void value(K key, short value)
            {
                ObjectShortHashMap.this.put(key, value);
            }
        });
    }

    public void removeKey(K key)
    {
        int index = this.probe(key);
        if (isNonSentinel(this.keys[index]) && nullSafeEquals(this.toNonSentinel(this.keys[index]), key))
        {
            this.keys[index] = REMOVED_KEY;
            this.values[index] = EMPTY_VALUE;
            this.occupiedWithData--;
            this.occupiedWithSentinels++;
            if (this.occupiedWithSentinels > this.maxOccupiedWithSentinels())
            {
                this.rehash();
            }
        }
    }

    public void remove(K key)
    {
        this.removeKey(key);
    }

    public short removeKeyIfAbsent(K key, short value)
    {
        int index = this.probe(key);
        if (isNonSentinel(this.keys[index]) && nullSafeEquals(this.toNonSentinel(this.keys[index]), key))
        {
            this.keys[index] = REMOVED_KEY;
            short oldValue = this.values[index];
            this.values[index] = EMPTY_VALUE;
            this.occupiedWithData--;
            this.occupiedWithSentinels++;
            if (this.occupiedWithSentinels > this.maxOccupiedWithSentinels())
            {
                this.rehash();
            }

            return oldValue;
        }
        return value;
    }

    public short getIfAbsentPut(K key, short value)
    {
        int index = this.probe(key);
        if (isNonSentinel(this.keys[index]) && nullSafeEquals(this.toNonSentinel(this.keys[index]), key))
        {
            return this.values[index];
        }
        if (this.keys[index] == REMOVED_KEY)
        {
            --this.occupiedWithSentinels;
        }
        this.keys[index] = toSentinelIfNull(key);
        this.values[index] = value;
        ++this.occupiedWithData;
        if (this.occupiedWithData > this.maxOccupiedWithData())
        {
            this.rehashAndGrow();
        }
        return value;
    }

    public short getIfAbsentPut(K key, ShortFunction0 function)
    {
        int index = this.probe(key);
        if (isNonSentinel(this.keys[index]) && nullSafeEquals(this.toNonSentinel(this.keys[index]), key))
        {
            return this.values[index];
        }
        if (this.keys[index] == REMOVED_KEY)
        {
            --this.occupiedWithSentinels;
        }
        this.keys[index] = toSentinelIfNull(key);
        short value = function.value();
        this.values[index] = value;
        ++this.occupiedWithData;
        if (this.occupiedWithData > this.maxOccupiedWithData())
        {
            this.rehashAndGrow();
        }
        return value;
    }

    public <P> short getIfAbsentPutWith(K key, ShortFunction<? super P> function, P parameter)
    {
        int index = this.probe(key);
        if (isNonSentinel(this.keys[index]) && nullSafeEquals(this.toNonSentinel(this.keys[index]), key))
        {
            return this.values[index];
        }
        if (this.keys[index] == REMOVED_KEY)
        {
            --this.occupiedWithSentinels;
        }
        this.keys[index] = toSentinelIfNull(key);
        short value = function.shortValueOf(parameter);
        this.values[index] = value;
        ++this.occupiedWithData;
        if (this.occupiedWithData > this.maxOccupiedWithData())
        {
            this.rehashAndGrow();
        }
        return value;
    }

    public short getIfAbsentPutWithKey(K key, ShortFunction<? super K> function)
    {
        int index = this.probe(key);
        if (isNonSentinel(this.keys[index]) && nullSafeEquals(this.toNonSentinel(this.keys[index]), key))
        {
            return this.values[index];
        }
        if (this.keys[index] == REMOVED_KEY)
        {
            --this.occupiedWithSentinels;
        }
        this.keys[index] = toSentinelIfNull(key);
        short value = function.shortValueOf(key);
        this.values[index] = value;
        ++this.occupiedWithData;
        if (this.occupiedWithData > this.maxOccupiedWithData())
        {
            this.rehashAndGrow();
        }
        return value;
    }

    public short updateValue(K key, short initialValueIfAbsent, ShortToShortFunction function)
    {
        int index = this.probe(key);
        if (isNonSentinel(this.keys[index]) && nullSafeEquals(this.toNonSentinel(this.keys[index]), key))
        {
            this.values[index] = function.valueOf(this.values[index]);
            return this.values[index];
        }
        if (this.keys[index] == REMOVED_KEY)
        {
            --this.occupiedWithSentinels;
        }
        this.keys[index] = toSentinelIfNull(key);
        short value = function.valueOf(initialValueIfAbsent);
        this.values[index] = value;
        ++this.occupiedWithData;
        if (this.occupiedWithData > this.maxOccupiedWithData())
        {
            this.rehashAndGrow();
        }
        return value;
    }

    public ObjectShortHashMap<K> withKeyValue(K key1, short value1)
    {
        this.put(key1, value1);
        return this;
    }

    public ObjectShortHashMap<K> withKeysValues(K key1, short value1, K key2, short value2)
    {
        this.put(key1, value1);
        this.put(key2, value2);
        return this;
    }

    public ObjectShortHashMap<K> withKeysValues(K key1, short value1, K key2, short value2, K key3, short value3)
    {
        this.put(key1, value1);
        this.put(key2, value2);
        this.put(key3, value3);
        return this;
    }

    public ObjectShortHashMap<K> withKeysValues(K key1, short value1, K key2, short value2, K key3, short value3, K key4, short value4)
    {
        this.put(key1, value1);
        this.put(key2, value2);
        this.put(key3, value3);
        this.put(key4, value4);
        return this;
    }

    public ObjectShortHashMap<K> withoutKey(K key)
    {
        this.removeKey(key);
        return this;
    }

    public ObjectShortHashMap<K> withoutAllKeys(Iterable<? extends K> keys)
    {
        for (K key : keys)
        {
            this.removeKey(key);
        }
        return this;
    }

    public MutableObjectShortMap<K> asUnmodifiable()
    {
        return new UnmodifiableObjectShortMap<K>(this);
    }

    public MutableObjectShortMap<K> asSynchronized()
    {
        return new SynchronizedObjectShortMap<K>(this);
    }

    public ImmutableObjectShortMap<K> toImmutable()
    {
        return ObjectShortMaps.immutable.withAll(this);
    }

    public short get(Object key)
    {
        return this.getIfAbsent(key, EMPTY_VALUE);
    }

    public short getOrThrow(Object key)
    {
        int index = this.probe(key);
        if (isNonSentinel(this.keys[index]))
        {
            return this.values[index];
        }
        throw new IllegalStateException("Key " + key + " not present.");
    }

    public short getIfAbsent(Object key, short ifAbsent)
    {
        int index = this.probe(key);
        if (isNonSentinel(this.keys[index]) && nullSafeEquals(this.toNonSentinel(this.keys[index]), key))
        {
            return this.values[index];
        }
        return ifAbsent;
    }

    public boolean containsKey(Object key)
    {
        int index = this.probe(key);
        return isNonSentinel(this.keys[index]) && nullSafeEquals(this.toNonSentinel(this.keys[index]), key);
    }

    public boolean containsValue(short value)
    {
        for (int i = 0; i < this.values.length; i++)
        {
            if (isNonSentinel(this.keys[i]) && this.values[i] == value)
            {
                return true;
            }
        }
        return false;
    }

    public void forEach(ShortProcedure procedure)
    {
        this.forEachValue(procedure);
    }

    public void forEachValue(ShortProcedure procedure)
    {
        for (int i = 0; i < this.keys.length; i++)
        {
            if (isNonSentinel(this.keys[i]))
            {
                procedure.value(this.values[i]);
            }
        }
    }

    public void forEachKey(Procedure<? super K> procedure)
    {
        for (int i = 0; i < this.keys.length; i++)
        {
            if (isNonSentinel(this.keys[i]))
            {
                procedure.value(this.toNonSentinel(this.keys[i]));
            }
        }
    }

    public void forEachKeyValue(ObjectShortProcedure<? super K> procedure)
    {
        for (int i = 0; i < this.keys.length; i++)
        {
            if (isNonSentinel(this.keys[i]))
            {
                procedure.value(this.toNonSentinel(this.keys[i]), this.values[i]);
            }
        }
    }

    public ObjectShortHashMap<K> select(ObjectShortPredicate<? super K> predicate)
    {
        ObjectShortHashMap<K> result = ObjectShortHashMap.newMap();

        for (int i = 0; i < this.keys.length; i++)
        {
            if (isNonSentinel(this.keys[i]) && predicate.accept(this.toNonSentinel(this.keys[i]), this.values[i]))
            {
                result.put(this.toNonSentinel(this.keys[i]), this.values[i]);
            }
        }
        return result;
    }

    public ObjectShortHashMap<K> reject(ObjectShortPredicate<? super K> predicate)
    {
        ObjectShortHashMap<K> result = ObjectShortHashMap.newMap();

        for (int i = 0; i < this.keys.length; i++)
        {
            if (isNonSentinel(this.keys[i]) && !predicate.accept(this.toNonSentinel(this.keys[i]), this.values[i]))
            {
                result.put(this.toNonSentinel(this.keys[i]), this.values[i]);
            }
        }
        return result;
    }

    public MutableShortCollection select(ShortPredicate predicate)
    {
        ShortArrayList result = new ShortArrayList();

        for (int i = 0; i < this.keys.length; i++)
        {
            if (isNonSentinel(this.keys[i]) && predicate.accept(this.values[i]))
            {
                result.add(this.values[i]);
            }
        }
        return result;
    }

    public MutableShortCollection reject(ShortPredicate predicate)
    {
        ShortArrayList result = new ShortArrayList();

        for (int i = 0; i < this.keys.length; i++)
        {
            if (isNonSentinel(this.keys[i]) && !predicate.accept(this.values[i]))
            {
                result.add(this.values[i]);
            }
        }
        return result;
    }

    public short detectIfNone(ShortPredicate predicate, short ifNone)
    {
        for (int i = 0; i < this.keys.length; i++)
        {
            if (isNonSentinel(this.keys[i]) && predicate.accept(this.values[i]))
            {
                return this.values[i];
            }
        }
        return ifNone;
    }

    public <V> MutableCollection<V> collect(ShortToObjectFunction<? extends V> function)
    {
        MutableList<V> result = FastList.newList(this.size());
        for (int i = 0; i < this.keys.length; i++)
        {
            if (isNonSentinel(this.keys[i]))
            {
                result.add(function.valueOf(this.values[i]));
            }
        }
        return result;
    }

    public int count(ShortPredicate predicate)
    {
        int count = 0;

        for (int i = 0; i < this.keys.length; i++)
        {
            if (isNonSentinel(this.keys[i]) && predicate.accept(this.values[i]))
            {
                count++;
            }
        }
        return count;
    }

    public boolean anySatisfy(ShortPredicate predicate)
    {
        for (int i = 0; i < this.keys.length; i++)
        {
            if (isNonSentinel(this.keys[i]) && predicate.accept(this.values[i]))
            {
                return true;
            }
        }
        return false;
    }

    public boolean allSatisfy(ShortPredicate predicate)
    {
        for (int i = 0; i < this.keys.length; i++)
        {
            if (isNonSentinel(this.keys[i]) && !predicate.accept(this.values[i]))
            {
                return false;
            }
        }
        return true;
    }

    public boolean noneSatisfy(ShortPredicate predicate)
    {
        for (int i = 0; i < this.keys.length; i++)
        {
            if (isNonSentinel(this.keys[i]) && predicate.accept(this.values[i]))
            {
                return false;
            }
        }
        return true;
    }

    public <V> V injectInto(V injectedValue, ObjectShortToObjectFunction<? super V, ? extends V> function)
    {
        V result = injectedValue;

        for (int i = 0; i < this.keys.length; i++)
        {
            if (isNonSentinel(this.keys[i]))
            {
                result = function.valueOf(result, this.values[i]);
            }
        }

        return result;
    }

    public long sum()
    {
        long sum = 0L;

        for (int i = 0; i < this.keys.length; i++)
        {
            if (isNonSentinel(this.keys[i]))
            {
                sum += this.values[i];
            }
        }
        return sum;
    }

    public short max()
    {
        if (this.isEmpty())
        {
            throw new NoSuchElementException();
        }
        short max = (short) 0;
        boolean isMaxSet = false;

        for (int i = 0; i < this.keys.length; i++)
        {
            if (isNonSentinel(this.keys[i]) && (!isMaxSet || max < this.values[i]))
            {
                max = this.values[i];
                isMaxSet = true;
            }
        }
        return max;
    }

    public short min()
    {
        if (this.isEmpty())
        {
            throw new NoSuchElementException();
        }
        short min = (short) 0;
        boolean isMinSet = false;

        for (int i = 0; i < this.keys.length; i++)
        {
            if (isNonSentinel(this.keys[i]) && (!isMinSet || this.values[i] < min))
            {
                min = this.values[i];
                isMinSet = true;
            }
        }
        return min;
    }

    public short maxIfEmpty(short defaultValue)
    {
        if (this.isEmpty())
        {
            return defaultValue;
        }
        short max = (short) 0;
        boolean isMaxSet = false;

        for (int i = 0; i < this.keys.length; i++)
        {
            if (isNonSentinel(this.keys[i]) && (!isMaxSet || max < this.values[i]))
            {
                max = this.values[i];
                isMaxSet = true;
            }
        }
        return max;
    }

    public short minIfEmpty(short defaultValue)
    {
        if (this.isEmpty())
        {
            return defaultValue;
        }
        short min = (short) 0;
        boolean isMinSet = false;

        for (int i = 0; i < this.keys.length; i++)
        {
            if (isNonSentinel(this.keys[i]) && (!isMinSet || this.values[i] < min))
            {
                min = this.values[i];
                isMinSet = true;
            }
        }
        return min;
    }

    public double average()
    {
        if (this.isEmpty())
        {
            throw new ArithmeticException();
        }
        return (double) this.sum() / (double) this.size();
    }

    public double median()
    {
        if (this.isEmpty())
        {
            throw new ArithmeticException();
        }
        short[] sortedArray = this.toSortedArray();
        int middleIndex = sortedArray.length >> 1;
        if (sortedArray.length > 1 && (sortedArray.length & 1) == 0)
        {
            short first = sortedArray[middleIndex];
            short second = sortedArray[middleIndex - 1];
            return ((double) first + (double) second) / 2.0;
        }
        return (double) sortedArray[middleIndex];
    }

    public MutableShortList toList()
    {
        MutableShortList result = new ShortArrayList(this.size());

        for (int i = 0; i < this.keys.length; i++)
        {
            if (isNonSentinel(this.keys[i]))
            {
                result.add(this.values[i]);
            }
        }
        return result;
    }

    public MutableShortSet toSet()
    {
        MutableShortSet result = new ShortHashSet(this.size());

        for (int i = 0; i < this.keys.length; i++)
        {
            if (isNonSentinel(this.keys[i]))
            {
                result.add(this.values[i]);
            }
        }
        return result;
    }

    public MutableShortBag toBag()
    {
        MutableShortBag result = new ShortHashBag(this.size());

        for (int i = 0; i < this.keys.length; i++)
        {
            if (isNonSentinel(this.keys[i]))
            {
                result.add(this.values[i]);
            }
        }
        return result;
    }

    public LazyShortIterable asLazy()
    {
        return new LazyShortIterableAdapter(this);
    }

    public short[] toSortedArray()
    {
        short[] array = this.toArray();
        Arrays.sort(array);
        return array;
    }

    public MutableShortList toSortedList()
    {
        return this.toList().sortThis();
    }

    public void writeExternal(ObjectOutput out) throws IOException
    {
        out.writeInt(this.size());
        for (int i = 0; i < this.keys.length; i++)
        {
            if (isNonSentinel(this.keys[i]))
            {
                out.writeObject(this.toNonSentinel(this.keys[i]));
                out.writeShort(this.values[i]);
            }
        }
    }

    public void readExternal(ObjectInput in) throws IOException, ClassNotFoundException
    {
        int size = in.readInt();
        int capacity = this.smallestPowerOfTwoGreaterThan(this.fastCeil(size * OCCUPIED_DATA_RATIO));
        this.allocateTable(capacity);

        for (int i = 0; i < size; i++)
        {
            this.put((K) in.readObject(), in.readShort());
        }
    }

    public LazyIterable<K> keysView()
    {
        return new KeysView();
    }

    public RichIterable<ObjectShortPair<K>> keyValuesView()
    {
        return new KeyValuesView();
    }

    /**
     * Rehashes every element in the set into a new backing table of the smallest possible size and eliminating removed sentinels.
     */
    public void compact()
    {
        this.rehash(this.smallestPowerOfTwoGreaterThan(this.size()));
    }

    private void rehash()
    {
        this.rehash(this.keys.length);
    }

    private void rehashAndGrow()
    {
        this.rehash(this.keys.length << 1);
    }

    private void rehash(int newCapacity)
    {
        int oldLength = this.keys.length;
        Object[] old = this.keys;
        short[] oldValues = this.values;
        this.allocateTable(newCapacity);
        this.occupiedWithData = 0;
        this.occupiedWithSentinels = 0;

        for (int i = 0; i < oldLength; i++)
        {
            if (isNonSentinel(old[i]))
            {
                this.put(this.toNonSentinel(old[i]), oldValues[i]);
            }
        }
    }

    // exposed for testing
    int probe(Object element)
    {
        int index = this.spread(element);

        int removedIndex = -1;
        if (isRemovedKey(this.keys[index]))
        {
            removedIndex = index;
        }

        else if (this.keys[index] == null || nullSafeEquals(this.toNonSentinel(this.keys[index]), element))
        {
            return index;
        }

        int nextIndex = index;
        int probe = 17;

        // loop until an empty slot is reached
        while (true)
        {
            // Probe algorithm: 17*n*(n+1)/2 where n = no. of collisions
            nextIndex += probe;
            probe += 17;
            nextIndex &= this.keys.length - 1;

            if (isRemovedKey(this.keys[nextIndex]))
            {
                if (removedIndex == -1)
                {
                    removedIndex = nextIndex;
                }
            }
            else if (nullSafeEquals(this.toNonSentinel(this.keys[nextIndex]), element))
            {
                return nextIndex;
            }
            else if (this.keys[nextIndex] == null)
            {
                return removedIndex == -1 ? nextIndex : removedIndex;
            }
        }
    }

    // exposed for testing
    int spread(Object element)
    {
        // This function ensures that hashCodes that differ only by
        // constant multiples at each bit position have a bounded
        // number of collisions (approximately 8 at default load factor).
        int h = element == null ? 0 : element.hashCode();
        h ^= h >>> 20 ^ h >>> 12;
        h ^= h >>> 7 ^ h >>> 4;
        return h & (this.keys.length - 1);
    }

    private static boolean nullSafeEquals(Object value, Object other)
    {
        if (value == null)
        {
            if (other == null)
            {
                return true;
            }
        }
        else if (other == value || value.equals(other))
        {
            return true;
        }
        return false;
    }

    private void allocateTable(int sizeToAllocate)
    {
        this.keys = new Object[sizeToAllocate];
        this.values = new short[sizeToAllocate];
    }

    private static boolean isRemovedKey(Object key)
    {
        return key == REMOVED_KEY;
    }

    private static <K> boolean isNonSentinel(K key)
    {
        return key != null && !isRemovedKey(key);
    }

    private K toNonSentinel(Object key)
    {
        return key == NULL_KEY ? null : (K) key;
    }

    private static Object toSentinelIfNull(Object key)
    {
        return key == null ? NULL_KEY : key;
    }

    private int maxOccupiedWithData()
    {
        int capacity = this.keys.length;
        // need at least one free slot for open addressing
        return Math.min(capacity - 1, capacity / OCCUPIED_DATA_RATIO);
    }

    private int maxOccupiedWithSentinels()
    {
        return this.keys.length / OCCUPIED_SENTINEL_RATIO;
    }

    private class InternalShortIterator implements ShortIterator
    {
        private int count;
        private int position;

        public short next()
        {
            if (!this.hasNext())
            {
                throw new NoSuchElementException();
            }

            Object[] keys = ObjectShortHashMap.this.keys;
            while (!isNonSentinel(keys[this.position]))
            {
                this.position++;
            }
            short result = ObjectShortHashMap.this.values[this.position];
            this.count++;
            this.position++;
            return result;
        }

        public boolean hasNext()
        {
            return this.count != ObjectShortHashMap.this.size();
        }
    }

    public Set<K> keySet()
    {
        return new KeySet();
    }

    public MutableShortCollection values()
    {
        return new ValuesCollection();
    }

    private class KeySet implements Set<K>
    {
        @Override
        public boolean equals(Object obj)
        {
            if (obj instanceof Set)
            {
                Set<?> other = (Set<?>) obj;
                if (other.size() == this.size())
                {
                    return this.containsAll(other);
                }
            }
            return false;
        }

        @Override
        public int hashCode()
        {
            int hashCode = 0;
            Object[] table = ObjectShortHashMap.this.keys;
            for (int i = 0; i < table.length; i++)
            {
                Object key = table[i];
                if (ObjectShortHashMap.isNonSentinel(key))
                {
                    K nonSentinelKey = ObjectShortHashMap.this.toNonSentinel(key);
                    hashCode += nonSentinelKey == null ? 0 : nonSentinelKey.hashCode();
                }
            }
            return hashCode;
        }

        public int size()
        {
            return ObjectShortHashMap.this.size();
        }

        public boolean isEmpty()
        {
            return ObjectShortHashMap.this.isEmpty();
        }

        public boolean contains(Object o)
        {
            return ObjectShortHashMap.this.containsKey(o);
        }

        public Object[] toArray()
        {
            int size = ObjectShortHashMap.this.size();
            Object[] result = new Object[size];
            this.copyKeys(result);
            return result;
        }

        public <T> T[] toArray(T[] result)
        {
            int size = ObjectShortHashMap.this.size();
            if (result.length < size)
            {
                result = (T[]) Array.newInstance(result.getClass().getComponentType(), size);
            }
            this.copyKeys(result);
            if (size < result.length)
            {
                result[size] = null;
            }
            return result;
        }

        public boolean add(K key)
        {
            throw new UnsupportedOperationException();
        }

        public boolean remove(Object key)
        {
            int oldSize = ObjectShortHashMap.this.size();
            ObjectShortHashMap.this.removeKey((K) key);
            return oldSize != ObjectShortHashMap.this.size();
        }

        public boolean containsAll(Collection<?> collection)
        {
            for (Object aCollection : collection)
            {
                if (!ObjectShortHashMap.this.containsKey(aCollection))
                {
                    return false;
                }
            }
            return true;
        }

        public boolean addAll(Collection<? extends K> collection)
        {
            throw new UnsupportedOperationException();
        }

        public boolean retainAll(Collection<?> collection)
        {
            int oldSize = ObjectShortHashMap.this.size();
            Iterator<K> iterator = this.iterator();
            while (iterator.hasNext())
            {
                K next = iterator.next();
                if (!collection.contains(next))
                {
                    this.remove(next);
                }
            }
            return oldSize != ObjectShortHashMap.this.size();
        }

        public boolean removeAll(Collection<?> collection)
        {
            int oldSize = ObjectShortHashMap.this.size();
            for (Object object : collection)
            {
                ObjectShortHashMap.this.removeKey((K) object);
            }
            return oldSize != ObjectShortHashMap.this.size();
        }

        public void clear()
        {
            ObjectShortHashMap.this.clear();
        }

        public Iterator<K> iterator()
        {
            return new KeySetIterator();
        }

        private void copyKeys(Object[] result)
        {
            int count = 0;
            for (int i = 0; i < ObjectShortHashMap.this.keys.length; i++)
            {
                Object key = ObjectShortHashMap.this.keys[i];
                if (ObjectShortHashMap.isNonSentinel(key))
                {
                    result[count++] = ObjectShortHashMap.this.keys[i];
                }
            }
        }
    }

    private class KeySetIterator implements Iterator<K>
    {
        private int count;
        private int position;
        private K currentKey;
        private boolean isCurrentKeySet;

        public boolean hasNext()
        {
            return this.count < ObjectShortHashMap.this.size();
        }

        public K next()
        {
            if (!this.hasNext())
            {
                throw new NoSuchElementException();
            }
            this.count++;
            Object[] keys = ObjectShortHashMap.this.keys;
            while (!isNonSentinel(keys[this.position]))
            {
                this.position++;
            }
            this.currentKey = (K) ObjectShortHashMap.this.keys[this.position];
            this.isCurrentKeySet = true;
            this.position++;
            return ObjectShortHashMap.this.toNonSentinel(this.currentKey);
        }

        public void remove()
        {
            if (!this.isCurrentKeySet)
            {
                throw new IllegalStateException();
            }

            this.isCurrentKeySet = false;
            this.count--;

            if (isNonSentinel(this.currentKey))
            {
                int index = this.position - 1;
                ObjectShortHashMap.this.keys[index] = REMOVED_KEY;
                ObjectShortHashMap.this.values[index] = EMPTY_VALUE;
                ObjectShortHashMap.this.occupiedWithData--;
                ObjectShortHashMap.this.occupiedWithSentinels++;
                if (ObjectShortHashMap.this.occupiedWithSentinels > ObjectShortHashMap.this.maxOccupiedWithSentinels())
                {
                    ObjectShortHashMap.this.rehash();
                }
            }
            else
            {
                ObjectShortHashMap.this.removeKey(this.currentKey);
            }
        }
    }

    private class ValuesCollection implements MutableShortCollection
    {
        public int size()
        {
            return ObjectShortHashMap.this.size();
        }

        public boolean isEmpty()
        {
            return ObjectShortHashMap.this.isEmpty();
        }

        public boolean notEmpty()
        {
            return ObjectShortHashMap.this.notEmpty();
        }

        public String makeString()
        {
            return this.makeString(", ");
        }

        public String makeString(String separator)
        {
            return this.makeString("", separator, "");
        }

        public String makeString(String start, String separator, String end)
        {
            Appendable stringBuilder = new StringBuilder();
            this.appendString(stringBuilder, start, separator, end);
            return stringBuilder.toString();
        }

        public void appendString(Appendable appendable)
        {
            this.appendString(appendable, ", ");
        }

        public void appendString(Appendable appendable, String separator)
        {
            this.appendString(appendable, "", separator, "");
        }

        public void appendString(Appendable appendable, String start, String separator, String end)
        {
            try
            {
                appendable.append(start);

                boolean first = true;

                for (int i = 0; i < ObjectShortHashMap.this.keys.length; i++)
                {
                    Object key = ObjectShortHashMap.this.keys[i];
                    if (isNonSentinel(key))
                    {
                        if (!first)
                        {
                            appendable.append(separator);
                        }
                        appendable.append(String.valueOf(ObjectShortHashMap.this.values[i]));
                        first = false;
                    }
                }
                appendable.append(end);
            }
            catch (IOException e)
            {
                throw new RuntimeException(e);
            }
        }

        public boolean add(short element)
        {
            throw new UnsupportedOperationException();
        }

        public boolean addAll(short... source)
        {
            throw new UnsupportedOperationException();
        }

        public boolean addAll(ShortIterable source)
        {
            throw new UnsupportedOperationException();
        }

        public boolean remove(short item)
        {
            int oldSize = ObjectShortHashMap.this.size();

            for (int i = 0; i < ObjectShortHashMap.this.keys.length; i++)
            {
                if (isNonSentinel(ObjectShortHashMap.this.keys[i]) && item == ObjectShortHashMap.this.values[i])
                {
                    ObjectShortHashMap.this.removeKey((K) ObjectShortHashMap.this.keys[i]);
                }
            }
            return oldSize != ObjectShortHashMap.this.size();
        }

        public boolean removeAll(ShortIterable source)
        {
            int oldSize = ObjectShortHashMap.this.size();

            ShortIterator iterator = source.shortIterator();
            while (iterator.hasNext())
            {
                this.remove(iterator.next());
            }
            return oldSize != ObjectShortHashMap.this.size();
        }

        public boolean removeAll(short... source)
        {
            int oldSize = ObjectShortHashMap.this.size();

            for (short item : source)
            {
                this.remove(item);
            }
            return oldSize != ObjectShortHashMap.this.size();
        }

        public void clear()
        {
            ObjectShortHashMap.this.clear();
        }

        public MutableShortCollection with(short element)
        {
            throw new UnsupportedOperationException();
        }

        public MutableShortCollection without(short element)
        {
            throw new UnsupportedOperationException();
        }

        public MutableShortCollection withAll(ShortIterable elements)
        {
            throw new UnsupportedOperationException();
        }

        public MutableShortCollection withoutAll(ShortIterable elements)
        {
            throw new UnsupportedOperationException();
        }

        public MutableShortCollection asUnmodifiable()
        {
            return UnmodifiableShortCollection.of(this);
        }

        public MutableShortCollection asSynchronized()
        {
            return SynchronizedShortCollection.of(this);
        }

        public ImmutableShortCollection toImmutable()
        {
            return ShortLists.immutable.withAll(this);
        }

        public ShortIterator shortIterator()
        {
            return ObjectShortHashMap.this.shortIterator();
        }

        public short[] toArray()
        {
            return ObjectShortHashMap.this.toArray();
        }

        public boolean contains(short value)
        {
            return ObjectShortHashMap.this.containsValue(value);
        }

        public boolean containsAll(short... source)
        {
            return ObjectShortHashMap.this.containsAll(source);
        }

        public boolean containsAll(ShortIterable source)
        {
            return ObjectShortHashMap.this.containsAll(source);
        }

        public void forEach(ShortProcedure procedure)
        {
            ObjectShortHashMap.this.forEach(procedure);
        }

        public MutableShortCollection select(ShortPredicate predicate)
        {
            return ObjectShortHashMap.this.select(predicate);
        }

        public MutableShortCollection reject(ShortPredicate predicate)
        {
            return ObjectShortHashMap.this.reject(predicate);
        }

        public <V> MutableCollection<V> collect(ShortToObjectFunction<? extends V> function)
        {
            return ObjectShortHashMap.this.collect(function);
        }

        public <T> T injectInto(T injectedValue, ObjectShortToObjectFunction<? super T, ? extends T> function)
        {
            return ObjectShortHashMap.this.injectInto(injectedValue, function);
        }

        public short detectIfNone(ShortPredicate predicate, short ifNone)
        {
            return ObjectShortHashMap.this.detectIfNone(predicate, ifNone);
        }

        public int count(ShortPredicate predicate)
        {
            return ObjectShortHashMap.this.count(predicate);
        }

        public boolean anySatisfy(ShortPredicate predicate)
        {
            return ObjectShortHashMap.this.anySatisfy(predicate);
        }

        public boolean allSatisfy(ShortPredicate predicate)
        {
            return ObjectShortHashMap.this.allSatisfy(predicate);
        }

        public boolean noneSatisfy(ShortPredicate predicate)
        {
            return ObjectShortHashMap.this.noneSatisfy(predicate);
        }

        public MutableShortList toList()
        {
            return ObjectShortHashMap.this.toList();
        }

        public MutableShortSet toSet()
        {
            return ObjectShortHashMap.this.toSet();
        }

        public MutableShortBag toBag()
        {
            return ObjectShortHashMap.this.toBag();
        }

        public LazyShortIterable asLazy()
        {
            return new LazyShortIterableAdapter(this);
        }

        public short[] toSortedArray()
        {
            return ObjectShortHashMap.this.toSortedArray();
        }

        public MutableShortList toSortedList()
        {
            return ObjectShortHashMap.this.toSortedList();
        }

        public long sum()
        {
            return ObjectShortHashMap.this.sum();
        }

        public short max()
        {
            return ObjectShortHashMap.this.max();
        }

        public short maxIfEmpty(short defaultValue)
        {
            return ObjectShortHashMap.this.maxIfEmpty(defaultValue);
        }

        public short min()
        {
            return ObjectShortHashMap.this.min();
        }

        public short minIfEmpty(short defaultValue)
        {
            return ObjectShortHashMap.this.minIfEmpty(defaultValue);
        }

        public double average()
        {
            return ObjectShortHashMap.this.average();
        }

        public double median()
        {
            return ObjectShortHashMap.this.median();
        }
    }

    private class KeysView extends AbstractLazyIterable<K>
    {
        public void forEach(Procedure<? super K> procedure)
        {
            for (int i = 0; i < ObjectShortHashMap.this.keys.length; i++)
            {
                if (ObjectShortHashMap.isNonSentinel(ObjectShortHashMap.this.keys[i]))
                {
                    procedure.value(ObjectShortHashMap.this.toNonSentinel(ObjectShortHashMap.this.keys[i]));
                }
            }
        }

        public void forEachWithIndex(ObjectIntProcedure<? super K> objectShortProcedure)
        {
            int index = 0;
            for (int i = 0; i < ObjectShortHashMap.this.keys.length; i++)
            {
                if (ObjectShortHashMap.isNonSentinel(ObjectShortHashMap.this.keys[i]))
                {
                    objectShortProcedure.value(ObjectShortHashMap.this.toNonSentinel(ObjectShortHashMap.this.keys[i]), index);
                    index++;
                }
            }
        }

        public <P> void forEachWith(Procedure2<? super K, ? super P> procedure, P parameter)
        {
            for (int i = 0; i < ObjectShortHashMap.this.keys.length; i++)
            {
                if (ObjectShortHashMap.isNonSentinel(ObjectShortHashMap.this.keys[i]))
                {
                    procedure.value(ObjectShortHashMap.this.toNonSentinel(ObjectShortHashMap.this.keys[i]), parameter);
                }
            }
        }

        public Iterator<K> iterator()
        {
            return new InternalKeysViewIterator<K>();
        }

        public class InternalKeysViewIterator<K> implements Iterator<K>
        {
            private int count;
            private int position;

            public K next()
            {
                if (!this.hasNext())
                {
                    throw new NoSuchElementException();
                }

                Object[] keys = ObjectShortHashMap.this.keys;
                while (!isNonSentinel(keys[this.position]))
                {
                    this.position++;
                }
                K result = (K) ObjectShortHashMap.this.keys[this.position];
                this.count++;
                this.position++;
                return result;
            }

            public void remove()
            {
                throw new UnsupportedOperationException();
            }

            public boolean hasNext()
            {
                return this.count != ObjectShortHashMap.this.size();
            }
        }
    }

    private class KeyValuesView extends AbstractLazyIterable<ObjectShortPair<K>>
    {
        public void forEach(Procedure<? super ObjectShortPair<K>> procedure)
        {
            for (int i = 0; i < ObjectShortHashMap.this.keys.length; i++)
            {
                if (ObjectShortHashMap.isNonSentinel(ObjectShortHashMap.this.keys[i]))
                {
                    procedure.value(PrimitiveTuples.pair(ObjectShortHashMap.this.toNonSentinel(ObjectShortHashMap.this.keys[i]), ObjectShortHashMap.this.values[i]));
                }
            }
        }

        public void forEachWithIndex(ObjectIntProcedure<? super ObjectShortPair<K>> objectIntProcedure)
        {
            int index = 0;
            for (int i = 0; i < ObjectShortHashMap.this.keys.length; i++)
            {
                if (ObjectShortHashMap.isNonSentinel(ObjectShortHashMap.this.keys[i]))
                {
                    objectIntProcedure.value(PrimitiveTuples.pair(ObjectShortHashMap.this.toNonSentinel(ObjectShortHashMap.this.keys[i]), ObjectShortHashMap.this.values[i]), index);
                    index++;
                }
            }
        }

        public <P> void forEachWith(Procedure2<? super ObjectShortPair<K>, ? super P> procedure, P parameter)
        {
            for (int i = 0; i < ObjectShortHashMap.this.keys.length; i++)
            {
                if (ObjectShortHashMap.isNonSentinel(ObjectShortHashMap.this.keys[i]))
                {
                    procedure.value(PrimitiveTuples.pair(ObjectShortHashMap.this.toNonSentinel(ObjectShortHashMap.this.keys[i]), ObjectShortHashMap.this.values[i]), parameter);
                }
            }
        }

        public Iterator<ObjectShortPair<K>> iterator()
        {
            return new InternalKeyValuesIterator();
        }

        public class InternalKeyValuesIterator implements Iterator<ObjectShortPair<K>>
        {
            private int count;
            private int position;

            public ObjectShortPair<K> next()
            {
                if (!this.hasNext())
                {
                    throw new NoSuchElementException();
                }

                Object[] keys = ObjectShortHashMap.this.keys;
                while (!isNonSentinel(keys[this.position]))
                {
                    this.position++;
                }
                ObjectShortPair<K> result = PrimitiveTuples.pair(ObjectShortHashMap.this.toNonSentinel(ObjectShortHashMap.this.keys[this.position]), ObjectShortHashMap.this.values[this.position]);
                this.count++;
                this.position++;
                return result;
            }

            public void remove()
            {
                throw new UnsupportedOperationException();
            }

            public boolean hasNext()
            {
                return this.count != ObjectShortHashMap.this.size();
            }
        }
    }
}
