/*
 * Copyright 2013 Goldman Sachs.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.gs.collections.impl.map.mutable.primitive;

import java.io.Externalizable;
import java.io.IOException;
import java.io.ObjectInput;
import java.io.ObjectOutput;
import java.util.Arrays;
import java.util.BitSet;
import java.util.NoSuchElementException;

import com.gs.collections.api.BooleanIterable;
import com.gs.collections.api.ShortIterable;
import com.gs.collections.api.LazyBooleanIterable;
import com.gs.collections.api.LazyShortIterable;
import com.gs.collections.api.LazyIterable;
import com.gs.collections.api.bag.primitive.MutableBooleanBag;
import com.gs.collections.api.bag.primitive.MutableShortBag;
import com.gs.collections.api.block.function.primitive.BooleanFunction;
import com.gs.collections.api.block.function.primitive.BooleanFunction0;
import com.gs.collections.api.block.function.primitive.BooleanToBooleanFunction;
import com.gs.collections.api.block.function.primitive.BooleanToObjectFunction;
import com.gs.collections.api.block.function.primitive.ShortToBooleanFunction;
import com.gs.collections.api.block.function.primitive.ShortToObjectFunction;
import com.gs.collections.api.block.function.primitive.ObjectShortToObjectFunction;
import com.gs.collections.api.block.function.primitive.ObjectBooleanToObjectFunction;
import com.gs.collections.api.block.predicate.primitive.BooleanPredicate;
import com.gs.collections.api.block.predicate.primitive.ShortBooleanPredicate;
import com.gs.collections.api.block.predicate.primitive.ShortPredicate;
import com.gs.collections.api.block.procedure.primitive.BooleanProcedure;
import com.gs.collections.api.block.procedure.primitive.ShortBooleanProcedure;
import com.gs.collections.api.block.procedure.primitive.ShortProcedure;
import com.gs.collections.api.collection.MutableCollection;
import com.gs.collections.api.collection.primitive.ImmutableBooleanCollection;
import com.gs.collections.api.collection.primitive.MutableBooleanCollection;
import com.gs.collections.api.iterator.BooleanIterator;
import com.gs.collections.api.iterator.ShortIterator;
import com.gs.collections.api.list.primitive.MutableBooleanList;
import com.gs.collections.api.list.primitive.MutableShortList;
import com.gs.collections.api.map.primitive.ShortBooleanMap;
import com.gs.collections.api.map.primitive.ImmutableShortBooleanMap;
import com.gs.collections.api.map.primitive.MutableShortBooleanMap;
import com.gs.collections.api.set.MutableSet;
import com.gs.collections.api.set.primitive.ShortSet;
import com.gs.collections.api.set.primitive.ImmutableShortSet;
import com.gs.collections.api.set.primitive.MutableBooleanSet;
import com.gs.collections.api.set.primitive.MutableShortSet;
import com.gs.collections.impl.bag.mutable.primitive.BooleanHashBag;
import com.gs.collections.impl.bag.mutable.primitive.ShortHashBag;
import com.gs.collections.impl.block.factory.primitive.ShortPredicates;
import com.gs.collections.impl.collection.mutable.primitive.SynchronizedBooleanCollection;
import com.gs.collections.impl.collection.mutable.primitive.UnmodifiableBooleanCollection;
import com.gs.collections.impl.factory.Sets;
import com.gs.collections.impl.factory.primitive.BooleanLists;
import com.gs.collections.impl.factory.primitive.ShortBooleanMaps;
import com.gs.collections.impl.factory.primitive.ShortSets;
import com.gs.collections.impl.lazy.primitive.CollectShortToObjectIterable;
import com.gs.collections.impl.lazy.primitive.LazyBooleanIterableAdapter;
import com.gs.collections.impl.lazy.primitive.LazyShortIterableAdapter;
import com.gs.collections.impl.lazy.primitive.SelectShortIterable;
import com.gs.collections.impl.list.mutable.FastList;
import com.gs.collections.impl.list.mutable.primitive.BooleanArrayList;
import com.gs.collections.impl.list.mutable.primitive.ShortArrayList;
import com.gs.collections.impl.set.mutable.primitive.BooleanHashSet;
import com.gs.collections.impl.set.mutable.primitive.ShortHashSet;
import com.gs.collections.impl.set.mutable.primitive.SynchronizedShortSet;
import com.gs.collections.impl.set.mutable.primitive.UnmodifiableShortSet;

/**
 * This file was automatically generated from template file primitiveBooleanHashMap.stg.
 *
 * @since 3.0.
 */
public class ShortBooleanHashMap implements MutableShortBooleanMap, Externalizable
{
    static final boolean EMPTY_VALUE = false;
    private static final long serialVersionUID = 1L;
    private static final short EMPTY_KEY = (short) 0;
    private static final short REMOVED_KEY = (short) 1;

    private static final float DEFAULT_LOAD_FACTOR = 0.5f;
    private static final int DEFAULT_INITIAL_CAPACITY = 8;
    private int occupied;
    private int maxSize;

    private short[] keys;
    private BitSet values;
    private float loadFactor = DEFAULT_LOAD_FACTOR;
    private SentinelValues sentinelValues;

    public ShortBooleanHashMap()
    {
        this.allocate(DEFAULT_INITIAL_CAPACITY << 1);
    }

    public ShortBooleanHashMap(int initialCapacity)
    {
        this(initialCapacity, DEFAULT_LOAD_FACTOR);
    }

    public ShortBooleanHashMap(ShortBooleanMap map)
    {
        this(Math.max(map.size(), DEFAULT_INITIAL_CAPACITY), DEFAULT_LOAD_FACTOR);

        this.putAll(map);
    }

    public ShortBooleanHashMap(int initialCapacity, float loadFactor)
    {
        if (initialCapacity < 0)
        {
            throw new IllegalArgumentException("initial capacity cannot be less than 0");
        }
        this.loadFactor = loadFactor;
        this.init(this.fastCeil(initialCapacity / loadFactor));
    }

    public MutableShortBooleanMap asUnmodifiable()
    {
        return new UnmodifiableShortBooleanMap(this);
    }

    public MutableShortBooleanMap asSynchronized()
    {
        return new SynchronizedShortBooleanMap(this);
    }

    public ImmutableShortBooleanMap toImmutable()
    {
        return ShortBooleanMaps.immutable.withAll(this);
    }

    public static ShortBooleanHashMap newWithKeysValues(short key1, boolean value1)
    {
        return new ShortBooleanHashMap(1).withKeyValue(key1, value1);
    }

    public static ShortBooleanHashMap newWithKeysValues(short key1, boolean value1, short key2, boolean value2)
    {
        return new ShortBooleanHashMap(2).withKeysValues(key1, value1, key2, value2);
    }

    public static ShortBooleanHashMap newWithKeysValues(short key1, boolean value1, short key2, boolean value2, short key3, boolean value3)
    {
        return new ShortBooleanHashMap(3).withKeysValues(key1, value1, key2, value2, key3, value3);
    }

    public static ShortBooleanHashMap newWithKeysValues(short key1, boolean value1, short key2, boolean value2, short key3, boolean value3, short key4, boolean value4)
    {
        return new ShortBooleanHashMap(4).withKeysValues(key1, value1, key2, value2, key3, value3, key4, value4);
    }

    public ShortBooleanHashMap withKeyValue(short key1, boolean value1)
    {
        this.put(key1, value1);
        return this;
    }

    public ShortBooleanHashMap withKeysValues(short key1, boolean value1, short key2, boolean value2)
    {
        this.put(key1, value1);
        this.put(key2, value2);
        return this;
    }

    public ShortBooleanHashMap withKeysValues(short key1, boolean value1, short key2, boolean value2, short key3, boolean value3)
    {
        this.put(key1, value1);
        this.put(key2, value2);
        this.put(key3, value3);
        return this;
    }

    public ShortBooleanHashMap withKeysValues(short key1, boolean value1, short key2, boolean value2, short key3, boolean value3, short key4, boolean value4)
    {
        this.put(key1, value1);
        this.put(key2, value2);
        this.put(key3, value3);
        this.put(key4, value4);
        return this;
    }

    public ShortBooleanHashMap withoutKey(short key)
    {
        this.removeKey(key);
        return this;
    }

    public ShortBooleanHashMap withoutAllKeys(ShortIterable keys)
    {
        keys.forEach(new ShortProcedure()
        {
            public void value(short key)
            {
                ShortBooleanHashMap.this.removeKey(key);
            }
        });
        return this;
    }

    private int init(int initialCapacity)
    {
        int capacity = 1;
        while (capacity < initialCapacity)
        {
            capacity <<= 1;
        }
        return this.allocate(capacity);
    }

    private int fastCeil(float v)
    {
        int possibleResult = (int) v;
        if (v - possibleResult > 0.0F)
        {
            possibleResult++;
        }
        return possibleResult;
    }

    private static boolean isEmptyKey(short key)
    {
        return key == EMPTY_KEY;
    }

    private static boolean isRemovedKey(short key)
    {
        return key == REMOVED_KEY;
    }

    private static boolean isNonSentinel(short key)
    {
        return !isEmptyKey(key) && !isRemovedKey(key);
    }

    private int allocate(int capacity)
    {
        this.allocateTable(capacity);
        this.computeMaxSize(capacity);
        return capacity;
    }

    private void allocateTable(int sizeToAllocate)
    {
        this.keys = new short[sizeToAllocate];
        this.values = new BitSet(sizeToAllocate);
    }

    private void computeMaxSize(int capacity)
    {
        // need at least one free slot for open addressing
        this.maxSize = Math.min(capacity - 1, (int) (capacity * this.loadFactor));
    }

    public int size()
    {
        return this.occupied + (this.sentinelValues == null ? 0 : this.sentinelValues.size());
    }

    public boolean isEmpty()
    {
        return this.occupied == 0 && (this.sentinelValues == null || this.sentinelValues.size() == 0);
    }

    public boolean notEmpty()
    {
        return this.occupied != 0 || (this.sentinelValues != null && this.sentinelValues.size() != 0);
    }

    private void rehash()
    {
        this.rehash(this.keys.length << 1);
    }

    private void rehash(int newCapacity)
    {
        int oldLength = this.keys.length;
        short[] old = this.keys;
        BitSet oldValues = this.values;
        this.allocate(newCapacity);
        this.occupied = 0;

        for (int i = 0; i < oldLength; i++)
        {
            if (isNonSentinel(old[i]))
            {
                this.put(old[i], oldValues.get(i));
            }
        }
    }

    // exposed for testing
    int probe(short element)
    {
        int index = this.spread(element);
        short keyAtIndex = this.keys[index];

        if (keyAtIndex == element || keyAtIndex == EMPTY_KEY)
        {
            return index;
        }

        int removedIndex = keyAtIndex == REMOVED_KEY ? index : -1;
        int nextIndex = index;
        int probe = 17;

        // loop until an empty slot is reached
        while (true)
        {
            // Probe algorithm: 17*n*(n+1)/2 where n = number of collisions
            nextIndex += probe;
            probe += 17;
            nextIndex &= this.keys.length - 1;

            if (this.keys[nextIndex] == element)
            {
                return nextIndex;
            }
            if (this.keys[nextIndex] == REMOVED_KEY)
            {
                if (removedIndex == -1)
                {
                    removedIndex = nextIndex;
                }
            }
            else if (this.keys[nextIndex] == EMPTY_KEY)
            {
                return removedIndex == -1 ? nextIndex : removedIndex;
            }
        }
    }

    // exposed for testing
    int spread(short element)
    {
        int code = (int) element;
        code ^= 61 ^ (code >> 16);
        code += code << 3;
        code ^= code >> 4;
        code *= 0x27d4eb2d;
        code ^= code >> 15;
        return code & (this.keys.length - 1);
    }

    public void clear()
    {
        this.sentinelValues = null;
        this.occupied = 0;
        Arrays.fill(this.keys, EMPTY_KEY);
        this.values.clear();
    }

    public void put(short key, boolean value)
    {
        if (isEmptyKey(key))
        {
            if (this.sentinelValues == null)
            {
                this.sentinelValues = new SentinelValues();
            }
            this.sentinelValues.containsZeroKey = true;
            this.sentinelValues.zeroValue = value;
            return;
        }

        if (isRemovedKey(key))
        {
            if (this.sentinelValues == null)
            {
                this.sentinelValues = new SentinelValues();
            }
            this.sentinelValues.containsOneKey = true;
            this.sentinelValues.oneValue = value;
            return;
        }

        int index = this.probe(key);

        if (this.keys[index] == key)
        {
            // key already present in map
            this.values.set(index, value);
            return;
        }

        this.keys[index] = key;
        this.values.set(index, value);
        ++this.occupied;
        if (this.occupied > this.maxSize)
        {
            this.rehash();
        }
    }

    public void putAll(ShortBooleanMap map)
    {
        map.forEachKeyValue(new ShortBooleanProcedure()
        {
            public void value(short key, boolean value)
            {
                ShortBooleanHashMap.this.put(key, value);
            }
        });
    }

    public boolean containsKey(short key)
    {
        if (isEmptyKey(key))
        {
            return this.sentinelValues != null && this.sentinelValues.containsZeroKey;
        }
        if (isRemovedKey(key))
        {
            return this.sentinelValues != null && this.sentinelValues.containsOneKey;
        }
        return this.keys[this.probe(key)] == key;
    }

    public boolean containsValue(boolean value)
    {
        if (this.sentinelValues != null && this.sentinelValues.containsValue(value))
        {
            return true;
        }
        for (int i = 0; i < this.keys.length; i++)
        {
            if (isNonSentinel(this.keys[i]) && this.values.get(i) == value)
            {
                return true;
            }
        }
        return false;
    }

    public boolean contains(boolean value)
    {
        return this.containsValue(value);
    }

    public boolean containsAll(boolean... source)
    {
        for (boolean each : source)
        {
            if (!this.contains(each))
            {
                return false;
            }
        }
        return true;
    }

    public boolean containsAll(BooleanIterable source)
    {
        return source.allSatisfy(new BooleanPredicate()
        {
            public boolean accept(boolean value)
            {
                return ShortBooleanHashMap.this.contains(value);
            }
        });
    }

    public boolean get(short key)
    {
        return this.getIfAbsent(key, EMPTY_VALUE);
    }

    public boolean getIfAbsent(short key, boolean ifAbsent)
    {
        if (isEmptyKey(key))
        {
            if (this.sentinelValues == null || !this.sentinelValues.containsZeroKey)
            {
                return ifAbsent;
            }
            return this.sentinelValues.zeroValue;
        }
        if (isRemovedKey(key))
        {
            if (this.sentinelValues == null || !this.sentinelValues.containsOneKey)
            {
                return ifAbsent;
            }
            return this.sentinelValues.oneValue;
        }
        int index = this.probe(key);
        if (isNonSentinel(this.keys[index]))
        {
            return this.values.get(index);
        }
        return ifAbsent;
    }

    public boolean getOrThrow(short key)
    {
        if (isEmptyKey(key))
        {
            if (this.sentinelValues == null || !this.sentinelValues.containsZeroKey)
            {
                throw new IllegalStateException("Key " + key + " not present.");
            }
            return this.sentinelValues.zeroValue;
        }
        if (isRemovedKey(key))
        {
            if (this.sentinelValues == null || !this.sentinelValues.containsOneKey)
            {
                throw new IllegalStateException("Key " + key + " not present.");
            }
            return this.sentinelValues.oneValue;
        }
        int index = this.probe(key);
        if (isNonSentinel(this.keys[index]))
        {
            return this.values.get(index);
        }
        throw new IllegalStateException("Key " + key + " not present.");
    }

    public boolean getIfAbsentPut(short key, boolean value)
    {
        if (isEmptyKey(key))
        {
            if (this.sentinelValues == null)
            {
                this.sentinelValues = new SentinelValues();
                this.sentinelValues.containsZeroKey = true;
                this.sentinelValues.zeroValue = value;
                return value;
            }
            if (this.sentinelValues.containsZeroKey)
            {
                return this.sentinelValues.zeroValue;
            }
            this.sentinelValues.containsZeroKey = true;
            this.sentinelValues.zeroValue = value;
            return value;
        }
        if (isRemovedKey(key))
        {
            if (this.sentinelValues == null)
            {
                this.sentinelValues = new SentinelValues();
                this.sentinelValues.containsOneKey = true;
                this.sentinelValues.oneValue = value;
                return value;
            }
            if (this.sentinelValues.containsOneKey)
            {
                return this.sentinelValues.oneValue;
            }
            this.sentinelValues.containsOneKey = true;
            this.sentinelValues.oneValue = value;
            return value;
        }
        int index = this.probe(key);
        if (this.keys[index] == key)
        {
            return this.values.get(index);
        }
        this.keys[index] = key;
        this.occupied++;
        this.values.set(index, value);
        return value;
    }

    public boolean getIfAbsentPut(short key, BooleanFunction0 function)
    {
        if (isEmptyKey(key))
        {
            if (this.sentinelValues == null)
            {
                boolean value = function.value();
                this.sentinelValues = new SentinelValues();
                this.sentinelValues.containsZeroKey = true;
                this.sentinelValues.zeroValue = value;
                return value;
            }
            if (this.sentinelValues.containsZeroKey)
            {
                return this.sentinelValues.zeroValue;
            }
            boolean value = function.value();
            this.sentinelValues.containsZeroKey = true;
            this.sentinelValues.zeroValue = value;
            return value;
        }
        if (isRemovedKey(key))
        {
            if (this.sentinelValues == null)
            {
                boolean value = function.value();
                this.sentinelValues = new SentinelValues();
                this.sentinelValues.containsOneKey = true;
                this.sentinelValues.oneValue = value;
                return value;
            }
            if (this.sentinelValues.containsOneKey)
            {
                return this.sentinelValues.oneValue;
            }
            boolean value = function.value();
            this.sentinelValues.containsOneKey = true;
            this.sentinelValues.oneValue = value;
            return value;
        }
        int index = this.probe(key);
        if (this.keys[index] == key)
        {
            return this.values.get(index);
        }
        this.keys[index] = key;
        this.occupied++;
        boolean value = function.value();
        this.values.set(index, value);
        return value;
    }

    public <P> boolean getIfAbsentPutWith(short key, BooleanFunction<? super P> function, P parameter)
    {
        if (isEmptyKey(key))
        {
            if (this.sentinelValues == null)
            {
                boolean value = function.booleanValueOf(parameter);
                this.sentinelValues = new SentinelValues();
                this.sentinelValues.containsZeroKey = true;
                this.sentinelValues.zeroValue = value;
                return value;
            }
            if (this.sentinelValues.containsZeroKey)
            {
                return this.sentinelValues.zeroValue;
            }
            boolean value = function.booleanValueOf(parameter);
            this.sentinelValues.containsZeroKey = true;
            this.sentinelValues.zeroValue = value;
            return value;
        }
        if (isRemovedKey(key))
        {
            if (this.sentinelValues == null)
            {
                boolean value = function.booleanValueOf(parameter);
                this.sentinelValues = new SentinelValues();
                this.sentinelValues.containsOneKey = true;
                this.sentinelValues.oneValue = value;
                return value;
            }
            if (this.sentinelValues.containsOneKey)
            {
                return this.sentinelValues.oneValue;
            }
            boolean value = function.booleanValueOf(parameter);
            this.sentinelValues.containsOneKey = true;
            this.sentinelValues.oneValue = value;
            return value;
        }
        int index = this.probe(key);
        if (this.keys[index] == key)
        {
            return this.values.get(index);
        }
        this.keys[index] = key;
        this.occupied++;
        boolean value = function.booleanValueOf(parameter);
        this.values.set(index, value);
        return value;
    }

    public boolean getIfAbsentPutWithKey(short key, ShortToBooleanFunction function)
    {
        if (isEmptyKey(key))
        {
            if (this.sentinelValues == null)
            {
                boolean value = function.valueOf(key);
                this.sentinelValues = new SentinelValues();
                this.sentinelValues.containsZeroKey = true;
                this.sentinelValues.zeroValue = value;
                return value;
            }
            if (this.sentinelValues.containsZeroKey)
            {
                return this.sentinelValues.zeroValue;
            }
            boolean value = function.valueOf(key);
            this.sentinelValues.containsZeroKey = true;
            this.sentinelValues.zeroValue = value;
            return value;
        }
        if (isRemovedKey(key))
        {
            if (this.sentinelValues == null)
            {
                boolean value = function.valueOf(key);
                this.sentinelValues = new SentinelValues();
                this.sentinelValues.containsOneKey = true;
                this.sentinelValues.oneValue = value;
                return value;
            }
            if (this.sentinelValues.containsOneKey)
            {
                return this.sentinelValues.oneValue;
            }
            boolean value = function.valueOf(key);
            this.sentinelValues.containsOneKey = true;
            this.sentinelValues.oneValue = value;
            return value;
        }
        int index = this.probe(key);
        if (this.keys[index] == key)
        {
            return this.values.get(index);
        }
        this.keys[index] = key;
        this.occupied++;
        boolean value = function.valueOf(key);
        this.values.set(index, value);
        return value;
    }

    public boolean updateValue(short key, boolean initialValueIfAbsent, BooleanToBooleanFunction function)
    {
        if (isEmptyKey(key))
        {
            if (this.sentinelValues == null)
            {
                this.sentinelValues = new SentinelValues();
                this.sentinelValues.containsZeroKey = true;
                this.sentinelValues.zeroValue = function.valueOf(initialValueIfAbsent);
            }
            else if (this.sentinelValues.containsZeroKey)
            {
                this.sentinelValues.zeroValue = function.valueOf(this.sentinelValues.zeroValue);
            }
            else
            {
                this.sentinelValues.containsZeroKey = true;
                this.sentinelValues.zeroValue = function.valueOf(initialValueIfAbsent);
            }
            return this.sentinelValues.zeroValue;
        }
        if (isRemovedKey(key))
        {
            if (this.sentinelValues == null)
            {
                this.sentinelValues = new SentinelValues();
                this.sentinelValues.containsOneKey = true;
                this.sentinelValues.oneValue = function.valueOf(initialValueIfAbsent);
            }
            else if (this.sentinelValues.containsOneKey)
            {
                this.sentinelValues.oneValue = function.valueOf(this.sentinelValues.oneValue);
            }
            else
            {
                this.sentinelValues.containsOneKey = true;
                this.sentinelValues.oneValue = function.valueOf(initialValueIfAbsent);
            }
            return this.sentinelValues.oneValue;
        }
        int index = this.probe(key);
        if (this.keys[index] == key)
        {
            this.values.set(index, function.valueOf(this.values.get(index)));
        }
        else
        {
            this.keys[index] = key;
            this.occupied++;
            this.values.set(index, function.valueOf(initialValueIfAbsent));
        }
        return this.values.get(index);
    }

    public void removeKey(short key)
    {
        if (isEmptyKey(key))
        {
            if (this.sentinelValues == null || !this.sentinelValues.containsZeroKey)
            {
                return;
            }
            if (this.sentinelValues.containsOneKey)
            {
                this.sentinelValues.containsZeroKey = false;
                this.sentinelValues.zeroValue = EMPTY_VALUE;
            }
            else
            {
                this.sentinelValues = null;
            }
            return;
        }
        if (isRemovedKey(key))
        {
            if (this.sentinelValues == null || !this.sentinelValues.containsOneKey)
            {
                return;
            }
            if (this.sentinelValues.containsZeroKey)
            {
                this.sentinelValues.containsOneKey = false;
                this.sentinelValues.oneValue = EMPTY_VALUE;
            }
            else
            {
                this.sentinelValues = null;
            }
            return;
        }
        int index = this.probe(key);
        if (this.keys[index] == key)
        {
            this.keys[index] = REMOVED_KEY;
            this.occupied--;
            this.values.set(index, EMPTY_VALUE);
        }
    }

    public void remove(short key)
    {
        this.removeKey(key);
    }

    public boolean removeKeyIfAbsent(short key, boolean value)
    {
        if (isEmptyKey(key))
        {
            if (this.sentinelValues == null || !this.sentinelValues.containsZeroKey)
            {
                return value;
            }
            boolean oldValue = this.sentinelValues.zeroValue;
            if (this.sentinelValues.containsOneKey)
            {
                this.sentinelValues.containsZeroKey = false;
                this.sentinelValues.zeroValue = EMPTY_VALUE;
            }
            else
            {
                this.sentinelValues = null;
            }
            return oldValue;
        }
        if (isRemovedKey(key))
        {
            if (this.sentinelValues == null || !this.sentinelValues.containsOneKey)
            {
                return value;
            }
            boolean oldValue = this.sentinelValues.oneValue;
            if (this.sentinelValues.containsZeroKey)
            {
                this.sentinelValues.containsOneKey = false;
                this.sentinelValues.oneValue = EMPTY_VALUE;
            }
            else
            {
                this.sentinelValues = null;
            }
            return oldValue;
        }
        int index = this.probe(key);
        if (this.keys[index] == key)
        {
            this.keys[index] = REMOVED_KEY;
            this.occupied--;
            boolean oldValue = this.values.get(index);
            this.values.set(index, EMPTY_VALUE);
            return oldValue;
        }
        return value;
    }

    @Override
    public boolean equals(Object obj)
    {
        if (this == obj)
        {
            return true;
        }

        if (!(obj instanceof ShortBooleanMap))
        {
            return false;
        }

        ShortBooleanMap other = (ShortBooleanMap) obj;

        if (this.size() != other.size())
        {
            return false;
        }

        if (this.sentinelValues == null)
        {
            if (other.containsKey(EMPTY_KEY) || other.containsKey(REMOVED_KEY))
            {
                return false;
            }
        }
        else
        {
            if (this.sentinelValues.containsZeroKey && (!other.containsKey(EMPTY_KEY) || this.sentinelValues.zeroValue != other.getOrThrow(EMPTY_KEY)))
            {
                return false;
            }

            if (this.sentinelValues.containsOneKey && (!other.containsKey(REMOVED_KEY) || this.sentinelValues.oneValue != other.getOrThrow(REMOVED_KEY)))
            {
                return false;
            }
        }

        for (int i = 0; i < this.keys.length; i++)
        {
            short key = this.keys[i];
            if (isNonSentinel(key) && (!other.containsKey(key) || this.values.get(i) != other.getOrThrow(key)))
            {
                return false;
            }
        }
        return true;
    }

    @Override
    public int hashCode()
    {
        int result = 0;

        if (this.sentinelValues != null)
        {
            if (this.sentinelValues.containsZeroKey)
            {
                result += (int) EMPTY_KEY ^ (this.sentinelValues.zeroValue ? 1231 : 1237);
            }
            if (this.sentinelValues.containsOneKey)
            {
                result += (int) REMOVED_KEY ^ (this.sentinelValues.oneValue ? 1231 : 1237);
            }
        }
        for (int i = 0; i < this.keys.length; i++)
        {
            if (isNonSentinel(this.keys[i]))
            {
                result += (int) this.keys[i] ^ (this.values.get(i) ? 1231 : 1237);
            }
        }

        return result;
    }

    @Override
    public String toString()
    {
        return this.makeString("{", ", ", "}");
    }

    public BooleanIterator booleanIterator()
    {
        return new InternalBooleanIterator();
    }

    public void forEach(BooleanProcedure procedure)
    {
        this.forEachValue(procedure);
    }

    public void forEachValue(BooleanProcedure procedure)
    {
        if (this.sentinelValues != null)
        {
            if (this.sentinelValues.containsZeroKey)
            {
                procedure.value(this.sentinelValues.zeroValue);
            }
            if (this.sentinelValues.containsOneKey)
            {
                procedure.value(this.sentinelValues.oneValue);
            }
        }
        for (int i = 0; i < this.keys.length; i++)
        {
            if (isNonSentinel(this.keys[i]))
            {
                procedure.value(this.values.get(i));
            }
        }
    }

    public void forEachKey(ShortProcedure procedure)
    {
        if (this.sentinelValues != null)
        {
            if (this.sentinelValues.containsZeroKey)
            {
                procedure.value(EMPTY_KEY);
            }
            if (this.sentinelValues.containsOneKey)
            {
                procedure.value(REMOVED_KEY);
            }
        }
        for (short key : this.keys)
        {
            if (isNonSentinel(key))
            {
                procedure.value(key);
            }
        }
    }

    public void forEachKeyValue(ShortBooleanProcedure procedure)
    {
        if (this.sentinelValues != null)
        {
            if (this.sentinelValues.containsZeroKey)
            {
                procedure.value(EMPTY_KEY, this.sentinelValues.zeroValue);
            }
            if (this.sentinelValues.containsOneKey)
            {
                procedure.value(REMOVED_KEY, this.sentinelValues.oneValue);
            }
        }
        for (int i = 0; i < this.keys.length; i++)
        {
            if (isNonSentinel(this.keys[i]))
            {
                procedure.value(this.keys[i], this.values.get(i));
            }
        }
    }

    public ShortBooleanHashMap select(ShortBooleanPredicate predicate)
    {
        ShortBooleanHashMap result = new ShortBooleanHashMap();

        if (this.sentinelValues != null)
        {
            if (this.sentinelValues.containsZeroKey && predicate.accept(EMPTY_KEY, this.sentinelValues.zeroValue))
            {
                result.put(EMPTY_KEY, this.sentinelValues.zeroValue);
            }
            if (this.sentinelValues.containsOneKey && predicate.accept(REMOVED_KEY, this.sentinelValues.oneValue))
            {
                result.put(REMOVED_KEY, this.sentinelValues.oneValue);
            }
        }
        for (int i = 0; i < this.keys.length; i++)
        {
            if (isNonSentinel(this.keys[i]) && predicate.accept(this.keys[i], this.values.get(i)))
            {
                result.put(this.keys[i], this.values.get(i));
            }
        }

        return result;
    }

    public ShortBooleanHashMap reject(ShortBooleanPredicate predicate)
    {
        ShortBooleanHashMap result = new ShortBooleanHashMap();

        if (this.sentinelValues != null)
        {
            if (this.sentinelValues.containsZeroKey && !predicate.accept(EMPTY_KEY, this.sentinelValues.zeroValue))
            {
                result.put(EMPTY_KEY, this.sentinelValues.zeroValue);
            }
            if (this.sentinelValues.containsOneKey && !predicate.accept(REMOVED_KEY, this.sentinelValues.oneValue))
            {
                result.put(REMOVED_KEY, this.sentinelValues.oneValue);
            }
        }
        for (int i = 0; i < this.keys.length; i++)
        {
            if (isNonSentinel(this.keys[i]) && !predicate.accept(this.keys[i], this.values.get(i)))
            {
                result.put(this.keys[i], this.values.get(i));
            }
        }
        return result;
    }

    public <V> V injectInto(V injectedValue, ObjectBooleanToObjectFunction<? super V, ? extends V> function)
    {
        V result = injectedValue;

        if (this.sentinelValues != null)
        {
            if (this.sentinelValues.containsZeroKey)
            {
                result = function.valueOf(result, this.sentinelValues.zeroValue);
            }
            if (this.sentinelValues.containsOneKey)
            {
                result = function.valueOf(result, this.sentinelValues.oneValue);
            }
        }

        for (int i = 0; i < this.keys.length; i++)
        {
            if (isNonSentinel(this.keys[i]))
            {
                result = function.valueOf(result, this.values.get(i));
            }
        }

        return result;
    }

    public String makeString()
    {
        return this.makeString(", ");
    }

    public String makeString(String separator)
    {
        return this.makeString("", separator, "");
    }

    public String makeString(String start, String separator, String end)
    {
        Appendable stringBuilder = new StringBuilder();
        this.appendString(stringBuilder, start, separator, end);
        return stringBuilder.toString();
    }

    public void appendString(Appendable appendable)
    {
        this.appendString(appendable, ", ");
    }

    public void appendString(Appendable appendable, String separator)
    {
        this.appendString(appendable, "", separator, "");
    }

    public void appendString(Appendable appendable, String start, String separator, String end)
    {
        try
        {
            appendable.append(start);

            boolean first = true;

            if (this.sentinelValues != null)
            {
                if (this.sentinelValues.containsZeroKey)
                {
                    appendable.append(String.valueOf(EMPTY_KEY)).append("=").append(String.valueOf(this.sentinelValues.zeroValue));
                    first = false;
                }
                if (this.sentinelValues.containsOneKey)
                {
                    if (!first)
                    {
                        appendable.append(separator);
                    }
                    appendable.append(String.valueOf(REMOVED_KEY)).append("=").append(String.valueOf(this.sentinelValues.oneValue));
                    first = false;
                }
            }
            for (int i = 0; i < this.keys.length; i++)
            {
                short key = this.keys[i];
                if (isNonSentinel(key))
                {
                    if (!first)
                    {
                        appendable.append(separator);
                    }
                    appendable.append(String.valueOf(key)).append("=").append(String.valueOf(this.values.get(i)));
                    first = false;
                }
            }
            appendable.append(end);
        }
        catch (IOException e)
        {
            throw new RuntimeException(e);
        }
    }

    public MutableBooleanCollection select(BooleanPredicate predicate)
    {
        BooleanArrayList result = new BooleanArrayList();

        if (this.sentinelValues != null)
        {
            if (this.sentinelValues.containsZeroKey && predicate.accept(this.sentinelValues.zeroValue))
            {
                result.add(this.sentinelValues.zeroValue);
            }
            if (this.sentinelValues.containsOneKey && predicate.accept(this.sentinelValues.oneValue))
            {
                result.add(this.sentinelValues.oneValue);
            }
        }
        for (int i = 0; i < this.keys.length; i++)
        {
            if (isNonSentinel(this.keys[i]) && predicate.accept(this.values.get(i)))
            {
                result.add(this.values.get(i));
            }
        }

        return result;
    }

    public MutableBooleanCollection reject(BooleanPredicate predicate)
    {
        BooleanArrayList result = new BooleanArrayList();
        if (this.sentinelValues != null)
        {
            if (this.sentinelValues.containsZeroKey && !predicate.accept(this.sentinelValues.zeroValue))
            {
                result.add(this.sentinelValues.zeroValue);
            }
            if (this.sentinelValues.containsOneKey && !predicate.accept(this.sentinelValues.oneValue))
            {
                result.add(this.sentinelValues.oneValue);
            }
        }
        for (int i = 0; i < this.keys.length; i++)
        {
            if (isNonSentinel(this.keys[i]) && !predicate.accept(this.values.get(i)))
            {
                result.add(this.values.get(i));
            }
        }
        return result;
    }

    public boolean detectIfNone(BooleanPredicate predicate, boolean value)
    {
        if (this.sentinelValues != null)
        {
            if (this.sentinelValues.containsZeroKey && predicate.accept(this.sentinelValues.zeroValue))
            {
                return this.sentinelValues.zeroValue;
            }
            if (this.sentinelValues.containsOneKey && predicate.accept(this.sentinelValues.oneValue))
            {
                return this.sentinelValues.oneValue;
            }
        }
        for (int i = 0; i < this.keys.length; i++)
        {
            if (isNonSentinel(this.keys[i]) && predicate.accept(this.values.get(i)))
            {
                return this.values.get(i);
            }
        }
        return value;
    }

    public <V> MutableCollection<V> collect(BooleanToObjectFunction<? extends V> function)
    {
        FastList<V> target = FastList.newList(this.size());
        if (this.sentinelValues != null)
        {
            if (this.sentinelValues.containsZeroKey)
            {
                target.add(function.valueOf(this.sentinelValues.zeroValue));
            }
            if (this.sentinelValues.containsOneKey)
            {
                target.add(function.valueOf(this.sentinelValues.oneValue));
            }
        }
        for (int i = 0; i < this.keys.length; i++)
        {
            if (isNonSentinel(this.keys[i]))
            {
                target.add(function.valueOf(this.values.get(i)));
            }
        }
        return target;
    }

    public int count(BooleanPredicate predicate)
    {
        int count = 0;
        if (this.sentinelValues != null)
        {
            if (this.sentinelValues.containsZeroKey && predicate.accept(this.sentinelValues.zeroValue))
            {
                count++;
            }
            if (this.sentinelValues.containsOneKey && predicate.accept(this.sentinelValues.oneValue))
            {
                count++;
            }
        }
        for (int i = 0; i < this.keys.length; i++)
        {
            if (isNonSentinel(this.keys[i]) && predicate.accept(this.values.get(i)))
            {
                count++;
            }
        }
        return count;
    }

    public boolean anySatisfy(BooleanPredicate predicate)
    {
        if (this.sentinelValues != null)
        {
            if (this.sentinelValues.containsZeroKey && predicate.accept(this.sentinelValues.zeroValue))
            {
                return true;
            }
            if (this.sentinelValues.containsOneKey && predicate.accept(this.sentinelValues.oneValue))
            {
                return true;
            }
        }
        for (int i = 0; i < this.keys.length; i++)
        {
            if (isNonSentinel(this.keys[i]) && predicate.accept(this.values.get(i)))
            {
                return true;
            }
        }
        return false;
    }

    public boolean allSatisfy(BooleanPredicate predicate)
    {
        if (this.sentinelValues != null)
        {
            if (this.sentinelValues.containsZeroKey && !predicate.accept(this.sentinelValues.zeroValue))
            {
                return false;
            }
            if (this.sentinelValues.containsOneKey && !predicate.accept(this.sentinelValues.oneValue))
            {
                return false;
            }
        }
        for (int i = 0; i < this.keys.length; i++)
        {
            if (isNonSentinel(this.keys[i]) && !predicate.accept(this.values.get(i)))
            {
                return false;
            }
        }
        return true;
    }

    public boolean noneSatisfy(BooleanPredicate predicate)
    {
        return !this.anySatisfy(predicate);
    }

    public boolean[] toArray()
    {
        boolean[] array = new boolean[this.size()];
        int index = 0;

        if (this.sentinelValues != null)
        {
            if (this.sentinelValues.containsZeroKey)
            {
                array[index] = this.sentinelValues.zeroValue;
                index++;
            }
            if (this.sentinelValues.containsOneKey)
            {
                array[index] = this.sentinelValues.oneValue;
                index++;
            }
        }
        for (int i = 0; i < this.keys.length; i++)
        {
            if (isNonSentinel(this.keys[i]))
            {
                array[index] = this.values.get(i);
                index++;
            }
        }

        return array;
    }

    public MutableBooleanList toList()
    {
        return BooleanArrayList.newList(this);
    }

    public MutableBooleanSet toSet()
    {
        return BooleanHashSet.newSet(this);
    }

    public MutableBooleanBag toBag()
    {
        return BooleanHashBag.newBag(this);
    }

    public LazyBooleanIterable asLazy()
    {
        return new LazyBooleanIterableAdapter(this);
    }

    public LazyShortIterable keysView()
    {
        return new KeysView();
    }

    public void writeExternal(ObjectOutput out) throws IOException
    {
        out.writeInt(this.size());
        out.writeFloat(this.loadFactor);
        if (this.sentinelValues != null)
        {
            if (this.sentinelValues.containsZeroKey)
            {
                out.writeShort(EMPTY_KEY);
                out.writeBoolean(this.sentinelValues.zeroValue);
            }
            if (this.sentinelValues.containsOneKey)
            {
                out.writeShort(REMOVED_KEY);
                out.writeBoolean(this.sentinelValues.oneValue);
            }
        }
        for (int i = 0; i < this.keys.length; i++)
        {
            if (isNonSentinel(this.keys[i]))
            {
                out.writeShort(this.keys[i]);
                out.writeBoolean(this.values.get(i));
            }
        }
    }

    public void readExternal(ObjectInput in) throws IOException, ClassNotFoundException
    {
        int size = in.readInt();
        this.loadFactor = in.readFloat();
        this.init(Math.max((int) (size / this.loadFactor) + 1,
                DEFAULT_INITIAL_CAPACITY));
        for (int i = 0; i < size; i++)
        {
            this.put(in.readShort(), in.readBoolean());
        }
    }

    private static final class SentinelValues
    {
        private boolean containsZeroKey;
        private boolean containsOneKey;
        private boolean zeroValue;
        private boolean oneValue;

        public int size()
        {
            return (this.containsZeroKey ? 1 : 0) + (this.containsOneKey ? 1 : 0);
        }

        public boolean containsValue(boolean value)
        {
            boolean valueEqualsZeroValue = this.containsZeroKey && this.zeroValue == value;
            boolean valueEqualsOneValue = this.containsOneKey && this.oneValue == value;
            return valueEqualsZeroValue || valueEqualsOneValue;
        }
    }

    private class InternalBooleanIterator implements BooleanIterator
    {
        private int count;
        private int position;
        private boolean handledZero;
        private boolean handledOne;

        public boolean hasNext()
        {
            return this.count < ShortBooleanHashMap.this.size();
        }

        public boolean next()
        {
            if (!this.hasNext())
            {
                throw new NoSuchElementException("next() called, but the iterator is exhausted");
            }
            this.count++;

            if (!this.handledZero)
            {
                this.handledZero = true;
                if (ShortBooleanHashMap.this.containsKey(EMPTY_KEY))
                {
                    return ShortBooleanHashMap.this.sentinelValues.zeroValue;
                }
            }
            if (!this.handledOne)
            {
                this.handledOne = true;
                if (ShortBooleanHashMap.this.containsKey(REMOVED_KEY))
                {
                    return ShortBooleanHashMap.this.sentinelValues.oneValue;
                }
            }

            short[] keys = ShortBooleanHashMap.this.keys;
            while (!isNonSentinel(keys[this.position]))
            {
                this.position++;
            }
            boolean result = ShortBooleanHashMap.this.values.get(this.position);
            this.position++;
            return result;
        }
    }

    private class KeysView implements LazyShortIterable
    {
        public boolean isEmpty()
        {
            return ShortBooleanHashMap.this.isEmpty();
        }

        public boolean notEmpty()
        {
            return ShortBooleanHashMap.this.notEmpty();
        }

        public int size()
        {
            return ShortBooleanHashMap.this.size();
        }

        public boolean contains(short key)
        {
            return ShortBooleanHashMap.this.containsKey(key);
        }

        public boolean containsAll(short... keys)
        {
            for (short key : keys)
            {
                if (!ShortBooleanHashMap.this.containsKey(key))
                {
                    return false;
                }
            }
            return true;
        }

        public boolean containsAll(ShortIterable source)
        {
            return source.allSatisfy(new ShortPredicate()
            {
                public boolean accept(short key)
                {
                    return ShortBooleanHashMap.this.containsKey(key);
                }
            });
        }

        public ShortIterator shortIterator()
        {
            return new KeySetIterator();
        }

        public void forEach(ShortProcedure procedure)
        {
            ShortBooleanHashMap.this.forEachKey(procedure);
        }

        @Override
        public String toString()
        {
            return this.makeString("[", ", ", "]");
        }

        public String makeString()
        {
            return this.makeString(", ");
        }

        public String makeString(String separator)
        {
            return this.makeString("", separator, "");
        }

        public String makeString(String start, String separator, String end)
        {
            Appendable stringBuilder = new StringBuilder();
            this.appendString(stringBuilder, start, separator, end);
            return stringBuilder.toString();
        }

        public void appendString(Appendable appendable)
        {
            this.appendString(appendable, ", ");
        }

        public void appendString(Appendable appendable, String separator)
        {
            this.appendString(appendable, "", separator, "");
        }

        public void appendString(Appendable appendable, String start, String separator, String end)
        {
            try
            {
                appendable.append(start);
                boolean first = true;
                if (ShortBooleanHashMap.this.sentinelValues != null)
                {
                    if (ShortBooleanHashMap.this.sentinelValues.containsZeroKey)
                    {
                        appendable.append(String.valueOf(EMPTY_KEY));
                        first = false;
                    }
                    if (ShortBooleanHashMap.this.sentinelValues.containsOneKey)
                    {
                        if (!first)
                        {
                            appendable.append(separator);
                        }
                        appendable.append(String.valueOf(REMOVED_KEY));
                        first = false;
                    }
                }
                for (short key : ShortBooleanHashMap.this.keys)
                {
                    if (isNonSentinel(key))
                    {
                        if (!first)
                        {
                            appendable.append(separator);
                        }
                        appendable.append(String.valueOf(key));
                        first = false;
                    }
                }
                appendable.append(end);
            }
            catch (IOException e)
            {
                throw new RuntimeException(e);
            }
        }

        public int count(ShortPredicate predicate)
        {
            int count = 0;
            if (ShortBooleanHashMap.this.sentinelValues != null)
            {
                if (ShortBooleanHashMap.this.sentinelValues.containsZeroKey && predicate.accept(EMPTY_KEY))
                {
                    count++;
                }
                if (ShortBooleanHashMap.this.sentinelValues.containsOneKey && predicate.accept(REMOVED_KEY))
                {
                    count++;
                }
            }
            for (short key : ShortBooleanHashMap.this.keys)
            {
                if (isNonSentinel(key) && predicate.accept(key))
                {
                    count++;
                }
            }
            return count;
        }

        public boolean anySatisfy(ShortPredicate predicate)
        {
            if (ShortBooleanHashMap.this.sentinelValues != null)
            {
                if (ShortBooleanHashMap.this.sentinelValues.containsZeroKey && predicate.accept(EMPTY_KEY))
                {
                    return true;
                }
                if (ShortBooleanHashMap.this.sentinelValues.containsOneKey && predicate.accept(REMOVED_KEY))
                {
                    return true;
                }
            }
            for (short key : ShortBooleanHashMap.this.keys)
            {
                if (isNonSentinel(key) && predicate.accept(key))
                {
                    return true;
                }
            }
            return false;
        }

        public boolean allSatisfy(ShortPredicate predicate)
        {
            if (ShortBooleanHashMap.this.sentinelValues != null)
            {
                if (ShortBooleanHashMap.this.sentinelValues.containsZeroKey && !predicate.accept(EMPTY_KEY))
                {
                    return false;
                }
                if (ShortBooleanHashMap.this.sentinelValues.containsOneKey && !predicate.accept(REMOVED_KEY))
                {
                    return false;
                }
            }
            for (short key : ShortBooleanHashMap.this.keys)
            {
                if (isNonSentinel(key) && !predicate.accept(key))
                {
                    return false;
                }
            }
            return true;
        }

        public boolean noneSatisfy(ShortPredicate predicate)
        {
            return !this.anySatisfy(predicate);
        }

        public LazyShortIterable select(ShortPredicate predicate)
        {
            return new SelectShortIterable(this, predicate);
        }

        public LazyShortIterable reject(ShortPredicate predicate)
        {
            return new SelectShortIterable(this, ShortPredicates.not(predicate));
        }

        public short detectIfNone(ShortPredicate predicate, short value)
        {
            if (ShortBooleanHashMap.this.sentinelValues != null)
            {
                if (ShortBooleanHashMap.this.sentinelValues.containsZeroKey && predicate.accept(EMPTY_KEY))
                {
                    return EMPTY_KEY;
                }
                if (ShortBooleanHashMap.this.sentinelValues.containsOneKey && predicate.accept(REMOVED_KEY))
                {
                    return REMOVED_KEY;
                }
            }
            for (short key : ShortBooleanHashMap.this.keys)
            {
                if (isNonSentinel(key) && predicate.accept(key))
                {
                    return key;
                }
            }
            return value;
        }

        public <V> LazyIterable<V> collect(ShortToObjectFunction<? extends V> function)
        {
            return new CollectShortToObjectIterable<V>(this, function);
        }

        public long sum()
        {
            long result = 0L;

            if (ShortBooleanHashMap.this.sentinelValues != null)
            {
                if (ShortBooleanHashMap.this.sentinelValues.containsZeroKey)
                {
                    result += EMPTY_KEY;
                }
                if (ShortBooleanHashMap.this.sentinelValues.containsOneKey)
                {
                    result += REMOVED_KEY;
                }
            }
            for (short key : ShortBooleanHashMap.this.keys)
            {
                if (isNonSentinel(key))
                {
                    result += key;
                }
            }

            return result;
        }

        public short max()
        {
            if (this.isEmpty())
            {
                throw new NoSuchElementException();
            }
            ShortIterator iterator = this.shortIterator();
            short max = iterator.next();
            while (iterator.hasNext())
            {
                short value = iterator.next();
                if (max < value)
                {
                    max = value;
                }
            }
            return max;
        }

        public short min()
        {
            if (this.isEmpty())
            {
                throw new NoSuchElementException();
            }
            ShortIterator iterator = this.shortIterator();
            short min = iterator.next();
            while (iterator.hasNext())
            {
                short value = iterator.next();
                if (value < min)
                {
                    min = value;
                }
            }
            return min;
        }

        public short minIfEmpty(short defaultValue)
        {
            if (this.isEmpty())
            {
                return defaultValue;
            }
            return this.min();
        }

        public short maxIfEmpty(short defaultValue)
        {
            if (this.isEmpty())
            {
                return defaultValue;
            }
            return this.max();
        }

        public double average()
        {
            if (this.isEmpty())
            {
                throw new ArithmeticException();
            }
            return (double) this.sum() / (double) this.size();
        }

        public double median()
        {
            if (this.isEmpty())
            {
                throw new ArithmeticException();
            }
            short[] sortedArray = this.toSortedArray();
            int middleIndex = sortedArray.length >> 1;
            if (sortedArray.length > 1 && (sortedArray.length & 1) == 0)
            {
                short first = sortedArray[middleIndex];
                short second = sortedArray[middleIndex - 1];
                return ((double) first + (double) second) / 2.0;
            }
            return (double) sortedArray[middleIndex];
        }

        public short[] toSortedArray()
        {
            short[] array = this.toArray();
            Arrays.sort(array);
            return array;
        }

        public short[] toArray()
        {
            int size = ShortBooleanHashMap.this.size();
            final short[] result = new short[size];
            ShortBooleanHashMap.this.forEachKey(new ShortProcedure()
            {
                private int index;

                public void value(short each)
                {
                    result[this.index] = each;
                    this.index++;
                }
            });
            return result;
        }

        public <T> T injectInto(T injectedValue, ObjectShortToObjectFunction<? super T, ? extends T> function)
        {
            T result = injectedValue;
            if (ShortBooleanHashMap.this.sentinelValues != null)
            {
                if (ShortBooleanHashMap.this.sentinelValues.containsZeroKey)
                {
                    result = function.valueOf(result, EMPTY_KEY);
                }
                if (ShortBooleanHashMap.this.sentinelValues.containsOneKey)
                {
                    result = function.valueOf(result, REMOVED_KEY);
                }
            }
            for (int i = 0; i < ShortBooleanHashMap.this.keys.length; i++)
            {
                if (isNonSentinel(ShortBooleanHashMap.this.keys[i]))
                {
                    result = function.valueOf(result, ShortBooleanHashMap.this.keys[i]);
                }
            }
            return result;
        }

        public MutableShortList toList()
        {
            return ShortArrayList.newList(this);
        }

        public MutableShortList toSortedList()
        {
            return ShortArrayList.newList(this).sortThis();
        }

        public MutableShortSet toSet()
        {
            return ShortHashSet.newSet(this);
        }

        public MutableShortBag toBag()
        {
            return ShortHashBag.newBag(this);
        }

        public LazyShortIterable asLazy()
        {
            return this;
        }
    }

    public MutableShortSet keySet()
    {
        return new KeySet();
    }

    private class KeySet implements MutableShortSet
    {
        public ShortIterator shortIterator()
        {
            return new KeySetIterator();
        }

        public void forEach(ShortProcedure procedure)
        {
            ShortBooleanHashMap.this.forEachKey(procedure);
        }

        public int count(ShortPredicate predicate)
        {
            int count = 0;
            if (ShortBooleanHashMap.this.sentinelValues != null)
            {
                if (ShortBooleanHashMap.this.sentinelValues.containsZeroKey && predicate.accept(EMPTY_KEY))
                {
                    count++;
                }
                if (ShortBooleanHashMap.this.sentinelValues.containsOneKey && predicate.accept(REMOVED_KEY))
                {
                    count++;
                }
            }
            for (short key : ShortBooleanHashMap.this.keys)
            {
                if (isNonSentinel(key) && predicate.accept(key))
                {
                    count++;
                }
            }
            return count;
        }

        public boolean anySatisfy(ShortPredicate predicate)
        {
            if (ShortBooleanHashMap.this.sentinelValues != null)
            {
                if (ShortBooleanHashMap.this.sentinelValues.containsZeroKey && predicate.accept(EMPTY_KEY))
                {
                    return true;
                }
                if (ShortBooleanHashMap.this.sentinelValues.containsOneKey && predicate.accept(REMOVED_KEY))
                {
                    return true;
                }
            }
            for (short key : ShortBooleanHashMap.this.keys)
            {
                if (isNonSentinel(key) && predicate.accept(key))
                {
                    return true;
                }
            }
            return false;
        }

        public boolean allSatisfy(ShortPredicate predicate)
        {
            if (ShortBooleanHashMap.this.sentinelValues != null)
            {
                if (ShortBooleanHashMap.this.sentinelValues.containsZeroKey && !predicate.accept(EMPTY_KEY))
                {
                    return false;
                }
                if (ShortBooleanHashMap.this.sentinelValues.containsOneKey && !predicate.accept(REMOVED_KEY))
                {
                    return false;
                }
            }
            for (short key : ShortBooleanHashMap.this.keys)
            {
                if (isNonSentinel(key) && !predicate.accept(key))
                {
                    return false;
                }
            }
            return true;
        }

        public boolean noneSatisfy(ShortPredicate predicate)
        {
            if (ShortBooleanHashMap.this.sentinelValues != null)
            {
                if (ShortBooleanHashMap.this.sentinelValues.containsZeroKey && predicate.accept(EMPTY_KEY))
                {
                    return false;
                }
                if (ShortBooleanHashMap.this.sentinelValues.containsOneKey && predicate.accept(REMOVED_KEY))
                {
                    return false;
                }
            }
            for (short key : ShortBooleanHashMap.this.keys)
            {
                if (isNonSentinel(key) && predicate.accept(key))
                {
                    return false;
                }
            }
            return true;
        }

        public boolean add(short element)
        {
            throw new UnsupportedOperationException();
        }

        public boolean addAll(short... source)
        {
            throw new UnsupportedOperationException();
        }

        public boolean addAll(ShortIterable source)
        {
            throw new UnsupportedOperationException();
        }

        public boolean remove(short key)
        {
            int oldSize = ShortBooleanHashMap.this.size();
            ShortBooleanHashMap.this.removeKey(key);
            return oldSize != ShortBooleanHashMap.this.size();
        }

        public boolean removeAll(ShortIterable source)
        {
            int oldSize = ShortBooleanHashMap.this.size();
            ShortIterator iterator = source.shortIterator();
            while (iterator.hasNext())
            {
                ShortBooleanHashMap.this.removeKey(iterator.next());
            }
            return oldSize != ShortBooleanHashMap.this.size();
        }

        public boolean removeAll(short... source)
        {
            int oldSize = ShortBooleanHashMap.this.size();
            for (short item : source)
            {
                ShortBooleanHashMap.this.removeKey(item);
            }
            return oldSize != ShortBooleanHashMap.this.size();
        }

        public void clear()
        {
            ShortBooleanHashMap.this.clear();
        }

        public MutableShortSet select(ShortPredicate predicate)
        {
            MutableShortSet result = new ShortHashSet();
            if (ShortBooleanHashMap.this.sentinelValues != null)
            {
                if (ShortBooleanHashMap.this.sentinelValues.containsZeroKey && predicate.accept(EMPTY_KEY))
                {
                    result.add(EMPTY_KEY);
                }
                if (ShortBooleanHashMap.this.sentinelValues.containsOneKey && predicate.accept(REMOVED_KEY))
                {
                    result.add(REMOVED_KEY);
                }
            }
            for (short key : ShortBooleanHashMap.this.keys)
            {
                if (isNonSentinel(key) && predicate.accept(key))
                {
                    result.add(key);
                }
            }
            return result;
        }

        public MutableShortSet reject(ShortPredicate predicate)
        {
            MutableShortSet result = new ShortHashSet();
            if (ShortBooleanHashMap.this.sentinelValues != null)
            {
                if (ShortBooleanHashMap.this.sentinelValues.containsZeroKey && !predicate.accept(EMPTY_KEY))
                {
                    result.add(EMPTY_KEY);
                }
                if (ShortBooleanHashMap.this.sentinelValues.containsOneKey && !predicate.accept(REMOVED_KEY))
                {
                    result.add(REMOVED_KEY);
                }
            }
            for (short key : ShortBooleanHashMap.this.keys)
            {
                if (isNonSentinel(key) && !predicate.accept(key))
                {
                    result.add(key);
                }
            }
            return result;
        }

        public MutableShortSet with(short element)
        {
            throw new UnsupportedOperationException();
        }

        public MutableShortSet without(short element)
        {
            throw new UnsupportedOperationException();
        }

        public MutableShortSet withAll(ShortIterable elements)
        {
            throw new UnsupportedOperationException();
        }

        public MutableShortSet withoutAll(ShortIterable elements)
        {
            throw new UnsupportedOperationException();
        }

        public short detectIfNone(ShortPredicate predicate, short ifNone)
        {
            if (ShortBooleanHashMap.this.sentinelValues != null)
            {
                if (ShortBooleanHashMap.this.sentinelValues.containsZeroKey && predicate.accept(EMPTY_KEY))
                {
                    return EMPTY_KEY;
                }
                if (ShortBooleanHashMap.this.sentinelValues.containsOneKey && predicate.accept(REMOVED_KEY))
                {
                    return REMOVED_KEY;
                }
            }
            for (short key : ShortBooleanHashMap.this.keys)
            {
                if (isNonSentinel(key) && predicate.accept(key))
                {
                    return key;
                }
            }
            return ifNone;
        }

        public <V> MutableSet<V> collect(ShortToObjectFunction<? extends V> function)
        {
            MutableSet<V> result = Sets.mutable.with();
            if (ShortBooleanHashMap.this.sentinelValues != null)
            {
                if (ShortBooleanHashMap.this.sentinelValues.containsZeroKey)
                {
                    result.add(function.valueOf(EMPTY_KEY));
                }
                if (ShortBooleanHashMap.this.sentinelValues.containsOneKey)
                {
                    result.add(function.valueOf(REMOVED_KEY));
                }
            }
            for (short key : ShortBooleanHashMap.this.keys)
            {
                if (isNonSentinel(key))
                {
                    result.add(function.valueOf(key));
                }
            }
            return result;
        }

        public MutableShortSet asUnmodifiable()
        {
            return UnmodifiableShortSet.of(this);
        }

        public MutableShortSet asSynchronized()
        {
            return SynchronizedShortSet.of(this);
        }

        public long sum()
        {
            long sum = 0L;
            if (ShortBooleanHashMap.this.sentinelValues != null)
            {
                if (ShortBooleanHashMap.this.sentinelValues.containsZeroKey)
                {
                    sum += EMPTY_KEY;
                }
                if (ShortBooleanHashMap.this.sentinelValues.containsOneKey)
                {
                    sum += REMOVED_KEY;
                }
            }
            for (short key : ShortBooleanHashMap.this.keys)
            {
                if (isNonSentinel(key))
                {
                    sum += key;
                }
            }
            return sum;
        }

        public short max()
        {
            if (ShortBooleanHashMap.this.isEmpty())
            {
                throw new NoSuchElementException();
            }

            short max = 0;
            boolean isMaxSet = false;

            if (ShortBooleanHashMap.this.sentinelValues != null)
            {
                if (ShortBooleanHashMap.this.sentinelValues.containsZeroKey)
                {
                    max = EMPTY_KEY;
                    isMaxSet = true;
                }
                if (ShortBooleanHashMap.this.sentinelValues.containsOneKey && (!isMaxSet || max < REMOVED_KEY))
                {
                    max = REMOVED_KEY;
                    isMaxSet = true;
                }
            }
            for (int i = 0; i < ShortBooleanHashMap.this.keys.length; i++)
            {
                if (isNonSentinel(ShortBooleanHashMap.this.keys[i]) && (!isMaxSet || max < ShortBooleanHashMap.this.keys[i]))
                {
                    max = ShortBooleanHashMap.this.keys[i];
                    isMaxSet = true;
                }
            }
            return max;
        }

        public short maxIfEmpty(short defaultValue)
        {
            if (ShortBooleanHashMap.this.isEmpty())
            {
                return defaultValue;
            }

            return this.max();
        }

        public short min()
        {
            if (ShortBooleanHashMap.this.isEmpty())
            {
                throw new NoSuchElementException();
            }

            short min = 0;
            boolean isMinSet = false;

            if (ShortBooleanHashMap.this.sentinelValues != null)
            {
                if (ShortBooleanHashMap.this.sentinelValues.containsZeroKey)
                {
                    min = EMPTY_KEY;
                    isMinSet = true;
                }
                if (ShortBooleanHashMap.this.sentinelValues.containsOneKey && (!isMinSet || REMOVED_KEY < min))
                {
                    min = REMOVED_KEY;
                    isMinSet = true;
                }
            }
            for (int i = 0; i < ShortBooleanHashMap.this.keys.length; i++)
            {
                if (isNonSentinel(ShortBooleanHashMap.this.keys[i]) && (!isMinSet || ShortBooleanHashMap.this.keys[i] < min))
                {
                    min = ShortBooleanHashMap.this.keys[i];
                    isMinSet = true;
                }
            }
            return min;
        }

        public short minIfEmpty(short defaultValue)
        {
            if (ShortBooleanHashMap.this.isEmpty())
            {
                return defaultValue;
            }

            return this.min();
        }

        public double average()
        {
            if (this.isEmpty())
            {
                throw new ArithmeticException();
            }
            return (double) this.sum() / (double) this.size();
        }

        public double median()
        {
            if (this.isEmpty())
            {
                throw new ArithmeticException();
            }
            short[] sortedArray = this.toSortedArray();
            int middleIndex = sortedArray.length >> 1;
            if (sortedArray.length > 1 && (sortedArray.length & 1) == 0)
            {
                short first = sortedArray[middleIndex];
                short second = sortedArray[middleIndex - 1];
                return ((double) first + (double) second) / 2.0;
            }
            return (double) sortedArray[middleIndex];
        }

        public short[] toSortedArray()
        {
            short[] array = this.toArray();
            Arrays.sort(array);
            return array;
        }

        public MutableShortList toSortedList()
        {
            return ShortArrayList.newList(this).sortThis();
        }

        public short[] toArray()
        {
            int size = ShortBooleanHashMap.this.size();
            final short[] result = new short[size];
            ShortBooleanHashMap.this.forEachKey(new ShortProcedure()
            {
                private int index;

                public void value(short each)
                {
                    result[this.index] = each;
                    this.index++;
                }
            });
            return result;
        }

        public boolean contains(short value)
        {
            return ShortBooleanHashMap.this.containsKey(value);
        }

        public boolean containsAll(short... source)
        {
            for (short item : source)
            {
                if (!ShortBooleanHashMap.this.containsKey(item))
                {
                    return false;
                }
            }
            return true;
        }

        public boolean containsAll(ShortIterable source)
        {
            ShortIterator iterator = source.shortIterator();
            while (iterator.hasNext())
            {
                if (!ShortBooleanHashMap.this.containsKey(iterator.next()))
                {
                    return false;
                }
            }
            return true;
        }

        public MutableShortList toList()
        {
            return ShortArrayList.newList(this);
        }

        public MutableShortSet toSet()
        {
            return ShortHashSet.newSet(this);
        }

        public MutableShortBag toBag()
        {
            return ShortHashBag.newBag(this);
        }

        public LazyShortIterable asLazy()
        {
            return new LazyShortIterableAdapter(this);
        }

        public <T> T injectInto(T injectedValue, ObjectShortToObjectFunction<? super T, ? extends T> function)
        {
            T result = injectedValue;
            if (ShortBooleanHashMap.this.sentinelValues != null)
            {
                if (ShortBooleanHashMap.this.sentinelValues.containsZeroKey)
                {
                    result = function.valueOf(result, EMPTY_KEY);
                }
                if (ShortBooleanHashMap.this.sentinelValues.containsOneKey)
                {
                    result = function.valueOf(result, REMOVED_KEY);
                }
            }
            for (int i = 0; i < ShortBooleanHashMap.this.keys.length; i++)
            {
                if (isNonSentinel(ShortBooleanHashMap.this.keys[i]))
                {
                    result = function.valueOf(result, ShortBooleanHashMap.this.keys[i]);
                }
            }
            return result;
        }

        public ShortSet freeze()
        {
            throw new UnsupportedOperationException();
        }

        public ImmutableShortSet toImmutable()
        {
            return ShortSets.immutable.withAll(this);
        }

        public int size()
        {
            return ShortBooleanHashMap.this.size();
        }

        public boolean isEmpty()
        {
            return ShortBooleanHashMap.this.isEmpty();
        }

        public boolean notEmpty()
        {
            return ShortBooleanHashMap.this.notEmpty();
        }

        @Override
        public boolean equals(Object obj)
        {
            if (this == obj)
            {
                return true;
            }

            if (!(obj instanceof ShortSet))
            {
                return false;
            }

            ShortSet other = (ShortSet) obj;
            return this.size() == other.size() && this.containsAll(other.toArray());
        }

        @Override
        public int hashCode()
        {
            int result = 0;

            if (ShortBooleanHashMap.this.sentinelValues != null)
            {
                if (ShortBooleanHashMap.this.sentinelValues.containsZeroKey)
                {
                    result += (int) EMPTY_KEY;
                }
                if (ShortBooleanHashMap.this.sentinelValues.containsOneKey)
                {
                    result += (int) REMOVED_KEY;
                }
            }
            for (int i = 0; i < ShortBooleanHashMap.this.keys.length; i++)
            {
                if (isNonSentinel(ShortBooleanHashMap.this.keys[i]))
                {
                    result += (int) ShortBooleanHashMap.this.keys[i];
                }
            }

            return result;
        }

        @Override
        public String toString()
        {
            return this.makeString("[", ", ", "]");
        }

        public String makeString()
        {
            return this.makeString(", ");
        }

        public String makeString(String separator)
        {
            return this.makeString("", separator, "");
        }

        public String makeString(String start, String separator, String end)
        {
            Appendable stringBuilder = new StringBuilder();
            this.appendString(stringBuilder, start, separator, end);
            return stringBuilder.toString();
        }

        public void appendString(Appendable appendable)
        {
            this.appendString(appendable, ", ");
        }

        public void appendString(Appendable appendable, String separator)
        {
            this.appendString(appendable, "", separator, "");
        }

        public void appendString(Appendable appendable, String start, String separator, String end)
        {
            try
            {
                appendable.append(start);
                boolean first = true;
                if (ShortBooleanHashMap.this.sentinelValues != null)
                {
                    if (ShortBooleanHashMap.this.sentinelValues.containsZeroKey)
                    {
                        appendable.append(String.valueOf(EMPTY_KEY));
                        first = false;
                    }
                    if (ShortBooleanHashMap.this.sentinelValues.containsOneKey)
                    {
                        if (!first)
                        {
                            appendable.append(separator);
                        }
                        appendable.append(String.valueOf(REMOVED_KEY));
                        first = false;
                    }
                }
                for (short key : ShortBooleanHashMap.this.keys)
                {
                    if (isNonSentinel(key))
                    {
                        if (!first)
                        {
                            appendable.append(separator);
                        }
                        appendable.append(String.valueOf(key));
                        first = false;
                    }
                }
                appendable.append(end);
            }
            catch (IOException e)
            {
                throw new RuntimeException(e);
            }
        }
    }

    public MutableBooleanCollection values()
    {
        return new ValuesCollection();
    }

    private class ValuesCollection implements MutableBooleanCollection
    {
        public void clear()
        {
            ShortBooleanHashMap.this.clear();
        }

        public MutableBooleanCollection select(BooleanPredicate predicate)
        {
            return ShortBooleanHashMap.this.select(predicate);
        }

        public MutableBooleanCollection reject(BooleanPredicate predicate)
        {
            return ShortBooleanHashMap.this.reject(predicate);
        }

        public boolean detectIfNone(BooleanPredicate predicate, boolean ifNone)
        {
            return ShortBooleanHashMap.this.detectIfNone(predicate, ifNone);
        }

        public <V> MutableCollection<V> collect(BooleanToObjectFunction<? extends V> function)
        {
            return ShortBooleanHashMap.this.collect(function);
        }

        public MutableBooleanCollection with(boolean element)
        {
            throw new UnsupportedOperationException();
        }

        public MutableBooleanCollection without(boolean element)
        {
            throw new UnsupportedOperationException();
        }

        public MutableBooleanCollection withAll(BooleanIterable elements)
        {
            throw new UnsupportedOperationException();
        }

        public MutableBooleanCollection withoutAll(BooleanIterable elements)
        {
            throw new UnsupportedOperationException();
        }

        public MutableBooleanCollection asSynchronized()
        {
            return SynchronizedBooleanCollection.of(this);
        }

        public MutableBooleanCollection asUnmodifiable()
        {
            return UnmodifiableBooleanCollection.of(this);
        }

        public ImmutableBooleanCollection toImmutable()
        {
            return BooleanLists.immutable.withAll(this);
        }

        public boolean contains(boolean value)
        {
            return ShortBooleanHashMap.this.containsValue(value);
        }

        public boolean containsAll(boolean... source)
        {
            return ShortBooleanHashMap.this.containsAll(source);
        }

        public boolean containsAll(BooleanIterable source)
        {
            return ShortBooleanHashMap.this.containsAll(source);
        }

        public MutableBooleanList toList()
        {
            return ShortBooleanHashMap.this.toList();
        }

        public MutableBooleanSet toSet()
        {
            return ShortBooleanHashMap.this.toSet();
        }

        public MutableBooleanBag toBag()
        {
            return ShortBooleanHashMap.this.toBag();
        }

        public LazyBooleanIterable asLazy()
        {
            return new LazyBooleanIterableAdapter(this);
        }

        public <V> V injectInto(V injectedValue, ObjectBooleanToObjectFunction<? super V, ? extends V> function)
        {
            return ShortBooleanHashMap.this.injectInto(injectedValue, function);
        }

        public boolean isEmpty()
        {
            return ShortBooleanHashMap.this.isEmpty();
        }

        public boolean notEmpty()
        {
            return ShortBooleanHashMap.this.notEmpty();
        }

        public String makeString()
        {
            return this.makeString(", ");
        }

        public String makeString(String separator)
        {
            return this.makeString("", separator, "");
        }

        public String makeString(String start, String separator, String end)
        {
            Appendable stringBuilder = new StringBuilder();
            this.appendString(stringBuilder, start, separator, end);
            return stringBuilder.toString();
        }

        public void appendString(Appendable appendable)
        {
            this.appendString(appendable, ", ");
        }

        public void appendString(Appendable appendable, String separator)
        {
            this.appendString(appendable, "", separator, "");
        }

        public void appendString(Appendable appendable, String start, String separator, String end)
        {
            try
            {
                appendable.append(start);

                boolean first = true;

                if (ShortBooleanHashMap.this.sentinelValues != null)
                {
                    if (ShortBooleanHashMap.this.sentinelValues.containsZeroKey)
                    {
                        appendable.append(String.valueOf(ShortBooleanHashMap.this.sentinelValues.zeroValue));
                        first = false;
                    }
                    if (ShortBooleanHashMap.this.sentinelValues.containsOneKey)
                    {
                        if (!first)
                        {
                            appendable.append(separator);
                        }
                        appendable.append(String.valueOf(ShortBooleanHashMap.this.sentinelValues.oneValue));
                        first = false;
                    }
                }
                for (int i = 0; i < ShortBooleanHashMap.this.keys.length; i++)
                {
                    short key = ShortBooleanHashMap.this.keys[i];
                    if (isNonSentinel(key))
                    {
                        if (!first)
                        {
                            appendable.append(separator);
                        }
                        appendable.append(String.valueOf(ShortBooleanHashMap.this.values.get(i)));
                        first = false;
                    }
                }
                appendable.append(end);
            }
            catch (IOException e)
            {
                throw new RuntimeException(e);
            }
        }

        public BooleanIterator booleanIterator()
        {
            return ShortBooleanHashMap.this.booleanIterator();
        }

        public void forEach(BooleanProcedure procedure)
        {
            ShortBooleanHashMap.this.forEach(procedure);
        }

        public int count(BooleanPredicate predicate)
        {
            return ShortBooleanHashMap.this.count(predicate);
        }

        public boolean anySatisfy(BooleanPredicate predicate)
        {
            return ShortBooleanHashMap.this.anySatisfy(predicate);
        }

        public boolean allSatisfy(BooleanPredicate predicate)
        {
            return ShortBooleanHashMap.this.allSatisfy(predicate);
        }

        public boolean noneSatisfy(BooleanPredicate predicate)
        {
            return ShortBooleanHashMap.this.noneSatisfy(predicate);
        }

        public boolean add(boolean element)
        {
            throw new UnsupportedOperationException();
        }

        public boolean addAll(boolean... source)
        {
            throw new UnsupportedOperationException();
        }

        public boolean addAll(BooleanIterable source)
        {
            throw new UnsupportedOperationException();
        }

        public boolean remove(boolean item)
        {
            int oldSize = ShortBooleanHashMap.this.size();

            if (ShortBooleanHashMap.this.sentinelValues != null)
            {
                if (ShortBooleanHashMap.this.sentinelValues.containsZeroKey && item == ShortBooleanHashMap.this.sentinelValues.zeroValue)
                {
                    ShortBooleanHashMap.this.removeKey(EMPTY_KEY);
                }
                if (ShortBooleanHashMap.this.sentinelValues.containsOneKey && item == ShortBooleanHashMap.this.sentinelValues.oneValue)
                {
                    ShortBooleanHashMap.this.removeKey(REMOVED_KEY);
                }
            }
            for (int i = 0; i < ShortBooleanHashMap.this.keys.length; i++)
            {
                if (isNonSentinel(ShortBooleanHashMap.this.keys[i]) && item == ShortBooleanHashMap.this.values.get(i))
                {
                    ShortBooleanHashMap.this.removeKey(ShortBooleanHashMap.this.keys[i]);
                }
            }
            return oldSize != ShortBooleanHashMap.this.size();
        }

        public boolean removeAll(BooleanIterable source)
        {
            int oldSize = ShortBooleanHashMap.this.size();

            BooleanIterator iterator = source.booleanIterator();
            while (iterator.hasNext())
            {
                this.remove(iterator.next());
            }
            return oldSize != ShortBooleanHashMap.this.size();
        }

        public boolean removeAll(boolean... source)
        {
            int oldSize = ShortBooleanHashMap.this.size();

            for (boolean item : source)
            {
                this.remove(item);
            }
            return oldSize != ShortBooleanHashMap.this.size();
        }

        public int size()
        {
            return ShortBooleanHashMap.this.size();
        }

        public boolean[] toArray()
        {
            return ShortBooleanHashMap.this.toArray();
        }
    }

    private class KeySetIterator implements ShortIterator
    {
        private int count;
        private int position;
        private boolean handledZero;
        private boolean handledOne;

        public boolean hasNext()
        {
            return this.count < ShortBooleanHashMap.this.size();
        }

        public short next()
        {
            if (!this.hasNext())
            {
                throw new NoSuchElementException("next() called, but the iterator is exhausted");
            }
            this.count++;

            if (!this.handledZero)
            {
                this.handledZero = true;
                if (ShortBooleanHashMap.this.containsKey(EMPTY_KEY))
                {
                    return EMPTY_KEY;
                }
            }
            if (!this.handledOne)
            {
                this.handledOne = true;
                if (ShortBooleanHashMap.this.containsKey(REMOVED_KEY))
                {
                    return REMOVED_KEY;
                }
            }

            short[] keys = ShortBooleanHashMap.this.keys;
            while (!isNonSentinel(keys[this.position]))
            {
                this.position++;
            }
            short result = keys[this.position];
            this.position++;
            return result;
        }
    }
}
