/*
 * Copyright 2013 Goldman Sachs.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.gs.collections.impl.set.immutable.primitive;

import java.io.IOException;
import java.io.Serializable;

import com.gs.collections.api.CharIterable;
import com.gs.collections.api.LazyCharIterable;
import com.gs.collections.api.bag.primitive.MutableCharBag;
import com.gs.collections.api.set.ImmutableSet;
import com.gs.collections.api.set.primitive.ImmutableCharSet;
import com.gs.collections.api.set.primitive.CharSet;
import com.gs.collections.api.set.primitive.MutableCharSet;
import com.gs.collections.api.block.function.primitive.CharToObjectFunction;
import com.gs.collections.api.block.function.primitive.ObjectCharToObjectFunction;
import com.gs.collections.api.block.predicate.primitive.CharPredicate;
import com.gs.collections.api.block.procedure.primitive.CharIntProcedure;
import com.gs.collections.api.block.procedure.primitive.CharProcedure;
import com.gs.collections.api.iterator.CharIterator;
import com.gs.collections.api.list.primitive.MutableCharList;
import com.gs.collections.impl.bag.mutable.primitive.CharHashBag;
import com.gs.collections.impl.set.mutable.UnifiedSet;
import com.gs.collections.impl.set.mutable.primitive.CharHashSet;
import com.gs.collections.impl.factory.primitive.CharSets;
import com.gs.collections.impl.lazy.primitive.LazyCharIterableAdapter;
import com.gs.collections.impl.list.mutable.primitive.CharArrayList;
import net.jcip.annotations.Immutable;

/**
 * ImmutableCharSingletonSet is an optimization for {@link ImmutableCharSet} of size 1.
 * This file was automatically generated from template file immutablePrimitiveSingletonSet.stg.
 *
 * @since 4.0.
 */
@Immutable
final class ImmutableCharSingletonSet implements ImmutableCharSet, Serializable
{
    private static final long serialVersionUID = 1L;

    private final char element;

    ImmutableCharSingletonSet (char element)
    {
        this.element = element;
    }

    public ImmutableCharSet newWith(char element)
    {
        return CharSets.immutable.with(this.element, element);
    }

    public ImmutableCharSet newWithout(char element)
    {
        return this.element == element ? CharSets.immutable.with() : this;
    }

    public ImmutableCharSet newWithAll(CharIterable elements)
    {
        return CharHashSet.newSet(elements).with(this.element).toImmutable();
    }

    public ImmutableCharSet newWithoutAll(CharIterable elements)
    {
        return elements.contains(this.element) ? CharSets.immutable.with() : this;
    }

    public int size()
    {
        return 1;
    }

    public boolean isEmpty()
    {
        return false;
    }

    public boolean notEmpty()
    {
        return true;
    }

    public boolean contains(char value)
    {
        return this.element == value;
    }

    public boolean containsAll(CharIterable source)
    {
        for (CharIterator iterator = source.charIterator(); iterator.hasNext(); )
        {
            if (this.element != iterator.next())
            {
                return false;
            }
        }
        return true;
    }

    public boolean containsAll(char... source)
    {
        for (char value : source)
        {
            if (this.element != value)
            {
                return false;
            }
        }
        return true;
    }

    public void forEach(CharProcedure procedure)
    {
        procedure.value(this.element);
    }

    public ImmutableCharSet select(CharPredicate predicate)
    {
        return predicate.accept(this.element) ? CharHashSet.newSetWith(this.element).toImmutable()
                : new CharHashSet().toImmutable();
    }

    public ImmutableCharSet reject(CharPredicate predicate)
    {
        return predicate.accept(this.element) ? new CharHashSet().toImmutable()
                : CharHashSet.newSetWith(this.element).toImmutable();
    }

    public <V> ImmutableSet<V> collect(CharToObjectFunction<? extends V> function)
    {
        return UnifiedSet.<V>newSetWith(function.valueOf(this.element)).toImmutable();
    }

    public MutableCharList toList()
    {
        return CharArrayList.newListWith(this.element);
    }

    public int sizeDistinct()
    {
        return 1;
    }

    public int occurrencesOf(char item)
    {
        return this.element == item ? 1 : 0;
    }

    public void forEachWithOccurrences(CharIntProcedure charIntProcedure)
    {
        charIntProcedure.value(this.element, 1);
    }

    public char detectIfNone(CharPredicate predicate, char ifNone)
    {
        return predicate.accept(this.element) ? this.element : ifNone;
    }

    public int count(CharPredicate predicate)
    {
        return predicate.accept(this.element) ? 1 : 0;
    }

    public boolean anySatisfy(CharPredicate predicate)
    {
        return predicate.accept(this.element);
    }

    public long sum()
    {
        return this.element;
    }

    public char min()
    {
        return this.element;
    }

    public char max()
    {
        return this.element;
    }

    public char maxIfEmpty(char defaultValue)
    {
        return this.element;
    }

    public char minIfEmpty(char defaultValue)
    {
        return this.element;
    }

    public double average()
    {
        return this.element;
    }

    public double median()
    {
        return this.element;
    }

    public char[] toSortedArray()
    {
        return new char[]{this.element};
    }

    public MutableCharList toSortedList()
    {
        return CharArrayList.newListWith(this.element);
    }

    public boolean noneSatisfy(CharPredicate predicate)
    {
        return !predicate.accept(this.element);
    }

    public boolean allSatisfy(CharPredicate predicate)
    {
        return predicate.accept(this.element);
    }

    public <T> T injectInto(T injectedValue, ObjectCharToObjectFunction<? super T, ? extends T> function)
    {
        return function.valueOf(injectedValue, this.element);
    }

    @Override
    public boolean equals(Object obj)
    {
        if (obj == this)
        {
            return true;
        }
        if (!(obj instanceof CharSet))
        {
            return false;
        }
        CharSet set = (CharSet) obj;
        if (set.size() != 1)
        {
            return false;
        }
        return set.contains(this.element);
    }

    @Override
    public int hashCode()
    {
        return (int) this.element ^ 1;
    }

    public MutableCharSet toSet()
    {
        return CharHashSet.newSetWith(this.element);
    }

    public MutableCharBag toBag()
    {
        return CharHashBag.newBagWith(this.element);
    }

    public CharSet freeze()
    {
        return this;
    }

    public ImmutableCharSet toImmutable()
    {
        return this;
    }

    public LazyCharIterable asLazy()
    {
        return new LazyCharIterableAdapter(this);
    }

    public char[] toArray()
    {
        return new char[]{this.element};
    }

    @Override
    public String toString()
    {
        return '[' + this.makeString() + ']';
    }

    public String makeString()
    {
        return this.makeString(", ");
    }

    public String makeString(String separator)
    {
        return this.makeString("", separator, "");
    }

    public String makeString(String start, String separator, String end)
    {
        Appendable stringBuilder = new StringBuilder();
        this.appendString(stringBuilder, start, separator, end);
        return stringBuilder.toString();
    }

    public void appendString(Appendable appendable)
    {
        this.appendString(appendable, ", ");
    }

    public void appendString(Appendable appendable, String separator)
    {
        this.appendString(appendable, "", separator, "");
    }

    public void appendString(Appendable appendable, String start, String separator, String end)
    {
        try
        {
            appendable.append(start);
            appendable.append(String.valueOf(this.element));
            appendable.append(end);
        }
        catch (IOException e)
        {
            throw new RuntimeException(e);
        }
    }

    public CharIterator charIterator()
    {
        return CharHashSet.newSetWith(this.element).charIterator();
    }
}

