/*
 * Copyright 2013 Goldman Sachs.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.gs.collections.impl.set.mutable.primitive;

import java.io.Externalizable;
import java.io.IOException;
import java.io.ObjectInput;
import java.io.ObjectOutput;
import java.io.Serializable;
import java.util.Arrays;
import java.util.NoSuchElementException;

import com.gs.collections.api.DoubleIterable;
import com.gs.collections.api.LazyDoubleIterable;
import com.gs.collections.api.bag.primitive.MutableDoubleBag;
import com.gs.collections.api.block.function.primitive.DoubleToObjectFunction;
import com.gs.collections.api.block.function.primitive.ObjectDoubleToObjectFunction;
import com.gs.collections.api.block.predicate.primitive.DoublePredicate;
import com.gs.collections.api.block.procedure.primitive.DoubleProcedure;
import com.gs.collections.api.iterator.DoubleIterator;
import com.gs.collections.api.list.primitive.MutableDoubleList;
import com.gs.collections.api.set.ImmutableSet;
import com.gs.collections.api.set.MutableSet;
import com.gs.collections.api.set.primitive.ImmutableDoubleSet;
import com.gs.collections.api.set.primitive.MutableDoubleSet;
import com.gs.collections.api.set.primitive.DoubleSet;
import com.gs.collections.impl.bag.mutable.primitive.DoubleHashBag;
import com.gs.collections.impl.factory.primitive.DoubleSets;
import com.gs.collections.impl.lazy.primitive.LazyDoubleIterableAdapter;
import com.gs.collections.impl.list.mutable.primitive.DoubleArrayList;
import com.gs.collections.impl.set.immutable.primitive.ImmutableDoubleSetSerializationProxy;
import com.gs.collections.impl.set.mutable.UnifiedSet;

/**
 * This file was automatically generated from template file primitiveHashSet.stg.
 *
 * @since 3.0.
 */
public final class DoubleHashSet implements MutableDoubleSet, Externalizable
{
    private static final long serialVersionUID = 1L;
    private static final int OCCUPIED_DATA_RATIO = 2;
    private static final int OCCUPIED_SENTINEL_RATIO = 4;
    private static final int DEFAULT_INITIAL_CAPACITY = 8;
    private static final double EMPTY = 0.0;
    private static final double REMOVED = 1.0;

    private double[] table;
    private int occupiedWithData;
    private int occupiedWithSentinels;
    // The 32 bits of this integer indicate whether the items 0.0 to 31.0 are present in the set.
    private int zeroToThirtyOne;
    private int zeroToThirtyOneOccupied;
    private transient boolean copyOnWrite;

    public DoubleHashSet()
    {
        this.allocateTable(DEFAULT_INITIAL_CAPACITY << 1);
    }

    public DoubleHashSet(int initialCapacity)
    {
        if (initialCapacity < 0)
        {
            throw new IllegalArgumentException("initial capacity cannot be less than 0");
        }
        int capacity = this.smallestPowerOfTwoGreaterThan(this.fastCeil(initialCapacity * OCCUPIED_DATA_RATIO));
        this.allocateTable(capacity);
    }

    private int smallestPowerOfTwoGreaterThan(int n)
    {
        return n > 1 ? Integer.highestOneBit(n - 1) << 1 : 1;
    }

    private int fastCeil(float v)
    {
        int possibleResult = (int) v;
        if (v - possibleResult > 0.0F)
        {
            possibleResult++;
        }
        return possibleResult;
    }

    public DoubleHashSet(DoubleHashSet set)
    {
        this.occupiedWithData = set.occupiedWithData;
        this.occupiedWithSentinels = set.occupiedWithSentinels;
        this.zeroToThirtyOneOccupied = set.zeroToThirtyOneOccupied;
        this.zeroToThirtyOne = set.zeroToThirtyOne;
        this.allocateTable(set.table.length);

        System.arraycopy(set.table, 0, this.table, 0, set.table.length);
    }

    public static DoubleHashSet newSet(DoubleIterable source)
    {
        if (source instanceof DoubleHashSet)
        {
            return new DoubleHashSet((DoubleHashSet) source);
        }

        return DoubleHashSet.newSetWith(source.toArray());
    }

    public static DoubleHashSet newSetWith(double... source)
    {
        DoubleHashSet result = new DoubleHashSet();
        result.addAll(source);
        return result;
    }

    private static boolean isBetweenZeroAndThirtyOne(double value)
    {
        return Double.compare(value, 0.0) >= 0 && Double.compare(value, 31.0) <= 0 && Double.compare(value, Math.floor(value)) == 0;
    }

    @Override
    public boolean equals(Object obj)
    {
        if (this == obj)
        {
            return true;
        }

        if (!(obj instanceof DoubleSet))
        {
            return false;
        }

        DoubleSet other = (DoubleSet) obj;
        return this.size() == other.size() && this.containsAll(other.toArray());
    }

    @Override
    public int hashCode()
    {
        int result = 0;
        int zeroToThirtyOne = this.zeroToThirtyOne;
        while (zeroToThirtyOne != 0)
        {
            double value = (double) Integer.numberOfTrailingZeros(zeroToThirtyOne);
            result += (int) (Double.doubleToLongBits(value) ^ Double.doubleToLongBits(value) >>> 32);
            zeroToThirtyOne &= ~(1 << (int) value);
        }
        if (this.table != null)
        {
            for (int i = 0; i < this.table.length; i++)
            {
                if (isNonSentinel(this.table[i]))
                {
                    result += (int) (Double.doubleToLongBits(this.table[i]) ^ Double.doubleToLongBits(this.table[i]) >>> 32);
                }
            }
        }
        return result;
    }

    @Override
    public String toString()
    {
        return this.makeString("[", ", ", "]");
    }

    public int size()
    {
        return this.occupiedWithData + this.zeroToThirtyOneOccupied;
    }

    public boolean isEmpty()
    {
        return this.size() == 0;
    }

    public boolean notEmpty()
    {
        return this.size() != 0;
    }

    public String makeString()
    {
        return this.makeString(", ");
    }

    public String makeString(String separator)
    {
        return this.makeString("", separator, "");
    }

    public String makeString(String start, String separator, String end)
    {
        Appendable stringBuilder = new StringBuilder();
        this.appendString(stringBuilder, start, separator, end);
        return stringBuilder.toString();
    }

    public void appendString(Appendable appendable)
    {
        this.appendString(appendable, ", ");
    }

    public void appendString(Appendable appendable, String separator)
    {
        this.appendString(appendable, "", separator, "");
    }

    public void appendString(Appendable appendable, String start, String separator, String end)
    {
        try
        {
            appendable.append(start);

            int count = 0;
            int zeroToThirtyOne = this.zeroToThirtyOne;
            while (zeroToThirtyOne != 0)
            {
                double value = (double) Integer.numberOfTrailingZeros(zeroToThirtyOne);
                if (count > 0)
                {
                    appendable.append(separator);
                }
                count++;
                appendable.append(String.valueOf(value));
                zeroToThirtyOne &= ~(1 << (int) value);
            }

            for (double value : this.table)
            {
                if (isNonSentinel(value))
                {
                    if (count > 0)
                    {
                        appendable.append(separator);
                    }
                    count++;
                    appendable.append(String.valueOf(value));
                }
            }
            appendable.append(end);
        }
        catch (IOException e)
        {
            throw new RuntimeException(e);
        }
    }

    public boolean add(double element)
    {
        if (isBetweenZeroAndThirtyOne(element))
        {
            int initial = this.zeroToThirtyOne;
            this.zeroToThirtyOne |= 1 << (int) element;
            if (this.zeroToThirtyOne != initial)
            {
                this.zeroToThirtyOneOccupied++;
                return true;
            }
            return false;
        }

        int index = this.probe(element);

        if (Double.compare(this.table[index], element) == 0)
        {
            // element already present in set
            return false;
        }

        if (this.copyOnWrite)
        {
            this.copyTable();
        }
        if (Double.compare(this.table[index], REMOVED) == 0)
        {
            --this.occupiedWithSentinels;
        }
        this.table[index] = element;
        ++this.occupiedWithData;
        if (this.occupiedWithData > this.maxOccupiedWithData())
        {
            this.rehashAndGrow();
        }
        return true;
    }

    public boolean addAll(double... source)
    {
        int oldSize = this.size();
        for (double item : source)
        {
            this.add(item);
        }
        return this.size() != oldSize;
    }

    public boolean addAll(DoubleIterable source)
    {
        if (source.isEmpty())
        {
            return false;
        }
        int oldSize = this.size();
        if (source instanceof DoubleHashSet)
        {
            DoubleHashSet hashSet = (DoubleHashSet) source;
            this.zeroToThirtyOne |= hashSet.zeroToThirtyOne;
            this.zeroToThirtyOneOccupied = Integer.bitCount(this.zeroToThirtyOne);
            for (double item : hashSet.table)
            {
                if (isNonSentinel(item))
                {
                    this.add(item);
                }
            }
        }
        else
        {
            DoubleIterator iterator = source.doubleIterator();
            while (iterator.hasNext())
            {
                double item = iterator.next();
                this.add(item);
            }
        }
        return this.size() != oldSize;
    }

    public boolean remove(double value)
    {
        if (isBetweenZeroAndThirtyOne(value))
        {
            int initial = this.zeroToThirtyOne;
            this.zeroToThirtyOne &= ~(1 << (int) value);
            if (this.zeroToThirtyOne == initial)
            {
                return false;
            }
            this.zeroToThirtyOneOccupied--;
            return true;
        }
        int index = this.probe(value);
        if (Double.compare(this.table[index], value) == 0)
        {
            if (this.copyOnWrite)
            {
                this.copyTable();
            }
            this.table[index] = REMOVED;
            this.occupiedWithData--;
            this.occupiedWithSentinels++;
            if (this.occupiedWithSentinels > this.maxOccupiedWithSentinels())
            {
                this.rehash();
            }

            return true;
        }
        return false;
    }

    public boolean removeAll(DoubleIterable source)
    {
        if (source.isEmpty())
        {
            return false;
        }
        int oldSize = this.size();
        if (source instanceof DoubleHashSet)
        {
            DoubleHashSet hashSet = (DoubleHashSet) source;
            this.zeroToThirtyOne &= ~hashSet.zeroToThirtyOne;
            this.zeroToThirtyOneOccupied = Integer.bitCount(this.zeroToThirtyOne);
            for (double item : hashSet.table)
            {
                if (isNonSentinel(item))
                {
                    this.remove(item);
                }
            }
        }
        else
        {
            DoubleIterator iterator = source.doubleIterator();
            while (iterator.hasNext())
            {
                double item = iterator.next();
                this.remove(item);
            }
        }
        return this.size() != oldSize;
    }

    public boolean removeAll(double... source)
    {
        if (source.length == 0)
        {
            return false;
        }
        int oldSize = this.size();
        for (double item : source)
        {
            this.remove(item);
        }
        return this.size() != oldSize;
    }

    public void clear()
    {
        this.zeroToThirtyOneOccupied = 0;
        this.occupiedWithData = 0;
        this.occupiedWithSentinels = 0;

        this.zeroToThirtyOne = 0;
        if (this.copyOnWrite)
        {
            this.table = new double[this.table.length];
            this.copyOnWrite = false;
        }
        else
        {
            Arrays.fill(this.table, EMPTY);
        }
    }

    public DoubleHashSet with(double element)
    {
        this.add(element);
        return this;
    }

    public DoubleHashSet without(double element)
    {
        this.remove(element);
        return this;
    }

    public DoubleHashSet withAll(DoubleIterable elements)
    {
        this.addAll(elements.toArray());
        return this;
    }

    public DoubleHashSet withoutAll(DoubleIterable elements)
    {
        this.removeAll(elements);
        return this;
    }

    public MutableDoubleSet asUnmodifiable()
    {
        return new UnmodifiableDoubleSet(this);
    }

    public MutableDoubleSet asSynchronized()
    {
        return new SynchronizedDoubleSet(this);
    }

    public ImmutableDoubleSet toImmutable()
    {
        if (this.size() == 0)
        {
            return DoubleSets.immutable.with();
        }
        if (this.size() == 1)
        {
            return DoubleSets.immutable.with(this.doubleIterator().next());
        }
        DoubleHashSet mutableSet = DoubleHashSet.newSetWith(this.toArray());
        return new ImmutableDoubleHashSet(mutableSet.table, mutableSet.occupiedWithData, mutableSet.zeroToThirtyOne, mutableSet.zeroToThirtyOneOccupied);
    }

    public DoubleIterator doubleIterator()
    {
        return new InternalDoubleIterator();
    }

    public double[] toArray()
    {
        double[] array = new double[this.size()];

        int j = 0;
        int zeroToThirtyOne = this.zeroToThirtyOne;
        while (zeroToThirtyOne != 0)
        {
            double value = (double) Integer.numberOfTrailingZeros(zeroToThirtyOne);
            array[j] = value;
            j++;
            zeroToThirtyOne &= ~(1 << (int) value);
        }

        for (int i = 0; i < this.table.length && j < this.size(); i++)
        {
            if (isNonSentinel(this.table[i]))
            {
                array[j] = this.table[i];
                j++;
            }
        }
        return array;
    }

    public boolean contains(double value)
    {
        if (isBetweenZeroAndThirtyOne(value))
        {
            int temp = this.zeroToThirtyOne;
            return ((temp >>> (int) value) & 1) != 0;
        }
        return Double.compare(this.table[this.probe(value)], value) == 0;
    }

    public boolean containsAll(double... source)
    {
        for (double item : source)
        {
            if (!this.contains(item))
            {
                return false;
            }
        }
        return true;
    }

    public boolean containsAll(DoubleIterable source)
    {
        for (DoubleIterator iterator = source.doubleIterator(); iterator.hasNext();)
        {
            if (!this.contains(iterator.next()))
            {
                return false;
            }
        }
        return true;
    }

    public void forEach(DoubleProcedure procedure)
    {
        int zeroToThirtyOne = this.zeroToThirtyOne;
        while (zeroToThirtyOne != 0)
        {
            double value = (double) Integer.numberOfTrailingZeros(zeroToThirtyOne);
            procedure.value(value);
            zeroToThirtyOne &= ~(1 << (int) value);
        }

        for (double value : this.table)
        {
            if (isNonSentinel(value))
            {
                procedure.value(value);
            }
        }
    }

    public DoubleHashSet select(DoublePredicate predicate)
    {
        DoubleHashSet result = new DoubleHashSet();

        int zeroToThirtyOne = this.zeroToThirtyOne;
        while (zeroToThirtyOne != 0)
        {
            double value = (double) Integer.numberOfTrailingZeros(zeroToThirtyOne);
            if (predicate.accept(value))
            {
                result.add(value);
            }
            zeroToThirtyOne &= ~(1 << (int) value);
        }

        for (double value : this.table)
        {
            if (isNonSentinel(value))
            {
                if (predicate.accept(value))
                {
                    result.add(value);
                }
            }
        }
        return result;
    }

    public MutableDoubleSet reject(DoublePredicate predicate)
    {
        DoubleHashSet result = new DoubleHashSet();

        int zeroToThirtyOne = this.zeroToThirtyOne;
        while (zeroToThirtyOne != 0)
        {
            double value = (double) Integer.numberOfTrailingZeros(zeroToThirtyOne);
            if (!predicate.accept(value))
            {
                result.add(value);
            }
            zeroToThirtyOne &= ~(1 << (int) value);
        }

        for (double value : this.table)
        {
            if (isNonSentinel(value))
            {
                if (!predicate.accept(value))
                {
                    result.add(value);
                }
            }
        }
        return result;
    }

    public <V> MutableSet<V> collect(DoubleToObjectFunction<? extends V> function)
    {
        MutableSet<V> target = UnifiedSet.newSet(this.size());

        int zeroToThirtyOne = this.zeroToThirtyOne;
        while (zeroToThirtyOne != 0)
        {
            double value = (double) Integer.numberOfTrailingZeros(zeroToThirtyOne);
            target.add(function.valueOf(value));
            zeroToThirtyOne &= ~(1 << (int) value);
        }

        for (double value : this.table)
        {
            if (isNonSentinel(value))
            {
                target.add(function.valueOf(value));
            }
        }
        return target;
    }

    public double detectIfNone(DoublePredicate predicate, double ifNone)
    {
        int zeroToThirtyOne = this.zeroToThirtyOne;
        while (zeroToThirtyOne != 0)
        {
            double value = (double) Integer.numberOfTrailingZeros(zeroToThirtyOne);
            if (predicate.accept(value))
            {
                return value;
            }
            zeroToThirtyOne &= ~(1 << (int) value);
        }

        for (double value : this.table)
        {
            if (isNonSentinel(value))
            {
                if (predicate.accept(value))
                {
                    return value;
                }
            }
        }
        return ifNone;
    }

    public int count(DoublePredicate predicate)
    {
        int count = 0;
        int zeroToThirtyOne = this.zeroToThirtyOne;
        while (zeroToThirtyOne != 0)
        {
            double value = (double) Integer.numberOfTrailingZeros(zeroToThirtyOne);
            if (predicate.accept(value))
            {
                count++;
            }
            zeroToThirtyOne &= ~(1 << (int) value);
        }

        for (double value : this.table)
        {
            if (isNonSentinel(value))
            {
                if (predicate.accept(value))
                {
                    count++;
                }
            }
        }
        return count;
    }

    public boolean anySatisfy(DoublePredicate predicate)
    {
        int zeroToThirtyOne = this.zeroToThirtyOne;
        while (zeroToThirtyOne != 0)
        {
            double value = (double) Integer.numberOfTrailingZeros(zeroToThirtyOne);
            if (predicate.accept(value))
            {
                return true;
            }
            zeroToThirtyOne &= ~(1 << (int) value);
        }

        for (double value : this.table)
        {
            if (isNonSentinel(value))
            {
                if (predicate.accept(value))
                {
                    return true;
                }
            }
        }
        return false;
    }

    public boolean allSatisfy(DoublePredicate predicate)
    {
        int zeroToThirtyOne = this.zeroToThirtyOne;
        while (zeroToThirtyOne != 0)
        {
            double value = (double) Integer.numberOfTrailingZeros(zeroToThirtyOne);
            if (!predicate.accept(value))
            {
                return false;
            }
            zeroToThirtyOne &= ~(1 << (int) value);
        }

        for (double value : this.table)
        {
            if (isNonSentinel(value))
            {
                if (!predicate.accept(value))
                {
                    return false;
                }
            }
        }
        return true;
    }

    public boolean noneSatisfy(DoublePredicate predicate)
    {
        int zeroToThirtyOne = this.zeroToThirtyOne;
        while (zeroToThirtyOne != 0)
        {
            double value = (double) Integer.numberOfTrailingZeros(zeroToThirtyOne);
            if (predicate.accept(value))
            {
                return false;
            }
            zeroToThirtyOne &= ~(1 << (int) value);
        }

        for (double value : this.table)
        {
            if (isNonSentinel(value))
            {
                if (predicate.accept(value))
                {
                    return false;
                }
            }
        }
        return true;
    }

    public MutableDoubleList toList()
    {
        return DoubleArrayList.newList(this);
    }

    public MutableDoubleSet toSet()
    {
        return DoubleHashSet.newSet(this);
    }

    public MutableDoubleBag toBag()
    {
        return DoubleHashBag.newBag(this);
    }

    public LazyDoubleIterable asLazy()
    {
        return new LazyDoubleIterableAdapter(this);
    }

    public double sum()
    {
        double result = 0.0;

        int zeroToThirtyOne = this.zeroToThirtyOne;
        while (zeroToThirtyOne != 0)
        {
            double value = (double) Integer.numberOfTrailingZeros(zeroToThirtyOne);
            result += value;
            zeroToThirtyOne &= ~(1 << (int) value);
        }

        for (double value : this.table)
        {
            if (isNonSentinel(value))
            {
                result += value;
            }
        }
        return result;
    }

    public double max()
    {
        if (this.isEmpty())
        {
            throw new NoSuchElementException();
        }
        double max = 31 - Integer.numberOfLeadingZeros(this.zeroToThirtyOne);
        boolean isMaxSet = this.zeroToThirtyOneOccupied != 0;

        for (double value : this.table)
        {
            if (isNonSentinel(value) && (!isMaxSet || Double.compare(max, value) < 0))
            {
                max = value;
                isMaxSet = true;
            }
        }
        return max;
    }

    public double maxIfEmpty(double defaultValue)
    {
        if (this.isEmpty())
        {
            return defaultValue;
        }
        return this.max();
    }

    public double min()
    {
        if (this.isEmpty())
        {
            throw new NoSuchElementException();
        }
        double min = (double) Integer.numberOfTrailingZeros(this.zeroToThirtyOne);
        boolean isMinSet = this.zeroToThirtyOneOccupied != 0;

        for (double value : this.table)
        {
            if (isNonSentinel(value) && (!isMinSet || Double.compare(value, min) < 0))
            {
                min = value;
                isMinSet = true;
            }
        }
        return min;
    }

    public double minIfEmpty(double defaultValue)
    {
        if (this.isEmpty())
        {
            return defaultValue;
        }
        return this.min();
    }

    public double average()
    {
        if (this.isEmpty())
        {
            throw new ArithmeticException();
        }
        return this.sum() / (double) this.size();
    }

    public double median()
    {
        if (this.isEmpty())
        {
            throw new ArithmeticException();
        }
        double[] sortedArray = this.toSortedArray();
        int middleIndex = sortedArray.length >> 1;
        if (sortedArray.length > 1 && (sortedArray.length & 1) == 0)
        {
            double first = sortedArray[middleIndex];
            double second = sortedArray[middleIndex - 1];
            return (first + second) / 2.0;
        }
        return sortedArray[middleIndex];
    }

    public double[] toSortedArray()
    {
        double[] array = this.toArray();
        Arrays.sort(array);
        return array;
    }

    public MutableDoubleList toSortedList()
    {
        return DoubleArrayList.newList(this).sortThis();
    }

    public DoubleSet freeze()
    {
        if (this.size() == 0)
        {
            return DoubleSets.immutable.with();
        }
        if (this.size() == 1)
        {
            return DoubleSets.immutable.with(this.doubleIterator().next());
        }
        this.copyOnWrite = true;
        return new ImmutableDoubleHashSet(this.table, this.occupiedWithData, this.zeroToThirtyOne, this.zeroToThirtyOneOccupied);
    }

    public void writeExternal(ObjectOutput out) throws IOException
    {
        out.writeInt(this.size());

        int zeroToThirtyOne = this.zeroToThirtyOne;
        while (zeroToThirtyOne != 0)
        {
            double value = (double) Integer.numberOfTrailingZeros(zeroToThirtyOne);
            out.writeDouble(value);
            zeroToThirtyOne &= ~(1 << (int) value);
        }

        for (double value : this.table)
        {
            if (isNonSentinel(value))
            {
                out.writeDouble(value);
            }
        }
    }

    public void readExternal(ObjectInput in) throws IOException
    {
        int size = in.readInt();

        for (int i = 0; i < size; i++)
        {
            this.add(in.readDouble());
        }
    }

    public <T> T injectInto(T injectedValue, ObjectDoubleToObjectFunction<? super T, ? extends T> function)
    {
        T result = injectedValue;
        int zeroToThirtyOne = this.zeroToThirtyOne;
        while (zeroToThirtyOne != 0)
        {
            double value = (double) Integer.numberOfTrailingZeros(zeroToThirtyOne);
            result = function.valueOf(result, value);
            zeroToThirtyOne &= ~(1 << (int) value);
        }

        for (double value : this.table)
        {
            if (isNonSentinel(value))
            {
                result = function.valueOf(result, value);
            }
        }
        return result;
    }

    /**
     * Rehashes every element in the set into a new backing table of the smallest possible size and eliminating removed sentinels.
     */
    public void compact()
    {
        this.rehash(this.smallestPowerOfTwoGreaterThan(this.size()));
    }

    private void rehash()
    {
        this.rehash(this.table.length);
    }

    private void rehashAndGrow()
    {
        this.rehash(this.table.length << 1);
    }

    private void rehash(int newCapacity)
    {
        int oldLength = this.table.length;
        double[] old = this.table;
        this.allocateTable(newCapacity);
        this.occupiedWithData = 0;
        this.occupiedWithSentinels = 0;

        for (int i = 0; i < oldLength; i++)
        {
            if (isNonSentinel(old[i]))
            {
                this.add(old[i]);
            }
        }
    }

    private void allocateTable(int sizeToAllocate)
    {
        this.table = new double[sizeToAllocate];
    }

    // exposed for testing
    int probe(double element)
    {
        int index = this.spread(element);
        double valueAtIndex = this.table[index];

        if (Double.compare(valueAtIndex, element) == 0 || Double.compare(valueAtIndex, EMPTY) == 0)
        {
            return index;
        }

        int removedIndex = Double.compare(valueAtIndex, REMOVED) == 0 ? index : -1;
        int nextIndex = index;
        int probe = 17;

        // loop until an empty slot is reached
        while (true)
        {
            // Probe algorithm: 17 * n * (n+1) / 2 where n = number of collisions
            nextIndex = (nextIndex + probe) & this.table.length - 1;
            probe += 17;

            if (Double.compare(this.table[nextIndex], element) == 0)
            {
                return nextIndex;
            }
            if (Double.compare(this.table[nextIndex], REMOVED) == 0)
            {
                if (removedIndex == -1)
                {
                    removedIndex = nextIndex;
                }
            }
            else if (Double.compare(this.table[nextIndex], EMPTY) == 0)
            {
                return removedIndex == -1 ? nextIndex : removedIndex;
            }
        }
    }

    // exposed for testing
    int spread(double element)
    {
        long code = Double.doubleToLongBits(element);
        code = ~code + (code << 18);
        code = (code << 18) - code - 1;
        code ^= code >>> 31;
        code *= 21;
        code += (code << 2) + (code << 4);
        code ^= code >>> 11;
        code += code << 6;
        code ^= code >>> 22;
        return (int) code & (this.table.length - 1);
    }

    private void copyTable()
    {
        this.copyOnWrite = false;
        double[] copy = new double[this.table.length];
        System.arraycopy(this.table, 0, copy, 0,
        Math.min(this.table.length, this.table.length));
        this.table = copy;
    }

    private int maxOccupiedWithData()
    {
        int capacity = this.table.length;
        // need at least one free slot for open addressing
        return Math.min(capacity - 1, capacity / OCCUPIED_DATA_RATIO);
    }

    private int maxOccupiedWithSentinels()
    {
        return this.table.length / OCCUPIED_SENTINEL_RATIO;
    }

    private static boolean isNonSentinel(double value)
    {
        return Double.compare(value, EMPTY) != 0 && Double.compare(value, REMOVED) != 0;
    }

    private static final class ImmutableDoubleHashSet implements ImmutableDoubleSet, Serializable
    {
        private static final long serialVersionUID = 1L;
        private final double[] table;
        private final int occupied;
        // The 32 bits of this integer indicate whether the items 0.0 to 31.0 are present in the set.
        private final int zeroToThirtyOne;
        private final int zeroToThirtyOneOccupied;

        private ImmutableDoubleHashSet(double[] table, int occupied, int zeroToThirtyOne, int zeroToThirtyOneOccupied)
        {
            this.checkOptimizedSize(occupied + zeroToThirtyOneOccupied);
            this.occupied = occupied;
            this.zeroToThirtyOneOccupied = zeroToThirtyOneOccupied;
            this.zeroToThirtyOne = zeroToThirtyOne;
            this.table = table;
        }

        private void checkOptimizedSize(int length)
        {
            if (length <= 1)
            {
                throw new IllegalArgumentException("Use DoubleSets.immutable.with() to instantiate an optimized collection");
            }
        }

        public static ImmutableDoubleSet newSetWith(double... elements)
        {
            return DoubleHashSet.newSetWith(elements).toImmutable();
        }

        @Override
        public boolean equals(Object obj)
        {
            if (this == obj)
            {
                return true;
            }

            if (!(obj instanceof DoubleSet))
            {
                return false;
            }

            DoubleSet other = (DoubleSet) obj;
            return this.size() == other.size() && this.containsAll(other.toArray());
        }

        @Override
        public int hashCode()
        {
            int result = 0;
            int zeroToThirtyOne = this.zeroToThirtyOne;
            while (zeroToThirtyOne != 0)
            {
                double value = (double) Integer.numberOfTrailingZeros(zeroToThirtyOne);
                result += (int) (Double.doubleToLongBits(value) ^ Double.doubleToLongBits(value) >>> 32);
                zeroToThirtyOne &= ~(1 << (int) value);
            }
            if (this.table != null)
            {
                for (int i = 0; i < this.table.length; i++)
                {
                    if (isNonSentinel(this.table[i]))
                    {
                        result += (int) (Double.doubleToLongBits(this.table[i]) ^ Double.doubleToLongBits(this.table[i]) >>> 32);
                    }
                }
            }
            return result;
        }

        @Override
        public String toString()
        {
            return this.makeString("[", ", ", "]");
        }

        public ImmutableDoubleSet newWith(double element)
        {
            return DoubleHashSet.newSet(this).with(element).toImmutable();
        }

        public ImmutableDoubleSet newWithout(double element)
        {
            return DoubleHashSet.newSet(this).without(element).toImmutable();
        }

        public ImmutableDoubleSet newWithAll(DoubleIterable elements)
        {
            return DoubleHashSet.newSet(this).withAll(elements).toImmutable();
        }

        public ImmutableDoubleSet newWithoutAll(DoubleIterable elements)
        {
            return DoubleHashSet.newSet(this).withoutAll(elements).toImmutable();
        }

        public int size()
        {
            return this.occupied + this.zeroToThirtyOneOccupied;
        }

        public boolean isEmpty()
        {
            return this.size() == 0;
        }

        public boolean notEmpty()
        {
            return this.size() != 0;
        }

        public String makeString()
        {
            return this.makeString(", ");
        }

        public String makeString(String separator)
        {
            return this.makeString("", separator, "");
        }

        public String makeString(String start, String separator, String end)
        {
            Appendable stringBuilder = new StringBuilder();
            this.appendString(stringBuilder, start, separator, end);
            return stringBuilder.toString();
        }

        public void appendString(Appendable appendable)
        {
            this.appendString(appendable, ", ");
        }

        public void appendString(Appendable appendable, String separator)
        {
            this.appendString(appendable, "", separator, "");
        }

        public void appendString(Appendable appendable, String start, String separator, String end)
        {
            try
            {
                appendable.append(start);

                int count = 0;
                int zeroToThirtyOne = this.zeroToThirtyOne;
                while (zeroToThirtyOne != 0)
                {
                    double value = (double) Integer.numberOfTrailingZeros(zeroToThirtyOne);
                    if (count > 0)
                    {
                        appendable.append(separator);
                    }
                    count++;
                    appendable.append(String.valueOf(value));
                    zeroToThirtyOne &= ~(1 << (int) value);
                }

                for (double value : this.table)
                {
                    if (isNonSentinel(value))
                    {
                        if (count > 0)
                        {
                            appendable.append(separator);
                        }
                        count++;
                        appendable.append(String.valueOf(value));
                    }
                }
                appendable.append(end);
            }
            catch (IOException e)
            {
                throw new RuntimeException(e);
            }
        }

        public DoubleIterator doubleIterator()
        {
            return new InternalDoubleIterator();
        }

        public double[] toArray()
        {
            double[] array = new double[this.size()];

            int j = 0;
            int zeroToThirtyOne = this.zeroToThirtyOne;
            while (zeroToThirtyOne != 0)
            {
                double value = (double) Integer.numberOfTrailingZeros(zeroToThirtyOne);
                array[j] = value;
                j++;
                zeroToThirtyOne &= ~(1 << (int) value);
            }

            for (int i = 0; i < this.table.length && j < this.size(); i++)
            {
                if (isNonSentinel(this.table[i]))
                {
                    array[j] = this.table[i];
                    j++;
                }
            }
            return array;
        }

        public boolean contains(double value)
        {
            if (isBetweenZeroAndThirtyOne(value))
            {
                int temp = this.zeroToThirtyOne;
                return ((temp >>> (int) value) & 1) != 0;
            }
            return Double.compare(this.table[this.probe(value)], value) == 0;
        }

        public boolean containsAll(double... source)
        {
            for (double item : source)
            {
                if (!this.contains(item))
                {
                    return false;
                }
            }
            return true;
        }

        public boolean containsAll(DoubleIterable source)
        {
            for (DoubleIterator iterator = source.doubleIterator(); iterator.hasNext();)
            {
                if (!this.contains(iterator.next()))
                {
                    return false;
                }
            }
            return true;
        }

        public void forEach(DoubleProcedure procedure)
        {
            int zeroToThirtyOne = this.zeroToThirtyOne;
            while (zeroToThirtyOne != 0)
            {
                double value = (double) Integer.numberOfTrailingZeros(zeroToThirtyOne);
                procedure.value(value);
                zeroToThirtyOne &= ~(1 << (int) value);
            }

            for (double value : this.table)
            {
                if (isNonSentinel(value))
                {
                    procedure.value(value);
                }
            }
        }

        public ImmutableDoubleSet select(DoublePredicate predicate)
        {
            DoubleHashSet result = new DoubleHashSet();

            int zeroToThirtyOne = this.zeroToThirtyOne;
            while (zeroToThirtyOne != 0)
            {
                double value = (double) Integer.numberOfTrailingZeros(zeroToThirtyOne);
                if (predicate.accept(value))
                {
                    result.add(value);
                }
                zeroToThirtyOne &= ~(1 << (int) value);
            }

            for (double value : this.table)
            {
                if (isNonSentinel(value))
                {
                    if (predicate.accept(value))
                    {
                        result.add(value);
                    }
                }
            }
            return result.toImmutable();
        }

        public ImmutableDoubleSet reject(DoublePredicate predicate)
        {
            DoubleHashSet result = new DoubleHashSet();

            int zeroToThirtyOne = this.zeroToThirtyOne;
            while (zeroToThirtyOne != 0)
            {
                double value = (double) Integer.numberOfTrailingZeros(zeroToThirtyOne);
                if (!predicate.accept(value))
                {
                    result.add(value);
                }
                zeroToThirtyOne &= ~(1 << (int) value);
            }

            for (double value : this.table)
            {
                if (isNonSentinel(value))
                {
                    if (!predicate.accept(value))
                    {
                        result.add(value);
                    }
                }
            }
            return result.toImmutable();
        }

        public <V> ImmutableSet<V> collect(DoubleToObjectFunction<? extends V> function)
        {
            MutableSet<V> target = UnifiedSet.newSet(this.size());

            int zeroToThirtyOne = this.zeroToThirtyOne;
            while (zeroToThirtyOne != 0)
            {
                double value = (double) Integer.numberOfTrailingZeros(zeroToThirtyOne);
                target.add(function.valueOf(value));
                zeroToThirtyOne &= ~(1 << (int) value);
            }

            for (double value : this.table)
            {
                if (isNonSentinel(value))
                {
                    target.add(function.valueOf(value));
                }
            }
            return target.toImmutable();
        }

        public double detectIfNone(DoublePredicate predicate, double ifNone)
        {
            int zeroToThirtyOne = this.zeroToThirtyOne;
            while (zeroToThirtyOne != 0)
            {
                double value = (double) Integer.numberOfTrailingZeros(zeroToThirtyOne);
                if (predicate.accept(value))
                {
                    return value;
                }
                zeroToThirtyOne &= ~(1 << (int) value);
            }

            for (double value : this.table)
            {
                if (isNonSentinel(value))
                {
                    if (predicate.accept(value))
                    {
                        return value;
                    }
                }
            }
            return ifNone;
        }

        public int count(DoublePredicate predicate)
        {
            int count = 0;
            int zeroToThirtyOne = this.zeroToThirtyOne;
            while (zeroToThirtyOne != 0)
            {
                double value = (double) Integer.numberOfTrailingZeros(zeroToThirtyOne);
                if (predicate.accept(value))
                {
                    count++;
                }
                zeroToThirtyOne &= ~(1 << (int) value);
            }

            for (double value : this.table)
            {
                if (isNonSentinel(value))
                {
                    if (predicate.accept(value))
                    {
                        count++;
                    }
                }
            }
            return count;
        }

        public boolean anySatisfy(DoublePredicate predicate)
        {
            int zeroToThirtyOne = this.zeroToThirtyOne;
            while (zeroToThirtyOne != 0)
            {
                double value = (double) Integer.numberOfTrailingZeros(zeroToThirtyOne);
                if (predicate.accept(value))
                {
                    return true;
                }
                zeroToThirtyOne &= ~(1 << (int) value);
            }

            for (double value : this.table)
            {
                if (isNonSentinel(value))
                {
                    if (predicate.accept(value))
                    {
                        return true;
                    }
                }
            }
            return false;
        }

        public boolean allSatisfy(DoublePredicate predicate)
        {
            int zeroToThirtyOne = this.zeroToThirtyOne;
            while (zeroToThirtyOne != 0)
            {
                double value = (double) Integer.numberOfTrailingZeros(zeroToThirtyOne);
                if (!predicate.accept(value))
                {
                    return false;
                }
                zeroToThirtyOne &= ~(1 << (int) value);
            }

            for (double value : this.table)
            {
                if (isNonSentinel(value))
                {
                    if (!predicate.accept(value))
                    {
                        return false;
                    }
                }
            }
            return true;
        }

        public boolean noneSatisfy(DoublePredicate predicate)
        {
            int zeroToThirtyOne = this.zeroToThirtyOne;
            while (zeroToThirtyOne != 0)
            {
                double value = (double) Integer.numberOfTrailingZeros(zeroToThirtyOne);
                if (predicate.accept(value))
                {
                    return false;
                }
                zeroToThirtyOne &= ~(1 << (int) value);
            }

            for (double value : this.table)
            {
                if (isNonSentinel(value))
                {
                    if (predicate.accept(value))
                    {
                        return false;
                    }
                }
            }
            return true;
        }

        public MutableDoubleList toList()
        {
            return DoubleArrayList.newList(this);
        }

        public MutableDoubleSet toSet()
        {
            return DoubleHashSet.newSet(this);
        }

        public MutableDoubleBag toBag()
        {
            return DoubleHashBag.newBag(this);
        }

        public LazyDoubleIterable asLazy()
        {
            return new LazyDoubleIterableAdapter(this);
        }

        public double sum()
        {
            double result = 0.0;

            int zeroToThirtyOne = this.zeroToThirtyOne;
            while (zeroToThirtyOne != 0)
            {
                double value = (double) Integer.numberOfTrailingZeros(zeroToThirtyOne);
                result += value;
                zeroToThirtyOne &= ~(1 << (int) value);
            }

            for (double value : this.table)
            {
                if (isNonSentinel(value))
                {
                    result += value;
                }
            }
            return result;
        }

        public double max()
        {
            if (this.isEmpty())
            {
                throw new NoSuchElementException();
            }
            double max = 31 - Integer.numberOfLeadingZeros(this.zeroToThirtyOne);
            boolean isMaxSet = this.zeroToThirtyOneOccupied != 0;

            for (double value : this.table)
            {
                if (isNonSentinel(value) && (!isMaxSet || Double.compare(max, value) < 0))
                {
                    max = value;
                    isMaxSet = true;
                }
            }
            return max;
        }

        public double maxIfEmpty(double defaultValue)
        {
            if (this.isEmpty())
            {
                return defaultValue;
            }
            return this.max();
        }

        public double min()
        {
            if (this.isEmpty())
            {
                throw new NoSuchElementException();
            }
            double min = (double) Integer.numberOfTrailingZeros(this.zeroToThirtyOne);
            boolean isMinSet = this.zeroToThirtyOneOccupied != 0;

            for (double value : this.table)
            {
                if (isNonSentinel(value) && (!isMinSet || Double.compare(value, min) < 0))
                {
                    min = value;
                    isMinSet = true;
                }
            }
            return min;
        }

        public double minIfEmpty(double defaultValue)
        {
            if (this.isEmpty())
            {
                return defaultValue;
            }
            return this.min();
        }

        public double average()
        {
            if (this.isEmpty())
            {
                throw new ArithmeticException();
            }
            return this.sum() / (double) this.size();
        }

        public double median()
        {
            if (this.isEmpty())
            {
                throw new ArithmeticException();
            }
            double[] sortedArray = this.toSortedArray();
            int middleIndex = sortedArray.length >> 1;
            if (sortedArray.length > 1 && (sortedArray.length & 1) == 0)
            {
                double first = sortedArray[middleIndex];
                double second = sortedArray[middleIndex - 1];
                return (first + second) / 2.0;
            }
            return sortedArray[middleIndex];
        }

        public double[] toSortedArray()
        {
            double[] array = this.toArray();
            Arrays.sort(array);
            return array;
        }

        public MutableDoubleList toSortedList()
        {
            return DoubleArrayList.newList(this).sortThis();
        }

        public <T> T injectInto(T injectedValue, ObjectDoubleToObjectFunction<? super T, ? extends T> function)
        {
            T result = injectedValue;
            int zeroToThirtyOne = this.zeroToThirtyOne;
            while (zeroToThirtyOne != 0)
            {
                double value = (double) Integer.numberOfTrailingZeros(zeroToThirtyOne);
                result = function.valueOf(result, value);
                zeroToThirtyOne &= ~(1 << (int) value);
            }

            for (double value : this.table)
            {
                if (isNonSentinel(value))
                {
                    result = function.valueOf(result, value);
                }
            }
            return result;
        }

        public DoubleSet freeze()
        {
            return this;
        }

        public ImmutableDoubleSet toImmutable()
        {
            return this;
        }

        private Object writeReplace()
        {
            return new ImmutableDoubleSetSerializationProxy(this);
        }

        // exposed for testing
        int probe(double element)
        {
            int index = this.spread(element);
            double valueAtIndex = this.table[index];

            if (Double.compare(valueAtIndex, element) == 0 || Double.compare(valueAtIndex, EMPTY) == 0)
            {
                return index;
            }

            int removedIndex = Double.compare(valueAtIndex, REMOVED) == 0 ? index : -1;
            int nextIndex = index;
            int probe = 17;

            // loop until an empty slot is reached
            while (true)
            {
                // Probe algorithm: 17 * n * (n+1) / 2 where n = number of collisions
                nextIndex = (nextIndex + probe) & this.table.length - 1;
                probe += 17;

                if (Double.compare(this.table[nextIndex], element) == 0)
                {
                    return nextIndex;
                }
                if (Double.compare(this.table[nextIndex], REMOVED) == 0)
                {
                    if (removedIndex == -1)
                    {
                        removedIndex = nextIndex;
                    }
                }
                else if (Double.compare(this.table[nextIndex], EMPTY) == 0)
                {
                    return removedIndex == -1 ? nextIndex : removedIndex;
                }
            }
        }

        // exposed for testing
        int spread(double element)
        {
            long code = Double.doubleToLongBits(element);
            code = ~code + (code << 18);
            code = (code << 18) - code - 1;
            code ^= code >>> 31;
            code *= 21;
            code += (code << 2) + (code << 4);
            code ^= code >>> 11;
            code += code << 6;
            code ^= code >>> 22;
            return (int) code & (this.table.length - 1);
        }

        private class InternalDoubleIterator implements DoubleIterator
        {
            private int count;
            private int position;
            private double zeroToThirtyOne;

            public boolean hasNext()
            {
                return this.count < ImmutableDoubleHashSet.this.size();
            }

            public double next()
            {
                if (!this.hasNext())
                {
                    throw new NoSuchElementException("next() called, but the iterator is exhausted");
                }
                this.count++;

                while (this.zeroToThirtyOne < 32)
                {
                    if (ImmutableDoubleHashSet.this.contains(this.zeroToThirtyOne))
                    {
                        double result = this.zeroToThirtyOne;
                        this.zeroToThirtyOne++;
                        return result;
                    }
                    this.zeroToThirtyOne++;
                }

                double[] table = ImmutableDoubleHashSet.this.table;
                while (!isNonSentinel(table[this.position]))
                {
                    this.position++;
                }
                double result = table[this.position];
                this.position++;
                return result;
            }
        }
    }

    private class InternalDoubleIterator implements DoubleIterator
    {
        private int count;
        private int position;
        private double zeroToThirtyOne;

        public boolean hasNext()
        {
            return this.count < DoubleHashSet.this.size();
        }

        public double next()
        {
            if (!this.hasNext())
            {
                throw new NoSuchElementException("next() called, but the iterator is exhausted");
            }
            this.count++;

            while (this.zeroToThirtyOne < 32)
            {
                if (DoubleHashSet.this.contains(this.zeroToThirtyOne))
                {
                    double result = this.zeroToThirtyOne;
                    this.zeroToThirtyOne++;
                    return result;
                }
                this.zeroToThirtyOne++;
            }

            double[] table = DoubleHashSet.this.table;
            while (!isNonSentinel(table[this.position]))
            {
                this.position++;
            }
            double result = table[this.position];
            this.position++;
            return result;
        }
    }
}
