/*
 * Copyright 2013 Goldman Sachs.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.gs.collections.impl.set.mutable.primitive;

import java.util.Collection;
import java.util.Collections;

import com.gs.collections.api.BooleanIterable;
import com.gs.collections.api.LazyBooleanIterable;
import com.gs.collections.api.block.function.primitive.BooleanToObjectFunction;
import com.gs.collections.api.block.predicate.primitive.BooleanPredicate;
import com.gs.collections.api.iterator.BooleanIterator;
import com.gs.collections.api.set.MutableSet;
import com.gs.collections.api.set.primitive.ImmutableBooleanSet;
import com.gs.collections.api.set.primitive.BooleanSet;
import com.gs.collections.api.set.primitive.MutableBooleanSet;
import com.gs.collections.impl.collection.mutable.primitive.AbstractSynchronizedBooleanCollection;
import com.gs.collections.impl.lazy.primitive.LazyBooleanIterableAdapter;
import net.jcip.annotations.GuardedBy;
import net.jcip.annotations.ThreadSafe;

/**
 * A synchronized view of a {@link MutableBooleanSet}. It is imperative that the user manually synchronize on the collection when iterating over it using the
 * {@link BooleanIterator}, as per {@link Collections#synchronizedCollection(Collection)}.
 * <p>
 * This file was automatically generated from template file synchronizedPrimitiveSet.stg.
 * </p>
 *
 * @see MutableBooleanSet#asSynchronized()
 * @see MutableSet#asSynchronized()
 * @since 3.1.
 */
@ThreadSafe
public final class SynchronizedBooleanSet
        extends AbstractSynchronizedBooleanCollection
        implements MutableBooleanSet
{
    private static final long serialVersionUID = 1L;

    SynchronizedBooleanSet(MutableBooleanSet set)
    {
        super(set);
    }

    SynchronizedBooleanSet(MutableBooleanSet set, Object newLock)
    {
        super(set, newLock);
    }

    /**
     * This method will take a MutableBooleanSet and wrap it directly in a SynchronizedBooleanSet.
     */
    public static SynchronizedBooleanSet of(MutableBooleanSet set)
    {
        return new SynchronizedBooleanSet(set);
    }

    /**
     * This method will take a MutableBooleanSet and wrap it directly in a SynchronizedBooleanSet.
     * Additionally, a developer specifies which lock to use with the collection.
     */
    public static SynchronizedBooleanSet of(MutableBooleanSet set, Object lock)
    {
        return new SynchronizedBooleanSet(set, lock);
    }

    @GuardedBy("getLock()")
    private MutableBooleanSet getMutableBooleanSet()
    {
        return (MutableBooleanSet) this.getBooleanCollection();
    }

    @Override
    public SynchronizedBooleanSet without(boolean element)
    {
        synchronized (this.getLock())
        {
            this.getMutableBooleanSet().remove(element);
        }
        return this;
    }

    @Override
    public SynchronizedBooleanSet with(boolean element)
    {
        synchronized (this.getLock())
        {
            this.getMutableBooleanSet().add(element);
        }
        return this;
    }

    @Override
    public SynchronizedBooleanSet withAll(BooleanIterable elements)
    {
        synchronized (this.getLock())
        {
            this.getMutableBooleanSet().addAll(elements.toArray());
        }
        return this;
    }

    @Override
    public SynchronizedBooleanSet withoutAll(BooleanIterable elements)
    {
        synchronized (this.getLock())
        {
            this.getMutableBooleanSet().removeAll(elements);
        }
        return this;
    }

    @Override
    public MutableBooleanSet select(BooleanPredicate predicate)
    {
        synchronized (this.getLock())
        {
            return this.getMutableBooleanSet().select(predicate);
        }
    }

    @Override
    public MutableBooleanSet reject(BooleanPredicate predicate)
    {
        synchronized (this.getLock())
        {
            return this.getMutableBooleanSet().reject(predicate);
        }
    }

    @Override
    public <V> MutableSet<V> collect(BooleanToObjectFunction<? extends V> function)
    {
        synchronized (this.getLock())
        {
            return this.getMutableBooleanSet().collect(function);
        }
    }

    @Override
    public boolean equals(Object otherSet)
    {
        synchronized (this.getLock())
        {
            return this.getMutableBooleanSet().equals(otherSet);
        }
    }

    @Override
    public int hashCode()
    {
        synchronized (this.getLock())
        {
            return this.getMutableBooleanSet().hashCode();
        }
    }

    @Override
    public LazyBooleanIterable asLazy()
    {
        synchronized (this.getLock())
        {
            return new LazyBooleanIterableAdapter(this);
        }
    }

    @Override
    public MutableBooleanSet asUnmodifiable()
    {
        return new UnmodifiableBooleanSet(this);
    }

    @Override
    public MutableBooleanSet asSynchronized()
    {
        return this;
    }

    public BooleanSet freeze()
    {
        synchronized (this.getLock())
        {
            return this.getMutableBooleanSet().freeze();
        }
    }

    @Override
    public ImmutableBooleanSet toImmutable()
    {
        synchronized (this.getLock())
        {
            return this.getMutableBooleanSet().toImmutable();
        }
    }
}
