/*
 * Copyright 2013 Goldman Sachs.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.gs.collections.impl.set.mutable.primitive;

import java.util.Collection;
import java.util.Collections;

import com.gs.collections.api.LongIterable;
import com.gs.collections.api.LazyLongIterable;
import com.gs.collections.api.block.function.primitive.LongToObjectFunction;
import com.gs.collections.api.block.predicate.primitive.LongPredicate;
import com.gs.collections.api.iterator.LongIterator;
import com.gs.collections.api.set.MutableSet;
import com.gs.collections.api.set.primitive.ImmutableLongSet;
import com.gs.collections.api.set.primitive.LongSet;
import com.gs.collections.api.set.primitive.MutableLongSet;
import com.gs.collections.impl.collection.mutable.primitive.AbstractSynchronizedLongCollection;
import com.gs.collections.impl.lazy.primitive.LazyLongIterableAdapter;
import net.jcip.annotations.GuardedBy;
import net.jcip.annotations.ThreadSafe;

/**
 * A synchronized view of a {@link MutableLongSet}. It is imperative that the user manually synchronize on the collection when iterating over it using the
 * {@link LongIterator}, as per {@link Collections#synchronizedCollection(Collection)}.
 * <p>
 * This file was automatically generated from template file synchronizedPrimitiveSet.stg.
 * </p>
 *
 * @see MutableLongSet#asSynchronized()
 * @see MutableSet#asSynchronized()
 * @since 3.1.
 */
@ThreadSafe
public final class SynchronizedLongSet
        extends AbstractSynchronizedLongCollection
        implements MutableLongSet
{
    private static final long serialVersionUID = 1L;

    SynchronizedLongSet(MutableLongSet set)
    {
        super(set);
    }

    SynchronizedLongSet(MutableLongSet set, Object newLock)
    {
        super(set, newLock);
    }

    /**
     * This method will take a MutableLongSet and wrap it directly in a SynchronizedLongSet.
     */
    public static SynchronizedLongSet of(MutableLongSet set)
    {
        return new SynchronizedLongSet(set);
    }

    /**
     * This method will take a MutableLongSet and wrap it directly in a SynchronizedLongSet.
     * Additionally, a developer specifies which lock to use with the collection.
     */
    public static SynchronizedLongSet of(MutableLongSet set, Object lock)
    {
        return new SynchronizedLongSet(set, lock);
    }

    @GuardedBy("getLock()")
    private MutableLongSet getMutableLongSet()
    {
        return (MutableLongSet) this.getLongCollection();
    }

    @Override
    public SynchronizedLongSet without(long element)
    {
        synchronized (this.getLock())
        {
            this.getMutableLongSet().remove(element);
        }
        return this;
    }

    @Override
    public SynchronizedLongSet with(long element)
    {
        synchronized (this.getLock())
        {
            this.getMutableLongSet().add(element);
        }
        return this;
    }

    @Override
    public SynchronizedLongSet withAll(LongIterable elements)
    {
        synchronized (this.getLock())
        {
            this.getMutableLongSet().addAll(elements.toArray());
        }
        return this;
    }

    @Override
    public SynchronizedLongSet withoutAll(LongIterable elements)
    {
        synchronized (this.getLock())
        {
            this.getMutableLongSet().removeAll(elements);
        }
        return this;
    }

    @Override
    public MutableLongSet select(LongPredicate predicate)
    {
        synchronized (this.getLock())
        {
            return this.getMutableLongSet().select(predicate);
        }
    }

    @Override
    public MutableLongSet reject(LongPredicate predicate)
    {
        synchronized (this.getLock())
        {
            return this.getMutableLongSet().reject(predicate);
        }
    }

    @Override
    public <V> MutableSet<V> collect(LongToObjectFunction<? extends V> function)
    {
        synchronized (this.getLock())
        {
            return this.getMutableLongSet().collect(function);
        }
    }

    @Override
    public boolean equals(Object otherSet)
    {
        synchronized (this.getLock())
        {
            return this.getMutableLongSet().equals(otherSet);
        }
    }

    @Override
    public int hashCode()
    {
        synchronized (this.getLock())
        {
            return this.getMutableLongSet().hashCode();
        }
    }

    @Override
    public LazyLongIterable asLazy()
    {
        synchronized (this.getLock())
        {
            return new LazyLongIterableAdapter(this);
        }
    }

    @Override
    public MutableLongSet asUnmodifiable()
    {
        return new UnmodifiableLongSet(this);
    }

    @Override
    public MutableLongSet asSynchronized()
    {
        return this;
    }

    public LongSet freeze()
    {
        synchronized (this.getLock())
        {
            return this.getMutableLongSet().freeze();
        }
    }

    @Override
    public ImmutableLongSet toImmutable()
    {
        synchronized (this.getLock())
        {
            return this.getMutableLongSet().toImmutable();
        }
    }
}
