/*
 * Copyright 2013 Goldman Sachs.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.gs.collections.impl.stack.immutable.primitive;

import java.io.Externalizable;
import java.io.IOException;
import java.io.ObjectInput;
import java.io.ObjectOutput;
import java.io.Serializable;

import com.gs.collections.api.LazyCharIterable;
import com.gs.collections.api.CharIterable;
import com.gs.collections.api.bag.primitive.MutableCharBag;
import com.gs.collections.api.block.function.primitive.CharToObjectFunction;
import com.gs.collections.api.block.function.primitive.ObjectCharToObjectFunction;
import com.gs.collections.api.block.predicate.primitive.CharPredicate;
import com.gs.collections.api.block.procedure.primitive.CharProcedure;
import com.gs.collections.api.iterator.CharIterator;
import com.gs.collections.api.list.primitive.CharList;
import com.gs.collections.api.list.primitive.MutableCharList;
import com.gs.collections.api.set.primitive.MutableCharSet;
import com.gs.collections.api.stack.ImmutableStack;
import com.gs.collections.api.stack.primitive.CharStack;
import com.gs.collections.api.stack.primitive.ImmutableCharStack;
import com.gs.collections.impl.bag.mutable.primitive.CharHashBag;
import com.gs.collections.impl.block.procedure.checked.primitive.CheckedCharProcedure;
import com.gs.collections.impl.factory.Stacks;
import com.gs.collections.impl.factory.primitive.CharStacks;
import com.gs.collections.impl.lazy.primitive.LazyCharIterableAdapter;
import com.gs.collections.impl.list.mutable.primitive.CharArrayList;
import com.gs.collections.impl.set.mutable.primitive.CharHashSet;
import com.gs.collections.impl.stack.mutable.primitive.CharArrayStack;
import net.jcip.annotations.Immutable;

/**
 * ImmutableCharArrayStack is the non-modifiable equivalent of {@link CharArrayStack}.
 * This file was automatically generated from template file immutablePrimitiveArrayStack.stg.
 *
 * @since 4.0.
 */
@Immutable
class ImmutableCharArrayStack
        implements ImmutableCharStack, Serializable
{
    private static final long serialVersionUID = 1L;
    private final CharArrayList delegate;

    private ImmutableCharArrayStack(char[] newElements)
    {
        this.checkOptimizedSize(newElements.length);
        this.delegate = new CharArrayList(newElements);
    }

    private ImmutableCharArrayStack(CharArrayList newElements)
    {
        this.checkOptimizedSize(newElements.size());
        this.delegate = newElements;
    }

    private void checkOptimizedSize(int length)
    {
        if (length <= 1)
        {
            throw new IllegalArgumentException("Use CharStacks.immutable.with() to instantiate an optimized collection");
        }
    }

    public static ImmutableCharArrayStack newStack(CharIterable iterable)
    {
        return new ImmutableCharArrayStack(iterable.toArray());
    }

    public static ImmutableCharArrayStack newStackWith(char... elements)
    {
        char[] newArray = new char[elements.length];
        System.arraycopy(elements, 0, newArray, 0, elements.length);
        return new ImmutableCharArrayStack(newArray);
    }

    public static ImmutableCharArrayStack newStackFromTopToBottom(char... items)
    {
        return new ImmutableCharArrayStack(CharArrayList.newListWith(items).reverseThis());
    }

    public static ImmutableCharArrayStack newStackFromTopToBottom(CharIterable items)
    {
        return new ImmutableCharArrayStack(CharArrayList.newList(items).reverseThis());
    }

    public ImmutableCharStack push(char item)
    {
        CharArrayList newDelegate = CharArrayList.newList(this.delegate);
        newDelegate.add(item);
        return new ImmutableCharArrayStack(newDelegate);
    }

    public ImmutableCharStack pop()
    {
        CharArrayList newDelegate = CharArrayList.newList(this.delegate);
        newDelegate.removeAtIndex(this.delegate.size() - 1);
        return CharStacks.immutable.with(newDelegate.toArray());
    }

    public ImmutableCharStack pop(int count)
    {
        this.checkNegativeCount(count);
        if (count == 0)
        {
            return this;
        }
        this.checkSizeLessThanCount(count);
        CharArrayList newDelegate = CharArrayList.newList(this.delegate);
        while (count > 0)
        {
            newDelegate.removeAtIndex(newDelegate.size() - 1);
            count--;
        }
        return CharStacks.immutable.with(newDelegate.toArray());
    }

    private void checkNegativeCount(int count)
    {
        if (count < 0)
        {
            throw new IllegalArgumentException("Count must be positive but was " + count);
        }
    }

    public char peek()
    {
        return this.delegate.getLast();
    }

    public CharList peek(int count)
    {
        this.checkNegativeCount(count);
        this.checkSizeLessThanCount(count);
        if (count == 0)
        {
            return new CharArrayList();
        }
        MutableCharList subList = new CharArrayList(count);
        int index = this.delegate.size() - 1;
        for (int i = 0; i < count; i++)
        {
            subList.add(this.delegate.get(index - i));
        }
        return subList;
    }

    public char peekAt(int index)
    {
        this.rangeCheck(index);
        return this.delegate.get(this.delegate.size() - 1 - index);
    }

    private void rangeCheck(int index)
    {
        if (index < 0 || index > this.delegate.size() - 1)
        {
            throw new IllegalArgumentException("Index " + index + " out of range.Should be between 0 and " + (this.delegate.size() - 1));
        }
    }

    public CharIterator charIterator()
    {
        return this.delegate.asReversed().charIterator();
    }

    public void forEach(CharProcedure procedure)
    {
        this.delegate.asReversed().forEach(procedure);
    }

    public int count(CharPredicate predicate)
    {
        return this.delegate.asReversed().count(predicate);
    }

    public boolean anySatisfy(CharPredicate predicate)
    {
        return this.delegate.asReversed().anySatisfy(predicate);
    }

    public boolean allSatisfy(CharPredicate predicate)
    {
        return this.delegate.asReversed().allSatisfy(predicate);
    }

    public boolean noneSatisfy(CharPredicate predicate)
    {
        return this.delegate.asReversed().noneSatisfy(predicate);
    }

    public ImmutableCharStack select(CharPredicate predicate)
    {
        return CharStacks.immutable.withAllReversed(this.delegate.asReversed().select(predicate));
    }

    public ImmutableCharStack reject(CharPredicate predicate)
    {
        return CharStacks.immutable.withAllReversed(this.delegate.asReversed().reject(predicate));
    }

    public char detectIfNone(CharPredicate predicate, char ifNone)
    {
        return this.delegate.asReversed().detectIfNone(predicate, ifNone);
    }

    public <V> ImmutableStack<V> collect(CharToObjectFunction<? extends V> function)
    {
        return Stacks.immutable.withAllReversed(this.delegate.asReversed().collect(function));
    }

    public long sum()
    {
        return this.delegate.sum();
    }

    public char max()
    {
        return this.delegate.max();
    }

    public char maxIfEmpty(char defaultValue)
    {
        return this.max();
    }

    public char min()
    {
        return this.delegate.min();
    }

    public char minIfEmpty(char defaultValue)
    {
        return this.min();
    }

    public double average()
    {
        return this.delegate.average();
    }

    public double median()
    {
        return this.delegate.median();
    }

    public char[] toSortedArray()
    {
        return this.delegate.toSortedArray();
    }

    public MutableCharList toSortedList()
    {
        return CharArrayList.newList(this).sortThis();
    }

    public char[] toArray()
    {
        return this.delegate.asReversed().toArray();
    }

    public boolean contains(char value)
    {
        return this.delegate.asReversed().contains(value);
    }

    public boolean containsAll(char... source)
    {
        return this.delegate.asReversed().containsAll(source);
    }

    public boolean containsAll(CharIterable source)
    {
        return this.delegate.asReversed().containsAll(source);
    }

    public MutableCharList toList()
    {
        return CharArrayList.newList(this);
    }

    public MutableCharSet toSet()
    {
        return CharHashSet.newSet(this);
    }

    public MutableCharBag toBag()
    {
        return CharHashBag.newBag(this);
    }

    public <V> V injectInto(V injectedValue, ObjectCharToObjectFunction<? super V, ? extends V> function)
    {
        return this.delegate.toReversed().injectInto(injectedValue, function);
    }

    public LazyCharIterable asLazy()
    {
        return new LazyCharIterableAdapter(this);
    }

    public ImmutableCharStack toImmutable()
    {
        return this;
    }

    public int size()
    {
        return this.delegate.size();
    }

    public boolean isEmpty()
    {
        return false;
    }

    public boolean notEmpty()
    {
        return true;
    }

    @Override
    public boolean equals(Object otherStack)
    {
        if (otherStack == this)
        {
            return true;
        }
        if (!(otherStack instanceof CharStack))
        {
            return false;
        }
        CharStack stack = (CharStack) otherStack;
        if (this.size() != stack.size())
        {
            return false;
        }
        for (int i = 0; i < this.size(); i++)
        {
            if (this.peekAt(i) != stack.peekAt(i))
            {
                return false;
            }
        }
        return true;
    }

    @Override
    public int hashCode()
    {
        int hashCode = 1;
        CharIterable iterable = this.delegate.asReversed();
        CharIterator iterator = iterable.charIterator();
        while (iterator.hasNext())
        {
            char item = iterator.next();
            hashCode = 31 * hashCode + (int) item;
        }
        return hashCode;
    }

    @Override
    public String toString()
    {
        return this.delegate.asReversed().toString();
    }

    public String makeString()
    {
        return this.delegate.asReversed().makeString();
    }

    public String makeString(String separator)
    {
        return this.delegate.asReversed().makeString(separator);
    }

    public String makeString(String start, String separator, String end)
    {
        return this.delegate.asReversed().makeString(start, separator, end);
    }

    public void appendString(Appendable appendable)
    {
        this.delegate.asReversed().appendString(appendable);
    }

    public void appendString(Appendable appendable, String separator)
    {
        this.delegate.asReversed().appendString(appendable, separator);
    }

    public void appendString(Appendable appendable, String start, String separator, String end)
    {
        this.delegate.asReversed().appendString(appendable, start, separator, end);
    }

    private void checkSizeLessThanCount(int count)
    {
        if (this.delegate.size() < count)
        {
            throw new IllegalArgumentException("Count must be less than size: Count = " + count + " Size = " + this.delegate.size());
        }
    }

    private Object writeReplace()
    {
        return new ImmutableCharStackSerializationProxy(this);
    }

    private static class ImmutableCharStackSerializationProxy implements Externalizable
    {
        private static final long serialVersionUID = 1L;

        private CharStack stack;

        @SuppressWarnings("UnusedDeclaration")
        public ImmutableCharStackSerializationProxy()
        {
            // Empty constructor for Externalizable class
        }

        protected ImmutableCharStackSerializationProxy(CharStack stack)
        {
            this.stack = stack;
        }

        public void writeExternal(final ObjectOutput out) throws IOException
        {
            out.writeInt(this.stack.size());
            try
            {
                this.stack.forEach(new CheckedCharProcedure()
                {
                    @Override
                    public void safeValue(char item) throws IOException
                    {
                        out.writeChar(item);
                    }
                });
            }
            catch (RuntimeException e)
            {
                if (e.getCause() instanceof IOException)
                {
                    throw (IOException) e.getCause();
                }
                throw e;
            }
        }

        public void readExternal(ObjectInput in) throws IOException, ClassNotFoundException
        {
            int size = in.readInt();
            CharArrayList deserializedDelegate = new CharArrayList(size);

            for (int i = 0; i < size; i++)
            {
                deserializedDelegate.add(in.readChar());
            }

            this.stack = ImmutableCharArrayStack.newStackFromTopToBottom(deserializedDelegate);
        }

        protected Object readResolve()
        {
            return this.stack;
        }
    }
}
