/*
 * Copyright 2013 Goldman Sachs.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.gs.collections.impl.stack.mutable.primitive;

import java.io.Externalizable;
import java.io.IOException;
import java.io.ObjectInput;
import java.io.ObjectOutput;
import java.util.EmptyStackException;

import com.gs.collections.api.CharIterable;
import com.gs.collections.api.LazyCharIterable;
import com.gs.collections.api.bag.primitive.MutableCharBag;
import com.gs.collections.api.block.function.primitive.CharToObjectFunction;
import com.gs.collections.api.block.function.primitive.ObjectCharToObjectFunction;
import com.gs.collections.api.block.predicate.primitive.CharPredicate;
import com.gs.collections.api.block.procedure.primitive.CharProcedure;
import com.gs.collections.api.iterator.CharIterator;
import com.gs.collections.api.list.primitive.CharList;
import com.gs.collections.api.list.primitive.MutableCharList;
import com.gs.collections.api.set.primitive.MutableCharSet;
import com.gs.collections.api.stack.MutableStack;
import com.gs.collections.api.stack.primitive.CharStack;
import com.gs.collections.api.stack.primitive.ImmutableCharStack;
import com.gs.collections.api.stack.primitive.MutableCharStack;
import com.gs.collections.impl.bag.mutable.primitive.CharHashBag;
import com.gs.collections.impl.factory.primitive.CharStacks;
import com.gs.collections.impl.lazy.primitive.LazyCharIterableAdapter;
import com.gs.collections.impl.list.mutable.primitive.CharArrayList;
import com.gs.collections.impl.set.mutable.primitive.CharHashSet;
import com.gs.collections.impl.stack.mutable.ArrayStack;
import net.jcip.annotations.NotThreadSafe;

/**
 * CharArrayStack is similar to {@link ArrayStack}, and is memory-optimized for char primitives.
 * This file was automatically generated from template file primitiveArrayStack.stg.
 */
@NotThreadSafe
public final class CharArrayStack
        implements MutableCharStack, Externalizable
{
    private static final long serialVersionUID = 1L;

    private transient CharArrayList delegate;

    public CharArrayStack()
    {
        this.delegate = new CharArrayList();
    }

    private CharArrayStack(int size)
    {
        this.delegate = new CharArrayList(size);
    }

    private CharArrayStack(char... items)
    {
        this.delegate = new CharArrayList(items);
    }

    public static CharArrayStack newStackFromTopToBottom(char... items)
    {
        CharArrayStack stack = new CharArrayStack(items.length);
        for (int i = items.length - 1; i >= 0; i--)
        {
            stack.push(items[i]);
        }
        return stack;
    }

    public static CharArrayStack newStackWith(char... items)
    {
        return new CharArrayStack(items);
    }

    public static CharArrayStack newStack(CharIterable items)
    {
        CharArrayStack stack = new CharArrayStack(items.size());
        stack.delegate = CharArrayList.newList(items);
        return stack;
    }

    public static CharArrayStack newStackFromTopToBottom(CharIterable items)
    {
        CharArrayStack stack = new CharArrayStack(items.size());
        stack.delegate = CharArrayList.newList(items).reverseThis();
        return stack;
    }

    public void push(char item)
    {
        this.delegate.add(item);
    }

    public char pop()
    {
        this.checkEmptyStack();
        return this.delegate.removeAtIndex(this.delegate.size() - 1);
    }

    private void checkEmptyStack()
    {
        if (this.delegate.isEmpty())
        {
            throw new EmptyStackException();
        }
    }

    public CharList pop(int count)
    {
        this.checkPositiveValueForCount(count);
        this.checkSizeLessThanCount(count);
        if (count == 0)
        {
            return new CharArrayList(0);
        }
        MutableCharList subList = new CharArrayList(count);
        while (count > 0)
        {
            subList.add(this.pop());
            count--;
        }
        return subList;
    }

    private void checkPositiveValueForCount(int count)
    {
        if (count < 0)
        {
            throw new IllegalArgumentException("Count must be positive but was " + count);
        }
    }

    private void checkSizeLessThanCount(int count)
    {
        if (this.delegate.size() < count)
        {
            throw new IllegalArgumentException("Count must be less than size: Count = " + count + " Size = " + this.delegate.size());
        }
    }

    public MutableCharStack select(CharPredicate predicate)
    {
        return newStackFromTopToBottom(this.delegate.asReversed().select(predicate));
    }

    public MutableCharStack reject(CharPredicate predicate)
    {
        return newStackFromTopToBottom(this.delegate.asReversed().reject(predicate));
    }

    public char peek()
    {
        this.checkEmptyStack();
        return this.delegate.getLast();
    }

    public CharList peek(int count)
    {
        this.checkPositiveValueForCount(count);
        this.checkSizeLessThanCount(count);
        if (count == 0)
        {
            return new CharArrayList(0);
        }
        MutableCharList subList = new CharArrayList(count);
        int index = this.delegate.size() - 1;
        for (int i = 0; i < count; i++)
        {
            subList.add(this.delegate.get(index - i));
        }
        return subList;
    }

    public char peekAt(int index)
    {
        this.rangeCheck(index);
        return this.delegate.get(this.delegate.size() - 1 - index);
    }

    private void rangeCheck(int index)
    {
        if (index < 0 || index > this.delegate.size() - 1)
        {
            throw new IllegalArgumentException("Index " + index + " out of range.Should be between 0 and " + (this.delegate.size() - 1));
        }
    }

    public CharIterator charIterator()
    {
        return this.delegate.asReversed().charIterator();
    }

    public void forEach(CharProcedure procedure)
    {
        this.delegate.asReversed().forEach(procedure);
    }

    public int size()
    {
        return this.delegate.size();
    }

    public boolean isEmpty()
    {
        return this.delegate.isEmpty();
    }

    public boolean notEmpty()
    {
        return this.delegate.notEmpty();
    }

    public int count(CharPredicate predicate)
    {
        return this.delegate.asReversed().count(predicate);
    }

    public boolean anySatisfy(CharPredicate predicate)
    {
        return this.delegate.asReversed().anySatisfy(predicate);
    }

    public boolean allSatisfy(CharPredicate predicate)
    {
        return this.delegate.asReversed().allSatisfy(predicate);
    }

    public boolean noneSatisfy(CharPredicate predicate)
    {
        return this.delegate.asReversed().noneSatisfy(predicate);
    }

    public char detectIfNone(CharPredicate predicate, char ifNone)
    {
        return this.delegate.asReversed().detectIfNone(predicate, ifNone);
    }

    public <V> MutableStack<V> collect(CharToObjectFunction<? extends V> function)
    {
        return ArrayStack.newStackFromTopToBottom(this.delegate.asReversed().collect(function));
    }

    public <V> V injectInto(V injectedValue, ObjectCharToObjectFunction<? super V, ? extends V> function)
    {
        //TODO changed to asReversed() once added to IntIterable
        return this.delegate.toReversed().injectInto(injectedValue, function);
    }

    public long sum()
    {
        return this.delegate.sum();
    }

    public char max()
    {
        return this.delegate.max();
    }

    public char min()
    {
        return this.delegate.min();
    }

    public char minIfEmpty(char defaultValue)
    {
        if (this.isEmpty())
        {
            return defaultValue;
        }
        return this.min();
    }

    public char maxIfEmpty(char defaultValue)
    {
        if (this.isEmpty())
        {
            return defaultValue;
        }
        return this.max();
    }

    public double average()
    {
        return this.delegate.average();
    }

    public double median()
    {
        return this.delegate.median();
    }

    public char[] toSortedArray()
    {
        return this.delegate.toSortedArray();
    }

    public char[] toArray()
    {
        return this.delegate.asReversed().toArray();
    }

    public boolean contains(char value)
    {
        return this.delegate.asReversed().contains(value);
    }

    public boolean containsAll(char... source)
    {
        return this.delegate.asReversed().containsAll(source);
    }

    public boolean containsAll(CharIterable source)
    {
        return this.delegate.asReversed().containsAll(source);
    }

    public void clear()
    {
        this.delegate.clear();
    }

    @Override
    public boolean equals(Object otherStack)
    {
        if (otherStack == this)
        {
            return true;
        }
        if (!(otherStack instanceof CharStack))
        {
            return false;
        }
        CharStack stack = (CharStack) otherStack;
        if (this.size() != stack.size())
        {
            return false;
        }
        for (int i = 0; i < this.size(); i++)
        {
            if (this.peekAt(i) != stack.peekAt(i))
            {
                return false;
            }
        }
        return true;
    }

    @Override
    public int hashCode()
    {
        int hashCode = 1;
        CharIterable iterable = this.delegate.asReversed();
        CharIterator iterator = iterable.charIterator();
        while (iterator.hasNext())
        {
            char item = iterator.next();
            hashCode = 31 * hashCode + (int) item;
        }
        return hashCode;
    }

    @Override
    public String toString()
    {
        return this.delegate.asReversed().toString();
    }

    public String makeString()
    {
        return this.delegate.asReversed().makeString();
    }

    public String makeString(String separator)
    {
        return this.delegate.asReversed().makeString(separator);
    }

    public String makeString(String start, String separator, String end)
    {
        return this.delegate.asReversed().makeString(start, separator, end);
    }

    public void appendString(Appendable appendable)
    {
        this.delegate.asReversed().appendString(appendable);
    }

    public void appendString(Appendable appendable, String separator)
    {
        this.delegate.asReversed().appendString(appendable, separator);
    }

    public void appendString(
            Appendable appendable,
            String start,
            String separator,
            String end)
    {
        this.delegate.asReversed().appendString(appendable, start, separator, end);
    }

    public MutableCharList toList()
    {
        return CharArrayList.newList(this);
    }

    public MutableCharList toSortedList()
    {
        return CharArrayList.newList(this).sortThis();
    }

    public MutableCharSet toSet()
    {
        return CharHashSet.newSet(this);
    }

    public MutableCharBag toBag()
    {
        return CharHashBag.newBag(this);
    }

    public LazyCharIterable asLazy()
    {
        return new LazyCharIterableAdapter(this);
    }

    public MutableCharStack asUnmodifiable()
    {
        return new UnmodifiableCharStack(this);
    }

    public MutableCharStack asSynchronized()
    {
        return new SynchronizedCharStack(this);
    }

    public ImmutableCharStack toImmutable()
    {
        return CharStacks.immutable.withAll(this.delegate);
    }

    public void writeExternal(ObjectOutput out) throws IOException
    {
        out.writeInt(this.size());
        CharIterator iterator = this.delegate.asReversed().charIterator();
        while (iterator.hasNext())
        {
            char each = iterator.next();
            out.writeChar(each);
        }
    }

    public void readExternal(ObjectInput in) throws IOException
    {
        int size = in.readInt();
        char[] array = new char[size];
        for (int i = size - 1; i >= 0; i--)
        {
            array[i] = in.readChar();
        }
        this.delegate = CharArrayList.newListWith(array);
    }
}