/*
 * Copyright 2013 Goldman Sachs.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.gs.collections.impl.stack.mutable.primitive;

import java.io.Externalizable;
import java.io.IOException;
import java.io.ObjectInput;
import java.io.ObjectOutput;
import java.util.EmptyStackException;

import com.gs.collections.api.ShortIterable;
import com.gs.collections.api.LazyShortIterable;
import com.gs.collections.api.bag.primitive.MutableShortBag;
import com.gs.collections.api.block.function.primitive.ShortToObjectFunction;
import com.gs.collections.api.block.function.primitive.ObjectShortToObjectFunction;
import com.gs.collections.api.block.predicate.primitive.ShortPredicate;
import com.gs.collections.api.block.procedure.primitive.ShortProcedure;
import com.gs.collections.api.iterator.ShortIterator;
import com.gs.collections.api.list.primitive.ShortList;
import com.gs.collections.api.list.primitive.MutableShortList;
import com.gs.collections.api.set.primitive.MutableShortSet;
import com.gs.collections.api.stack.MutableStack;
import com.gs.collections.api.stack.primitive.ShortStack;
import com.gs.collections.api.stack.primitive.ImmutableShortStack;
import com.gs.collections.api.stack.primitive.MutableShortStack;
import com.gs.collections.impl.bag.mutable.primitive.ShortHashBag;
import com.gs.collections.impl.factory.primitive.ShortStacks;
import com.gs.collections.impl.lazy.primitive.LazyShortIterableAdapter;
import com.gs.collections.impl.list.mutable.primitive.ShortArrayList;
import com.gs.collections.impl.set.mutable.primitive.ShortHashSet;
import com.gs.collections.impl.stack.mutable.ArrayStack;
import net.jcip.annotations.NotThreadSafe;

/**
 * ShortArrayStack is similar to {@link ArrayStack}, and is memory-optimized for short primitives.
 * This file was automatically generated from template file primitiveArrayStack.stg.
 */
@NotThreadSafe
public final class ShortArrayStack
        implements MutableShortStack, Externalizable
{
    private static final long serialVersionUID = 1L;

    private transient ShortArrayList delegate;

    public ShortArrayStack()
    {
        this.delegate = new ShortArrayList();
    }

    private ShortArrayStack(int size)
    {
        this.delegate = new ShortArrayList(size);
    }

    private ShortArrayStack(short... items)
    {
        this.delegate = new ShortArrayList(items);
    }

    public static ShortArrayStack newStackFromTopToBottom(short... items)
    {
        ShortArrayStack stack = new ShortArrayStack(items.length);
        for (int i = items.length - 1; i >= 0; i--)
        {
            stack.push(items[i]);
        }
        return stack;
    }

    public static ShortArrayStack newStackWith(short... items)
    {
        return new ShortArrayStack(items);
    }

    public static ShortArrayStack newStack(ShortIterable items)
    {
        ShortArrayStack stack = new ShortArrayStack(items.size());
        stack.delegate = ShortArrayList.newList(items);
        return stack;
    }

    public static ShortArrayStack newStackFromTopToBottom(ShortIterable items)
    {
        ShortArrayStack stack = new ShortArrayStack(items.size());
        stack.delegate = ShortArrayList.newList(items).reverseThis();
        return stack;
    }

    public void push(short item)
    {
        this.delegate.add(item);
    }

    public short pop()
    {
        this.checkEmptyStack();
        return this.delegate.removeAtIndex(this.delegate.size() - 1);
    }

    private void checkEmptyStack()
    {
        if (this.delegate.isEmpty())
        {
            throw new EmptyStackException();
        }
    }

    public ShortList pop(int count)
    {
        this.checkPositiveValueForCount(count);
        this.checkSizeLessThanCount(count);
        if (count == 0)
        {
            return new ShortArrayList(0);
        }
        MutableShortList subList = new ShortArrayList(count);
        while (count > 0)
        {
            subList.add(this.pop());
            count--;
        }
        return subList;
    }

    private void checkPositiveValueForCount(int count)
    {
        if (count < 0)
        {
            throw new IllegalArgumentException("Count must be positive but was " + count);
        }
    }

    private void checkSizeLessThanCount(int count)
    {
        if (this.delegate.size() < count)
        {
            throw new IllegalArgumentException("Count must be less than size: Count = " + count + " Size = " + this.delegate.size());
        }
    }

    public MutableShortStack select(ShortPredicate predicate)
    {
        return newStackFromTopToBottom(this.delegate.asReversed().select(predicate));
    }

    public MutableShortStack reject(ShortPredicate predicate)
    {
        return newStackFromTopToBottom(this.delegate.asReversed().reject(predicate));
    }

    public short peek()
    {
        this.checkEmptyStack();
        return this.delegate.getLast();
    }

    public ShortList peek(int count)
    {
        this.checkPositiveValueForCount(count);
        this.checkSizeLessThanCount(count);
        if (count == 0)
        {
            return new ShortArrayList(0);
        }
        MutableShortList subList = new ShortArrayList(count);
        int index = this.delegate.size() - 1;
        for (int i = 0; i < count; i++)
        {
            subList.add(this.delegate.get(index - i));
        }
        return subList;
    }

    public short peekAt(int index)
    {
        this.rangeCheck(index);
        return this.delegate.get(this.delegate.size() - 1 - index);
    }

    private void rangeCheck(int index)
    {
        if (index < 0 || index > this.delegate.size() - 1)
        {
            throw new IllegalArgumentException("Index " + index + " out of range.Should be between 0 and " + (this.delegate.size() - 1));
        }
    }

    public ShortIterator shortIterator()
    {
        return this.delegate.asReversed().shortIterator();
    }

    public void forEach(ShortProcedure procedure)
    {
        this.delegate.asReversed().forEach(procedure);
    }

    public int size()
    {
        return this.delegate.size();
    }

    public boolean isEmpty()
    {
        return this.delegate.isEmpty();
    }

    public boolean notEmpty()
    {
        return this.delegate.notEmpty();
    }

    public int count(ShortPredicate predicate)
    {
        return this.delegate.asReversed().count(predicate);
    }

    public boolean anySatisfy(ShortPredicate predicate)
    {
        return this.delegate.asReversed().anySatisfy(predicate);
    }

    public boolean allSatisfy(ShortPredicate predicate)
    {
        return this.delegate.asReversed().allSatisfy(predicate);
    }

    public boolean noneSatisfy(ShortPredicate predicate)
    {
        return this.delegate.asReversed().noneSatisfy(predicate);
    }

    public short detectIfNone(ShortPredicate predicate, short ifNone)
    {
        return this.delegate.asReversed().detectIfNone(predicate, ifNone);
    }

    public <V> MutableStack<V> collect(ShortToObjectFunction<? extends V> function)
    {
        return ArrayStack.newStackFromTopToBottom(this.delegate.asReversed().collect(function));
    }

    public <V> V injectInto(V injectedValue, ObjectShortToObjectFunction<? super V, ? extends V> function)
    {
        //TODO changed to asReversed() once added to IntIterable
        return this.delegate.toReversed().injectInto(injectedValue, function);
    }

    public long sum()
    {
        return this.delegate.sum();
    }

    public short max()
    {
        return this.delegate.max();
    }

    public short min()
    {
        return this.delegate.min();
    }

    public short minIfEmpty(short defaultValue)
    {
        if (this.isEmpty())
        {
            return defaultValue;
        }
        return this.min();
    }

    public short maxIfEmpty(short defaultValue)
    {
        if (this.isEmpty())
        {
            return defaultValue;
        }
        return this.max();
    }

    public double average()
    {
        return this.delegate.average();
    }

    public double median()
    {
        return this.delegate.median();
    }

    public short[] toSortedArray()
    {
        return this.delegate.toSortedArray();
    }

    public short[] toArray()
    {
        return this.delegate.asReversed().toArray();
    }

    public boolean contains(short value)
    {
        return this.delegate.asReversed().contains(value);
    }

    public boolean containsAll(short... source)
    {
        return this.delegate.asReversed().containsAll(source);
    }

    public boolean containsAll(ShortIterable source)
    {
        return this.delegate.asReversed().containsAll(source);
    }

    public void clear()
    {
        this.delegate.clear();
    }

    @Override
    public boolean equals(Object otherStack)
    {
        if (otherStack == this)
        {
            return true;
        }
        if (!(otherStack instanceof ShortStack))
        {
            return false;
        }
        ShortStack stack = (ShortStack) otherStack;
        if (this.size() != stack.size())
        {
            return false;
        }
        for (int i = 0; i < this.size(); i++)
        {
            if (this.peekAt(i) != stack.peekAt(i))
            {
                return false;
            }
        }
        return true;
    }

    @Override
    public int hashCode()
    {
        int hashCode = 1;
        ShortIterable iterable = this.delegate.asReversed();
        ShortIterator iterator = iterable.shortIterator();
        while (iterator.hasNext())
        {
            short item = iterator.next();
            hashCode = 31 * hashCode + (int) item;
        }
        return hashCode;
    }

    @Override
    public String toString()
    {
        return this.delegate.asReversed().toString();
    }

    public String makeString()
    {
        return this.delegate.asReversed().makeString();
    }

    public String makeString(String separator)
    {
        return this.delegate.asReversed().makeString(separator);
    }

    public String makeString(String start, String separator, String end)
    {
        return this.delegate.asReversed().makeString(start, separator, end);
    }

    public void appendString(Appendable appendable)
    {
        this.delegate.asReversed().appendString(appendable);
    }

    public void appendString(Appendable appendable, String separator)
    {
        this.delegate.asReversed().appendString(appendable, separator);
    }

    public void appendString(
            Appendable appendable,
            String start,
            String separator,
            String end)
    {
        this.delegate.asReversed().appendString(appendable, start, separator, end);
    }

    public MutableShortList toList()
    {
        return ShortArrayList.newList(this);
    }

    public MutableShortList toSortedList()
    {
        return ShortArrayList.newList(this).sortThis();
    }

    public MutableShortSet toSet()
    {
        return ShortHashSet.newSet(this);
    }

    public MutableShortBag toBag()
    {
        return ShortHashBag.newBag(this);
    }

    public LazyShortIterable asLazy()
    {
        return new LazyShortIterableAdapter(this);
    }

    public MutableShortStack asUnmodifiable()
    {
        return new UnmodifiableShortStack(this);
    }

    public MutableShortStack asSynchronized()
    {
        return new SynchronizedShortStack(this);
    }

    public ImmutableShortStack toImmutable()
    {
        return ShortStacks.immutable.withAll(this.delegate);
    }

    public void writeExternal(ObjectOutput out) throws IOException
    {
        out.writeInt(this.size());
        ShortIterator iterator = this.delegate.asReversed().shortIterator();
        while (iterator.hasNext())
        {
            short each = iterator.next();
            out.writeShort(each);
        }
    }

    public void readExternal(ObjectInput in) throws IOException
    {
        int size = in.readInt();
        short[] array = new short[size];
        for (int i = size - 1; i >= 0; i--)
        {
            array[i] = in.readShort();
        }
        this.delegate = ShortArrayList.newListWith(array);
    }
}