/*
 * Copyright 2014 Goldman Sachs.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.gs.collections.impl.bag.immutable.primitive;

import java.io.Externalizable;
import java.io.IOException;
import java.io.ObjectInput;
import java.io.ObjectOutput;
import java.io.Serializable;

import com.gs.collections.api.CharIterable;
import com.gs.collections.api.LazyCharIterable;
import com.gs.collections.api.bag.ImmutableBag;
import com.gs.collections.api.bag.primitive.CharBag;
import com.gs.collections.api.bag.primitive.ImmutableCharBag;
import com.gs.collections.api.bag.primitive.MutableCharBag;
import com.gs.collections.api.block.function.primitive.CharToObjectFunction;
import com.gs.collections.api.block.function.primitive.ObjectCharToObjectFunction;
import com.gs.collections.api.block.predicate.primitive.CharPredicate;
import com.gs.collections.api.block.procedure.primitive.CharIntProcedure;
import com.gs.collections.api.block.procedure.primitive.CharProcedure;
import com.gs.collections.api.iterator.CharIterator;
import com.gs.collections.api.list.primitive.MutableCharList;
import com.gs.collections.api.set.primitive.MutableCharSet;
import com.gs.collections.impl.bag.mutable.primitive.CharHashBag;
import com.gs.collections.impl.block.procedure.checked.primitive.CheckedCharIntProcedure;

/**
 * ImmutableCharHashBag is the non-modifiable equivalent of {@link CharHashBag}.
 * This file was automatically generated from template file immutablePrimitiveHashBag.stg.
 *
 * @since 4.0.
 */
final class ImmutableCharHashBag implements ImmutableCharBag, Serializable
{
    private static final long serialVersionUID = 1L;

    private final MutableCharBag delegate;

    private ImmutableCharHashBag(char[] newElements)
    {
        this.checkOptimizedSize(newElements.length);
        this.delegate = CharHashBag.newBagWith(newElements);
    }

    private void checkOptimizedSize(int length)
    {
        if (length <= 1)
        {
            throw new IllegalArgumentException("Use CharBags.immutable.with() to instantiate an optimized collection");
        }
    }

    public static ImmutableCharHashBag newBagWith(char... elements)
    {
        return new ImmutableCharHashBag(elements);
    }

    public ImmutableCharBag newWith(char element)
    {
        return CharHashBag.newBag(this.delegate).with(element).toImmutable();
    }

    public ImmutableCharBag newWithout(char element)
    {
        CharHashBag hashBag = CharHashBag.newBag(this.delegate);
        hashBag.remove(element);
        return hashBag.toImmutable();
    }

    public ImmutableCharBag newWithAll(CharIterable elements)
    {
        CharHashBag bag = CharHashBag.newBag(this.delegate);
        bag.addAll(elements);
        return bag.toImmutable();
    }

    public ImmutableCharBag newWithoutAll(CharIterable elements)
    {
        CharHashBag bag = CharHashBag.newBag(this.delegate);
        bag.removeAll(elements);
        return bag.toImmutable();
    }

    public int size()
    {
        return this.delegate.size();
    }

    public boolean isEmpty()
    {
        return this.delegate.isEmpty();
    }

    public boolean notEmpty()
    {
        return this.delegate.notEmpty();
    }

    public boolean contains(char value)
    {
        return this.delegate.contains(value);
    }

    public boolean containsAll(CharIterable source)
    {
        return this.delegate.containsAll(source);
    }

    public boolean containsAll(char... elements)
    {
        return this.delegate.containsAll(elements);
    }

    public void forEach(CharProcedure procedure)
    {
        this.delegate.forEach(procedure);
    }

    public ImmutableCharBag select(CharPredicate predicate)
    {
        return this.delegate.select(predicate).toImmutable();
    }

    public ImmutableCharBag reject(CharPredicate predicate)
    {
        return this.delegate.reject(predicate).toImmutable();
    }

    public <V> ImmutableBag<V> collect(CharToObjectFunction<? extends V> function)
    {
        return this.delegate.collect(function).toImmutable();
    }

    public MutableCharList toList()
    {
        return this.delegate.toList();
    }

    public int sizeDistinct()
    {
        return this.delegate.sizeDistinct();
    }

    public int occurrencesOf(char item)
    {
        return this.delegate.occurrencesOf(item);
    }

    public void forEachWithOccurrences(CharIntProcedure charIntProcedure)
    {
        this.delegate.forEachWithOccurrences(charIntProcedure);
    }

    public char detectIfNone(CharPredicate predicate, char ifNone)
    {
        return this.delegate.detectIfNone(predicate, ifNone);
    }

    public int count(CharPredicate predicate)
    {
        return this.delegate.count(predicate);
    }

    public boolean anySatisfy(CharPredicate predicate)
    {
        return this.delegate.anySatisfy(predicate);
    }

    public long sum()
    {
        return this.delegate.sum();
    }

    public char min()
    {
        return this.delegate.min();
    }

    public char max()
    {
        return this.delegate.max();
    }

    public char maxIfEmpty(char defaultValue)
    {
        return this.delegate.maxIfEmpty(defaultValue);
    }

    public char minIfEmpty(char defaultValue)
    {
        return this.delegate.minIfEmpty(defaultValue);
    }

    public double average()
    {
        return this.delegate.average();
    }

    public double median()
    {
        return this.delegate.median();
    }

    public char[] toSortedArray()
    {
        return this.delegate.toSortedArray();
    }

    public MutableCharList toSortedList()
    {
        return this.delegate.toSortedList();
    }

    public boolean noneSatisfy(CharPredicate predicate)
    {
        return this.delegate.noneSatisfy(predicate);
    }

    public boolean allSatisfy(CharPredicate predicate)
    {
        return this.delegate.allSatisfy(predicate);
    }

    public <T> T injectInto(T injectedValue, ObjectCharToObjectFunction<? super T, ? extends T> function)
    {
        return ((CharHashBag) this.delegate).injectInto(injectedValue, function);
    }

    @Override
    public boolean equals(Object obj)
    {
        return this.delegate.equals(obj);
    }

    @Override
    public int hashCode()
    {
        return this.delegate.hashCode();
    }

    public MutableCharSet toSet()
    {
        return this.delegate.toSet();
    }

    public MutableCharBag toBag()
    {
        return this.delegate.toBag();
    }

    public ImmutableCharBag toImmutable()
    {
        return this;
    }

    public LazyCharIterable asLazy()
    {
        return this.delegate.asLazy();
    }

    public char[] toArray()
    {
        return this.delegate.toArray();
    }

    @Override
    public String toString()
    {
        return this.delegate.toString();
    }

    public String makeString()
    {
        return this.delegate.makeString();
    }

    public String makeString(String separator)
    {
        return this.delegate.makeString(separator);
    }

    public String makeString(String start, String separator, String end)
    {
        return this.delegate.makeString(start, separator, end);
    }

    public void appendString(Appendable appendable)
    {
        this.delegate.appendString(appendable);
    }

    public void appendString(Appendable appendable, String separator)
    {
        this.delegate.appendString(appendable, separator);
    }

    public void appendString(Appendable appendable, String start, String separator, String end)
    {
        this.delegate.appendString(appendable, start, separator, end);
    }

    public CharIterator charIterator()
    {
        return this.delegate.charIterator();
    }

    private Object writeReplace()
    {
        return new ImmutableCharBagSerializationProxy(this);
    }

    protected static class ImmutableCharBagSerializationProxy implements Externalizable
    {
        private static final long serialVersionUID = 1L;

        private CharBag bag;

        @SuppressWarnings("UnusedDeclaration")
        public ImmutableCharBagSerializationProxy()
        {
            // Empty constructor for Externalizable class
        }

        protected ImmutableCharBagSerializationProxy(CharBag bag)
        {
            this.bag = bag;
        }

        public void writeExternal(final ObjectOutput out) throws IOException
        {
            out.writeInt(this.bag.sizeDistinct());
            try
            {
                this.bag.forEachWithOccurrences(new CheckedCharIntProcedure()
                {
                    @Override
                    public void safeValue(char item, int count) throws IOException
                    {
                        out.writeChar(item);
                        out.writeInt(count);
                    }
                });
            }
            catch (RuntimeException e)
            {
                if (e.getCause() instanceof IOException)
                {
                    throw (IOException) e.getCause();
                }
                throw e;
            }
        }

        public void readExternal(ObjectInput in) throws IOException, ClassNotFoundException
        {
            int size = in.readInt();
            MutableCharBag deserializedBag = new CharHashBag();

            for (int i = 0; i < size; i++)
            {
                deserializedBag.addOccurrences(in.readChar(), in.readInt());
            }

            this.bag = deserializedBag;
        }

        protected Object readResolve()
        {
            return this.bag.toImmutable();
        }
    }
}
