/*
 * Copyright 2014 Goldman Sachs.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.gs.collections.impl.map.mutable.primitive;

import java.io.Externalizable;
import java.io.IOException;
import java.io.ObjectInput;
import java.io.ObjectOutput;
import java.util.Arrays;
import java.util.Iterator;
import java.util.NoSuchElementException;

import com.gs.collections.api.LongIterable;
import com.gs.collections.api.LazyLongIterable;
import com.gs.collections.api.LazyIterable;
import com.gs.collections.api.RichIterable;
import com.gs.collections.api.bag.primitive.MutableLongBag;
import com.gs.collections.api.block.function.primitive.LongFunction;
import com.gs.collections.api.block.function.primitive.LongFunction0;
import com.gs.collections.api.block.function.primitive.LongToLongFunction;
import com.gs.collections.api.block.function.primitive.LongToObjectFunction;
import com.gs.collections.api.block.function.primitive.ObjectLongToObjectFunction;
import com.gs.collections.api.block.predicate.primitive.LongLongPredicate;
import com.gs.collections.api.block.predicate.primitive.LongPredicate;
import com.gs.collections.api.block.procedure.Procedure;
import com.gs.collections.api.block.procedure.Procedure2;
import com.gs.collections.api.block.procedure.primitive.LongProcedure;
import com.gs.collections.api.block.procedure.primitive.LongLongProcedure;
import com.gs.collections.api.block.procedure.primitive.ObjectIntProcedure;
import com.gs.collections.api.collection.MutableCollection;
import com.gs.collections.api.collection.primitive.ImmutableLongCollection;
import com.gs.collections.api.collection.primitive.MutableLongCollection;
import com.gs.collections.api.iterator.LongIterator;
import com.gs.collections.api.list.primitive.MutableLongList;
import com.gs.collections.api.map.primitive.LongLongMap;
import com.gs.collections.api.map.primitive.ImmutableLongLongMap;
import com.gs.collections.api.map.primitive.MutableLongLongMap;
import com.gs.collections.api.set.MutableSet;
import com.gs.collections.api.set.primitive.LongSet;
import com.gs.collections.api.set.primitive.ImmutableLongSet;
import com.gs.collections.api.set.primitive.MutableLongSet;
import com.gs.collections.api.tuple.primitive.LongLongPair;
import com.gs.collections.impl.bag.mutable.primitive.LongHashBag;
import com.gs.collections.impl.block.factory.primitive.LongPredicates;
import com.gs.collections.impl.collection.mutable.primitive.SynchronizedLongCollection;
import com.gs.collections.impl.collection.mutable.primitive.UnmodifiableLongCollection;
import com.gs.collections.impl.factory.Sets;
import com.gs.collections.impl.factory.primitive.LongLists;
import com.gs.collections.impl.factory.primitive.LongLongMaps;
import com.gs.collections.impl.factory.primitive.LongSets;
import com.gs.collections.impl.lazy.AbstractLazyIterable;
import com.gs.collections.impl.lazy.primitive.AbstractLazyLongIterable;
import com.gs.collections.impl.lazy.primitive.LazyLongIterableAdapter;
import com.gs.collections.impl.list.mutable.FastList;
import com.gs.collections.impl.list.mutable.primitive.LongArrayList;
import com.gs.collections.impl.set.mutable.primitive.LongHashSet;
import com.gs.collections.impl.set.mutable.primitive.SynchronizedLongSet;
import com.gs.collections.impl.set.mutable.primitive.UnmodifiableLongSet;
import com.gs.collections.impl.tuple.primitive.PrimitiveTuples;

/**
 * This file was automatically generated from template file primitivePrimitiveHashMap.stg.
 *
 * @since 3.0.
 */
public class LongLongHashMap implements MutableLongLongMap, Externalizable
{
    static final long EMPTY_VALUE = 0L;
    private static final long serialVersionUID = 1L;
    private static final long EMPTY_KEY = 0L;
    private static final long REMOVED_KEY = 1L;

    private static final int OCCUPIED_DATA_RATIO = 2;
    private static final int OCCUPIED_SENTINEL_RATIO = 4;
    private static final int DEFAULT_INITIAL_CAPACITY = 8;

    private long[] keys;
    private long[] values;

    private int occupiedWithData;
    private int occupiedWithSentinels;

    private SentinelValues sentinelValues;

    public LongLongHashMap()
    {
        this.allocateTable(DEFAULT_INITIAL_CAPACITY << 1);
    }

    public LongLongHashMap(int initialCapacity)
    {
        if (initialCapacity < 0)
        {
            throw new IllegalArgumentException("initial capacity cannot be less than 0");
        }
        int capacity = this.smallestPowerOfTwoGreaterThan(this.fastCeil(initialCapacity * OCCUPIED_DATA_RATIO));
        this.allocateTable(capacity);
    }

    private int smallestPowerOfTwoGreaterThan(int n)
    {
        return n > 1 ? Integer.highestOneBit(n - 1) << 1 : 1;
    }

    public LongLongHashMap(LongLongMap map)
    {
        this(Math.max(map.size(), DEFAULT_INITIAL_CAPACITY));
        this.putAll(map);
    }

    private int fastCeil(float v)
    {
        int possibleResult = (int) v;
        if (v - possibleResult > 0.0F)
        {
            possibleResult++;
        }
        return possibleResult;
    }

    public static LongLongHashMap newWithKeysValues(long key1, long value1)
    {
        return new LongLongHashMap(1).withKeyValue(key1, value1);
    }

    public static LongLongHashMap newWithKeysValues(long key1, long value1, long key2, long value2)
    {
        return new LongLongHashMap(2).withKeysValues(key1, value1, key2, value2);
    }

    public static LongLongHashMap newWithKeysValues(long key1, long value1, long key2, long value2, long key3, long value3)
    {
        return new LongLongHashMap(3).withKeysValues(key1, value1, key2, value2, key3, value3);
    }

    public static LongLongHashMap newWithKeysValues(long key1, long value1, long key2, long value2, long key3, long value3, long key4, long value4)
    {
        return new LongLongHashMap(4).withKeysValues(key1, value1, key2, value2, key3, value3, key4, value4);
    }

    @Override
    public boolean equals(Object obj)
    {
        if (this == obj)
        {
            return true;
        }

        if (!(obj instanceof LongLongMap))
        {
            return false;
        }

        LongLongMap other = (LongLongMap) obj;

        if (this.size() != other.size())
        {
            return false;
        }

        if (this.sentinelValues == null)
        {
            if (other.containsKey(EMPTY_KEY) || other.containsKey(REMOVED_KEY))
            {
                return false;
            }
        }
        else
        {
            if (this.sentinelValues.containsZeroKey && (!other.containsKey(EMPTY_KEY) || this.sentinelValues.zeroValue != other.getOrThrow(EMPTY_KEY)))
            {
                return false;
            }

            if (this.sentinelValues.containsOneKey && (!other.containsKey(REMOVED_KEY) || this.sentinelValues.oneValue != other.getOrThrow(REMOVED_KEY)))
            {
                return false;
            }
        }

        for (int i = 0; i < this.keys.length; i++)
        {
            long key = this.keys[i];
            if (isNonSentinel(key) && (!other.containsKey(key) || this.values[i] != other.getOrThrow(key)))
            {
                return false;
            }
        }
        return true;
    }

    @Override
    public int hashCode()
    {
        int result = 0;

        if (this.sentinelValues != null)
        {
            if (this.sentinelValues.containsZeroKey)
            {
                result += (int) (EMPTY_KEY ^ EMPTY_KEY >>> 32) ^ (int) (this.sentinelValues.zeroValue ^ this.sentinelValues.zeroValue >>> 32);
            }
            if (this.sentinelValues.containsOneKey)
            {
                result += (int) (REMOVED_KEY ^ REMOVED_KEY >>> 32) ^ (int) (this.sentinelValues.oneValue ^ this.sentinelValues.oneValue >>> 32);
            }
        }
        for (int i = 0; i < this.keys.length; i++)
        {
            if (isNonSentinel(this.keys[i]))
            {
                result += (int) (this.keys[i] ^ this.keys[i] >>> 32) ^ (int) (this.values[i] ^ this.values[i] >>> 32);
            }
        }

        return result;
    }

    @Override
    public String toString()
    {
        StringBuilder appendable = new StringBuilder();

        appendable.append("{");

        boolean first = true;

        if (this.sentinelValues != null)
        {
            if (this.sentinelValues.containsZeroKey)
            {
                appendable.append(String.valueOf(EMPTY_KEY)).append("=").append(String.valueOf(this.sentinelValues.zeroValue));
                first = false;
            }
            if (this.sentinelValues.containsOneKey)
            {
                if (!first)
                {
                    appendable.append(", ");
                }
                appendable.append(String.valueOf(REMOVED_KEY)).append("=").append(String.valueOf(this.sentinelValues.oneValue));
                first = false;
            }
        }
        for (int i = 0; i < this.keys.length; i++)
        {
            long key = this.keys[i];
            if (isNonSentinel(key))
            {
                if (!first)
                {
                    appendable.append(", ");
                }
                appendable.append(String.valueOf(key)).append("=").append(String.valueOf(this.values[i]));
                first = false;
            }
        }
        appendable.append("}");

        return appendable.toString();
    }

    public int size()
    {
        return this.occupiedWithData + (this.sentinelValues == null ? 0 : this.sentinelValues.size());
    }

    public boolean isEmpty()
    {
        return this.occupiedWithData == 0 && (this.sentinelValues == null || this.sentinelValues.size() == 0);
    }

    public boolean notEmpty()
    {
        return this.occupiedWithData != 0 || (this.sentinelValues != null && this.sentinelValues.size() != 0);
    }

    public String makeString()
    {
        return this.makeString(", ");
    }

    public String makeString(String separator)
    {
        return this.makeString("", separator, "");
    }

    public String makeString(String start, String separator, String end)
    {
        Appendable stringBuilder = new StringBuilder();
        this.appendString(stringBuilder, start, separator, end);
        return stringBuilder.toString();
    }

    public void appendString(Appendable appendable)
    {
        this.appendString(appendable, ", ");
    }

    public void appendString(Appendable appendable, String separator)
    {
        this.appendString(appendable, "", separator, "");
    }

    public void appendString(Appendable appendable, String start, String separator, String end)
    {
        try
        {
            appendable.append(start);

            boolean first = true;

            if (this.sentinelValues != null)
            {
                if (this.sentinelValues.containsZeroKey)
                {
                    appendable.append(String.valueOf(this.sentinelValues.zeroValue));
                    first = false;
                }
                if (this.sentinelValues.containsOneKey)
                {
                    if (!first)
                    {
                        appendable.append(separator);
                    }
                    appendable.append(String.valueOf(this.sentinelValues.oneValue));
                    first = false;
                }
            }
            for (int i = 0; i < this.keys.length; i++)
            {
                long key = this.keys[i];
                if (isNonSentinel(key))
                {
                    if (!first)
                    {
                        appendable.append(separator);
                    }
                    appendable.append(String.valueOf(this.values[i]));
                    first = false;
                }
            }
            appendable.append(end);
        }
        catch (IOException e)
        {
            throw new RuntimeException(e);
        }
    }

    public LongIterator longIterator()
    {
        return new InternalLongIterator();
    }

    public long[] toArray()
    {
        long[] array = new long[this.size()];
        int index = 0;

        if (this.sentinelValues != null)
        {
            if (this.sentinelValues.containsZeroKey)
            {
                array[index] = this.sentinelValues.zeroValue;
                index++;
            }
            if (this.sentinelValues.containsOneKey)
            {
                array[index] = this.sentinelValues.oneValue;
                index++;
            }
        }
        for (int i = 0; i < this.keys.length; i++)
        {
            if (isNonSentinel(this.keys[i]))
            {
                array[index] = this.values[i];
                index++;
            }
        }

        return array;
    }

    public boolean contains(long value)
    {
        return this.containsValue(value);
    }

    public boolean containsAll(long... source)
    {
        for (long each : source)
        {
            if (!this.contains(each))
            {
                return false;
            }
        }
        return true;
    }

    public boolean containsAll(LongIterable source)
    {
        return source.allSatisfy(new LongPredicate()
        {
            public boolean accept(long value)
            {
                return LongLongHashMap.this.contains(value);
            }
        });
    }

    public void forEach(LongProcedure procedure)
    {
        this.forEachValue(procedure);
    }

    public MutableLongCollection select(LongPredicate predicate)
    {
        LongArrayList result = new LongArrayList();

        if (this.sentinelValues != null)
        {
            if (this.sentinelValues.containsZeroKey && predicate.accept(this.sentinelValues.zeroValue))
            {
                result.add(this.sentinelValues.zeroValue);
            }
            if (this.sentinelValues.containsOneKey && predicate.accept(this.sentinelValues.oneValue))
            {
                result.add(this.sentinelValues.oneValue);
            }
        }
        for (int i = 0; i < this.keys.length; i++)
        {
            if (isNonSentinel(this.keys[i]) && predicate.accept(this.values[i]))
            {
                result.add(this.values[i]);
            }
        }

        return result;
    }

    public MutableLongCollection reject(LongPredicate predicate)
    {
        LongArrayList result = new LongArrayList();
        if (this.sentinelValues != null)
        {
            if (this.sentinelValues.containsZeroKey && !predicate.accept(this.sentinelValues.zeroValue))
            {
                result.add(this.sentinelValues.zeroValue);
            }
            if (this.sentinelValues.containsOneKey && !predicate.accept(this.sentinelValues.oneValue))
            {
                result.add(this.sentinelValues.oneValue);
            }
        }
        for (int i = 0; i < this.keys.length; i++)
        {
            if (isNonSentinel(this.keys[i]) && !predicate.accept(this.values[i]))
            {
                result.add(this.values[i]);
            }
        }
        return result;
    }

    public <V> MutableCollection<V> collect(LongToObjectFunction<? extends V> function)
    {
        FastList<V> target = FastList.newList(this.size());
        if (this.sentinelValues != null)
        {
            if (this.sentinelValues.containsZeroKey)
            {
                target.add(function.valueOf(this.sentinelValues.zeroValue));
            }
            if (this.sentinelValues.containsOneKey)
            {
                target.add(function.valueOf(this.sentinelValues.oneValue));
            }
        }
        for (int i = 0; i < this.keys.length; i++)
        {
            if (isNonSentinel(this.keys[i]))
            {
                target.add(function.valueOf(this.values[i]));
            }
        }
        return target;
    }

    public long detectIfNone(LongPredicate predicate, long value)
    {
        if (this.sentinelValues != null)
        {
            if (this.sentinelValues.containsZeroKey && predicate.accept(this.sentinelValues.zeroValue))
            {
                return this.sentinelValues.zeroValue;
            }
            if (this.sentinelValues.containsOneKey && predicate.accept(this.sentinelValues.oneValue))
            {
                return this.sentinelValues.oneValue;
            }
        }
        for (int i = 0; i < this.keys.length; i++)
        {
            if (isNonSentinel(this.keys[i]) && predicate.accept(this.values[i]))
            {
                return this.values[i];
            }
        }
        return value;
    }

    public int count(LongPredicate predicate)
    {
        int count = 0;
        if (this.sentinelValues != null)
        {
            if (this.sentinelValues.containsZeroKey && predicate.accept(this.sentinelValues.zeroValue))
            {
                count++;
            }
            if (this.sentinelValues.containsOneKey && predicate.accept(this.sentinelValues.oneValue))
            {
                count++;
            }
        }
        for (int i = 0; i < this.keys.length; i++)
        {
            if (isNonSentinel(this.keys[i]) && predicate.accept(this.values[i]))
            {
                count++;
            }
        }
        return count;
    }

    public boolean anySatisfy(LongPredicate predicate)
    {
        if (this.sentinelValues != null)
        {
            if (this.sentinelValues.containsZeroKey && predicate.accept(this.sentinelValues.zeroValue))
            {
                return true;
            }
            if (this.sentinelValues.containsOneKey && predicate.accept(this.sentinelValues.oneValue))
            {
                return true;
            }
        }
        for (int i = 0; i < this.keys.length; i++)
        {
            if (isNonSentinel(this.keys[i]) && predicate.accept(this.values[i]))
            {
                return true;
            }
        }
        return false;
    }

    public boolean allSatisfy(LongPredicate predicate)
    {
        if (this.sentinelValues != null)
        {
            if (this.sentinelValues.containsZeroKey && !predicate.accept(this.sentinelValues.zeroValue))
            {
                return false;
            }
            if (this.sentinelValues.containsOneKey && !predicate.accept(this.sentinelValues.oneValue))
            {
                return false;
            }
        }
        for (int i = 0; i < this.keys.length; i++)
        {
            if (isNonSentinel(this.keys[i]) && !predicate.accept(this.values[i]))
            {
                return false;
            }
        }
        return true;
    }

    public boolean noneSatisfy(LongPredicate predicate)
    {
        if (this.sentinelValues != null)
        {
            if (this.sentinelValues.containsZeroKey && predicate.accept(this.sentinelValues.zeroValue))
            {
                return false;
            }
            if (this.sentinelValues.containsOneKey && predicate.accept(this.sentinelValues.oneValue))
            {
                return false;
            }
        }
        for (int i = 0; i < this.keys.length; i++)
        {
            if (isNonSentinel(this.keys[i]) && predicate.accept(this.values[i]))
            {
                return false;
            }
        }
        return true;
    }

    public <V> V injectInto(V injectedValue, ObjectLongToObjectFunction<? super V, ? extends V> function)
    {
        V result = injectedValue;

        if (this.sentinelValues != null)
        {
            if (this.sentinelValues.containsZeroKey)
            {
                result = function.valueOf(result, this.sentinelValues.zeroValue);
            }
            if (this.sentinelValues.containsOneKey)
            {
                result = function.valueOf(result, this.sentinelValues.oneValue);
            }
        }
        for (int i = 0; i < this.keys.length; i++)
        {
            if (isNonSentinel(this.keys[i]))
            {
                result = function.valueOf(result, this.values[i]);
            }
        }

        return result;
    }

    public MutableLongList toList()
    {
        return LongArrayList.newList(this);
    }

    public MutableLongSet toSet()
    {
        return LongHashSet.newSet(this);
    }

    public MutableLongBag toBag()
    {
        return LongHashBag.newBag(this);
    }

    public LazyLongIterable asLazy()
    {
        return new LazyLongIterableAdapter(this);
    }

    public void clear()
    {
        this.sentinelValues = null;
        this.occupiedWithData = 0;
        this.occupiedWithSentinels = 0;
        Arrays.fill(this.keys, EMPTY_KEY);
        Arrays.fill(this.values, EMPTY_VALUE);
    }

    public void put(long key, long value)
    {
        if (isEmptyKey(key))
        {
            if (this.sentinelValues == null)
            {
                this.sentinelValues = new SentinelValues();
            }
            this.addEmptyKeyValue(value);
            return;
        }

        if (isRemovedKey(key))
        {
            if (this.sentinelValues == null)
            {
                this.sentinelValues = new SentinelValues();
            }
            this.addRemovedKeyValue(value);
            return;
        }

        int index = this.probe(key);

        if (this.keys[index] == key)
        {
            // key already present in map
            this.values[index] = value;
            return;
        }

        this.addKeyValueAtIndex(key, value, index);
    }

    public void putAll(LongLongMap map)
    {
        map.forEachKeyValue(new LongLongProcedure()
        {
            public void value(long key, long value)
            {
                LongLongHashMap.this.put(key, value);
            }
        });
    }

    public void removeKey(long key)
    {
        if (isEmptyKey(key))
        {
            if (this.sentinelValues == null || !this.sentinelValues.containsZeroKey)
            {
                return;
            }
            this.removeEmptyKey();
            return;
        }
        if (isRemovedKey(key))
        {
            if (this.sentinelValues == null || !this.sentinelValues.containsOneKey)
            {
                return;
            }
            this.removeRemovedKey();
            return;
        }
        int index = this.probe(key);
        if (this.keys[index] == key)
        {
            this.keys[index] = REMOVED_KEY;
            this.values[index] = EMPTY_VALUE;
            this.occupiedWithData--;
            this.occupiedWithSentinels++;
            if (this.occupiedWithSentinels > this.maxOccupiedWithSentinels())
            {
                this.rehash();
            }
        }
    }

    public void remove(long key)
    {
        this.removeKey(key);
    }

    public long removeKeyIfAbsent(long key, long value)
    {
        if (isEmptyKey(key))
        {
            if (this.sentinelValues == null || !this.sentinelValues.containsZeroKey)
            {
                return value;
            }
            long oldValue = this.sentinelValues.zeroValue;
            this.removeEmptyKey();
            return oldValue;
        }
        if (isRemovedKey(key))
        {
            if (this.sentinelValues == null || !this.sentinelValues.containsOneKey)
            {
                return value;
            }
            long oldValue = this.sentinelValues.oneValue;
            this.removeRemovedKey();
            return oldValue;
        }
        int index = this.probe(key);
        if (this.keys[index] == key)
        {
            this.keys[index] = REMOVED_KEY;
            long oldValue = this.values[index];
            this.values[index] = EMPTY_VALUE;
            this.occupiedWithData--;
            this.occupiedWithSentinels++;
            if (this.occupiedWithSentinels > this.maxOccupiedWithSentinels())
            {
                this.rehash();
            }

            return oldValue;
        }
        return value;
    }

    public long getIfAbsentPut(long key, long value)
    {
        if (isEmptyKey(key))
        {
            if (this.sentinelValues == null)
            {
                this.sentinelValues = new SentinelValues();
                this.addEmptyKeyValue(value);
                return value;
            }
            if (this.sentinelValues.containsZeroKey)
            {
                return this.sentinelValues.zeroValue;
            }
            this.addEmptyKeyValue(value);
            return value;
        }
        if (isRemovedKey(key))
        {
            if (this.sentinelValues == null)
            {
                this.sentinelValues = new SentinelValues();
                this.addRemovedKeyValue(value);
                return value;
            }
            if (this.sentinelValues.containsOneKey)
            {
                return this.sentinelValues.oneValue;
            }
            this.addRemovedKeyValue(value);
            return value;
        }
        int index = this.probe(key);
        if (this.keys[index] == key)
        {
            return this.values[index];
        }
        this.addKeyValueAtIndex(key, value, index);
        return value;
    }

    public long getIfAbsentPut(long key, LongFunction0 function)
    {
        if (isEmptyKey(key))
        {
            if (this.sentinelValues == null)
            {
                long value = function.value();
                this.sentinelValues = new SentinelValues();
                this.addEmptyKeyValue(value);
                return value;
            }
            if (this.sentinelValues.containsZeroKey)
            {
                return this.sentinelValues.zeroValue;
            }
            long value = function.value();
            this.addEmptyKeyValue(value);
            return value;
        }
        if (isRemovedKey(key))
        {
            if (this.sentinelValues == null)
            {
                long value = function.value();
                this.sentinelValues = new SentinelValues();
                this.addRemovedKeyValue(value);
                return value;
            }
            if (this.sentinelValues.containsOneKey)
            {
                return this.sentinelValues.oneValue;
            }
            long value = function.value();
            this.addRemovedKeyValue(value);
            return value;
        }
        int index = this.probe(key);
        if (this.keys[index] == key)
        {
            return this.values[index];
        }
        long value = function.value();
        this.addKeyValueAtIndex(key, value, index);
        return value;
    }

    public <P> long getIfAbsentPutWith(long key, LongFunction<? super P> function, P parameter)
    {
        if (isEmptyKey(key))
        {
            if (this.sentinelValues == null)
            {
                long value = function.longValueOf(parameter);
                this.sentinelValues = new SentinelValues();
                this.addEmptyKeyValue(value);
                return value;
            }
            if (this.sentinelValues.containsZeroKey)
            {
                return this.sentinelValues.zeroValue;
            }
            long value = function.longValueOf(parameter);
            this.addEmptyKeyValue(value);
            return value;
        }
        if (isRemovedKey(key))
        {
            if (this.sentinelValues == null)
            {
                long value = function.longValueOf(parameter);
                this.sentinelValues = new SentinelValues();
                this.addRemovedKeyValue(value);
                return value;
            }
            if (this.sentinelValues.containsOneKey)
            {
                return this.sentinelValues.oneValue;
            }
            long value = function.longValueOf(parameter);
            this.addRemovedKeyValue(value);
            return value;
        }
        int index = this.probe(key);
        if (this.keys[index] == key)
        {
            return this.values[index];
        }
        long value = function.longValueOf(parameter);
        this.addKeyValueAtIndex(key, value, index);
        return value;
    }

    public long getIfAbsentPutWithKey(long key, LongToLongFunction function)
    {
        if (isEmptyKey(key))
        {
            if (this.sentinelValues == null)
            {
                long value = function.valueOf(key);
                this.sentinelValues = new SentinelValues();
                this.addEmptyKeyValue(value);
                return value;
            }
            if (this.sentinelValues.containsZeroKey)
            {
                return this.sentinelValues.zeroValue;
            }
            long value = function.valueOf(key);
            this.addEmptyKeyValue(value);
            return value;
        }
        if (isRemovedKey(key))
        {
            if (this.sentinelValues == null)
            {
                long value = function.valueOf(key);
                this.sentinelValues = new SentinelValues();
                this.addRemovedKeyValue(value);
                return value;
            }
            if (this.sentinelValues.containsOneKey)
            {
                return this.sentinelValues.oneValue;
            }
            long value = function.valueOf(key);
            this.addRemovedKeyValue(value);
            return value;
        }
        int index = this.probe(key);
        if (this.keys[index] == key)
        {
            return this.values[index];
        }
        long value = function.valueOf(key);
        this.addKeyValueAtIndex(key, value, index);
        return value;
    }

    public long addToValue(long key, long toBeAdded)
    {
        if (isEmptyKey(key))
        {
            if (this.sentinelValues == null)
            {
                this.sentinelValues = new SentinelValues();
                this.addEmptyKeyValue(toBeAdded);
            }
            else if (this.sentinelValues.containsZeroKey)
            {
                this.sentinelValues.zeroValue += toBeAdded;
            }
            else
            {
                this.addEmptyKeyValue(toBeAdded);
            }
            return this.sentinelValues.zeroValue;
        }
        if (isRemovedKey(key))
        {
            if (this.sentinelValues == null)
            {
                this.sentinelValues = new SentinelValues();
                this.addRemovedKeyValue(toBeAdded);
            }
            else if (this.sentinelValues.containsOneKey)
            {
                this.sentinelValues.oneValue += toBeAdded;
            }
            else
            {
                this.addRemovedKeyValue(toBeAdded);
            }
            return this.sentinelValues.oneValue;
        }
        int index = this.probe(key);
        if (this.keys[index] == key)
        {
            this.values[index] += toBeAdded;
            return this.values[index];
        }
        this.addKeyValueAtIndex(key, toBeAdded, index);
        return this.values[index];
    }

    private void addKeyValueAtIndex(long key, long value, int index)
    {
        if (this.keys[index] == REMOVED_KEY)
        {
            this.occupiedWithSentinels--;
        }
        this.keys[index] = key;
        this.values[index] = value;
        this.occupiedWithData++;
        if (this.occupiedWithData > this.maxOccupiedWithData())
        {
            this.rehashAndGrow();
        }
    }

    private void addEmptyKeyValue(long value)
    {
        this.sentinelValues.containsZeroKey = true;
        this.sentinelValues.zeroValue = value;
    }

    private void removeEmptyKey()
    {
        if (this.sentinelValues.containsOneKey)
        {
            this.sentinelValues.containsZeroKey = false;
            this.sentinelValues.zeroValue = EMPTY_VALUE;
        }
        else
        {
            this.sentinelValues = null;
        }
    }

    private void addRemovedKeyValue(long value)
    {
        this.sentinelValues.containsOneKey = true;
        this.sentinelValues.oneValue = value;
    }

    private void removeRemovedKey()
    {
        if (this.sentinelValues.containsZeroKey)
        {
            this.sentinelValues.containsOneKey = false;
            this.sentinelValues.oneValue = EMPTY_VALUE;
        }
        else
        {
            this.sentinelValues = null;
        }
    }

    public long updateValue(long key, long initialValueIfAbsent, LongToLongFunction function)
    {
        if (isEmptyKey(key))
        {
            if (this.sentinelValues == null)
            {
                this.sentinelValues = new SentinelValues();
                this.addEmptyKeyValue(function.valueOf(initialValueIfAbsent));
            }
            else if (this.sentinelValues.containsZeroKey)
            {
                this.sentinelValues.zeroValue = function.valueOf(this.sentinelValues.zeroValue);
            }
            else
            {
                this.addEmptyKeyValue(function.valueOf(initialValueIfAbsent));
            }
            return this.sentinelValues.zeroValue;
        }
        if (isRemovedKey(key))
        {
            if (this.sentinelValues == null)
            {
                this.sentinelValues = new SentinelValues();
                this.addRemovedKeyValue(function.valueOf(initialValueIfAbsent));
            }
            else if (this.sentinelValues.containsOneKey)
            {
                this.sentinelValues.oneValue = function.valueOf(this.sentinelValues.oneValue);
            }
            else
            {
                this.addRemovedKeyValue(function.valueOf(initialValueIfAbsent));
            }
            return this.sentinelValues.oneValue;
        }
        int index = this.probe(key);
        if (this.keys[index] == key)
        {
            this.values[index] = function.valueOf(this.values[index]);
            return this.values[index];
        }
        long value = function.valueOf(initialValueIfAbsent);
        this.addKeyValueAtIndex(key, value, index);
        return value;
    }

    public LongLongHashMap withKeyValue(long key1, long value1)
    {
        this.put(key1, value1);
        return this;
    }

    public LongLongHashMap withKeysValues(long key1, long value1, long key2, long value2)
    {
        this.put(key1, value1);
        this.put(key2, value2);
        return this;
    }

    public LongLongHashMap withKeysValues(long key1, long value1, long key2, long value2, long key3, long value3)
    {
        this.put(key1, value1);
        this.put(key2, value2);
        this.put(key3, value3);
        return this;
    }

    public LongLongHashMap withKeysValues(long key1, long value1, long key2, long value2, long key3, long value3, long key4, long value4)
    {
        this.put(key1, value1);
        this.put(key2, value2);
        this.put(key3, value3);
        this.put(key4, value4);
        return this;
    }

    public LongLongHashMap withoutKey(long key)
    {
        this.removeKey(key);
        return this;
    }

    public LongLongHashMap withoutAllKeys(LongIterable keys)
    {
        keys.forEach(new LongProcedure()
        {
            public void value(long key)
            {
                LongLongHashMap.this.removeKey(key);
            }
        });
        return this;
    }

    public MutableLongLongMap asUnmodifiable()
    {
        return new UnmodifiableLongLongMap(this);
    }

    public MutableLongLongMap asSynchronized()
    {
        return new SynchronizedLongLongMap(this);
    }

    public ImmutableLongLongMap toImmutable()
    {
        return LongLongMaps.immutable.ofAll(this);
    }

    public long get(long key)
    {
        return this.getIfAbsent(key, EMPTY_VALUE);
    }

    public long getIfAbsent(long key, long ifAbsent)
    {
        if (isEmptyKey(key))
        {
            if (this.sentinelValues == null || !this.sentinelValues.containsZeroKey)
            {
                return ifAbsent;
            }
            return this.sentinelValues.zeroValue;
        }
        if (isRemovedKey(key))
        {
            if (this.sentinelValues == null || !this.sentinelValues.containsOneKey)
            {
                return ifAbsent;
            }
            return this.sentinelValues.oneValue;
        }
        int index = this.probe(key);
        if (this.keys[index] == key)
        {
            return this.values[index];
        }
        return ifAbsent;
    }

    public long getOrThrow(long key)
    {
        if (isEmptyKey(key))
        {
            if (this.sentinelValues == null || !this.sentinelValues.containsZeroKey)
            {
                throw new IllegalStateException("Key " + key + " not present.");
            }
            return this.sentinelValues.zeroValue;
        }
        if (isRemovedKey(key))
        {
            if (this.sentinelValues == null || !this.sentinelValues.containsOneKey)
            {
                throw new IllegalStateException("Key " + key + " not present.");
            }
            return this.sentinelValues.oneValue;
        }
        int index = this.probe(key);
        if (isNonSentinel(this.keys[index]))
        {
            return this.values[index];
        }
        throw new IllegalStateException("Key " + key + " not present.");
    }

    public boolean containsKey(long key)
    {
        if (isEmptyKey(key))
        {
            return this.sentinelValues != null && this.sentinelValues.containsZeroKey;
        }
        if (isRemovedKey(key))
        {
            return this.sentinelValues != null && this.sentinelValues.containsOneKey;
        }
        return this.keys[this.probe(key)] == key;
    }

    public boolean containsValue(long value)
    {
        if (this.sentinelValues != null && this.sentinelValues.containsValue(value))
        {
            return true;
        }
        for (int i = 0; i < this.values.length; i++)
        {
            if (isNonSentinel(this.keys[i]) && this.values[i] == value)
            {
                return true;
            }
        }
        return false;
    }

    public void forEachValue(LongProcedure procedure)
    {
        if (this.sentinelValues != null)
        {
            if (this.sentinelValues.containsZeroKey)
            {
                procedure.value(this.sentinelValues.zeroValue);
            }
            if (this.sentinelValues.containsOneKey)
            {
                procedure.value(this.sentinelValues.oneValue);
            }
        }
        for (int i = 0; i < this.keys.length; i++)
        {
            if (isNonSentinel(this.keys[i]))
            {
                procedure.value(this.values[i]);
            }
        }
    }

    public void forEachKey(LongProcedure procedure)
    {
        if (this.sentinelValues != null)
        {
            if (this.sentinelValues.containsZeroKey)
            {
                procedure.value(EMPTY_KEY);
            }
            if (this.sentinelValues.containsOneKey)
            {
                procedure.value(REMOVED_KEY);
            }
        }
        for (long key : this.keys)
        {
            if (isNonSentinel(key))
            {
                procedure.value(key);
            }
        }
    }

    public void forEachKeyValue(LongLongProcedure procedure)
    {
        if (this.sentinelValues != null)
        {
            if (this.sentinelValues.containsZeroKey)
            {
                procedure.value(EMPTY_KEY, this.sentinelValues.zeroValue);
            }
            if (this.sentinelValues.containsOneKey)
            {
                procedure.value(REMOVED_KEY, this.sentinelValues.oneValue);
            }
        }
        for (int i = 0; i < this.keys.length; i++)
        {
            if (isNonSentinel(this.keys[i]))
            {
                procedure.value(this.keys[i], this.values[i]);
            }
        }
    }

    public LazyLongIterable keysView()
    {
        return new KeysView();
    }

    public RichIterable<LongLongPair> keyValuesView()
    {
        return new KeyValuesView();
    }

    public LongLongHashMap select(LongLongPredicate predicate)
    {
        LongLongHashMap result = new LongLongHashMap();

        if (this.sentinelValues != null)
        {
            if (this.sentinelValues.containsZeroKey && predicate.accept(EMPTY_KEY, this.sentinelValues.zeroValue))
            {
                result.put(EMPTY_KEY, this.sentinelValues.zeroValue);
            }
            if (this.sentinelValues.containsOneKey && predicate.accept(REMOVED_KEY, this.sentinelValues.oneValue))
            {
                result.put(REMOVED_KEY, this.sentinelValues.oneValue);
            }
        }
        for (int i = 0; i < this.keys.length; i++)
        {
            if (isNonSentinel(this.keys[i]) && predicate.accept(this.keys[i], this.values[i]))
            {
                result.put(this.keys[i], this.values[i]);
            }
        }

        return result;
    }

    public LongLongHashMap reject(LongLongPredicate predicate)
    {
        LongLongHashMap result = new LongLongHashMap();

        if (this.sentinelValues != null)
        {
            if (this.sentinelValues.containsZeroKey && !predicate.accept(EMPTY_KEY, this.sentinelValues.zeroValue))
            {
                result.put(EMPTY_KEY, this.sentinelValues.zeroValue);
            }
            if (this.sentinelValues.containsOneKey && !predicate.accept(REMOVED_KEY, this.sentinelValues.oneValue))
            {
                result.put(REMOVED_KEY, this.sentinelValues.oneValue);
            }
        }
        for (int i = 0; i < this.keys.length; i++)
        {
            if (isNonSentinel(this.keys[i]) && !predicate.accept(this.keys[i], this.values[i]))
            {
                result.put(this.keys[i], this.values[i]);
            }
        }
        return result;
    }

    public long sum()
    {
        long result = 0L;

        if (this.sentinelValues != null)
        {
            if (this.sentinelValues.containsZeroKey)
            {
                result += this.sentinelValues.zeroValue;
            }
            if (this.sentinelValues.containsOneKey)
            {
                result += this.sentinelValues.oneValue;
            }
        }
        for (int i = 0; i < this.keys.length; i++)
        {
            if (isNonSentinel(this.keys[i]))
            {
                result += this.values[i];
            }
        }

        return result;
    }

    public long max()
    {
        if (this.isEmpty())
        {
            throw new NoSuchElementException();
        }
        LongIterator iterator = this.longIterator();
        long max = iterator.next();
        while (iterator.hasNext())
        {
            long value = iterator.next();
            if (max < value)
            {
                max = value;
            }
        }
        return max;
    }

    public long maxIfEmpty(long defaultValue)
    {
        if (this.isEmpty())
        {
            return defaultValue;
        }
        return this.max();
    }

    public long min()
    {
        if (this.isEmpty())
        {
            throw new NoSuchElementException();
        }
        LongIterator iterator = this.longIterator();
        long min = iterator.next();
        while (iterator.hasNext())
        {
            long value = iterator.next();
            if (value < min)
            {
                min = value;
            }
        }
        return min;
    }

    public long minIfEmpty(long defaultValue)
    {
        if (this.isEmpty())
        {
            return defaultValue;
        }
        return this.min();
    }

    public double average()
    {
        if (this.isEmpty())
        {
            throw new ArithmeticException();
        }
        return (double) this.sum() / (double) this.size();
    }

    public double median()
    {
        if (this.isEmpty())
        {
            throw new ArithmeticException();
        }
        long[] sortedArray = this.toSortedArray();
        int middleIndex = sortedArray.length >> 1;
        if (sortedArray.length > 1 && (sortedArray.length & 1) == 0)
        {
            long first = sortedArray[middleIndex];
            long second = sortedArray[middleIndex - 1];
            return ((double) first + (double) second) / 2.0;
        }
        return (double) sortedArray[middleIndex];
    }

    public long[] toSortedArray()
    {
        long[] array = this.toArray();
        Arrays.sort(array);
        return array;
    }

    public MutableLongList toSortedList()
    {
        return LongArrayList.newList(this).sortThis();
    }

    public void writeExternal(ObjectOutput out) throws IOException
    {
        out.writeInt(this.size());
        if (this.sentinelValues != null)
        {
            if (this.sentinelValues.containsZeroKey)
            {
                out.writeLong(EMPTY_KEY);
                out.writeLong(this.sentinelValues.zeroValue);
            }
            if (this.sentinelValues.containsOneKey)
            {
                out.writeLong(REMOVED_KEY);
                out.writeLong(this.sentinelValues.oneValue);
            }
        }
        for (int i = 0; i < this.keys.length; i++)
        {
            if (isNonSentinel(this.keys[i]))
            {
                out.writeLong(this.keys[i]);
                out.writeLong(this.values[i]);
            }
        }
    }

    public void readExternal(ObjectInput in) throws IOException, ClassNotFoundException
    {
        int size = in.readInt();
        for (int i = 0; i < size; i++)
        {
            this.put(in.readLong(), in.readLong());
        }
    }

    /**
     * Rehashes every element in the set into a new backing table of the smallest possible size and eliminating removed sentinels.
     */
    public void compact()
    {
        this.rehash(this.smallestPowerOfTwoGreaterThan(this.size()));
    }

    private void rehash()
    {
        this.rehash(this.keys.length);
    }

    private void rehashAndGrow()
    {
        this.rehash(this.keys.length << 1);
    }

    private void rehash(int newCapacity)
    {
        int oldLength = this.keys.length;
        long[] old = this.keys;
        long[] oldValues = this.values;
        this.allocateTable(newCapacity);
        this.occupiedWithData = 0;
        this.occupiedWithSentinels = 0;

        for (int i = 0; i < oldLength; i++)
        {
            if (isNonSentinel(old[i]))
            {
                this.put(old[i], oldValues[i]);
            }
        }
    }

    // exposed for testing
    int probe(long element)
    {
        int index = this.spread(element);
        long keyAtIndex = this.keys[index];

        if (keyAtIndex == element || keyAtIndex == EMPTY_KEY)
        {
            return index;
        }

        int removedIndex = keyAtIndex == REMOVED_KEY ? index : -1;
        int nextIndex = index;
        int probe = 17;

        // loop until an empty slot is reached
        while (true)
        {
            // Probe algorithm: 17*n*(n+1)/2 where n = number of collisions
            nextIndex += probe;
            probe += 17;
            nextIndex &= this.keys.length - 1;

            if (this.keys[nextIndex] == element)
            {
                return nextIndex;
            }
            if (this.keys[nextIndex] == REMOVED_KEY)
            {
                if (removedIndex == -1)
                {
                    removedIndex = nextIndex;
                }
            }
            else if (this.keys[nextIndex] == EMPTY_KEY)
            {
                return removedIndex == -1 ? nextIndex : removedIndex;
            }
        }
    }

    // exposed for testing
    int spread(long element)
    {
        long code = element;
        code = ~code + (code << 18);
        code = (code << 18) - code - 1;
        code ^= code >>> 31;
        code *= 21;
        code += (code << 2) + (code << 4);
        code ^= code >>> 11;
        code += code << 6;
        code ^= code >>> 22;
        return (int) code & (this.keys.length - 1);
    }

    private void allocateTable(int sizeToAllocate)
    {
        this.keys = new long[sizeToAllocate];
        this.values = new long[sizeToAllocate];
    }

    private static boolean isEmptyKey(long key)
    {
        return key == EMPTY_KEY;
    }

    private static boolean isRemovedKey(long key)
    {
        return key == REMOVED_KEY;
    }

    private static boolean isNonSentinel(long key)
    {
        return !isEmptyKey(key) && !isRemovedKey(key);
    }

    private int maxOccupiedWithData()
    {
        int capacity = this.keys.length;
        // need at least one free slot for open addressing
        return Math.min(capacity - 1, capacity / OCCUPIED_DATA_RATIO);
    }

    private int maxOccupiedWithSentinels()
    {
        return this.keys.length / OCCUPIED_SENTINEL_RATIO;
    }

    private static final class SentinelValues
    {
        private boolean containsZeroKey;
        private boolean containsOneKey;
        private long zeroValue;
        private long oneValue;

        public int size()
        {
            return (this.containsZeroKey ? 1 : 0) + (this.containsOneKey ? 1 : 0);
        }

        public boolean containsValue(long value)
        {
            boolean valueEqualsZeroValue = this.containsZeroKey && this.zeroValue == value;
            boolean valueEqualsOneValue = this.containsOneKey && this.oneValue == value;
            return valueEqualsZeroValue || valueEqualsOneValue;
        }
    }

    private class InternalLongIterator implements LongIterator
    {
        private int count;
        private int position;
        private boolean handledZero;
        private boolean handledOne;

        public boolean hasNext()
        {
            return this.count < LongLongHashMap.this.size();
        }

        public long next()
        {
            if (!this.hasNext())
            {
                throw new NoSuchElementException("next() called, but the iterator is exhausted");
            }
            this.count++;

            if (!this.handledZero)
            {
                this.handledZero = true;
                if (LongLongHashMap.this.containsKey(EMPTY_KEY))
                {
                    return LongLongHashMap.this.get(EMPTY_KEY);
                }
            }
            if (!this.handledOne)
            {
                this.handledOne = true;
                if (LongLongHashMap.this.containsKey(REMOVED_KEY))
                {
                    return LongLongHashMap.this.get(REMOVED_KEY);
                }
            }

            long[] keys = LongLongHashMap.this.keys;
            while (!isNonSentinel(keys[this.position]))
            {
                this.position++;
            }
            long result = LongLongHashMap.this.values[this.position];
            this.position++;
            return result;
        }
    }

    private class KeysView extends AbstractLazyLongIterable
    {
        public LongIterator longIterator()
        {
            return new KeySetIterator();
        }

        public void forEach(LongProcedure procedure)
        {
            LongLongHashMap.this.forEachKey(procedure);
        }
    }

    private class KeySetIterator implements LongIterator
    {
        private int count;
        private int position;
        private boolean handledZero;
        private boolean handledOne;

        public boolean hasNext()
        {
            return this.count < LongLongHashMap.this.size();
        }

        public long next()
        {
            if (!this.hasNext())
            {
                throw new NoSuchElementException("next() called, but the iterator is exhausted");
            }
            this.count++;

            if (!this.handledZero)
            {
                this.handledZero = true;
                if (LongLongHashMap.this.containsKey(EMPTY_KEY))
                {
                    return EMPTY_KEY;
                }
            }
            if (!this.handledOne)
            {
                this.handledOne = true;
                if (LongLongHashMap.this.containsKey(REMOVED_KEY))
                {
                    return REMOVED_KEY;
                }
            }

            long[] keys = LongLongHashMap.this.keys;
            while (!isNonSentinel(keys[this.position]))
            {
                this.position++;
            }
            long result = keys[this.position];
            this.position++;
            return result;
        }
    }

    public MutableLongSet keySet()
    {
        return new KeySet();
    }

    private class KeySet implements MutableLongSet
    {
        public LongIterator longIterator()
        {
            return new KeySetIterator();
        }

        public void forEach(LongProcedure procedure)
        {
            LongLongHashMap.this.forEachKey(procedure);
        }

        public int count(LongPredicate predicate)
        {
            int count = 0;
            if (LongLongHashMap.this.sentinelValues != null)
            {
                if (LongLongHashMap.this.sentinelValues.containsZeroKey && predicate.accept(EMPTY_KEY))
                {
                    count++;
                }
                if (LongLongHashMap.this.sentinelValues.containsOneKey && predicate.accept(REMOVED_KEY))
                {
                    count++;
                }
            }
            for (long key : LongLongHashMap.this.keys)
            {
                if (isNonSentinel(key) && predicate.accept(key))
                {
                    count++;
                }
            }
            return count;
        }

        public boolean anySatisfy(LongPredicate predicate)
        {
            if (LongLongHashMap.this.sentinelValues != null)
            {
                if (LongLongHashMap.this.sentinelValues.containsZeroKey && predicate.accept(EMPTY_KEY))
                {
                    return true;
                }
                if (LongLongHashMap.this.sentinelValues.containsOneKey && predicate.accept(REMOVED_KEY))
                {
                    return true;
                }
            }
            for (long key : LongLongHashMap.this.keys)
            {
                if (isNonSentinel(key) && predicate.accept(key))
                {
                    return true;
                }
            }
            return false;
        }

        public boolean allSatisfy(LongPredicate predicate)
        {
            if (LongLongHashMap.this.sentinelValues != null)
            {
                if (LongLongHashMap.this.sentinelValues.containsZeroKey && !predicate.accept(EMPTY_KEY))
                {
                    return false;
                }
                if (LongLongHashMap.this.sentinelValues.containsOneKey && !predicate.accept(REMOVED_KEY))
                {
                    return false;
                }
            }
            for (long key : LongLongHashMap.this.keys)
            {
                if (isNonSentinel(key) && !predicate.accept(key))
                {
                    return false;
                }
            }
            return true;
        }

        public boolean noneSatisfy(LongPredicate predicate)
        {
            if (LongLongHashMap.this.sentinelValues != null)
            {
                if (LongLongHashMap.this.sentinelValues.containsZeroKey && predicate.accept(EMPTY_KEY))
                {
                    return false;
                }
                if (LongLongHashMap.this.sentinelValues.containsOneKey && predicate.accept(REMOVED_KEY))
                {
                    return false;
                }
            }
            for (long key : LongLongHashMap.this.keys)
            {
                if (isNonSentinel(key) && predicate.accept(key))
                {
                    return false;
                }
            }
            return true;
        }

        public boolean add(long element)
        {
            throw new UnsupportedOperationException("Cannot call add() on " + this.getClass().getSimpleName());
        }

        public boolean addAll(long... source)
        {
            throw new UnsupportedOperationException("Cannot call addAll() on " + this.getClass().getSimpleName());
        }

        public boolean addAll(LongIterable source)
        {
            throw new UnsupportedOperationException("Cannot call addAll() on " + this.getClass().getSimpleName());
        }

        public boolean remove(long key)
        {
            int oldSize = LongLongHashMap.this.size();
            LongLongHashMap.this.removeKey(key);
            return oldSize != LongLongHashMap.this.size();
        }

        public boolean removeAll(LongIterable source)
        {
            int oldSize = LongLongHashMap.this.size();
            LongIterator iterator = source.longIterator();
            while (iterator.hasNext())
            {
                LongLongHashMap.this.removeKey(iterator.next());
            }
            return oldSize != LongLongHashMap.this.size();
        }

        public boolean removeAll(long... source)
        {
            int oldSize = LongLongHashMap.this.size();
            for (long item : source)
            {
                LongLongHashMap.this.removeKey(item);
            }
            return oldSize != LongLongHashMap.this.size();
        }

        public boolean retainAll(LongIterable source)
        {
            int oldSize = this.size();
            final LongSet sourceSet = source instanceof LongSet ? (LongSet) source : source.toSet();
            LongLongHashMap retained = LongLongHashMap.this.select(new LongLongPredicate()
            {
                public boolean accept(long key, long value)
                {
                    return sourceSet.contains(key);
                }
            });
            if (retained.size() != oldSize)
            {
                LongLongHashMap.this.keys = retained.keys;
                LongLongHashMap.this.values = retained.values;
                LongLongHashMap.this.sentinelValues = retained.sentinelValues;
                LongLongHashMap.this.occupiedWithData = retained.occupiedWithData;
                LongLongHashMap.this.occupiedWithSentinels = retained.occupiedWithSentinels;
                return true;
            }
            return false;
        }

        public boolean retainAll(long... source)
        {
            return this.retainAll(LongHashSet.newSetWith(source));
        }

        public void clear()
        {
            LongLongHashMap.this.clear();
        }

        public MutableLongSet select(LongPredicate predicate)
        {
            MutableLongSet result = new LongHashSet();
            if (LongLongHashMap.this.sentinelValues != null)
            {
                if (LongLongHashMap.this.sentinelValues.containsZeroKey && predicate.accept(EMPTY_KEY))
                {
                    result.add(EMPTY_KEY);
                }
                if (LongLongHashMap.this.sentinelValues.containsOneKey && predicate.accept(REMOVED_KEY))
                {
                    result.add(REMOVED_KEY);
                }
            }
            for (long key : LongLongHashMap.this.keys)
            {
                if (isNonSentinel(key) && predicate.accept(key))
                {
                    result.add(key);
                }
            }
            return result;
        }

        public MutableLongSet reject(LongPredicate predicate)
        {
            MutableLongSet result = new LongHashSet();
            if (LongLongHashMap.this.sentinelValues != null)
            {
                if (LongLongHashMap.this.sentinelValues.containsZeroKey && !predicate.accept(EMPTY_KEY))
                {
                    result.add(EMPTY_KEY);
                }
                if (LongLongHashMap.this.sentinelValues.containsOneKey && !predicate.accept(REMOVED_KEY))
                {
                    result.add(REMOVED_KEY);
                }
            }
            for (long key : LongLongHashMap.this.keys)
            {
                if (isNonSentinel(key) && !predicate.accept(key))
                {
                    result.add(key);
                }
            }
            return result;
        }

        public MutableLongSet with(long element)
        {
            throw new UnsupportedOperationException("Cannot call with() on " + this.getClass().getSimpleName());
        }

        public MutableLongSet without(long element)
        {
            throw new UnsupportedOperationException("Cannot call without() on " + this.getClass().getSimpleName());
        }

        public MutableLongSet withAll(LongIterable elements)
        {
            throw new UnsupportedOperationException("Cannot call withAll() on " + this.getClass().getSimpleName());
        }

        public MutableLongSet withoutAll(LongIterable elements)
        {
            throw new UnsupportedOperationException("Cannot call withoutAll() on " + this.getClass().getSimpleName());
        }

        public long detectIfNone(LongPredicate predicate, long ifNone)
        {
            if (LongLongHashMap.this.sentinelValues != null)
            {
                if (LongLongHashMap.this.sentinelValues.containsZeroKey && predicate.accept(EMPTY_KEY))
                {
                    return EMPTY_KEY;
                }
                if (LongLongHashMap.this.sentinelValues.containsOneKey && predicate.accept(REMOVED_KEY))
                {
                    return REMOVED_KEY;
                }
            }
            for (long key : LongLongHashMap.this.keys)
            {
                if (isNonSentinel(key) && predicate.accept(key))
                {
                    return key;
                }
            }
            return ifNone;
        }

        public <V> MutableSet<V> collect(LongToObjectFunction<? extends V> function)
        {
            MutableSet<V> result = Sets.mutable.with();
            if (LongLongHashMap.this.sentinelValues != null)
            {
                if (LongLongHashMap.this.sentinelValues.containsZeroKey)
                {
                    result.add(function.valueOf(EMPTY_KEY));
                }
                if (LongLongHashMap.this.sentinelValues.containsOneKey)
                {
                    result.add(function.valueOf(REMOVED_KEY));
                }
            }
            for (long key : LongLongHashMap.this.keys)
            {
                if (isNonSentinel(key))
                {
                    result.add(function.valueOf(key));
                }
            }
            return result;
        }

        public MutableLongSet asUnmodifiable()
        {
            return UnmodifiableLongSet.of(this);
        }

        public MutableLongSet asSynchronized()
        {
            return SynchronizedLongSet.of(this);
        }

        public long sum()
        {
            long sum = 0L;
            if (LongLongHashMap.this.sentinelValues != null)
            {
                if (LongLongHashMap.this.sentinelValues.containsZeroKey)
                {
                    sum += EMPTY_KEY;
                }
                if (LongLongHashMap.this.sentinelValues.containsOneKey)
                {
                    sum += REMOVED_KEY;
                }
            }
            for (long key : LongLongHashMap.this.keys)
            {
                if (isNonSentinel(key))
                {
                    sum += key;
                }
            }
            return sum;
        }

        public long max()
        {
            if (LongLongHashMap.this.isEmpty())
            {
                throw new NoSuchElementException();
            }

            long max = 0;
            boolean isMaxSet = false;

            if (LongLongHashMap.this.sentinelValues != null)
            {
                if (LongLongHashMap.this.sentinelValues.containsZeroKey)
                {
                    max = EMPTY_KEY;
                    isMaxSet = true;
                }
                if (LongLongHashMap.this.sentinelValues.containsOneKey && (!isMaxSet || max < REMOVED_KEY))
                {
                    max = REMOVED_KEY;
                    isMaxSet = true;
                }
            }
            for (int i = 0; i < LongLongHashMap.this.keys.length; i++)
            {
                if (isNonSentinel(LongLongHashMap.this.keys[i]) && (!isMaxSet || max < LongLongHashMap.this.keys[i]))
                {
                    max = LongLongHashMap.this.keys[i];
                    isMaxSet = true;
                }
            }
            return max;
        }

        public long maxIfEmpty(long defaultValue)
        {
            if (LongLongHashMap.this.isEmpty())
            {
                return defaultValue;
            }

            return this.max();
        }

        public long min()
        {
            if (LongLongHashMap.this.isEmpty())
            {
                throw new NoSuchElementException();
            }

            long min = 0;
            boolean isMinSet = false;

            if (LongLongHashMap.this.sentinelValues != null)
            {
                if (LongLongHashMap.this.sentinelValues.containsZeroKey)
                {
                    min = EMPTY_KEY;
                    isMinSet = true;
                }
                if (LongLongHashMap.this.sentinelValues.containsOneKey && (!isMinSet || REMOVED_KEY < min))
                {
                    min = REMOVED_KEY;
                    isMinSet = true;
                }
            }
            for (int i = 0; i < LongLongHashMap.this.keys.length; i++)
            {
                if (isNonSentinel(LongLongHashMap.this.keys[i]) && (!isMinSet || LongLongHashMap.this.keys[i] < min))
                {
                    min = LongLongHashMap.this.keys[i];
                    isMinSet = true;
                }
            }
            return min;
        }

        public long minIfEmpty(long defaultValue)
        {
            if (LongLongHashMap.this.isEmpty())
            {
                return defaultValue;
            }

            return this.min();
        }

        public double average()
        {
            if (this.isEmpty())
            {
                throw new ArithmeticException();
            }
            return (double) this.sum() / (double) this.size();
        }

        public double median()
        {
            if (this.isEmpty())
            {
                throw new ArithmeticException();
            }
            long[] sortedArray = this.toSortedArray();
            int middleIndex = sortedArray.length >> 1;
            if (sortedArray.length > 1 && (sortedArray.length & 1) == 0)
            {
                long first = sortedArray[middleIndex];
                long second = sortedArray[middleIndex - 1];
                return ((double) first + (double) second) / 2.0;
            }
            return (double) sortedArray[middleIndex];
        }

        public long[] toSortedArray()
        {
            long[] array = this.toArray();
            Arrays.sort(array);
            return array;
        }

        public MutableLongList toSortedList()
        {
            return LongArrayList.newList(this).sortThis();
        }

        public long[] toArray()
        {
            int size = LongLongHashMap.this.size();
            final long[] result = new long[size];
            LongLongHashMap.this.forEachKey(new LongProcedure()
            {
                private int index;

                public void value(long each)
                {
                    result[this.index] = each;
                    this.index++;
                }
            });
            return result;
        }

        public boolean contains(long value)
        {
            return LongLongHashMap.this.containsKey(value);
        }

        public boolean containsAll(long... source)
        {
            for (long item : source)
            {
                if (!LongLongHashMap.this.containsKey(item))
                {
                    return false;
                }
            }
            return true;
        }

        public boolean containsAll(LongIterable source)
        {
            LongIterator iterator = source.longIterator();
            while (iterator.hasNext())
            {
                if (!LongLongHashMap.this.containsKey(iterator.next()))
                {
                    return false;
                }
            }
            return true;
        }

        public MutableLongList toList()
        {
            return LongArrayList.newList(this);
        }

        public MutableLongSet toSet()
        {
            return LongHashSet.newSet(this);
        }

        public MutableLongBag toBag()
        {
            return LongHashBag.newBag(this);
        }

        public LazyLongIterable asLazy()
        {
            return new LazyLongIterableAdapter(this);
        }

        public <T> T injectInto(T injectedValue, ObjectLongToObjectFunction<? super T, ? extends T> function)
        {
            T result = injectedValue;
            if (LongLongHashMap.this.sentinelValues != null)
            {
                if (LongLongHashMap.this.sentinelValues.containsZeroKey)
                {
                    result = function.valueOf(result, EMPTY_KEY);
                }
                if (LongLongHashMap.this.sentinelValues.containsOneKey)
                {
                    result = function.valueOf(result, REMOVED_KEY);
                }
            }
            for (int i = 0; i < LongLongHashMap.this.keys.length; i++)
            {
                if (isNonSentinel(LongLongHashMap.this.keys[i]))
                {
                    result = function.valueOf(result, LongLongHashMap.this.keys[i]);
                }
            }
            return result;
        }

        public LongSet freeze()
        {
            throw new UnsupportedOperationException(this.getClass().getSimpleName() + ".freeze() not implemented yet");
        }

        public ImmutableLongSet toImmutable()
        {
            return LongSets.immutable.withAll(this);
        }

        public int size()
        {
            return LongLongHashMap.this.size();
        }

        public boolean isEmpty()
        {
            return LongLongHashMap.this.isEmpty();
        }

        public boolean notEmpty()
        {
            return LongLongHashMap.this.notEmpty();
        }

        @Override
        public boolean equals(Object obj)
        {
            if (this == obj)
            {
                return true;
            }

            if (!(obj instanceof LongSet))
            {
                return false;
            }

            LongSet other = (LongSet) obj;
            return this.size() == other.size() && this.containsAll(other.toArray());
        }

        @Override
        public int hashCode()
        {
            int result = 0;

            if (LongLongHashMap.this.sentinelValues != null)
            {
                if (LongLongHashMap.this.sentinelValues.containsZeroKey)
                {
                    result += (int) (EMPTY_KEY ^ EMPTY_KEY >>> 32);
                }
                if (LongLongHashMap.this.sentinelValues.containsOneKey)
                {
                    result += (int) (REMOVED_KEY ^ REMOVED_KEY >>> 32);
                }
            }
            for (int i = 0; i < LongLongHashMap.this.keys.length; i++)
            {
                if (isNonSentinel(LongLongHashMap.this.keys[i]))
                {
                    result += (int) (LongLongHashMap.this.keys[i] ^ LongLongHashMap.this.keys[i] >>> 32);
                }
            }

            return result;
        }

        @Override
        public String toString()
        {
            return this.makeString("[", ", ", "]");
        }

        public String makeString()
        {
            return this.makeString(", ");
        }

        public String makeString(String separator)
        {
            return this.makeString("", separator, "");
        }

        public String makeString(String start, String separator, String end)
        {
            Appendable stringBuilder = new StringBuilder();
            this.appendString(stringBuilder, start, separator, end);
            return stringBuilder.toString();
        }

        public void appendString(Appendable appendable)
        {
            this.appendString(appendable, ", ");
        }

        public void appendString(Appendable appendable, String separator)
        {
            this.appendString(appendable, "", separator, "");
        }

        public void appendString(Appendable appendable, String start, String separator, String end)
        {
            try
            {
                appendable.append(start);
                boolean first = true;
                if (LongLongHashMap.this.sentinelValues != null)
                {
                    if (LongLongHashMap.this.sentinelValues.containsZeroKey)
                    {
                        appendable.append(String.valueOf(EMPTY_KEY));
                        first = false;
                    }
                    if (LongLongHashMap.this.sentinelValues.containsOneKey)
                    {
                        if (!first)
                        {
                            appendable.append(separator);
                        }
                        appendable.append(String.valueOf(REMOVED_KEY));
                        first = false;
                    }
                }
                for (long key : LongLongHashMap.this.keys)
                {
                    if (isNonSentinel(key))
                    {
                        if (!first)
                        {
                            appendable.append(separator);
                        }
                        appendable.append(String.valueOf(key));
                        first = false;
                    }
                }
                appendable.append(end);
            }
            catch (IOException e)
            {
                throw new RuntimeException(e);
            }
        }
    }

    public MutableLongCollection values()
    {
        return new ValuesCollection();
    }

    private class ValuesCollection implements MutableLongCollection
    {
        public void clear()
        {
            LongLongHashMap.this.clear();
        }

        public MutableLongCollection select(LongPredicate predicate)
        {
            return LongLongHashMap.this.select(predicate);
        }

        public MutableLongCollection reject(LongPredicate predicate)
        {
            return LongLongHashMap.this.reject(predicate);
        }

        public long detectIfNone(LongPredicate predicate, long ifNone)
        {
            return LongLongHashMap.this.detectIfNone(predicate, ifNone);
        }

        public <V> MutableCollection<V> collect(LongToObjectFunction<? extends V> function)
        {
            return LongLongHashMap.this.collect(function);
        }

        public <T> T injectInto(T injectedValue, ObjectLongToObjectFunction<? super T, ? extends T> function)
        {
            return LongLongHashMap.this.injectInto(injectedValue, function);
        }

        public long sum()
        {
            return LongLongHashMap.this.sum();
        }

        public long max()
        {
            return LongLongHashMap.this.max();
        }

        public long maxIfEmpty(long defaultValue)
        {
            return LongLongHashMap.this.maxIfEmpty(defaultValue);
        }

        public long min()
        {
            return LongLongHashMap.this.min();
        }

        public long minIfEmpty(long defaultValue)
        {
            return LongLongHashMap.this.minIfEmpty(defaultValue);
        }

        public double average()
        {
            return LongLongHashMap.this.average();
        }

        public double median()
        {
            return LongLongHashMap.this.median();
        }

        public long[] toSortedArray()
        {
            return LongLongHashMap.this.toSortedArray();
        }

        public MutableLongList toSortedList()
        {
            return LongLongHashMap.this.toSortedList();
        }

        public MutableLongCollection with(long element)
        {
            throw new UnsupportedOperationException("Cannot call with() on " + this.getClass().getSimpleName());
        }

        public MutableLongCollection without(long element)
        {
            throw new UnsupportedOperationException("Cannot call without() on " + this.getClass().getSimpleName());
        }

        public MutableLongCollection withAll(LongIterable elements)
        {
            throw new UnsupportedOperationException("Cannot call withAll() on " + this.getClass().getSimpleName());
        }

        public MutableLongCollection withoutAll(LongIterable elements)
        {
            throw new UnsupportedOperationException("Cannot call withoutAll() on " + this.getClass().getSimpleName());
        }

        public MutableLongCollection asUnmodifiable()
        {
            return UnmodifiableLongCollection.of(this);
        }

        public MutableLongCollection asSynchronized()
        {
            return SynchronizedLongCollection.of(this);
        }

        public ImmutableLongCollection toImmutable()
        {
            return LongLists.immutable.withAll(this);
        }

        public boolean contains(long value)
        {
            return LongLongHashMap.this.containsValue(value);
        }

        public boolean containsAll(long... source)
        {
            return LongLongHashMap.this.containsAll(source);
        }

        public boolean containsAll(LongIterable source)
        {
            return LongLongHashMap.this.containsAll(source);
        }

        public MutableLongList toList()
        {
            return LongLongHashMap.this.toList();
        }

        public MutableLongSet toSet()
        {
            return LongLongHashMap.this.toSet();
        }

        public MutableLongBag toBag()
        {
            return LongLongHashMap.this.toBag();
        }

        public LazyLongIterable asLazy()
        {
            return new LazyLongIterableAdapter(this);
        }

        public boolean isEmpty()
        {
            return LongLongHashMap.this.isEmpty();
        }

        public boolean notEmpty()
        {
            return LongLongHashMap.this.notEmpty();
        }

        public String makeString()
        {
            return this.makeString(", ");
        }

        public String makeString(String separator)
        {
            return this.makeString("", separator, "");
        }

        public String makeString(String start, String separator, String end)
        {
            Appendable stringBuilder = new StringBuilder();
            this.appendString(stringBuilder, start, separator, end);
            return stringBuilder.toString();
        }

        public void appendString(Appendable appendable)
        {
            this.appendString(appendable, ", ");
        }

        public void appendString(Appendable appendable, String separator)
        {
            this.appendString(appendable, "", separator, "");
        }

        public void appendString(Appendable appendable, String start, String separator, String end)
        {
            try
            {
                appendable.append(start);

                boolean first = true;

                if (LongLongHashMap.this.sentinelValues != null)
                {
                    if (LongLongHashMap.this.sentinelValues.containsZeroKey)
                    {
                        appendable.append(String.valueOf(LongLongHashMap.this.sentinelValues.zeroValue));
                        first = false;
                    }
                    if (LongLongHashMap.this.sentinelValues.containsOneKey)
                    {
                        if (!first)
                        {
                            appendable.append(separator);
                        }
                        appendable.append(String.valueOf(LongLongHashMap.this.sentinelValues.oneValue));
                        first = false;
                    }
                }
                for (int i = 0; i < LongLongHashMap.this.keys.length; i++)
                {
                    long key = LongLongHashMap.this.keys[i];
                    if (isNonSentinel(key))
                    {
                        if (!first)
                        {
                            appendable.append(separator);
                        }
                        appendable.append(String.valueOf(LongLongHashMap.this.values[i]));
                        first = false;
                    }
                }
                appendable.append(end);
            }
            catch (IOException e)
            {
                throw new RuntimeException(e);
            }
        }

        public LongIterator longIterator()
        {
            return LongLongHashMap.this.longIterator();
        }

        public void forEach(LongProcedure procedure)
        {
            LongLongHashMap.this.forEach(procedure);
        }

        public int count(LongPredicate predicate)
        {
            return LongLongHashMap.this.count(predicate);
        }

        public boolean anySatisfy(LongPredicate predicate)
        {
            return LongLongHashMap.this.anySatisfy(predicate);
        }

        public boolean allSatisfy(LongPredicate predicate)
        {
            return LongLongHashMap.this.allSatisfy(predicate);
        }

        public boolean noneSatisfy(LongPredicate predicate)
        {
            return LongLongHashMap.this.noneSatisfy(predicate);
        }

        public boolean add(long element)
        {
            throw new UnsupportedOperationException("Cannot call add() on " + this.getClass().getSimpleName());
        }

        public boolean addAll(long... source)
        {
            throw new UnsupportedOperationException("Cannot call addAll() on " + this.getClass().getSimpleName());
        }

        public boolean addAll(LongIterable source)
        {
            throw new UnsupportedOperationException("Cannot call addAll() on " + this.getClass().getSimpleName());
        }

        public boolean remove(long item)
        {
            int oldSize = LongLongHashMap.this.size();

            if (LongLongHashMap.this.sentinelValues != null)
            {
                if (LongLongHashMap.this.sentinelValues.containsZeroKey && item == LongLongHashMap.this.sentinelValues.zeroValue)
                {
                    LongLongHashMap.this.removeKey(EMPTY_KEY);
                }
                if (LongLongHashMap.this.sentinelValues.containsOneKey && item == LongLongHashMap.this.sentinelValues.oneValue)
                {
                    LongLongHashMap.this.removeKey(REMOVED_KEY);
                }
            }
            for (int i = 0; i < LongLongHashMap.this.keys.length; i++)
            {
                if (isNonSentinel(LongLongHashMap.this.keys[i]) && item == LongLongHashMap.this.values[i])
                {
                    LongLongHashMap.this.removeKey(LongLongHashMap.this.keys[i]);
                }
            }
            return oldSize != LongLongHashMap.this.size();
        }

        public boolean removeAll(LongIterable source)
        {
            int oldSize = LongLongHashMap.this.size();

            LongIterator iterator = source.longIterator();
            while (iterator.hasNext())
            {
                this.remove(iterator.next());
            }
            return oldSize != LongLongHashMap.this.size();
        }

        public boolean removeAll(long... source)
        {
            int oldSize = LongLongHashMap.this.size();

            for (long item : source)
            {
                this.remove(item);
            }
            return oldSize != LongLongHashMap.this.size();
        }

        public boolean retainAll(LongIterable source)
        {
            int oldSize = this.size();
            final LongSet sourceSet = source instanceof LongSet ? (LongSet) source : source.toSet();
            LongLongHashMap retained = LongLongHashMap.this.select(new LongLongPredicate()
            {
                public boolean accept(long key, long value)
                {
                    return sourceSet.contains(value);
                }
            });
            if (retained.size() != oldSize)
            {
                LongLongHashMap.this.keys = retained.keys;
                LongLongHashMap.this.values = retained.values;
                LongLongHashMap.this.sentinelValues = retained.sentinelValues;
                LongLongHashMap.this.occupiedWithData = retained.occupiedWithData;
                LongLongHashMap.this.occupiedWithSentinels = retained.occupiedWithSentinels;
                return true;
            }
            return false;
        }

        public boolean retainAll(long... source)
        {
            return this.retainAll(LongHashSet.newSetWith(source));
        }

        public int size()
        {
            return LongLongHashMap.this.size();
        }

        public long[] toArray()
        {
            return LongLongHashMap.this.toArray();
        }
    }

    private class KeyValuesView extends AbstractLazyIterable<LongLongPair>
    {
        public void forEach(Procedure<? super LongLongPair> procedure)
        {
            if (LongLongHashMap.this.sentinelValues != null)
            {
                if (LongLongHashMap.this.sentinelValues.containsZeroKey)
                {
                    procedure.value(PrimitiveTuples.pair(EMPTY_KEY, LongLongHashMap.this.sentinelValues.zeroValue));
                }
                if (LongLongHashMap.this.sentinelValues.containsOneKey)
                {
                    procedure.value(PrimitiveTuples.pair(REMOVED_KEY, LongLongHashMap.this.sentinelValues.oneValue));
                }
            }
            for (int i = 0; i < LongLongHashMap.this.keys.length; i++)
            {
                if (isNonSentinel(LongLongHashMap.this.keys[i]))
                {
                    procedure.value(PrimitiveTuples.pair(LongLongHashMap.this.keys[i], LongLongHashMap.this.values[i]));
                }
            }
        }

        public void forEachWithIndex(ObjectIntProcedure<? super LongLongPair> objectIntProcedure)
        {
            int index = 0;
            if (LongLongHashMap.this.sentinelValues != null)
            {
                if (LongLongHashMap.this.sentinelValues.containsZeroKey)
                {
                    objectIntProcedure.value(PrimitiveTuples.pair(EMPTY_KEY, LongLongHashMap.this.sentinelValues.zeroValue), index);
                    index++;
                }
                if (LongLongHashMap.this.sentinelValues.containsOneKey)
                {
                    objectIntProcedure.value(PrimitiveTuples.pair(REMOVED_KEY, LongLongHashMap.this.sentinelValues.oneValue), index);
                    index++;
                }
            }
            for (int i = 0; i < LongLongHashMap.this.keys.length; i++)
            {
                if (isNonSentinel(LongLongHashMap.this.keys[i]))
                {
                    objectIntProcedure.value(PrimitiveTuples.pair(LongLongHashMap.this.keys[i], LongLongHashMap.this.values[i]), index);
                    index++;
                }
            }
        }

        public <P> void forEachWith(Procedure2<? super LongLongPair, ? super P> procedure, P parameter)
        {
            if (LongLongHashMap.this.sentinelValues != null)
            {
                if (LongLongHashMap.this.sentinelValues.containsZeroKey)
                {
                    procedure.value(PrimitiveTuples.pair(EMPTY_KEY, LongLongHashMap.this.sentinelValues.zeroValue), parameter);
                }
                if (LongLongHashMap.this.sentinelValues.containsOneKey)
                {
                    procedure.value(PrimitiveTuples.pair(REMOVED_KEY, LongLongHashMap.this.sentinelValues.oneValue), parameter);
                }
            }
            for (int i = 0; i < LongLongHashMap.this.keys.length; i++)
            {
                if (isNonSentinel(LongLongHashMap.this.keys[i]))
                {
                    procedure.value(PrimitiveTuples.pair(LongLongHashMap.this.keys[i], LongLongHashMap.this.values[i]), parameter);
                }
            }
        }

        public Iterator<LongLongPair> iterator()
        {
            return new InternalKeyValuesIterator();
        }

        public class InternalKeyValuesIterator implements Iterator<LongLongPair>
        {
            private int count;
            private int position;
            private boolean handledZero;
            private boolean handledOne;

            public LongLongPair next()
            {
                if (!this.hasNext())
                {
                    throw new NoSuchElementException("next() called, but the iterator is exhausted");
                }
                this.count++;

                if (!this.handledZero)
                {
                    this.handledZero = true;
                    if (LongLongHashMap.this.containsKey(EMPTY_KEY))
                    {
                        return PrimitiveTuples.pair(EMPTY_KEY, LongLongHashMap.this.sentinelValues.zeroValue);
                    }
                }
                if (!this.handledOne)
                {
                    this.handledOne = true;
                    if (LongLongHashMap.this.containsKey(REMOVED_KEY))
                    {
                        return PrimitiveTuples.pair(REMOVED_KEY, LongLongHashMap.this.sentinelValues.oneValue);
                    }
                }

                long[] keys = LongLongHashMap.this.keys;
                while (!isNonSentinel(keys[this.position]))
                {
                    this.position++;
                }
                LongLongPair result = PrimitiveTuples.pair(keys[this.position], LongLongHashMap.this.values[this.position]);
                this.position++;
                return result;
            }

            public void remove()
            {
                throw new UnsupportedOperationException("Cannot call remove() on " + this.getClass().getSimpleName());
            }

            public boolean hasNext()
            {
                return this.count != LongLongHashMap.this.size();
            }
        }
    }
}
