/*
 * Copyright 2014 Goldman Sachs.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.gs.collections.impl.map.mutable.primitive;

import java.io.Serializable;
import java.util.Collection;
import java.util.Collections;
import java.util.Comparator;
import java.util.Iterator;

import com.gs.collections.api.ByteIterable;
import com.gs.collections.api.LazyByteIterable;
import com.gs.collections.api.LazyIterable;
import com.gs.collections.api.RichIterable;
import com.gs.collections.api.bag.MutableBag;
import com.gs.collections.api.block.function.Function;
import com.gs.collections.api.block.function.Function0;
import com.gs.collections.api.block.function.Function2;
import com.gs.collections.api.block.function.primitive.BooleanFunction;
import com.gs.collections.api.block.function.primitive.ByteFunction;
import com.gs.collections.api.block.function.primitive.CharFunction;
import com.gs.collections.api.block.function.primitive.DoubleFunction;
import com.gs.collections.api.block.function.primitive.DoubleObjectToDoubleFunction;
import com.gs.collections.api.block.function.primitive.FloatFunction;
import com.gs.collections.api.block.function.primitive.FloatObjectToFloatFunction;
import com.gs.collections.api.block.function.primitive.IntFunction;
import com.gs.collections.api.block.function.primitive.IntObjectToIntFunction;
import com.gs.collections.api.block.function.primitive.ByteToObjectFunction;
import com.gs.collections.api.block.function.primitive.LongFunction;
import com.gs.collections.api.block.function.primitive.LongObjectToLongFunction;
import com.gs.collections.api.block.function.primitive.ShortFunction;
import com.gs.collections.api.block.predicate.Predicate;
import com.gs.collections.api.block.predicate.Predicate2;
import com.gs.collections.api.block.predicate.primitive.ByteObjectPredicate;
import com.gs.collections.api.block.procedure.Procedure;
import com.gs.collections.api.block.procedure.Procedure2;
import com.gs.collections.api.block.procedure.primitive.ByteObjectProcedure;
import com.gs.collections.api.block.procedure.primitive.ByteProcedure;
import com.gs.collections.api.block.procedure.primitive.ObjectIntProcedure;
import com.gs.collections.api.collection.MutableCollection;
import com.gs.collections.api.collection.primitive.MutableBooleanCollection;
import com.gs.collections.api.collection.primitive.MutableByteCollection;
import com.gs.collections.api.collection.primitive.MutableCharCollection;
import com.gs.collections.api.collection.primitive.MutableDoubleCollection;
import com.gs.collections.api.collection.primitive.MutableFloatCollection;
import com.gs.collections.api.collection.primitive.MutableIntCollection;
import com.gs.collections.api.collection.primitive.MutableLongCollection;
import com.gs.collections.api.collection.primitive.MutableShortCollection;
import com.gs.collections.api.list.MutableList;
import com.gs.collections.api.map.MutableMap;
import com.gs.collections.api.map.primitive.ByteObjectMap;
import com.gs.collections.api.map.primitive.ImmutableByteObjectMap;
import com.gs.collections.api.map.primitive.MutableByteObjectMap;
import com.gs.collections.api.map.sorted.MutableSortedMap;
import com.gs.collections.api.multimap.MutableMultimap;
import com.gs.collections.api.partition.PartitionMutableCollection;
import com.gs.collections.api.set.MutableSet;
import com.gs.collections.api.set.primitive.MutableByteSet;
import com.gs.collections.api.set.sorted.MutableSortedSet;
import com.gs.collections.api.tuple.Pair;
import com.gs.collections.api.tuple.primitive.ByteObjectPair;
import com.gs.collections.impl.SynchronizedRichIterable;
import com.gs.collections.impl.collection.mutable.CollectionAdapter;
import com.gs.collections.impl.collection.mutable.SynchronizedMutableCollection;
import com.gs.collections.impl.factory.primitive.ByteObjectMaps;
import com.gs.collections.impl.primitive.SynchronizedByteIterable;
import com.gs.collections.impl.set.mutable.primitive.SynchronizedByteSet;
import net.jcip.annotations.GuardedBy;

/**
 * A synchronized view of a {@link MutableByteObjectMap}. It is imperative that the user manually synchronize on the collection when iterating over it using the
 * standard JDK iterator or JDK 5 for loop, as per {@link Collections#synchronizedCollection(Collection)}.
 * <p>
 * This file was automatically generated from template file synchronizedPrimitiveObjectMap.stg.
 * </p>
 *
 * @see MutableByteObjectMap#asSynchronized()
 * @see MutableMap#asSynchronized()
 * @since 3.1.
 */
public final class SynchronizedByteObjectMap<V>
        implements MutableByteObjectMap<V>, Serializable
{
    private static final long serialVersionUID = 1L;
    private final Object lock;
    @GuardedBy("this.lock")
    private final MutableByteObjectMap<V> map;

    SynchronizedByteObjectMap(MutableByteObjectMap<V> map)
    {
        this(map, null);
    }

    SynchronizedByteObjectMap(MutableByteObjectMap<V> map, Object newLock)
    {
        this.map = map;
        this.lock = newLock == null ? this : newLock;
    }

    public V put(byte key, V value)
    {
        synchronized (this.lock)
        {
            return this.map.put(key, value);
        }
    }

    public void putAll(ByteObjectMap<? extends V> map)
    {
        synchronized (this.lock)
        {
            map.forEachKeyValue(new ByteObjectProcedure<V>()
            {
                public void value(byte key, V value)
                {
                    SynchronizedByteObjectMap.this.map.put(key, value);
                }
            });
        }
    }

    public V removeKey(byte key)
    {
        synchronized (this.lock)
        {
            return this.map.removeKey(key);
        }
    }

    public V remove(byte key)
    {
        synchronized (this.lock)
        {
            return this.map.remove(key);
        }
    }

    public V getIfAbsentPut(byte key, V value)
    {
        synchronized (this.lock)
        {
            return this.map.getIfAbsentPut(key, value);
        }
    }

    public V getIfAbsentPut(byte key, Function0<? extends V> function)
    {
        synchronized (this.lock)
        {
            return this.map.getIfAbsentPut(key, function);
        }
    }

    public V getIfAbsentPutWithKey(byte key, ByteToObjectFunction<? extends V> function)
    {
        synchronized (this.lock)
        {
            return this.map.getIfAbsentPutWithKey(key, function);
        }
    }

    public <P> V getIfAbsentPutWith(byte key, Function<? super P, ? extends V> function, P parameter)
    {
        synchronized (this.lock)
        {
            return this.map.getIfAbsentPutWith(key, function, parameter);
        }
    }

    public V updateValue(byte key, Function0<? extends V> factory, Function<? super V, ? extends V> function)
    {
        synchronized (this.lock)
        {
            return this.map.updateValue(key, factory, function);
        }
    }

    public <P> V updateValueWith(byte key, Function0<? extends V> factory, Function2<? super V, ? super P, ? extends V> function, P parameter)
    {
        synchronized (this.lock)
        {
            return this.map.updateValueWith(key, factory, function, parameter);
        }
    }

    public V get(byte key)
    {
        synchronized (this.lock)
        {
            return this.map.get(key);
        }
    }

    public V getIfAbsent(byte key, Function0<? extends V> ifAbsent)
    {
        synchronized (this.lock)
        {
            return this.map.getIfAbsent(key, ifAbsent);
        }
    }

    public boolean containsKey(byte key)
    {
        synchronized (this.lock)
        {
            return this.map.containsKey(key);
        }
    }

    public boolean containsValue(Object value)
    {
        synchronized (this.lock)
        {
            return this.map.containsValue(value);
        }
    }

    public void forEachValue(Procedure<? super V> procedure)
    {
        synchronized (this.lock)
        {
            this.map.forEachValue(procedure);
        }
    }

    public void forEachKey(ByteProcedure procedure)
    {
        synchronized (this.lock)
        {
            this.map.forEachKey(procedure);
        }
    }

    public void forEachKeyValue(ByteObjectProcedure<? super V> procedure)
    {
        synchronized (this.lock)
        {
            this.map.forEachKeyValue(procedure);
        }
    }

    public MutableByteObjectMap<V> select(ByteObjectPredicate<? super V> predicate)
    {
        synchronized (this.lock)
        {
            return this.map.select(predicate);
        }
    }

    public MutableByteObjectMap<V> reject(ByteObjectPredicate<? super V> predicate)
    {
        synchronized (this.lock)
        {
            return this.map.reject(predicate);
        }
    }

    public int size()
    {
        synchronized (this.lock)
        {
            return this.map.size();
        }
    }

    public boolean isEmpty()
    {
        synchronized (this.lock)
        {
            return this.map.isEmpty();
        }
    }

    public boolean notEmpty()
    {
        synchronized (this.lock)
        {
            return this.map.notEmpty();
        }
    }

    public V getFirst()
    {
        synchronized (this.lock)
        {
            return this.map.getFirst();
        }
    }

    public V getLast()
    {
        synchronized (this.lock)
        {
            return this.map.getLast();
        }
    }

    public boolean contains(Object object)
    {
        synchronized (this.lock)
        {
            return this.map.contains(object);
        }
    }

    public boolean containsAllIterable(Iterable<?> source)
    {
        synchronized (this.lock)
        {
            return this.map.containsAllIterable(source);
        }
    }

    public boolean containsAll(Collection<?> source)
    {
        synchronized (this.lock)
        {
            return this.map.containsAll(source);
        }
    }

    public boolean containsAllArguments(Object... elements)
    {
        synchronized (this.lock)
        {
            return this.map.containsAllArguments(elements);
        }
    }

    public MutableCollection<V> select(Predicate<? super V> predicate)
    {
        synchronized (this.lock)
        {
            return this.map.select(predicate);
        }
    }

    public <R extends Collection<V>> R select(Predicate<? super V> predicate, R target)
    {
        synchronized (this.lock)
        {
            return this.map.select(predicate, target);
        }
    }

    public <P> MutableCollection<V> selectWith(Predicate2<? super V, ? super P> predicate, P parameter)
    {
        synchronized (this.lock)
        {
            return this.map.selectWith(predicate, parameter);
        }
    }

    public <P, R extends Collection<V>> R selectWith(Predicate2<? super V, ? super P> predicate, P parameter, R targetCollection)
    {
        synchronized (this.lock)
        {
            return this.map.selectWith(predicate, parameter, targetCollection);
        }
    }

    public PartitionMutableCollection<V> partition(Predicate<? super V> predicate)
    {
        synchronized (this.lock)
        {
            return this.map.partition(predicate);
        }
    }

    public <P> PartitionMutableCollection<V> partitionWith(Predicate2<? super V, ? super P> predicate, P parameter)
    {
        synchronized (this.lock)
        {
            return this.map.partitionWith(predicate, parameter);
        }
    }

    public <S> MutableCollection<S> selectInstancesOf(Class<S> clazz)
    {
        synchronized (this.lock)
        {
            return this.map.selectInstancesOf(clazz);
        }
    }

    public MutableList<Pair<V, Integer>> zipWithIndex()
    {
        synchronized (this.lock)
        {
            return (MutableList<Pair<V, Integer>>) this.map.zipWithIndex();
        }
    }

    public <R extends Collection<Pair<V, Integer>>> R zipWithIndex(R target)
    {
        synchronized (this.lock)
        {
            return this.map.zipWithIndex(target);
        }
    }

    public RichIterable<RichIterable<V>> chunk(int size)
    {
        synchronized (this.lock)
        {
            return this.map.chunk(size);
        }
    }

    public <K, VV> MutableMap<K, VV> aggregateInPlaceBy(Function<? super V, ? extends K> groupBy, Function0<? extends VV> zeroValueFactory, Procedure2<? super VV, ? super V> mutatingAggregator)
    {
        synchronized (this.lock)
        {
            return this.map.aggregateInPlaceBy(groupBy, zeroValueFactory, mutatingAggregator);
        }
    }

    public <K, VV> MutableMap<K, VV> aggregateBy(Function<? super V, ? extends K> groupBy, Function0<? extends VV> zeroValueFactory, Function2<? super VV, ? super V, ? extends VV> nonMutatingAggregator)
    {
        synchronized (this.lock)
        {
            return this.map.aggregateBy(groupBy, zeroValueFactory, nonMutatingAggregator);
        }
    }

    public MutableCollection<V> reject(Predicate<? super V> predicate)
    {
        synchronized (this.lock)
        {
            return this.map.reject(predicate);
        }
    }

    public <R extends Collection<V>> R reject(Predicate<? super V> predicate, R target)
    {
        synchronized (this.lock)
        {
            return this.map.reject(predicate, target);
        }
    }

    public <P> MutableCollection<V> rejectWith(Predicate2<? super V, ? super P> predicate, P parameter)
    {
        synchronized (this.lock)
        {
            return this.map.rejectWith(predicate, parameter);
        }
    }

    public <P, R extends Collection<V>> R rejectWith(Predicate2<? super V, ? super P> predicate, P parameter, R targetCollection)
    {
        synchronized (this.lock)
        {
            return this.map.rejectWith(predicate, parameter, targetCollection);
        }
    }

    public void clear()
    {
        synchronized (this.lock)
        {
            this.map.clear();
        }
    }

    public <VV> MutableCollection<VV> collect(Function<? super V, ? extends VV> function)
    {
        synchronized (this.lock)
        {
            return this.map.collect(function);
        }
    }

    public MutableBooleanCollection collectBoolean(BooleanFunction<? super V> booleanFunction)
    {
        synchronized (this.lock)
        {
            return this.map.collectBoolean(booleanFunction);
        }
    }

    public <R extends MutableBooleanCollection> R collectBoolean(BooleanFunction<? super V> booleanFunction, R target)
    {
        synchronized (this.lock)
        {
            return this.map.collectBoolean(booleanFunction, target);
        }
    }

    public MutableByteCollection collectByte(ByteFunction<? super V> byteFunction)
    {
        synchronized (this.lock)
        {
            return this.map.collectByte(byteFunction);
        }
    }

    public <R extends MutableByteCollection> R collectByte(ByteFunction<? super V> byteFunction, R target)
    {
        synchronized (this.lock)
        {
            return this.map.collectByte(byteFunction, target);
        }
    }

    public MutableCharCollection collectChar(CharFunction<? super V> charFunction)
    {
        synchronized (this.lock)
        {
            return this.map.collectChar(charFunction);
        }
    }

    public <R extends MutableCharCollection> R collectChar(CharFunction<? super V> charFunction, R target)
    {
        synchronized (this.lock)
        {
            return this.map.collectChar(charFunction, target);
        }
    }

    public MutableDoubleCollection collectDouble(DoubleFunction<? super V> doubleFunction)
    {
        synchronized (this.lock)
        {
            return this.map.collectDouble(doubleFunction);
        }
    }

    public <R extends MutableDoubleCollection> R collectDouble(DoubleFunction<? super V> doubleFunction, R target)
    {
        synchronized (this.lock)
        {
            return this.map.collectDouble(doubleFunction, target);
        }
    }

    public MutableFloatCollection collectFloat(FloatFunction<? super V> floatFunction)
    {
        synchronized (this.lock)
        {
            return this.map.collectFloat(floatFunction);
        }
    }

    public <R extends MutableFloatCollection> R collectFloat(FloatFunction<? super V> floatFunction, R target)
    {
        synchronized (this.lock)
        {
            return this.map.collectFloat(floatFunction, target);
        }
    }

    public MutableIntCollection collectInt(IntFunction<? super V> intFunction)
    {
        synchronized (this.lock)
        {
            return this.map.collectInt(intFunction);
        }
    }

    public <R extends MutableIntCollection> R collectInt(IntFunction<? super V> intFunction, R target)
    {
        synchronized (this.lock)
        {
            return this.map.collectInt(intFunction, target);
        }
    }

    public MutableLongCollection collectLong(LongFunction<? super V> longFunction)
    {
        synchronized (this.lock)
        {
            return this.map.collectLong(longFunction);
        }
    }

    public <R extends MutableLongCollection> R collectLong(LongFunction<? super V> longFunction, R target)
    {
        synchronized (this.lock)
        {
            return this.map.collectLong(longFunction, target);
        }
    }

    public MutableShortCollection collectShort(ShortFunction<? super V> shortFunction)
    {
        synchronized (this.lock)
        {
            return this.map.collectShort(shortFunction);
        }
    }

    public <R extends MutableShortCollection> R collectShort(ShortFunction<? super V> shortFunction, R target)
    {
        synchronized (this.lock)
        {
            return this.map.collectShort(shortFunction, target);
        }
    }

    public <P, VV> MutableCollection<VV> collectWith(Function2<? super V, ? super P, ? extends VV> function, P parameter)
    {
        synchronized (this.lock)
        {
            return this.map.collectWith(function, parameter);
        }
    }

    public <P, VV, R extends Collection<VV>> R collectWith(Function2<? super V, ? super P, ? extends VV> function, P parameter, R targetCollection)
    {
        synchronized (this.lock)
        {
            return this.map.collectWith(function, parameter, targetCollection);
        }
    }

    public <VV, R extends Collection<VV>> R collect(Function<? super V, ? extends VV> function, R target)
    {
        synchronized (this.lock)
        {
            return this.map.collect(function, target);
        }
    }

    public <VV> MutableCollection<VV> collectIf(Predicate<? super V> predicate, Function<? super V, ? extends VV> function)
    {
        synchronized (this.lock)
        {
            return this.map.collectIf(predicate, function);
        }
    }

    public <VV, R extends Collection<VV>> R collectIf(Predicate<? super V> predicate, Function<? super V, ? extends VV> function, R target)
    {
        synchronized (this.lock)
        {
            return this.map.collectIf(predicate, function, target);
        }
    }

    public <VV> RichIterable<VV> flatCollect(Function<? super V, ? extends Iterable<VV>> function)
    {
        synchronized (this.lock)
        {
            return this.map.flatCollect(function);
        }
    }

    public <VV, R extends Collection<VV>> R flatCollect(Function<? super V, ? extends Iterable<VV>> function, R target)
    {
        synchronized (this.lock)
        {
            return this.map.flatCollect(function, target);
        }
    }

    public V detect(Predicate<? super V> predicate)
    {
        synchronized (this.lock)
        {
            return this.map.detect(predicate);
        }
    }

    public <P> V detectWith(Predicate2<? super V, ? super P> predicate, P parameter)
    {
        synchronized (this.lock)
        {
            return this.map.detectWith(predicate, parameter);
        }
    }

    public V detectIfNone(Predicate<? super V> predicate, Function0<? extends V> function)
    {
        synchronized (this.lock)
        {
            return this.map.detectIfNone(predicate, function);
        }
    }

    public <P> V detectWithIfNone(Predicate2<? super V, ? super P> predicate, P parameter, Function0<? extends V> function)
    {
        synchronized (this.lock)
        {
            return this.map.detectWithIfNone(predicate, parameter, function);
        }
    }

    public int count(Predicate<? super V> predicate)
    {
        synchronized (this.lock)
        {
            return this.map.count(predicate);
        }
    }

    public <P> int countWith(Predicate2<? super V, ? super P> predicate, P parameter)
    {
        synchronized (this.lock)
        {
            return this.map.countWith(predicate, parameter);
        }
    }

    public boolean anySatisfy(Predicate<? super V> predicate)
    {
        synchronized (this.lock)
        {
            return this.map.anySatisfy(predicate);
        }
    }

    public <P> boolean anySatisfyWith(Predicate2<? super V, ? super P> predicate, P parameter)
    {
        synchronized (this.lock)
        {
            return this.map.anySatisfyWith(predicate, parameter);
        }
    }

    public boolean allSatisfy(Predicate<? super V> predicate)
    {
        synchronized (this.lock)
        {
            return this.map.allSatisfy(predicate);
        }
    }

    public <P> boolean allSatisfyWith(Predicate2<? super V, ? super P> predicate, P parameter)
    {
        synchronized (this.lock)
        {
            return this.map.allSatisfyWith(predicate, parameter);
        }
    }

    public boolean noneSatisfy(Predicate<? super V> predicate)
    {
        synchronized (this.lock)
        {
            return this.map.noneSatisfy(predicate);
        }
    }

    public <P> boolean noneSatisfyWith(Predicate2<? super V, ? super P> predicate, P parameter)
    {
        synchronized (this.lock)
        {
            return this.map.noneSatisfyWith(predicate, parameter);
        }
    }

    public <IV> IV injectInto(IV injectedValue, Function2<? super IV, ? super V, ? extends IV> function)
    {
        synchronized (this.lock)
        {
            return this.map.injectInto(injectedValue, function);
        }
    }

    public int injectInto(int injectedValue, IntObjectToIntFunction<? super V> function)
    {
        synchronized (this.lock)
        {
            return this.map.injectInto(injectedValue, function);
        }
    }

    public long injectInto(long injectedValue, LongObjectToLongFunction<? super V> function)
    {
        synchronized (this.lock)
        {
            return this.map.injectInto(injectedValue, function);
        }
    }

    public float injectInto(float injectedValue, FloatObjectToFloatFunction<? super V> function)
    {
        synchronized (this.lock)
        {
            return this.map.injectInto(injectedValue, function);
        }
    }

    public double injectInto(double injectedValue, DoubleObjectToDoubleFunction<? super V> function)
    {
        synchronized (this.lock)
        {
            return this.map.injectInto(injectedValue, function);
        }
    }

    public MutableList<V> toList()
    {
        synchronized (this.lock)
        {
            return this.map.toList();
        }
    }

    public MutableList<V> toSortedList()
    {
        synchronized (this.lock)
        {
            return this.map.toSortedList();
        }
    }

    public MutableList<V> toSortedList(Comparator<? super V> comparator)
    {
        synchronized (this.lock)
        {
            return this.map.toSortedList(comparator);
        }
    }

    public <VV extends Comparable<? super VV>> MutableList<V> toSortedListBy(Function<? super V, ? extends VV> function)
    {
        synchronized (this.lock)
        {
            return this.map.toSortedListBy(function);
        }
    }

    public MutableSet<V> toSet()
    {
        synchronized (this.lock)
        {
            return this.map.toSet();
        }
    }

    public MutableSortedSet<V> toSortedSet()
    {
        synchronized (this.lock)
        {
            return this.map.toSortedSet();
        }
    }

    public MutableSortedSet<V> toSortedSet(Comparator<? super V> comparator)
    {
        synchronized (this.lock)
        {
            return this.map.toSortedSet(comparator);
        }
    }

    public <VV extends Comparable<? super VV>> MutableSortedSet<V> toSortedSetBy(Function<? super V, ? extends VV> function)
    {
        synchronized (this.lock)
        {
            return this.map.toSortedSetBy(function);
        }
    }

    public MutableBag<V> toBag()
    {
        synchronized (this.lock)
        {
            return this.map.toBag();
        }
    }

    public <NK, NV> MutableMap<NK, NV> toMap(Function<? super V, ? extends NK> keyFunction, Function<? super V, ? extends NV> valueFunction)
    {
        synchronized (this.lock)
        {
            return this.map.toMap(keyFunction, valueFunction);
        }
    }

    public <NK, NV> MutableSortedMap<NK, NV> toSortedMap(Function<? super V, ? extends NK> keyFunction, Function<? super V, ? extends NV> valueFunction)
    {
        synchronized (this.lock)
        {
            return this.map.toSortedMap(keyFunction, valueFunction);
        }
    }

    public <NK, NV> MutableSortedMap<NK, NV> toSortedMap(Comparator<? super NK> comparator, Function<? super V, ? extends NK> keyFunction, Function<? super V, ? extends NV> valueFunction)
    {
        synchronized (this.lock)
        {
            return this.map.toSortedMap(comparator, keyFunction, valueFunction);
        }
    }

    public LazyIterable<V> asLazy()
    {
        synchronized (this.lock)
        {
            return this.map.asLazy();
        }
    }

    public Object[] toArray()
    {
        synchronized (this.lock)
        {
            return this.map.toArray();
        }
    }

    public <T> T[] toArray(T[] a)
    {
        synchronized (this.lock)
        {
            return this.map.toArray(a);
        }
    }

    public V min(Comparator<? super V> comparator)
    {
        synchronized (this.lock)
        {
            return this.map.min(comparator);
        }
    }

    public V max(Comparator<? super V> comparator)
    {
        synchronized (this.lock)
        {
            return this.map.max(comparator);
        }
    }

    public V min()
    {
        synchronized (this.lock)
        {
            return this.map.min();
        }
    }

    public V max()
    {
        synchronized (this.lock)
        {
            return this.map.max();
        }
    }

    public <VV extends Comparable<? super VV>> V maxBy(Function<? super V, ? extends VV> function)
    {
        synchronized (this.lock)
        {
            return this.map.maxBy(function);
        }
    }

    public <VV extends Comparable<? super VV>> V minBy(Function<? super V, ? extends VV> function)
    {
        synchronized (this.lock)
        {
            return this.map.minBy(function);
        }
    }

    public long sumOfInt(IntFunction<? super V> function)
    {
        synchronized (this.lock)
        {
            return this.map.sumOfInt(function);
        }
    }

    public double sumOfFloat(FloatFunction<? super V> function)
    {
        synchronized (this.lock)
        {
            return this.map.sumOfFloat(function);
        }
    }

    public long sumOfLong(LongFunction<? super V> function)
    {
        synchronized (this.lock)
        {
            return this.map.sumOfLong(function);
        }
    }

    public double sumOfDouble(DoubleFunction<? super V> function)
    {
        synchronized (this.lock)
        {
            return this.map.sumOfDouble(function);
        }
    }

    public MutableByteSet keySet()
    {
        synchronized (this.lock)
        {
            return SynchronizedByteSet.of(this.map.keySet(), this.lock);
        }
    }

    public Collection<V> values()
    {
        synchronized (this.lock)
        {
            return SynchronizedMutableCollection.of(CollectionAdapter.adapt(this.map.values()), this.lock);
        }
    }

    public LazyByteIterable keysView()
    {
        synchronized (this.lock)
        {
            return SynchronizedByteIterable.of(this.map.keysView(), this.lock).asLazy();
        }
    }

    public RichIterable<ByteObjectPair<V>> keyValuesView()
    {
        synchronized (this.lock)
        {
            return SynchronizedRichIterable.of(this.map.keyValuesView(), this.lock).asLazy();
        }
    }

    @Override
    public boolean equals(Object obj)
    {
        synchronized (this.lock)
        {
            return this.map.equals(obj);
        }
    }

    @Override
    public int hashCode()
    {
        synchronized (this.lock)
        {
            return this.map.hashCode();
        }
    }

    @Override
    public String toString()
    {
        synchronized (this.lock)
        {
            return this.map.toString();
        }
    }

    public String makeString()
    {
        synchronized (this.lock)
        {
            return this.map.makeString();
        }
    }

    public String makeString(String separator)
    {
        synchronized (this.lock)
        {
            return this.map.makeString(separator);
        }
    }

    public String makeString(String start, String separator, String end)
    {
        synchronized (this.lock)
        {
            return this.map.makeString(start, separator, end);
        }
    }

    public void appendString(Appendable appendable)
    {
        synchronized (this.lock)
        {
            this.map.appendString(appendable);
        }
    }

    public void appendString(Appendable appendable, String separator)
    {
        synchronized (this.lock)
        {
            this.map.appendString(appendable, separator);
        }
    }

    public void appendString(Appendable appendable, String start, String separator, String end)
    {
        synchronized (this.lock)
        {
            this.map.appendString(appendable, start, separator, end);
        }
    }

    public <VV> MutableMultimap<VV, V> groupBy(Function<? super V, ? extends VV> function)
    {
        synchronized (this.lock)
        {
            return this.map.groupBy(function);
        }
    }

    public <VV, R extends MutableMultimap<VV, V>> R groupBy(Function<? super V, ? extends VV> function, R target)
    {
        synchronized (this.lock)
        {
            return this.map.groupBy(function, target);
        }
    }

    public <VV> MutableMultimap<VV, V> groupByEach(Function<? super V, ? extends Iterable<VV>> function)
    {
        synchronized (this.lock)
        {
            return this.map.groupByEach(function);
        }
    }

    public <VV, R extends MutableMultimap<VV, V>> R groupByEach(Function<? super V, ? extends Iterable<VV>> function, R target)
    {
        synchronized (this.lock)
        {
            return this.map.groupByEach(function, target);
        }
    }

    public <VV> MutableMap<VV, V> groupByUniqueKey(Function<? super V, ? extends VV> function)
    {
        synchronized (this.lock)
        {
            return this.map.groupByUniqueKey(function);
        }
    }

    public <S> MutableCollection<Pair<V, S>> zip(Iterable<S> that)
    {
        synchronized (this.lock)
        {
            return this.map.zip(that);
        }
    }

    public <S, R extends Collection<Pair<V, S>>> R zip(Iterable<S> that, R target)
    {
        synchronized (this.lock)
        {
            return this.map.zip(that, target);
        }
    }

    public MutableByteObjectMap<V> withKeyValue(byte key, V value)
    {
        synchronized (this.lock)
        {
            this.map.withKeyValue(key, value);
        }
        return this;
    }

    public MutableByteObjectMap<V> withoutKey(byte key)
    {
        synchronized (this.lock)
        {
            this.map.withoutKey(key);
        }
        return this;
    }

    public MutableByteObjectMap<V> withoutAllKeys(ByteIterable keys)
    {
        synchronized (this.lock)
        {
            this.map.withoutAllKeys(keys);
        }
        return this;
    }

    public MutableByteObjectMap<V> asUnmodifiable()
    {
        return new UnmodifiableByteObjectMap<V>(this);
    }

    public MutableByteObjectMap<V> asSynchronized()
    {
        return this;
    }

    public ImmutableByteObjectMap<V> toImmutable()
    {
        return ByteObjectMaps.immutable.withAll(this);
    }

    public void forEach(Procedure<? super V> procedure)
    {
        synchronized (this.lock)
        {
            this.map.forEach(procedure);
        }
    }

    public void forEachWithIndex(ObjectIntProcedure<? super V> objectIntProcedure)
    {
        synchronized (this.lock)
        {
            this.map.forEachWithIndex(objectIntProcedure);
        }
    }

    public <P> void forEachWith(Procedure2<? super V, ? super P> procedure, P parameter)
    {
        synchronized (this.lock)
        {
            this.map.forEachWith(procedure, parameter);
        }
    }

    /**
     * Must be called in a synchronized block.
     */
    public Iterator<V> iterator()
    {
        return this.map.iterator();
    }
}
