/*
 * Copyright 2014 Goldman Sachs.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.gs.collections.impl.stack.mutable.primitive;

import java.io.Serializable;
import java.util.Collection;
import java.util.Collections;

import com.gs.collections.api.FloatIterable;
import com.gs.collections.api.LazyFloatIterable;
import com.gs.collections.api.bag.primitive.MutableFloatBag;
import com.gs.collections.api.block.function.primitive.FloatToObjectFunction;
import com.gs.collections.api.block.function.primitive.ObjectFloatToObjectFunction;
import com.gs.collections.api.block.predicate.primitive.FloatPredicate;
import com.gs.collections.api.block.procedure.primitive.FloatProcedure;
import com.gs.collections.api.iterator.FloatIterator;
import com.gs.collections.api.list.primitive.FloatList;
import com.gs.collections.api.list.primitive.MutableFloatList;
import com.gs.collections.api.set.primitive.MutableFloatSet;
import com.gs.collections.api.stack.MutableStack;
import com.gs.collections.api.stack.primitive.ImmutableFloatStack;
import com.gs.collections.api.stack.primitive.MutableFloatStack;
import com.gs.collections.impl.factory.primitive.FloatStacks;
import com.gs.collections.impl.lazy.primitive.LazyFloatIterableAdapter;
import net.jcip.annotations.GuardedBy;
import net.jcip.annotations.ThreadSafe;

/**
 * A synchronized view of a {@link MutableFloatStack}. It is imperative that the user manually synchronize on the collection when iterating over it using the
 * {@link FloatIterator}, as per {@link Collections#synchronizedCollection(Collection)}.
 * <p>
 * This file was automatically generated from template file synchronizedPrimitiveStack.stg.
 * </p>
 *
 * @see MutableFloatStack#asSynchronized()
 * @see MutableStack#asSynchronized()
 * @since 3.1.
 */
@ThreadSafe
public final class SynchronizedFloatStack
        implements MutableFloatStack, Serializable
{
    private static final long serialVersionUID = 1L;
    private final Object lock;
    @GuardedBy("this.lock")
    private final MutableFloatStack stack;

    SynchronizedFloatStack(MutableFloatStack stack)
    {
        this(stack, null);
    }

    SynchronizedFloatStack(MutableFloatStack stack, Object newLock)
    {
        this.stack = stack;
        this.lock = newLock == null ? this : newLock;
    }

    public void push(float item)
    {
        synchronized (this.lock)
        {
            this.stack.push(item);
        }
    }

    public float pop()
    {
        synchronized (this.lock)
        {
            return this.stack.pop();
        }
    }

    public FloatList pop(int count)
    {
        synchronized (this.lock)
        {
            return this.stack.pop(count);
        }
    }

    public float peek()
    {
        synchronized (this.lock)
        {
            return this.stack.peek();
        }
    }

    public FloatList peek(int count)
    {
        synchronized (this.lock)
        {
            return this.stack.peek(count);
        }
    }

    public float peekAt(int index)
    {
        synchronized (this.lock)
        {
            return this.stack.peekAt(index);
        }
    }

    public int size()
    {
        synchronized (this.lock)
        {
            return this.stack.size();
        }
    }

    public boolean isEmpty()
    {
        synchronized (this.lock)
        {
            return this.stack.isEmpty();
        }
    }

    public boolean notEmpty()
    {
        synchronized (this.lock)
        {
            return this.stack.notEmpty();
        }
    }

    public void clear()
    {
        synchronized (this.lock)
        {
            this.stack.clear();
        }
    }

    public boolean contains(float value)
    {
        synchronized (this.lock)
        {
            return this.stack.contains(value);
        }
    }

    public boolean containsAll(float... source)
    {
        synchronized (this.lock)
        {
            return this.stack.containsAll(source);
        }
    }

    public boolean containsAll(FloatIterable source)
    {
        synchronized (this.lock)
        {
            return this.stack.containsAll(source);
        }
    }

    /**
     * Must be called in a synchronized block.
     */
    public FloatIterator floatIterator()
    {
        return this.stack.floatIterator();
    }

    public void forEach(FloatProcedure procedure)
    {
        synchronized (this.lock)
        {
            this.stack.forEach(procedure);
        }
    }

    public int count(FloatPredicate predicate)
    {
        synchronized (this.lock)
        {
            return this.stack.count(predicate);
        }
    }

    public boolean anySatisfy(FloatPredicate predicate)
    {
        synchronized (this.lock)
        {
            return this.stack.anySatisfy(predicate);
        }
    }

    public boolean allSatisfy(FloatPredicate predicate)
    {
        synchronized (this.lock)
        {
            return this.stack.allSatisfy(predicate);
        }
    }

    public boolean noneSatisfy(FloatPredicate predicate)
    {
        synchronized (this.lock)
        {
            return this.stack.noneSatisfy(predicate);
        }
    }

    public float detectIfNone(FloatPredicate predicate, float ifNone)
    {
        synchronized (this.lock)
        {
            return this.stack.detectIfNone(predicate, ifNone);
        }
    }

    public MutableFloatStack select(FloatPredicate predicate)
    {
        synchronized (this.lock)
        {
            return this.stack.select(predicate);
        }
    }

    public MutableFloatStack reject(FloatPredicate predicate)
    {
        synchronized (this.lock)
        {
            return this.stack.reject(predicate);
        }
    }

    public <V> MutableStack<V> collect(FloatToObjectFunction<? extends V> function)
    {
        synchronized (this.lock)
        {
            return this.stack.collect(function);
        }
    }

    public double sum()
    {
        synchronized (this.lock)
        {
            return this.stack.sum();
        }
    }

    public float max()
    {
        synchronized (this.lock)
        {
            return this.stack.max();
        }
    }

    public float min()
    {
        synchronized (this.lock)
        {
            return this.stack.min();
        }
    }

    public float minIfEmpty(float defaultValue)
    {
        synchronized (this.lock)
        {
            return this.stack.minIfEmpty(defaultValue);
        }
    }

    public float maxIfEmpty(float defaultValue)
    {
        synchronized (this.lock)
        {
            return this.stack.maxIfEmpty(defaultValue);
        }
    }

    public double average()
    {
        synchronized (this.lock)
        {
            return this.stack.average();
        }
    }

    public double median()
    {
        synchronized (this.lock)
        {
            return this.stack.median();
        }
    }

    public MutableFloatList toSortedList()
    {
        synchronized (this.lock)
        {
            return this.stack.toSortedList();
        }
    }

    public float[] toSortedArray()
    {
        synchronized (this.lock)
        {
            return this.stack.toSortedArray();
        }
    }

    public float[] toArray()
    {
        synchronized (this.lock)
        {
            return this.stack.toArray();
        }
    }

    @Override
    public String toString()
    {
        synchronized (this.lock)
        {
            return this.stack.toString();
        }
    }

    public String makeString()
    {
        synchronized (this.lock)
        {
            return this.stack.makeString();
        }
    }

    public String makeString(String separator)
    {
        synchronized (this.lock)
        {
            return this.stack.makeString(separator);
        }
    }

    public String makeString(String start, String separator, String end)
    {
        synchronized (this.lock)
        {
            return this.stack.makeString(start, separator, end);
        }
    }

    public void appendString(Appendable appendable)
    {
        synchronized (this.lock)
        {
            this.stack.appendString(appendable);
        }
    }

    public void appendString(Appendable appendable, String separator)
    {
        synchronized (this.lock)
        {
            this.stack.appendString(appendable, separator);
        }
    }

    public void appendString(
            Appendable appendable,
            String start,
            String separator,
            String end)
    {
        synchronized (this.lock)
        {
            this.stack.appendString(appendable, start, separator, end);
        }
    }

    public MutableFloatList toList()
    {
        synchronized (this.lock)
        {
            return this.stack.toList();
        }
    }

    public MutableFloatSet toSet()
    {
        synchronized (this.lock)
        {
            return this.stack.toSet();
        }
    }

    public MutableFloatBag toBag()
    {
        synchronized (this.lock)
        {
            return this.stack.toBag();
        }
    }

    @Override
    public boolean equals(Object otherStack)
    {
        synchronized (this.lock)
        {
            return this.stack.equals(otherStack);
        }
    }

    @Override
    public int hashCode()
    {
        synchronized (this.lock)
        {
            return this.stack.hashCode();
        }
    }

    public LazyFloatIterable asLazy()
    {
        synchronized (this.lock)
        {
            return new LazyFloatIterableAdapter(this);
        }
    }

    public MutableFloatStack asUnmodifiable()
    {
        return new UnmodifiableFloatStack(this);
    }

    public MutableFloatStack asSynchronized()
    {
        return this;
    }

    public ImmutableFloatStack toImmutable()
    {
        return FloatStacks.immutable.withAllReversed(this);
    }

    public <T> T injectInto(T injectedValue, ObjectFloatToObjectFunction<? super T, ? extends T> function)
    {
        synchronized (this.lock)
        {
            return this.stack.injectInto(injectedValue, function);
        }
    }
}