/*
 * Copyright 2014 Goldman Sachs.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.gs.collections.impl.utility.primitive;

import com.gs.collections.api.DoubleIterable;
import com.gs.collections.api.LazyDoubleIterable;
import com.gs.collections.api.LazyIterable;
import com.gs.collections.api.block.function.primitive.DoubleToObjectFunction;
import com.gs.collections.api.block.predicate.primitive.DoublePredicate;
import com.gs.collections.impl.factory.primitive.DoubleLists;
import com.gs.collections.impl.lazy.primitive.CollectDoubleToObjectIterable;
import com.gs.collections.impl.lazy.primitive.LazyDoubleIterableAdapter;
import com.gs.collections.impl.lazy.primitive.SelectDoubleIterable;

/**
 * LazyDoubleIterate is a factory class which creates "deferred" double iterables around the specified double iterables. A "deferred"
 * double iterable performs some operation, such as filtering or transforming, when the result double iterable is iterated over.  This
 * makes the operation very memory efficient, because you don't have to create intermediate collections during the
 * operation.
 * This file was automatically generated from template file lazyPrimitiveIterate.stg.
 *
 * @since 5.0
 */
public final class LazyDoubleIterate
{
    private static final LazyDoubleIterable EMPTY_ITERABLE = DoubleLists.immutable.of().asLazy();

    private LazyDoubleIterate()
    {
        throw new AssertionError("Suppress default constructor for noninstantiability");
    }

    /**
     * Creates a deferred double iterable for the specified double iterable.
     */
    public static LazyDoubleIterable adapt(DoubleIterable iterable)
    {
        return new LazyDoubleIterableAdapter(iterable);
    }

    /**
     * Creates a deferred filtering double iterable for the specified double iterable.
     */
    public static LazyDoubleIterable select(DoubleIterable iterable, DoublePredicate predicate)
    {
        return new SelectDoubleIterable(iterable, predicate);
    }

    /**
     * Creates a deferred transforming double iterable for the specified double iterable.
     */
    public static <V> LazyIterable<V> collect(
            DoubleIterable iterable,
            DoubleToObjectFunction<? extends V> function)
    {
        return new CollectDoubleToObjectIterable<V>(iterable, function);
    }

    /**
     * Creates a deferred filtering and transforming double iterable for the specified double iterable.
     */
    public static <V> LazyIterable<V> collectIf(
            DoubleIterable iterable,
            DoublePredicate predicate,
            DoubleToObjectFunction<? extends V> function)
    {
        return LazyDoubleIterate.select(iterable, predicate).collect(function);
    }

    public static LazyDoubleIterable empty()
    {
        return EMPTY_ITERABLE;
    }
}
