/*
 * Copyright 2014 Goldman Sachs.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.gs.collections.impl.utility.primitive;

import com.gs.collections.api.LongIterable;
import com.gs.collections.api.LazyLongIterable;
import com.gs.collections.api.LazyIterable;
import com.gs.collections.api.block.function.primitive.LongToObjectFunction;
import com.gs.collections.api.block.predicate.primitive.LongPredicate;
import com.gs.collections.impl.factory.primitive.LongLists;
import com.gs.collections.impl.lazy.primitive.CollectLongToObjectIterable;
import com.gs.collections.impl.lazy.primitive.LazyLongIterableAdapter;
import com.gs.collections.impl.lazy.primitive.SelectLongIterable;

/**
 * LazyLongIterate is a factory class which creates "deferred" long iterables around the specified long iterables. A "deferred"
 * long iterable performs some operation, such as filtering or transforming, when the result long iterable is iterated over.  This
 * makes the operation very memory efficient, because you don't have to create intermediate collections during the
 * operation.
 * This file was automatically generated from template file lazyPrimitiveIterate.stg.
 *
 * @since 5.0
 */
public final class LazyLongIterate
{
    private static final LazyLongIterable EMPTY_ITERABLE = LongLists.immutable.of().asLazy();

    private LazyLongIterate()
    {
        throw new AssertionError("Suppress default constructor for noninstantiability");
    }

    /**
     * Creates a deferred long iterable for the specified long iterable.
     */
    public static LazyLongIterable adapt(LongIterable iterable)
    {
        return new LazyLongIterableAdapter(iterable);
    }

    /**
     * Creates a deferred filtering long iterable for the specified long iterable.
     */
    public static LazyLongIterable select(LongIterable iterable, LongPredicate predicate)
    {
        return new SelectLongIterable(iterable, predicate);
    }

    /**
     * Creates a deferred transforming long iterable for the specified long iterable.
     */
    public static <V> LazyIterable<V> collect(
            LongIterable iterable,
            LongToObjectFunction<? extends V> function)
    {
        return new CollectLongToObjectIterable<V>(iterable, function);
    }

    /**
     * Creates a deferred filtering and transforming long iterable for the specified long iterable.
     */
    public static <V> LazyIterable<V> collectIf(
            LongIterable iterable,
            LongPredicate predicate,
            LongToObjectFunction<? extends V> function)
    {
        return LazyLongIterate.select(iterable, predicate).collect(function);
    }

    public static LazyLongIterable empty()
    {
        return EMPTY_ITERABLE;
    }
}
