/*
 * Copyright 2014 Goldman Sachs.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.gs.collections.impl.lazy.primitive;

import com.gs.collections.api.FloatIterable;
import com.gs.collections.api.LazyFloatIterable;
import com.gs.collections.api.LazyIterable;
import com.gs.collections.api.bag.primitive.MutableFloatBag;
import com.gs.collections.api.block.function.primitive.FloatToObjectFunction;
import com.gs.collections.api.block.function.primitive.ObjectFloatToObjectFunction;
import com.gs.collections.api.block.predicate.primitive.FloatPredicate;
import com.gs.collections.api.block.procedure.primitive.FloatProcedure;
import com.gs.collections.api.list.primitive.MutableFloatList;
import com.gs.collections.api.set.primitive.MutableFloatSet;
import com.gs.collections.impl.bag.mutable.primitive.FloatHashBag;
import com.gs.collections.impl.block.factory.primitive.FloatPredicates;
import com.gs.collections.impl.factory.primitive.FloatSets;
import com.gs.collections.impl.list.mutable.primitive.FloatArrayList;
import com.gs.collections.impl.set.mutable.primitive.FloatHashSet;
import com.gs.collections.impl.utility.internal.primitive.FloatIterableIterate;
import com.gs.collections.impl.utility.primitive.LazyFloatIterate;

import java.util.NoSuchElementException;

/**
 * This file was automatically generated from template file abstractLazyPrimitiveIterable.stg.
 *
 * @since 5.0
 */
public abstract class AbstractLazyFloatIterable implements LazyFloatIterable
{
    public int size()
    {
        return this.count(FloatPredicates.alwaysTrue());
    }

    @Override
    public String toString()
    {
        return this.makeString("[", ", ", "]");
    }

    public boolean isEmpty()
    {
        return FloatIterableIterate.isEmpty(this);
    }

    public boolean notEmpty()
    {
        return FloatIterableIterate.notEmpty(this);
    }

    public String makeString()
    {
        return this.makeString(", ");
    }

    public String makeString(String separator)
    {
        return this.makeString("", separator, "");
    }

    public String makeString(String start, String separator, String end)
    {
        Appendable stringBuilder = new StringBuilder();
        this.appendString(stringBuilder, start, separator, end);
        return stringBuilder.toString();
    }

    public void appendString(Appendable appendable)
    {
        this.appendString(appendable, ", ");
    }

    public void appendString(Appendable appendable, String separator)
    {
        this.appendString(appendable, "", separator, "");
    }

    public void appendString(Appendable appendable, String start, String separator, String end)
    {
        FloatIterableIterate.appendString(this, appendable, start, separator, end);
    }

    public boolean contains(float value)
    {
        return this.anySatisfy(FloatPredicates.equal(value));
    }

    public boolean containsAll(float... source)
    {
        return this.containsAll(FloatSets.immutable.of(source));
    }

    public boolean containsAll(FloatIterable source)
    {
        return source.allSatisfy(new FloatPredicate()
        {
            public boolean accept(float value)
            {
                return AbstractLazyFloatIterable.this.contains(value);
            }
        });
    }

    public LazyFloatIterable select(FloatPredicate predicate)
    {
        return LazyFloatIterate.select(this, predicate);
    }

    public LazyFloatIterable reject(FloatPredicate predicate)
    {
        return LazyFloatIterate.select(this, FloatPredicates.not(predicate));
    }

    public <V> LazyIterable<V> collect(FloatToObjectFunction<? extends V> function)
    {
        return LazyFloatIterate.collect(this, function);
    }

    public float detectIfNone(FloatPredicate predicate, float ifNone)
    {
        return FloatIterableIterate.detectIfNone(this, predicate, ifNone);
    }

    public int count(FloatPredicate predicate)
    {
        return FloatIterableIterate.count(this, predicate);
    }

    public boolean anySatisfy(FloatPredicate predicate)
    {
        return FloatIterableIterate.anySatisfy(this, predicate);
    }

    public boolean allSatisfy(FloatPredicate predicate)
    {
        return FloatIterableIterate.allSatisfy(this, predicate);
    }

    public boolean noneSatisfy(FloatPredicate predicate)
    {
        return FloatIterableIterate.noneSatisfy(this, predicate);
    }

    public <T> T injectInto(T injectedValue, ObjectFloatToObjectFunction<? super T, ? extends T> function)
    {
        return FloatIterableIterate.injectInto(this, injectedValue, function);
    }

    public float[] toArray()
    {
        return this.toList().toArray();
    }

    public MutableFloatList toList()
    {
        final MutableFloatList list = new FloatArrayList();
        this.forEach(new FloatProcedure()
        {
            public void value(float each)
            {
                list.add(each);
            }
        });
        return list;
    }

    public MutableFloatSet toSet()
    {
        final MutableFloatSet set = new FloatHashSet();
        this.forEach(new FloatProcedure()
        {
            public void value(float each)
            {
                set.add(each);
            }
        });
        return set;
    }

    public MutableFloatBag toBag()
    {
        final MutableFloatBag bag = new FloatHashBag();
        this.forEach(new FloatProcedure()
        {
            public void value(float each)
            {
                bag.add(each);
            }
        });
        return bag;
    }

    public double sum()
    {
        FloatSumProcedure procedure = new FloatSumProcedure();
        this.forEach(procedure);
        return procedure.getValue();
    }

    public float max()
    {
        if (this.isEmpty())
        {
            throw new NoSuchElementException();
        }
        FloatMaxProcedure procedure = new FloatMaxProcedure();
        this.forEach(procedure);
        return procedure.getValue();
    }

    public float maxIfEmpty(float ifEmpty)
    {
        if (this.isEmpty())
        {
            return ifEmpty;
        }
        return this.max();
    }

    public float min()
    {
        if (this.isEmpty())
        {
            throw new NoSuchElementException();
        }
        FloatMinProcedure procedure = new FloatMinProcedure();
        this.forEach(procedure);
        return procedure.getValue();
    }

    public float minIfEmpty(float ifEmpty)
    {
        if (this.isEmpty())
        {
            return ifEmpty;
        }
        return this.min();
    }

    public double average()
    {
        if (this.isEmpty())
        {
            throw new ArithmeticException();
        }
        return (double) this.sum() / (double) this.size();
    }

    public double median()
    {
        if (this.isEmpty())
        {
            throw new ArithmeticException();
        }
        float[] sortedArray = this.toSortedArray();
        int middleIndex = sortedArray.length >> 1;
        if (sortedArray.length > 1 && (sortedArray.length & 1) == 0)
        {
            float first = sortedArray[middleIndex];
            float second = sortedArray[middleIndex - 1];
            return ((double) first + (double) second) / 2.0;
        }
        return (double) sortedArray[middleIndex];
    }

    public float[] toSortedArray()
    {
        return this.toSortedList().toArray();
    }

    public MutableFloatList toSortedList()
    {
        return FloatArrayList.newList(this).sortThis();
    }

    private static final class FloatMaxProcedure implements FloatProcedure
    {
        private boolean visitedOnce;
        private float max;

        public void value(float each)
        {
            if (this.visitedOnce)
            {
                if (Float.compare(this.max, each) < 0)
                {
                    this.max = each;
                }
            }
            else
            {
                this.max = each;
                this.visitedOnce = true;
            }
        }

        public float getValue()
        {
            return this.max;
        }
    }

    private static final class FloatMinProcedure implements FloatProcedure
    {
        private boolean visitedOnce;
        private float min;

        public void value(float each)
        {
            if (this.visitedOnce)
            {
                if (Float.compare(each, this.min) < 0)
                {
                    this.min = each;
                }
            }
            else
            {
                this.min = each;
                this.visitedOnce = true;
            }
        }

        public float getValue()
        {
            return this.min;
        }
    }

    private static final class FloatSumProcedure implements FloatProcedure
    {
        private float sum = 0;

        public void value(float each)
        {
            this.sum += each;
        }

        public float getValue()
        {
            return this.sum;
        }
    }

    public LazyFloatIterable asLazy()
    {
        return this;
    }
}
