/*
 * Copyright 2014 Goldman Sachs.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.gs.collections.impl.list.mutable.primitive;

import java.io.Externalizable;
import java.io.IOException;
import java.io.ObjectInput;
import java.io.ObjectOutput;
import java.util.Arrays;
import java.util.NoSuchElementException;

import com.gs.collections.api.ShortIterable;
import com.gs.collections.api.LazyShortIterable;
import com.gs.collections.api.block.function.primitive.ObjectShortIntToObjectFunction;
import com.gs.collections.api.block.function.primitive.ObjectShortToObjectFunction;
import com.gs.collections.api.block.function.primitive.ShortToObjectFunction;
import com.gs.collections.api.block.predicate.primitive.ShortPredicate;
import com.gs.collections.api.block.procedure.primitive.ShortIntProcedure;
import com.gs.collections.api.block.procedure.primitive.ShortProcedure;
import com.gs.collections.api.iterator.MutableShortIterator;
import com.gs.collections.api.list.MutableList;
import com.gs.collections.api.list.primitive.ShortList;
import com.gs.collections.api.list.primitive.ImmutableShortList;
import com.gs.collections.api.list.primitive.MutableShortList;
import com.gs.collections.api.set.primitive.ShortSet;
import com.gs.collections.api.set.primitive.MutableShortSet;
import com.gs.collections.impl.factory.primitive.ShortLists;
import com.gs.collections.impl.lazy.primitive.ReverseShortIterable;
import com.gs.collections.impl.list.mutable.FastList;
import com.gs.collections.impl.primitive.AbstractShortIterable;
import com.gs.collections.impl.set.mutable.primitive.ShortHashSet;
import net.jcip.annotations.NotThreadSafe;

/**
 * ShortArrayList is similar to {@link FastList}, and is memory-optimized for short primitives.
 * This file was automatically generated from template file primitiveArrayList.stg.
 *
 * @since 3.0.
 */
@NotThreadSafe
public final class ShortArrayList extends AbstractShortIterable
        implements MutableShortList, Externalizable
{
    private static final long serialVersionUID = 1L;
    private static final short[] DEFAULT_SIZED_EMPTY_ARRAY = {};
    private static final short[] ZERO_SIZED_ARRAY = {};
    private static final int MAXIMUM_ARRAY_SIZE = Integer.MAX_VALUE - 8;
    private int size;
    private transient short[] items = DEFAULT_SIZED_EMPTY_ARRAY;

    public ShortArrayList()
    {
    }

    public ShortArrayList(int initialCapacity)
    {
        this.items = initialCapacity == 0 ? ZERO_SIZED_ARRAY : new short[initialCapacity];
    }

    public ShortArrayList(short... array)
    {
        this.size = array.length;
        this.items = array;
    }

    /**
     * Creates a new list using the passed {@code elements} argument as the backing store.
     * <p>
     * !!! WARNING: This method uses the passed in array, so can be very unsafe if the original
     * array is held onto anywhere else. !!!
     */
    public static ShortArrayList newListWith(short... elements)
    {
        return new ShortArrayList(elements);
    }

    public static ShortArrayList newList(ShortIterable source)
    {
        return ShortArrayList.newListWith(source.toArray());
    }

    public static ShortArrayList newWithNValues(int size, short value)
    {
        ShortArrayList newList = new ShortArrayList(size);
        for (int i = 0; i < size; i++)
        {
            newList.add(value);
        }
        return newList;
    }

    public int size()
    {
        return this.size;
    }

    public void clear()
    {
        Arrays.fill(this.items, (short) 0);
        this.size = 0;
    }

    public boolean contains(short value)
    {
        for (int i = 0; i < this.size; i++)
        {
            if (this.items[i] == value)
            {
                return true;
            }
        }
        return false;
    }

    public short get(int index)
    {
        if (index < this.size)
        {
            return this.items[index];
        }
        throw this.newIndexOutOfBoundsException(index);
    }

    private IndexOutOfBoundsException newIndexOutOfBoundsException(int index)
    {
        return new IndexOutOfBoundsException("Index: " + index + " Size: " + this.size);
    }

    public short getFirst()
    {
        this.checkEmpty();
        return this.items[0];
    }

    public short getLast()
    {
        this.checkEmpty();
        return this.items[this.size() - 1];
    }

    private void checkEmpty()
    {
        if (this.isEmpty())
        {
            throw this.newIndexOutOfBoundsException(0);
        }
    }

    public int indexOf(short value)
    {
        for (int i = 0; i < this.size; i++)
        {
            if (this.items[i] == value)
            {
                return i;
            }
        }
        return -1;
    }

    public int lastIndexOf(short value)
    {
        for (int i = this.size - 1; i >= 0; i--)
        {
            if (this.items[i] == value)
            {
                return i;
            }
        }
        return -1;
    }

    public void trimToSize()
    {
        if (this.size < this.items.length)
        {
            this.transferItemsToNewArrayWithCapacity(this.size);
        }
    }

    private void transferItemsToNewArrayWithCapacity(int newCapacity)
    {
        this.items = this.copyItemsWithNewCapacity(newCapacity);
    }

    private short[] copyItemsWithNewCapacity(int newCapacity)
    {
        short[] newItems = new short[newCapacity];
        System.arraycopy(this.items, 0, newItems, 0, Math.min(this.size, newCapacity));
        return newItems;
    }

    private int sizePlusFiftyPercent(int oldSize)
    {
        int result = oldSize + (oldSize >> 1) + 1;
        return result < oldSize ? MAXIMUM_ARRAY_SIZE : result;
    }

    public void ensureCapacity(int minCapacity)
    {
        int oldCapacity = this.items.length;
        if (minCapacity > oldCapacity)
        {
            int newCapacity = Math.max(this.sizePlusFiftyPercent(oldCapacity), minCapacity);
            this.transferItemsToNewArrayWithCapacity(newCapacity);
        }
    }

    private void ensureCapacityForAdd()
    {
        if (this.items == DEFAULT_SIZED_EMPTY_ARRAY)
        {
            this.items = new short[10];
        }
        else
        {
            this.transferItemsToNewArrayWithCapacity(this.sizePlusFiftyPercent(this.size));
        }
    }

    public boolean add(short newItem)
    {
        if (this.items.length == this.size)
        {
            this.ensureCapacityForAdd();
        }
        this.items[this.size] = newItem;
        this.size++;
        return true;
    }

    public boolean addAll(short... source)
    {
        if (source.length < 1)
        {
            return false;
        }
        int sourceSize = source.length;
        int newSize = this.size + sourceSize;
        this.ensureCapacity(newSize);
        System.arraycopy(source, 0, this.items, this.size, sourceSize);
        this.size = newSize;
        return true;
    }

    public boolean addAll(ShortIterable source)
    {
        return this.addAll(source.toArray());
    }

    private void throwOutOfBounds(int index)
    {
        throw this.newIndexOutOfBoundsException(index);
    }

    public void addAtIndex(int index, short element)
    {
        if (index > -1 && index < this.size)
        {
            this.addAtIndexLessThanSize(index, element);
        }
        else if (index == this.size)
        {
            this.add(element);
        }
        else
        {
            this.throwOutOfBounds(index);
        }
    }

    private void addAtIndexLessThanSize(int index, short element)
    {
        int oldSize = this.size;
        this.size++;
        if (this.items.length == oldSize)
        {
            short[] newItems = new short[this.sizePlusFiftyPercent(oldSize)];
            if (index > 0)
            {
                System.arraycopy(this.items, 0, newItems, 0, index);
            }
            System.arraycopy(this.items, index, newItems, index + 1, oldSize - index);
            this.items = newItems;
        }
        else
        {
            System.arraycopy(this.items, index, this.items, index + 1, oldSize - index);
        }
        this.items[index] = element;
    }

    public boolean addAllAtIndex(int index, short... source)
    {
        if (index > this.size || index < 0)
        {
            this.throwOutOfBounds(index);
        }
        if (source.length == 0)
        {
            return false;
        }
        int sourceSize = source.length;
        int newSize = this.size + sourceSize;
        this.ensureCapacity(newSize);
        this.shiftElementsAtIndex(index, sourceSize);
        System.arraycopy(source, 0, this.items, index, sourceSize);
        this.size = newSize;
        return true;
    }

    public boolean addAllAtIndex(int index, ShortIterable source)
    {
        return this.addAllAtIndex(index, source.toArray());
    }

    private void shiftElementsAtIndex(int index, int sourceSize)
    {
        int numberToMove = this.size - index;
        if (numberToMove > 0)
        {
            System.arraycopy(this.items, index, this.items, index + sourceSize, numberToMove);
        }
    }

    public boolean remove(short value)
    {
        int index = this.indexOf(value);
        if (index >= 0)
        {
            this.removeAtIndex(index);
            return true;
        }
        return false;
    }

    public boolean removeAll(ShortIterable source)
    {
        boolean modified = false;
        for (int index = 0; index < this.size; index++)
        {
            if (source.contains(this.get(index)))
            {
                this.removeAtIndex(index);
                index--;
                modified = true;
            }
        }
        return modified;
    }

    public boolean removeAll(short... source)
    {
        ShortHashSet set = ShortHashSet.newSetWith(source);
        short[] newItems = new short[this.size];
        int count = 0;
        int oldSize = this.size;
        for (int index = 0; index < this.size; index++)
        {
            if (!set.contains(this.items[index]))
            {
                newItems[count] = this.items[index];
                count++;
            }
        }
        this.items = newItems;
        this.size = count;
        return oldSize != this.size;
    }

    public boolean retainAll(ShortIterable source)
    {
        int oldSize = this.size();
        final ShortSet sourceSet = source instanceof ShortSet ? (ShortSet) source : source.toSet();
        ShortArrayList retained = this.select(new ShortPredicate()
        {
            public boolean accept(short value)
            {
                return sourceSet.contains(value);
            }
        });
        this.size = retained.size;
        this.items = retained.items;
        return oldSize != this.size();
    }

    public boolean retainAll(short... source)
    {
        return this.retainAll(ShortHashSet.newSetWith(source));
    }

    public short removeAtIndex(int index)
    {
        short previous = this.get(index);
        int totalOffset = this.size - index - 1;
        if (totalOffset > 0)
        {
            System.arraycopy(this.items, index + 1, this.items, index, totalOffset);
        }
        --this.size;
        this.items[this.size] = (short) 0;
        return previous;
    }

    public short set(int index, short element)
    {
        short previous = this.get(index);
        this.items[index] = element;
        return previous;
    }

    public ShortArrayList with(short element)
    {
        this.add(element);
        return this;
    }

    public ShortArrayList without(short element)
    {
        this.remove(element);
        return this;
    }

    public ShortArrayList withAll(ShortIterable elements)
    {
        this.addAll(elements.toArray());
        return this;
    }

    public ShortArrayList withoutAll(ShortIterable elements)
    {
        this.removeAll(elements);
        return this;
    }

    public ShortArrayList with(short element1, short element2)
    {
        this.add(element1);
        this.add(element2);
        return this;
    }

    public ShortArrayList with(short element1, short element2, short element3)
    {
        this.add(element1);
        this.add(element2);
        this.add(element3);
        return this;
    }

    public ShortArrayList with(short element1, short element2, short element3, short... elements)
    {
        this.add(element1);
        this.add(element2);
        this.add(element3);
        return this.withArrayCopy(elements, 0, elements.length);
    }

    private ShortArrayList withArrayCopy(short[] elements, int begin, int length)
    {
        this.ensureCapacity(this.size + length);
        System.arraycopy(elements, begin, this.items, this.size, length);
        this.size += length;
        return this;
    }

    public MutableShortIterator shortIterator()
    {
        return new InternalShortIterator();
    }

    public void forEach(ShortProcedure procedure)
    {
        for (int i = 0; i < this.size; i++)
        {
            procedure.value(this.items[i]);
        }
    }

    public void forEachWithIndex(ShortIntProcedure procedure)
    {
        for (int i = 0; i < this.size; i++)
        {
            procedure.value(this.items[i], i);
        }
    }

    public <T> T injectInto(T injectedValue, ObjectShortToObjectFunction<? super T, ? extends T> function)
    {
        T result = injectedValue;
        for (int i = 0; i < this.size; i++)
        {
            result = function.valueOf(result, this.items[i]);
        }
        return result;
    }

    public <T> T injectIntoWithIndex(T injectedValue, ObjectShortIntToObjectFunction<? super T, ? extends T> function)
    {
        T result = injectedValue;
        for (int i = 0; i < this.size; i++)
        {
            result = function.valueOf(result, this.items[i], i);
        }
        return result;
    }

    public int count(ShortPredicate predicate)
    {
        int count = 0;
        for (int i = 0; i < this.size; i++)
        {
            if (predicate.accept(this.items[i]))
            {
                count++;
            }
        }
        return count;
    }

    public boolean anySatisfy(ShortPredicate predicate)
    {
        for (int i = 0; i < this.size; i++)
        {
            if (predicate.accept(this.items[i]))
            {
                return true;
            }
        }
        return false;
    }

    public boolean allSatisfy(ShortPredicate predicate)
    {
        for (int i = 0; i < this.size; i++)
        {
            if (!predicate.accept(this.items[i]))
            {
                return false;
            }
        }
        return true;
    }

    public boolean noneSatisfy(ShortPredicate predicate)
    {
        for (int i = 0; i < this.size; i++)
        {
            if (predicate.accept(this.items[i]))
            {
                return false;
            }
        }
        return true;
    }

    public ShortArrayList select(ShortPredicate predicate)
    {
        ShortArrayList result = new ShortArrayList();
        for (int i = 0; i < this.size; i++)
        {
            short item = this.items[i];
            if (predicate.accept(item))
            {
                result.add(item);
            }
        }
        return result;
    }

    public ShortArrayList reject(ShortPredicate predicate)
    {
        ShortArrayList result = new ShortArrayList();
        for (int i = 0; i < this.size; i++)
        {
            short item = this.items[i];
            if (!predicate.accept(item))
            {
                result.add(item);
            }
        }
        return result;
    }

    public short detectIfNone(ShortPredicate predicate, short ifNone)
    {
        for (int i = 0; i < this.size; i++)
        {
            short item = this.items[i];
            if (predicate.accept(item))
            {
                return item;
            }
        }
        return ifNone;
    }

    public <V> MutableList<V> collect(ShortToObjectFunction<? extends V> function)
    {
        FastList<V> target = FastList.newList(this.size);
        for (int i = 0; i < this.size; i++)
        {
            target.add(function.valueOf(this.items[i]));
        }
        return target;
    }

    public short max()
    {
        if (this.isEmpty())
        {
            throw new NoSuchElementException();
        }
        short max = this.items[0];
        for (int i = 1; i < this.size; i++)
        {
            short value = this.items[i];
            if (max < value)
            {
                max = value;
            }
        }
        return max;
    }

    public short min()
    {
        if (this.isEmpty())
        {
            throw new NoSuchElementException();
        }
        short min = this.items[0];
        for (int i = 1; i < this.size; i++)
        {
            short value = this.items[i];
            if (value < min)
            {
                min = value;
            }
        }
        return min;
    }

    public long sum()
    {
        long result = 0L;
        for (int i = 0; i < this.size; i++)
        {
            result += this.items[i];
        }
        return result;
    }

    public long dotProduct(ShortList list)
    {
        if (this.size != list.size())
        {
            throw new IllegalArgumentException("Lists used in dotProduct must be the same size");
        }
        long sum = 0L;
        for (int i = 0; i < this.size; i++)
        {
            sum += (long) this.items[i] * list.get(i);
        }
        return sum;
    }

    public short[] toArray()
    {
        short[] newItems = new short[this.size];
        System.arraycopy(this.items, 0, newItems, 0, this.size);
        return newItems;
    }

    @Override
    public boolean equals(Object otherList)
    {
        if (otherList == this)
        {
            return true;
        }
        if (!(otherList instanceof ShortList))
        {
            return false;
        }
        ShortList list = (ShortList) otherList;
        if (this.size != list.size())
        {
            return false;
        }
        for (int i = 0; i < this.size; i++)
        {
            if (this.items[i] != list.get(i))
            {
                return false;
            }
        }
        return true;
    }

    @Override
    public int hashCode()
    {
        int hashCode = 1;
        for (int i = 0; i < this.size; i++)
        {
            short item = this.items[i];
            hashCode = 31 * hashCode + (int) item;
        }
        return hashCode;
    }

    public void appendString(
            Appendable appendable,
            String start,
            String separator,
            String end)
    {
        try
        {
            appendable.append(start);
            for (int i = 0; i < this.size; i++)
            {
                if (i > 0)
                {
                    appendable.append(separator);
                }
                short value = this.items[i];
                appendable.append(String.valueOf(value));
            }
            appendable.append(end);
        }
        catch (IOException e)
        {
            throw new RuntimeException(e);
        }
    }

    public MutableShortList asUnmodifiable()
    {
        return new UnmodifiableShortList(this);
    }

    public MutableShortList asSynchronized()
    {
        return new SynchronizedShortList(this);
    }

    public ImmutableShortList toImmutable()
    {
        if (this.size == 0)
        {
            return ShortLists.immutable.empty();
        }
        if (this.size == 1)
        {
            return ShortLists.immutable.with(this.items[0]);
        }
        return ShortLists.immutable.with(this.toArray());
    }

    public void writeExternal(ObjectOutput out) throws IOException
    {
        out.writeInt(this.size);
        for (int i = 0; i < this.size; i++)
        {
            out.writeShort(this.items[i]);
        }
    }

    public void readExternal(ObjectInput in) throws IOException
    {
        this.size = in.readInt();
        this.items = new short[this.size];
        for (int i = 0; i < this.size; i++)
        {
            this.items[i] = in.readShort();
        }
    }

    public LazyShortIterable asReversed()
    {
        return ReverseShortIterable.adapt(this);
    }

    public ShortArrayList reverseThis()
    {
        int size = this.items.length;
        int endIndex = size - 1;
        for (int i = 0; i < size / 2; i++)
        {
            short tempSwapValue = this.items[i];
            this.items[i] = this.items[endIndex - i];
            this.items[endIndex - i] = tempSwapValue;
        }
        return this;
    }

    public ShortArrayList sortThis()
    {
        Arrays.sort(this.items, 0, this.size);
        return this;
    }

    public ShortArrayList toReversed()
    {
        return ShortArrayList.newList(this.asReversed());
    }

    public MutableShortList distinct()
    {
        ShortArrayList target = new ShortArrayList();
        MutableShortSet seenSoFar = new ShortHashSet(this.size());

        for (int i = 0; i < this.size; i++)
        {
            short each = this.items[i];
            if (seenSoFar.add(each))
            {
                target.add(each);
            }
        }
        return target;
    }

    public MutableShortList subList(int fromIndex, int toIndex)
    {
        throw new UnsupportedOperationException("subList not yet implemented!");
    }

    private class InternalShortIterator implements MutableShortIterator
    {
        /**
         * Index of element to be returned by subsequent call to next.
         */
        private int currentIndex;
        private int lastIndex = -1;

        public boolean hasNext()
        {
            return this.currentIndex != ShortArrayList.this.size();
        }

        public short next()
        {
            if (!this.hasNext())
            {
                throw new NoSuchElementException();
            }
            short next = ShortArrayList.this.items[this.currentIndex];
            this.lastIndex = this.currentIndex++;
            return next;
        }

        public void remove()
        {
            if (this.lastIndex == -1)
            {
                throw new IllegalStateException();
            }
            ShortArrayList.this.removeAtIndex(this.lastIndex);
            this.currentIndex--;
            this.lastIndex = -1;
        }
    }
}
