/*
 * Copyright 2014 Goldman Sachs.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.gs.collections.impl.map.mutable.primitive;

import java.io.IOException;
import java.io.Serializable;
import java.util.NoSuchElementException;
import com.gs.collections.api.block.function.primitive.DoubleToObjectFunction;
import com.gs.collections.api.block.function.primitive.ObjectDoubleToObjectFunction;
import com.gs.collections.api.block.predicate.primitive.DoublePredicate;
import com.gs.collections.api.block.procedure.primitive.DoubleProcedure;
import com.gs.collections.api.iterator.DoubleIterator;
import com.gs.collections.api.set.ImmutableSet;
import com.gs.collections.api.set.MutableSet;
import com.gs.collections.api.set.primitive.ImmutableDoubleSet;
import com.gs.collections.impl.set.immutable.primitive.ImmutableDoubleSetSerializationProxy;
import com.gs.collections.impl.set.immutable.primitive.AbstractImmutableDoubleSet;
import com.gs.collections.impl.set.mutable.UnifiedSet;
import com.gs.collections.impl.set.mutable.primitive.DoubleHashSet;


/**
 * This file was automatically generated from template file immutablePrimitivePrimitiveMapKeySet.stg
 *
 * @since 6.0.
 */
class ImmutableDoubleDoubleMapKeySet extends AbstractImmutableDoubleSet implements Serializable
{
    private static final long serialVersionUID = 1L;
    private static final double EMPTY_KEY = 0.0;
    private static final double REMOVED_KEY = 1.0;

    private final double[] keysValues;
    private final int occupiedWithData;
    private final boolean containsZeroKey;
    private final boolean containsOneKey;

    ImmutableDoubleDoubleMapKeySet(double[] keysValues, int occupiedWithData, boolean containsZeroKey, boolean containsOneKey)
    {
        this.keysValues = keysValues;
        this.occupiedWithData = occupiedWithData;
        this.containsZeroKey = containsZeroKey;
        this.containsOneKey = containsOneKey;
    }

    private static boolean isEmptyKey(double key)
    {
        return Double.compare(key, EMPTY_KEY) == 0;
    }

    private static boolean isRemovedKey(double key)
    {
        return Double.compare(key, REMOVED_KEY) == 0;
    }

    private static boolean isNonSentinel(double key)
    {
        return !isEmptyKey(key) && !isRemovedKey(key);
    }

    @Override
    public int hashCode()
    {
        int result = 0;
        if (this.containsZeroKey)
        {
            result += (int) (Double.doubleToLongBits(EMPTY_KEY) ^ Double.doubleToLongBits(EMPTY_KEY) >>> 32);
        }
        if (this.containsOneKey)
        {
            result += (int) (Double.doubleToLongBits(REMOVED_KEY) ^ Double.doubleToLongBits(REMOVED_KEY) >>> 32);
        }
        if (this.keysValues != null)
        {
            for (int i = 0; i < this.keysValues.length; i += 2)
            {
                if (isNonSentinel(this.keysValues[i]))
                {
                    result += (int) (Double.doubleToLongBits(this.keysValues[i]) ^ Double.doubleToLongBits(this.keysValues[i]) >>> 32);
                }
            }
        }
        return result;
    }

    public int size()
    {
        return this.occupiedWithData + (this.containsOneKey ? 1 : 0) + (this.containsZeroKey ? 1 : 0);
    }

    public void appendString(Appendable appendable, String start, String separator, String end)
    {
        try
        {
            appendable.append(start);
            int count = 0;
            if (this.containsZeroKey)
            {
                appendable.append(String.valueOf(EMPTY_KEY));
                count++;
            }
            if (this.containsOneKey)
            {
                if (count > 0)
                {
                    appendable.append(separator);
                }
                count++;
                appendable.append(String.valueOf(REMOVED_KEY));
            }
            for (int i = 0; i < this.keysValues.length; i += 2)
            {
                if (isNonSentinel(this.keysValues[i]))
                {
                    if (count > 0)
                    {
                        appendable.append(separator);
                    }
                    count++;
                    appendable.append(String.valueOf(this.keysValues[i]));
                }
            }
            appendable.append(end);
        }
        catch (IOException e)
        {
            throw new RuntimeException(e);
        }
    }

    public DoubleIterator doubleIterator()
    {
        return new InternalDoubleIterator();
    }

    public double[] toArray()
    {
        double[] array = new double[this.size()];
        int index = 0;
        if (this.containsZeroKey)
        {
            array[index] = EMPTY_KEY;
            index++;
        }
        if (this.containsOneKey)
        {
            array[index] = REMOVED_KEY;
            index++;
        }
        if (this.keysValues != null)
        {
            for (int i = 0; i < this.keysValues.length; i += 2)
            {
                if (isNonSentinel(this.keysValues[i]))
                {
                    array[index] = this.keysValues[i];
                    index++;
                }
            }
        }
        return array;
    }

    public boolean contains(double value)
    {
        if (Double.compare(value, EMPTY_KEY) == 0)
        {
            return this.containsZeroKey;
        }
        if (Double.compare(value, REMOVED_KEY) == 0)
        {
            return this.containsOneKey;
        }
        return Double.compare(this.keysValues[this.probe(value)], value) == 0;
    }

    public void forEach(DoubleProcedure procedure)
    {
        if (this.containsZeroKey)
        {
            procedure.value(EMPTY_KEY);
        }
        if (this.containsOneKey)
        {
            procedure.value(REMOVED_KEY);
        }
        if (this.keysValues != null)
        {
            for (int i = 0; i < this.keysValues.length; i += 2)
            {
                if (isNonSentinel(this.keysValues[i]))
                {
                    procedure.value(this.keysValues[i]);
                }
            }
        }
    }

    public ImmutableDoubleSet select(DoublePredicate predicate)
    {
        DoubleHashSet result = new DoubleHashSet();
        if (this.containsZeroKey && predicate.accept(EMPTY_KEY))
        {
            result.add(EMPTY_KEY);
        }
        if (this.containsOneKey && predicate.accept(REMOVED_KEY))
        {
            result.add(REMOVED_KEY);
        }
        for (int i = 0; i < this.keysValues.length; i += 2)
        {
            if (isNonSentinel(this.keysValues[i]) && predicate.accept(this.keysValues[i]))
            {
                result.add(this.keysValues[i]);
            }
        }
        return result.toImmutable();
    }

    public ImmutableDoubleSet reject(DoublePredicate predicate)
    {
        DoubleHashSet result = new DoubleHashSet();
        if (this.containsZeroKey && !predicate.accept(EMPTY_KEY))
        {
            result.add(EMPTY_KEY);
        }
        if (this.containsOneKey && !predicate.accept(REMOVED_KEY))
        {
            result.add(REMOVED_KEY);
        }
        for (int i = 0; i < this.keysValues.length; i += 2)
        {
            if (isNonSentinel(this.keysValues[i]) && !predicate.accept(this.keysValues[i]))
            {
                result.add(this.keysValues[i]);
            }
        }
        return result.toImmutable();
    }

    public <V> ImmutableSet<V> collect(DoubleToObjectFunction<? extends V> function)
    {
        MutableSet<V> target = UnifiedSet.newSet(this.size());
        if (this.containsZeroKey)
        {
            target.add(function.valueOf(EMPTY_KEY));
        }
        if (this.containsOneKey)
        {
            target.add(function.valueOf(REMOVED_KEY));
        }
        if (this.keysValues != null)
        {
            for (int i = 0; i < this.keysValues.length; i += 2)
            {
                if (isNonSentinel(this.keysValues[i]))
                {
                    target.add(function.valueOf(this.keysValues[i]));
                }
            }
        }
        return target.toImmutable();
    }

    public double detectIfNone(DoublePredicate predicate, double ifNone)
    {
        if (this.containsZeroKey && predicate.accept(EMPTY_KEY))
        {
            return EMPTY_KEY;
        }
        if (this.containsOneKey && predicate.accept(REMOVED_KEY))
        {
            return REMOVED_KEY;
        }
        for (int i = 0; i < this.keysValues.length; i += 2)
        {
            if (isNonSentinel(this.keysValues[i]) && predicate.accept(this.keysValues[i]))
            {
                return this.keysValues[i];
            }
        }
        return ifNone;
    }

    public int count(DoublePredicate predicate)
    {
        int count = 0;
        if (this.containsZeroKey && predicate.accept(EMPTY_KEY))
        {
            count++;
        }
        if (this.containsOneKey && predicate.accept(REMOVED_KEY))
        {
            count++;
        }
        for (int i = 0; i < this.keysValues.length; i += 2)
        {
            if (isNonSentinel(this.keysValues[i]) && predicate.accept(this.keysValues[i]))
            {
                count++;
            }
        }
        return count;
    }

    public boolean anySatisfy(DoublePredicate predicate)
    {
        if (this.containsZeroKey && predicate.accept(EMPTY_KEY))
        {
            return true;
        }
        if (this.containsOneKey && predicate.accept(REMOVED_KEY))
        {
            return true;
        }
        for (int i = 0; i < this.keysValues.length; i += 2)
        {
            if (isNonSentinel(this.keysValues[i]) && predicate.accept(this.keysValues[i]))
            {
                return true;
            }
        }
        return false;
    }

    public boolean allSatisfy(DoublePredicate predicate)
    {
        if (this.containsZeroKey && !predicate.accept(EMPTY_KEY))
        {
            return false;
        }
        if (this.containsOneKey && !predicate.accept(REMOVED_KEY))
        {
            return false;
        }
        for (int i = 0; i < this.keysValues.length; i += 2)
        {
            if (isNonSentinel(this.keysValues[i]) && !predicate.accept(this.keysValues[i]))
            {
                return false;
            }
        }
        return true;
    }

    public boolean noneSatisfy(DoublePredicate predicate)
    {
        if (this.containsZeroKey && predicate.accept(EMPTY_KEY))
        {
            return false;
        }
        if (this.containsOneKey && predicate.accept(REMOVED_KEY))
        {
            return false;
        }
        for (int i = 0; i < this.keysValues.length; i += 2)
        {
            if (isNonSentinel(this.keysValues[i]) && predicate.accept(this.keysValues[i]))
            {
                return false;
            }
        }
        return true;
    }

    public double sum()
    {
        double result = 0.0;
        if (this.containsZeroKey)
        {
            result += EMPTY_KEY;
        }
        if (this.containsOneKey)
        {
            result += REMOVED_KEY;
        }
        if (this.keysValues != null)
        {
            for (int i = 0; i < this.keysValues.length; i += 2)
            {
                if (isNonSentinel(this.keysValues[i]))
                {
                    result += this.keysValues[i];
                }
            }
        }
        return result;
    }

    public double max()
    {
        if (this.isEmpty())
        {
            throw new NoSuchElementException();
        }

        double max = 0;
        boolean isMaxSet = false;

        if (this.containsZeroKey)
        {
            max = EMPTY_KEY;
            isMaxSet = true;
        }
        if (this.containsOneKey && (!isMaxSet || Double.compare(max, REMOVED_KEY) < 0))
        {
            max = REMOVED_KEY;
            isMaxSet = true;
        }
        for (int i = 0; i < this.keysValues.length; i += 2)
        {
            if (isNonSentinel(this.keysValues[i]) && (!isMaxSet || Double.compare(max, this.keysValues[i]) < 0))
            {
                max = this.keysValues[i];
                isMaxSet = true;
            }
        }
        return max;
    }

    public double min()
    {
        if (this.isEmpty())
        {
            throw new NoSuchElementException();
        }

        double min = 0;
        boolean isMinSet = false;

        if (this.containsZeroKey)
        {
            min = EMPTY_KEY;
            isMinSet = true;
        }
        if (this.containsOneKey && (!isMinSet || REMOVED_KEY < min))
        {
            min = REMOVED_KEY;
            isMinSet = true;
        }

        for (int i = 0; i < this.keysValues.length; i += 2)
        {
            if (isNonSentinel(this.keysValues[i]) && (!isMinSet || Double.compare(this.keysValues[i], min) < 0))
            {
                min = this.keysValues[i];
                isMinSet = true;
            }
        }
        return min;
    }

    public <T> T injectInto(T injectedValue, ObjectDoubleToObjectFunction<? super T, ? extends T> function)
    {
        T result = injectedValue;
        if (this.containsZeroKey)
        {
            result = function.valueOf(result, EMPTY_KEY);
        }
        if (this.containsOneKey)
        {
            result = function.valueOf(result, REMOVED_KEY);
        }
        if (this.keysValues != null)
        {
            for (int i = 0; i < this.keysValues.length; i += 2)
            {
                if (isNonSentinel(this.keysValues[i]))
                {
                    result = function.valueOf(result, this.keysValues[i]);
                }
            }
        }
        return result;
    }

    private Object writeReplace()
    {
        return new ImmutableDoubleSetSerializationProxy(this);
    }

    // exposed for testing
    int probe(double element)
    {
        int index = this.spread(element) << 1;
        double keyAtIndex = this.keysValues[index];

        if (Double.compare(keyAtIndex, element) == 0 || Double.compare(keyAtIndex, EMPTY_KEY) == 0)
        {
            return index;
        }

        int removedIndex = Double.compare(keyAtIndex, REMOVED_KEY) == 0 ? index : -1;
        int nextIndex = index;
        int probe = 34;
        // loop until an empty slot is reached
        while (true)
        {
            // Probe algorithm: 17*n*(n+1)/2 where n = number of collisions
            nextIndex += probe;
            probe += 34;
            nextIndex &= this.keysValues.length - 1;
            if (Double.compare(this.keysValues[nextIndex], element) == 0)
            {
                return nextIndex;
            }
            if (Double.compare(this.keysValues[nextIndex], REMOVED_KEY) == 0)
            {
                if (removedIndex == -1)
                {
                    removedIndex = nextIndex;
                }
            }
            else if (Double.compare(this.keysValues[nextIndex], EMPTY_KEY) == 0)
            {
                return removedIndex == -1 ? nextIndex : removedIndex;
            }
        }
    }

    // exposed for testing
    int spread(double element)
    {
        long code = Double.doubleToLongBits(element);
        code ^= code >>> 23;
        code *= -6261870919139520145L;
        code ^= code >>> 39;
        code *= 2747051607443084853L;
        code ^= code >>> 37;
        return this.mask((int) code);
    }

    private int mask(int spread)
    {
        return spread & ((this.keysValues.length >> 1) - 1);
    }

    private class InternalDoubleIterator implements DoubleIterator
    {
        private int count;
        private int position;
        private boolean handledZero;
        private boolean handledOne;

        public boolean hasNext()
        {
            return this.count < ImmutableDoubleDoubleMapKeySet.this.size();
        }

        public double next()
        {
            if (!this.hasNext())
            {
                throw new NoSuchElementException("next() called, but the iterator is exhausted");
            }
            this.count++;

            if (!this.handledZero)
            {
                this.handledZero = true;
                if (ImmutableDoubleDoubleMapKeySet.this.containsZeroKey)
                {
                    return ImmutableDoubleDoubleMapKeySet.EMPTY_KEY;
                }
            }
            if (!this.handledOne)
            {
                this.handledOne = true;
                if (ImmutableDoubleDoubleMapKeySet.this.containsOneKey)
                {
                    return ImmutableDoubleDoubleMapKeySet.REMOVED_KEY;
                }
            }

            double[] keysValues = ImmutableDoubleDoubleMapKeySet.this.keysValues;
            while (!ImmutableDoubleDoubleMapKeySet.isNonSentinel(keysValues[this.position]))
            {
                this.position += 2;
            }
            double result = ImmutableDoubleDoubleMapKeySet.this.keysValues[this.position];
            this.position += 2;
            return result;
        }
    }
}
