/*
 * Copyright 2014 Goldman Sachs.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.gs.collections.impl.utility.internal.primitive;

import java.util.Collection;

import com.gs.collections.api.block.function.primitive.ByteToObjectFunction;
import com.gs.collections.api.block.function.primitive.ObjectByteToObjectFunction;
import com.gs.collections.api.block.predicate.primitive.BytePredicate;
import com.gs.collections.api.block.procedure.primitive.ByteProcedure;
import com.gs.collections.api.collection.primitive.MutableByteCollection;
import com.gs.collections.api.iterator.ByteIterator;

/**
 * The ByteIteratorIterate class provides implementations of the various iteration patterns for use with the {@link ByteIterator}.
 * This file was automatically generated from template file primitiveIteratorIterate.stg.
 *
 * @since 5.0
 */
public final class ByteIteratorIterate
{
    private ByteIteratorIterate()
    {
        throw new AssertionError("Suppress default constructor for noninstantiability");
    }

    public static void forEach(ByteIterator iterator, ByteProcedure procedure)
    {
        while (iterator.hasNext())
        {
            procedure.value(iterator.next());
        }
    }

    public static <R extends MutableByteCollection> R select(
            ByteIterator iterator,
            BytePredicate predicate,
            R targetCollection)
    {
        while (iterator.hasNext())
        {
            byte item = iterator.next();
            if (predicate.accept(item))
            {
                targetCollection.add(item);
            }
        }
        return targetCollection;
    }

    public static <R extends MutableByteCollection> R reject(
            ByteIterator iterator,
            BytePredicate predicate,
            R targetCollection)
    {
        while (iterator.hasNext())
        {
            byte item = iterator.next();
            if (!predicate.accept(item))
            {
                targetCollection.add(item);
            }
        }
        return targetCollection;
    }

    public static <V, R extends Collection<V>> R collect(
            ByteIterator iterator,
            ByteToObjectFunction<? extends V> function,
            R targetCollection)
    {
        while (iterator.hasNext())
        {
            byte item = iterator.next();
            targetCollection.add(function.valueOf(item));
        }
        return targetCollection;
    }

    public static byte detectIfNone(ByteIterator iterator, BytePredicate predicate, byte ifNone)
    {
        while (iterator.hasNext())
        {
            byte item = iterator.next();
            if (predicate.accept(item))
            {
                return item;
            }
        }
        return ifNone;
    }

    public static int count(ByteIterator iterator, BytePredicate predicate)
    {
        int count = 0;
        while (iterator.hasNext())
        {
            if (predicate.accept(iterator.next()))
            {
                count++;
            }
        }
        return count;
    }

    public static boolean anySatisfy(ByteIterator iterator, BytePredicate predicate)
    {
        while (iterator.hasNext())
        {
            if (predicate.accept(iterator.next()))
            {
                return true;
            }
        }
        return false;
    }

    public static boolean allSatisfy(ByteIterator iterator, BytePredicate predicate)
    {
        while (iterator.hasNext())
        {
            if (!predicate.accept(iterator.next()))
            {
                return false;
            }
        }
        return true;
    }

    public static boolean noneSatisfy(ByteIterator iterator, BytePredicate predicate)
    {
        while (iterator.hasNext())
        {
            if (predicate.accept(iterator.next()))
            {
                return false;
            }
        }
        return true;
    }

    public static <T> T injectInto(ByteIterator iterator, T injectedValue, ObjectByteToObjectFunction<? super T, ? extends T> function)
    {
        T result = injectedValue;
        while (iterator.hasNext())
        {
            result = function.valueOf(result, iterator.next());
        }
        return result;
    }


    public static long sum(ByteIterator iterator)
    {
        long sum = 0L;
        while (iterator.hasNext())
        {
            sum += iterator.next();
        }
        return sum;
    }

    public static byte max(ByteIterator iterator)
    {
        byte max = iterator.next();
        while (iterator.hasNext())
        {
            byte next = iterator.next();
            if (max < next)
            {
                max = next;
            }
        }
        return max;
    }

    public static byte min(ByteIterator iterator)
    {
        byte min = iterator.next();
        while (iterator.hasNext())
        {
            byte next = iterator.next();
            if (next < min)
            {
                min = next;
            }
        }
        return min;
    }
}
