/*
 * Copyright 2014 Goldman Sachs.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.gs.collections.impl.stack.immutable.primitive;

import java.io.Externalizable;
import java.io.IOException;
import java.io.ObjectInput;
import java.io.ObjectOutput;
import java.io.Serializable;

import com.gs.collections.api.LazyShortIterable;
import com.gs.collections.api.ShortIterable;
import com.gs.collections.api.bag.primitive.MutableShortBag;
import com.gs.collections.api.block.function.primitive.ShortToObjectFunction;
import com.gs.collections.api.block.function.primitive.ObjectShortToObjectFunction;
import com.gs.collections.api.block.predicate.primitive.ShortPredicate;
import com.gs.collections.api.block.procedure.primitive.ShortProcedure;
import com.gs.collections.api.iterator.ShortIterator;
import com.gs.collections.api.list.primitive.ShortList;
import com.gs.collections.api.list.primitive.MutableShortList;
import com.gs.collections.api.set.primitive.MutableShortSet;
import com.gs.collections.api.stack.ImmutableStack;
import com.gs.collections.api.stack.primitive.ShortStack;
import com.gs.collections.api.stack.primitive.ImmutableShortStack;
import com.gs.collections.impl.bag.mutable.primitive.ShortHashBag;
import com.gs.collections.impl.block.procedure.checked.primitive.CheckedShortProcedure;
import com.gs.collections.impl.factory.Stacks;
import com.gs.collections.impl.factory.primitive.ShortStacks;
import com.gs.collections.impl.iterator.UnmodifiableShortIterator;
import com.gs.collections.impl.lazy.primitive.LazyShortIterableAdapter;
import com.gs.collections.impl.list.mutable.primitive.ShortArrayList;
import com.gs.collections.impl.set.mutable.primitive.ShortHashSet;
import com.gs.collections.impl.stack.mutable.primitive.ShortArrayStack;
import net.jcip.annotations.Immutable;

/**
 * ImmutableShortArrayStack is the non-modifiable equivalent of {@link ShortArrayStack}.
 * This file was automatically generated from template file immutablePrimitiveArrayStack.stg.
 *
 * @since 4.0.
 */
@Immutable
final class ImmutableShortArrayStack
        implements ImmutableShortStack, Serializable
{
    private static final long serialVersionUID = 1L;
    private final ShortArrayList delegate;

    private ImmutableShortArrayStack(short[] newElements)
    {
        this.checkOptimizedSize(newElements.length);
        this.delegate = new ShortArrayList(newElements);
    }

    private ImmutableShortArrayStack(ShortArrayList newElements)
    {
        this.checkOptimizedSize(newElements.size());
        this.delegate = newElements;
    }

    private void checkOptimizedSize(int length)
    {
        if (length <= 1)
        {
            throw new IllegalArgumentException("Use ShortStacks.immutable.with() to instantiate an optimized collection");
        }
    }

    public static ImmutableShortArrayStack newStack(ShortIterable iterable)
    {
        return new ImmutableShortArrayStack(iterable.toArray());
    }

    public static ImmutableShortArrayStack newStackWith(short... elements)
    {
        short[] newArray = new short[elements.length];
        System.arraycopy(elements, 0, newArray, 0, elements.length);
        return new ImmutableShortArrayStack(newArray);
    }

    public static ImmutableShortArrayStack newStackFromTopToBottom(short... items)
    {
        return new ImmutableShortArrayStack(ShortArrayList.newListWith(items).reverseThis());
    }

    public static ImmutableShortArrayStack newStackFromTopToBottom(ShortIterable items)
    {
        return new ImmutableShortArrayStack(ShortArrayList.newList(items).reverseThis());
    }

    public ImmutableShortStack push(short item)
    {
        ShortArrayList newDelegate = ShortArrayList.newList(this.delegate);
        newDelegate.add(item);
        return new ImmutableShortArrayStack(newDelegate);
    }

    public ImmutableShortStack pop()
    {
        ShortArrayList newDelegate = ShortArrayList.newList(this.delegate);
        newDelegate.removeAtIndex(this.delegate.size() - 1);
        return ShortStacks.immutable.with(newDelegate.toArray());
    }

    public ImmutableShortStack pop(int count)
    {
        this.checkNegativeCount(count);
        if (count == 0)
        {
            return this;
        }
        this.checkSizeLessThanCount(count);
        ShortArrayList newDelegate = ShortArrayList.newList(this.delegate);
        while (count > 0)
        {
            newDelegate.removeAtIndex(newDelegate.size() - 1);
            count--;
        }
        return ShortStacks.immutable.with(newDelegate.toArray());
    }

    private void checkNegativeCount(int count)
    {
        if (count < 0)
        {
            throw new IllegalArgumentException("Count must be positive but was " + count);
        }
    }

    public short peek()
    {
        return this.delegate.getLast();
    }

    public ShortList peek(int count)
    {
        this.checkNegativeCount(count);
        this.checkSizeLessThanCount(count);
        if (count == 0)
        {
            return new ShortArrayList();
        }
        MutableShortList subList = new ShortArrayList(count);
        int index = this.delegate.size() - 1;
        for (int i = 0; i < count; i++)
        {
            subList.add(this.delegate.get(index - i));
        }
        return subList;
    }

    public short peekAt(int index)
    {
        this.rangeCheck(index);
        return this.delegate.get(this.delegate.size() - 1 - index);
    }

    private void rangeCheck(int index)
    {
        if (index < 0 || index > this.delegate.size() - 1)
        {
            throw new IllegalArgumentException("Index " + index + " out of range.Should be between 0 and " + (this.delegate.size() - 1));
        }
    }

    public ShortIterator shortIterator()
    {
        return new UnmodifiableShortIterator(this.delegate.asReversed().shortIterator());
    }

    public void forEach(ShortProcedure procedure)
    {
        this.delegate.asReversed().forEach(procedure);
    }

    public int count(ShortPredicate predicate)
    {
        return this.delegate.asReversed().count(predicate);
    }

    public boolean anySatisfy(ShortPredicate predicate)
    {
        return this.delegate.asReversed().anySatisfy(predicate);
    }

    public boolean allSatisfy(ShortPredicate predicate)
    {
        return this.delegate.asReversed().allSatisfy(predicate);
    }

    public boolean noneSatisfy(ShortPredicate predicate)
    {
        return this.delegate.asReversed().noneSatisfy(predicate);
    }

    public ImmutableShortStack select(ShortPredicate predicate)
    {
        return ShortStacks.immutable.withAllReversed(this.delegate.asReversed().select(predicate));
    }

    public ImmutableShortStack reject(ShortPredicate predicate)
    {
        return ShortStacks.immutable.withAllReversed(this.delegate.asReversed().reject(predicate));
    }

    public short detectIfNone(ShortPredicate predicate, short ifNone)
    {
        return this.delegate.asReversed().detectIfNone(predicate, ifNone);
    }

    public <V> ImmutableStack<V> collect(ShortToObjectFunction<? extends V> function)
    {
        return Stacks.immutable.withAllReversed(this.delegate.asReversed().collect(function));
    }

    public long sum()
    {
        return this.delegate.sum();
    }

    public short max()
    {
        return this.delegate.max();
    }

    public short maxIfEmpty(short defaultValue)
    {
        return this.max();
    }

    public short min()
    {
        return this.delegate.min();
    }

    public short minIfEmpty(short defaultValue)
    {
        return this.min();
    }

    public double average()
    {
        return this.delegate.average();
    }

    public double median()
    {
        return this.delegate.median();
    }

    public short[] toSortedArray()
    {
        return this.delegate.toSortedArray();
    }

    public MutableShortList toSortedList()
    {
        return ShortArrayList.newList(this).sortThis();
    }

    public short[] toArray()
    {
        return this.delegate.asReversed().toArray();
    }

    public boolean contains(short value)
    {
        return this.delegate.asReversed().contains(value);
    }

    public boolean containsAll(short... source)
    {
        return this.delegate.asReversed().containsAll(source);
    }

    public boolean containsAll(ShortIterable source)
    {
        return this.delegate.asReversed().containsAll(source);
    }

    public MutableShortList toList()
    {
        return ShortArrayList.newList(this);
    }

    public MutableShortSet toSet()
    {
        return ShortHashSet.newSet(this);
    }

    public MutableShortBag toBag()
    {
        return ShortHashBag.newBag(this);
    }

    public <V> V injectInto(V injectedValue, ObjectShortToObjectFunction<? super V, ? extends V> function)
    {
        return this.delegate.toReversed().injectInto(injectedValue, function);
    }

    public LazyShortIterable asLazy()
    {
        return new LazyShortIterableAdapter(this);
    }

    public ImmutableShortStack toImmutable()
    {
        return this;
    }

    public int size()
    {
        return this.delegate.size();
    }

    public boolean isEmpty()
    {
        return false;
    }

    public boolean notEmpty()
    {
        return true;
    }

    @Override
    public boolean equals(Object otherStack)
    {
        if (otherStack == this)
        {
            return true;
        }
        if (!(otherStack instanceof ShortStack))
        {
            return false;
        }
        ShortStack stack = (ShortStack) otherStack;
        if (this.size() != stack.size())
        {
            return false;
        }
        for (int i = 0; i < this.size(); i++)
        {
            if (this.peekAt(i) != stack.peekAt(i))
            {
                return false;
            }
        }
        return true;
    }

    @Override
    public int hashCode()
    {
        int hashCode = 1;
        ShortIterable iterable = this.delegate.asReversed();
        ShortIterator iterator = iterable.shortIterator();
        while (iterator.hasNext())
        {
            short item = iterator.next();
            hashCode = 31 * hashCode + (int) item;
        }
        return hashCode;
    }

    @Override
    public String toString()
    {
        return this.delegate.asReversed().toString();
    }

    public String makeString()
    {
        return this.delegate.asReversed().makeString();
    }

    public String makeString(String separator)
    {
        return this.delegate.asReversed().makeString(separator);
    }

    public String makeString(String start, String separator, String end)
    {
        return this.delegate.asReversed().makeString(start, separator, end);
    }

    public void appendString(Appendable appendable)
    {
        this.delegate.asReversed().appendString(appendable);
    }

    public void appendString(Appendable appendable, String separator)
    {
        this.delegate.asReversed().appendString(appendable, separator);
    }

    public void appendString(Appendable appendable, String start, String separator, String end)
    {
        this.delegate.asReversed().appendString(appendable, start, separator, end);
    }

    private void checkSizeLessThanCount(int count)
    {
        if (this.delegate.size() < count)
        {
            throw new IllegalArgumentException("Count must be less than size: Count = " + count + " Size = " + this.delegate.size());
        }
    }

    private Object writeReplace()
    {
        return new ImmutableShortStackSerializationProxy(this);
    }

    private static class ImmutableShortStackSerializationProxy implements Externalizable
    {
        private static final long serialVersionUID = 1L;

        private ShortStack stack;

        @SuppressWarnings("UnusedDeclaration")
        public ImmutableShortStackSerializationProxy()
        {
            // Empty constructor for Externalizable class
        }

        protected ImmutableShortStackSerializationProxy(ShortStack stack)
        {
            this.stack = stack;
        }

        public void writeExternal(final ObjectOutput out) throws IOException
        {
            out.writeInt(this.stack.size());
            try
            {
                this.stack.forEach(new CheckedShortProcedure()
                {
                    @Override
                    public void safeValue(short item) throws IOException
                    {
                        out.writeShort(item);
                    }
                });
            }
            catch (RuntimeException e)
            {
                if (e.getCause() instanceof IOException)
                {
                    throw (IOException) e.getCause();
                }
                throw e;
            }
        }

        public void readExternal(ObjectInput in) throws IOException, ClassNotFoundException
        {
            int size = in.readInt();
            ShortArrayList deserializedDelegate = new ShortArrayList(size);

            for (int i = 0; i < size; i++)
            {
                deserializedDelegate.add(in.readShort());
            }

            this.stack = ImmutableShortArrayStack.newStackFromTopToBottom(deserializedDelegate);
        }

        protected Object readResolve()
        {
            return this.stack;
        }
    }
}
