/*
 * Copyright 2014 Goldman Sachs.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.gs.collections.impl.utility.primitive;

import com.gs.collections.api.CharIterable;
import com.gs.collections.api.LazyCharIterable;
import com.gs.collections.api.LazyIterable;
import com.gs.collections.api.block.function.primitive.CharToObjectFunction;
import com.gs.collections.api.block.predicate.primitive.CharPredicate;
import com.gs.collections.impl.factory.primitive.CharLists;
import com.gs.collections.impl.lazy.primitive.CollectCharToObjectIterable;
import com.gs.collections.impl.lazy.primitive.LazyCharIterableAdapter;
import com.gs.collections.impl.lazy.primitive.SelectCharIterable;

/**
 * LazyCharIterate is a factory class which creates "deferred" char iterables around the specified char iterables. A "deferred"
 * char iterable performs some operation, such as filtering or transforming, when the result char iterable is iterated over.  This
 * makes the operation very memory efficient, because you don't have to create intermediate collections during the
 * operation.
 * This file was automatically generated from template file lazyPrimitiveIterate.stg.
 *
 * @since 5.0
 */
public final class LazyCharIterate
{
    private static final LazyCharIterable EMPTY_ITERABLE = CharLists.immutable.of().asLazy();

    private LazyCharIterate()
    {
        throw new AssertionError("Suppress default constructor for noninstantiability");
    }

    /**
     * Creates a deferred char iterable for the specified char iterable.
     */
    public static LazyCharIterable adapt(CharIterable iterable)
    {
        return new LazyCharIterableAdapter(iterable);
    }

    /**
     * Creates a deferred filtering char iterable for the specified char iterable.
     */
    public static LazyCharIterable select(CharIterable iterable, CharPredicate predicate)
    {
        return new SelectCharIterable(iterable, predicate);
    }

    /**
     * Creates a deferred transforming char iterable for the specified char iterable.
     */
    public static <V> LazyIterable<V> collect(
            CharIterable iterable,
            CharToObjectFunction<? extends V> function)
    {
        return new CollectCharToObjectIterable<V>(iterable, function);
    }

    /**
     * Creates a deferred filtering and transforming char iterable for the specified char iterable.
     */
    public static <V> LazyIterable<V> collectIf(
            CharIterable iterable,
            CharPredicate predicate,
            CharToObjectFunction<? extends V> function)
    {
        return LazyCharIterate.select(iterable, predicate).collect(function);
    }

    public static LazyCharIterable empty()
    {
        return EMPTY_ITERABLE;
    }
}
