/*
 * Copyright 2014 Goldman Sachs.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.gs.collections.impl.bag.immutable.primitive;

import java.io.Externalizable;
import java.io.IOException;
import java.io.ObjectInput;
import java.io.ObjectOutput;
import java.io.Serializable;

import com.gs.collections.api.IntIterable;
import com.gs.collections.api.LazyIntIterable;
import com.gs.collections.api.bag.ImmutableBag;
import com.gs.collections.api.bag.primitive.IntBag;
import com.gs.collections.api.bag.primitive.ImmutableIntBag;
import com.gs.collections.api.bag.primitive.MutableIntBag;
import com.gs.collections.api.block.function.primitive.IntToObjectFunction;
import com.gs.collections.api.block.function.primitive.ObjectIntToObjectFunction;
import com.gs.collections.api.block.predicate.primitive.IntPredicate;
import com.gs.collections.api.block.procedure.primitive.IntIntProcedure;
import com.gs.collections.api.block.procedure.primitive.IntProcedure;
import com.gs.collections.api.iterator.IntIterator;
import com.gs.collections.api.list.primitive.MutableIntList;
import com.gs.collections.api.set.primitive.MutableIntSet;
import com.gs.collections.impl.bag.mutable.primitive.IntHashBag;
import com.gs.collections.impl.block.procedure.checked.primitive.CheckedIntIntProcedure;
import com.gs.collections.impl.iterator.UnmodifiableIntIterator;

/**
 * ImmutableIntHashBag is the non-modifiable equivalent of {@link IntHashBag}.
 * This file was automatically generated from template file immutablePrimitiveHashBag.stg.
 *
 * @since 4.0.
 */
final class ImmutableIntHashBag implements ImmutableIntBag, Serializable
{
    private static final long serialVersionUID = 1L;

    private final MutableIntBag delegate;

    private ImmutableIntHashBag(int[] newElements)
    {
        this.checkOptimizedSize(newElements.length);
        this.delegate = IntHashBag.newBagWith(newElements);
    }

    private void checkOptimizedSize(int length)
    {
        if (length <= 1)
        {
            throw new IllegalArgumentException("Use IntBags.immutable.with() to instantiate an optimized collection");
        }
    }

    public static ImmutableIntHashBag newBagWith(int... elements)
    {
        return new ImmutableIntHashBag(elements);
    }

    public ImmutableIntBag newWith(int element)
    {
        return IntHashBag.newBag(this.delegate).with(element).toImmutable();
    }

    public ImmutableIntBag newWithout(int element)
    {
        IntHashBag hashBag = IntHashBag.newBag(this.delegate);
        hashBag.remove(element);
        return hashBag.toImmutable();
    }

    public ImmutableIntBag newWithAll(IntIterable elements)
    {
        IntHashBag bag = IntHashBag.newBag(this.delegate);
        bag.addAll(elements);
        return bag.toImmutable();
    }

    public ImmutableIntBag newWithoutAll(IntIterable elements)
    {
        IntHashBag bag = IntHashBag.newBag(this.delegate);
        bag.removeAll(elements);
        return bag.toImmutable();
    }

    public int size()
    {
        return this.delegate.size();
    }

    public boolean isEmpty()
    {
        return this.delegate.isEmpty();
    }

    public boolean notEmpty()
    {
        return this.delegate.notEmpty();
    }

    public boolean contains(int value)
    {
        return this.delegate.contains(value);
    }

    public boolean containsAll(IntIterable source)
    {
        return this.delegate.containsAll(source);
    }

    public boolean containsAll(int... elements)
    {
        return this.delegate.containsAll(elements);
    }

    public void forEach(IntProcedure procedure)
    {
        this.each(procedure);
    }

    /**
     * @since 7.0.
     */
    public void each(IntProcedure procedure)
    {
        this.delegate.forEach(procedure);
    }

    public ImmutableIntBag select(IntPredicate predicate)
    {
        return this.delegate.select(predicate).toImmutable();
    }

    public ImmutableIntBag reject(IntPredicate predicate)
    {
        return this.delegate.reject(predicate).toImmutable();
    }

    public <V> ImmutableBag<V> collect(IntToObjectFunction<? extends V> function)
    {
        return this.delegate.collect(function).toImmutable();
    }

    public MutableIntList toList()
    {
        return this.delegate.toList();
    }

    public int sizeDistinct()
    {
        return this.delegate.sizeDistinct();
    }

    public int occurrencesOf(int item)
    {
        return this.delegate.occurrencesOf(item);
    }

    public void forEachWithOccurrences(IntIntProcedure intIntProcedure)
    {
        this.delegate.forEachWithOccurrences(intIntProcedure);
    }

    public int detectIfNone(IntPredicate predicate, int ifNone)
    {
        return this.delegate.detectIfNone(predicate, ifNone);
    }

    public int count(IntPredicate predicate)
    {
        return this.delegate.count(predicate);
    }

    public boolean anySatisfy(IntPredicate predicate)
    {
        return this.delegate.anySatisfy(predicate);
    }

    public long sum()
    {
        return this.delegate.sum();
    }

    public int min()
    {
        return this.delegate.min();
    }

    public int max()
    {
        return this.delegate.max();
    }

    public int maxIfEmpty(int defaultValue)
    {
        return this.delegate.maxIfEmpty(defaultValue);
    }

    public int minIfEmpty(int defaultValue)
    {
        return this.delegate.minIfEmpty(defaultValue);
    }

    public double average()
    {
        return this.delegate.average();
    }

    public double median()
    {
        return this.delegate.median();
    }

    public int[] toSortedArray()
    {
        return this.delegate.toSortedArray();
    }

    public MutableIntList toSortedList()
    {
        return this.delegate.toSortedList();
    }

    public boolean noneSatisfy(IntPredicate predicate)
    {
        return this.delegate.noneSatisfy(predicate);
    }

    public boolean allSatisfy(IntPredicate predicate)
    {
        return this.delegate.allSatisfy(predicate);
    }

    public <T> T injectInto(T injectedValue, ObjectIntToObjectFunction<? super T, ? extends T> function)
    {
        return this.delegate.injectInto(injectedValue, function);
    }

    @Override
    public boolean equals(Object obj)
    {
        return this.delegate.equals(obj);
    }

    @Override
    public int hashCode()
    {
        return this.delegate.hashCode();
    }

    public MutableIntSet toSet()
    {
        return this.delegate.toSet();
    }

    public MutableIntBag toBag()
    {
        return this.delegate.toBag();
    }

    public ImmutableIntBag toImmutable()
    {
        return this;
    }

    public LazyIntIterable asLazy()
    {
        return this.delegate.asLazy();
    }

    public int[] toArray()
    {
        return this.delegate.toArray();
    }

    @Override
    public String toString()
    {
        return this.delegate.toString();
    }

    public String makeString()
    {
        return this.delegate.makeString();
    }

    public String makeString(String separator)
    {
        return this.delegate.makeString(separator);
    }

    public String makeString(String start, String separator, String end)
    {
        return this.delegate.makeString(start, separator, end);
    }

    public void appendString(Appendable appendable)
    {
        this.delegate.appendString(appendable);
    }

    public void appendString(Appendable appendable, String separator)
    {
        this.delegate.appendString(appendable, separator);
    }

    public void appendString(Appendable appendable, String start, String separator, String end)
    {
        this.delegate.appendString(appendable, start, separator, end);
    }

    public IntIterator intIterator()
    {
        return new UnmodifiableIntIterator(this.delegate.intIterator());
    }

    private Object writeReplace()
    {
        return new ImmutableIntBagSerializationProxy(this);
    }

    protected static class ImmutableIntBagSerializationProxy implements Externalizable
    {
        private static final long serialVersionUID = 1L;

        private IntBag bag;

        @SuppressWarnings("UnusedDeclaration")
        public ImmutableIntBagSerializationProxy()
        {
            // Empty constructor for Externalizable class
        }

        protected ImmutableIntBagSerializationProxy(IntBag bag)
        {
            this.bag = bag;
        }

        public void writeExternal(final ObjectOutput out) throws IOException
        {
            out.writeInt(this.bag.sizeDistinct());
            try
            {
                this.bag.forEachWithOccurrences(new CheckedIntIntProcedure()
                {
                    @Override
                    public void safeValue(int item, int count) throws IOException
                    {
                        out.writeInt(item);
                        out.writeInt(count);
                    }
                });
            }
            catch (RuntimeException e)
            {
                if (e.getCause() instanceof IOException)
                {
                    throw (IOException) e.getCause();
                }
                throw e;
            }
        }

        public void readExternal(ObjectInput in) throws IOException, ClassNotFoundException
        {
            int size = in.readInt();
            MutableIntBag deserializedBag = new IntHashBag();

            for (int i = 0; i < size; i++)
            {
                deserializedBag.addOccurrences(in.readInt(), in.readInt());
            }

            this.bag = deserializedBag;
        }

        protected Object readResolve()
        {
            return this.bag.toImmutable();
        }
    }
}
