/*
 * Copyright 2014 Goldman Sachs.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.gs.collections.impl.lazy.primitive;

import com.gs.collections.api.IntIterable;
import com.gs.collections.api.LazyBooleanIterable;
import com.gs.collections.api.LazyByteIterable;
import com.gs.collections.api.LazyCharIterable;
import com.gs.collections.api.LazyDoubleIterable;
import com.gs.collections.api.LazyFloatIterable;
import com.gs.collections.api.LazyIntIterable;
import com.gs.collections.api.LazyIterable;
import com.gs.collections.api.LazyLongIterable;
import com.gs.collections.api.LazyShortIterable;
import com.gs.collections.api.LazyIterable;
import com.gs.collections.api.bag.primitive.MutableIntBag;
import com.gs.collections.api.block.function.primitive.IntToObjectFunction;
import com.gs.collections.api.block.function.primitive.IntToBooleanFunction;
import com.gs.collections.api.block.function.primitive.IntToByteFunction;
import com.gs.collections.api.block.function.primitive.IntToCharFunction;
import com.gs.collections.api.block.function.primitive.IntToDoubleFunction;
import com.gs.collections.api.block.function.primitive.IntToFloatFunction;
import com.gs.collections.api.block.function.primitive.IntToIntFunction;
import com.gs.collections.api.block.function.primitive.IntToLongFunction;
import com.gs.collections.api.block.function.primitive.IntToObjectFunction;
import com.gs.collections.api.block.function.primitive.IntToShortFunction;
import com.gs.collections.api.block.function.primitive.ObjectIntToObjectFunction;
import com.gs.collections.api.block.predicate.primitive.IntPredicate;
import com.gs.collections.api.block.procedure.primitive.IntProcedure;
import com.gs.collections.api.list.primitive.MutableIntList;
import com.gs.collections.api.set.primitive.MutableIntSet;
import com.gs.collections.impl.bag.mutable.primitive.IntHashBag;
import com.gs.collections.impl.block.factory.primitive.IntPredicates;
import com.gs.collections.impl.factory.primitive.IntSets;
import com.gs.collections.impl.list.mutable.primitive.IntArrayList;
import com.gs.collections.impl.set.mutable.primitive.IntHashSet;
import com.gs.collections.impl.utility.internal.primitive.IntIterableIterate;
import com.gs.collections.impl.utility.primitive.LazyIntIterate;

import java.util.NoSuchElementException;

/**
 * This file was automatically generated from template file abstractLazyPrimitiveIterable.stg.
 *
 * @since 5.0
 */
public abstract class AbstractLazyIntIterable implements LazyIntIterable
{
    public void forEach(IntProcedure procedure)
    {
        this.each(procedure);
    }

    public int size()
    {
        return this.count(IntPredicates.alwaysTrue());
    }

    @Override
    public String toString()
    {
        return this.makeString("[", ", ", "]");
    }

    public boolean isEmpty()
    {
        return IntIterableIterate.isEmpty(this);
    }

    public boolean notEmpty()
    {
        return IntIterableIterate.notEmpty(this);
    }

    public String makeString()
    {
        return this.makeString(", ");
    }

    public String makeString(String separator)
    {
        return this.makeString("", separator, "");
    }

    public String makeString(String start, String separator, String end)
    {
        Appendable stringBuilder = new StringBuilder();
        this.appendString(stringBuilder, start, separator, end);
        return stringBuilder.toString();
    }

    public void appendString(Appendable appendable)
    {
        this.appendString(appendable, ", ");
    }

    public void appendString(Appendable appendable, String separator)
    {
        this.appendString(appendable, "", separator, "");
    }

    public void appendString(Appendable appendable, String start, String separator, String end)
    {
        IntIterableIterate.appendString(this, appendable, start, separator, end);
    }

    public boolean contains(int value)
    {
        return this.anySatisfy(IntPredicates.equal(value));
    }

    public boolean containsAll(int... source)
    {
        return this.containsAll(IntSets.immutable.of(source));
    }

    public boolean containsAll(IntIterable source)
    {
        return source.allSatisfy(new IntPredicate()
        {
            public boolean accept(int value)
            {
                return AbstractLazyIntIterable.this.contains(value);
            }
        });
    }

    public LazyIntIterable select(IntPredicate predicate)
    {
        return LazyIntIterate.select(this, predicate);
    }

    public LazyIntIterable reject(IntPredicate predicate)
    {
        return LazyIntIterate.select(this, IntPredicates.not(predicate));
    }

    public <V> LazyIterable<V> collect(IntToObjectFunction<? extends V> function)
    {
        return LazyIntIterate.collect(this, function);
    }

    /**
     * @since 7.0
     */
    public LazyBooleanIterable collectBoolean(IntToBooleanFunction function)
    {
        return new CollectIntToBooleanIterable(this, function);
    }

    /**
     * @since 7.0
     */
    public LazyByteIterable collectByte(IntToByteFunction function)
    {
        return new CollectIntToByteIterable(this, function);
    }

    /**
     * @since 7.0
     */
    public LazyCharIterable collectChar(IntToCharFunction function)
    {
        return new CollectIntToCharIterable(this, function);
    }

    /**
     * @since 7.0
     */
    public LazyShortIterable collectShort(IntToShortFunction function)
    {
        return new CollectIntToShortIterable(this, function);
    }

    /**
     * @since 7.0
     */
    public LazyIntIterable collectInt(IntToIntFunction function)
    {
        return new CollectIntToIntIterable(this, function);
    }

    /**
     * @since 7.0
     */
    public LazyFloatIterable collectFloat(IntToFloatFunction function)
    {
        return new CollectIntToFloatIterable(this, function);
    }

    /**
     * @since 7.0
     */
    public LazyLongIterable collectLong(IntToLongFunction function)
    {
        return new CollectIntToLongIterable(this, function);
    }

    /**
     * @since 7.0
     */
    public LazyDoubleIterable collectDouble(IntToDoubleFunction function)
    {
        return new CollectIntToDoubleIterable(this, function);
    }

    public int detectIfNone(IntPredicate predicate, int ifNone)
    {
        return IntIterableIterate.detectIfNone(this, predicate, ifNone);
    }

    public int count(IntPredicate predicate)
    {
        return IntIterableIterate.count(this, predicate);
    }

    public boolean anySatisfy(IntPredicate predicate)
    {
        return IntIterableIterate.anySatisfy(this, predicate);
    }

    public boolean allSatisfy(IntPredicate predicate)
    {
        return IntIterableIterate.allSatisfy(this, predicate);
    }

    public boolean noneSatisfy(IntPredicate predicate)
    {
        return IntIterableIterate.noneSatisfy(this, predicate);
    }

    public <T> T injectInto(T injectedValue, ObjectIntToObjectFunction<? super T, ? extends T> function)
    {
        return IntIterableIterate.injectInto(this, injectedValue, function);
    }

    public int[] toArray()
    {
        return this.toList().toArray();
    }

    public MutableIntList toList()
    {
        final MutableIntList list = new IntArrayList();
        this.forEach(new IntProcedure()
        {
            public void value(int each)
            {
                list.add(each);
            }
        });
        return list;
    }

    public MutableIntSet toSet()
    {
        final MutableIntSet set = new IntHashSet();
        this.forEach(new IntProcedure()
        {
            public void value(int each)
            {
                set.add(each);
            }
        });
        return set;
    }

    public MutableIntBag toBag()
    {
        final MutableIntBag bag = new IntHashBag();
        this.forEach(new IntProcedure()
        {
            public void value(int each)
            {
                bag.add(each);
            }
        });
        return bag;
    }

    public long sum()
    {
        IntSumProcedure procedure = new IntSumProcedure();
        this.forEach(procedure);
        return procedure.getValue();
    }

    public int max()
    {
        if (this.isEmpty())
        {
            throw new NoSuchElementException();
        }
        IntMaxProcedure procedure = new IntMaxProcedure();
        this.forEach(procedure);
        return procedure.getValue();
    }

    public int maxIfEmpty(int ifEmpty)
    {
        if (this.isEmpty())
        {
            return ifEmpty;
        }
        return this.max();
    }

    public int min()
    {
        if (this.isEmpty())
        {
            throw new NoSuchElementException();
        }
        IntMinProcedure procedure = new IntMinProcedure();
        this.forEach(procedure);
        return procedure.getValue();
    }

    public int minIfEmpty(int ifEmpty)
    {
        if (this.isEmpty())
        {
            return ifEmpty;
        }
        return this.min();
    }

    public double average()
    {
        if (this.isEmpty())
        {
            throw new ArithmeticException();
        }
        return (double) this.sum() / (double) this.size();
    }

    public double median()
    {
        if (this.isEmpty())
        {
            throw new ArithmeticException();
        }
        int[] sortedArray = this.toSortedArray();
        int middleIndex = sortedArray.length >> 1;
        if (sortedArray.length > 1 && (sortedArray.length & 1) == 0)
        {
            int first = sortedArray[middleIndex];
            int second = sortedArray[middleIndex - 1];
            return ((double) first + (double) second) / 2.0;
        }
        return (double) sortedArray[middleIndex];
    }

    public int[] toSortedArray()
    {
        return this.toSortedList().toArray();
    }

    public MutableIntList toSortedList()
    {
        return IntArrayList.newList(this).sortThis();
    }

    private static final class IntMaxProcedure implements IntProcedure
    {
        private boolean visitedOnce;
        private int max;

        public void value(int each)
        {
            if (this.visitedOnce)
            {
                if (this.max < each)
                {
                    this.max = each;
                }
            }
            else
            {
                this.max = each;
                this.visitedOnce = true;
            }
        }

        public int getValue()
        {
            return this.max;
        }
    }

    private static final class IntMinProcedure implements IntProcedure
    {
        private boolean visitedOnce;
        private int min;

        public void value(int each)
        {
            if (this.visitedOnce)
            {
                if (each < this.min)
                {
                    this.min = each;
                }
            }
            else
            {
                this.min = each;
                this.visitedOnce = true;
            }
        }

        public int getValue()
        {
            return this.min;
        }
    }

    private static final class IntSumProcedure implements IntProcedure
    {
        private int sum = 0;

        public void value(int each)
        {
            this.sum += each;
        }

        public int getValue()
        {
            return this.sum;
        }
    }


    public LazyIntIterable asLazy()
    {
        return this;
    }
}
