/*
 * Copyright 2014 Goldman Sachs.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.gs.collections.impl.map.mutable.primitive;

import java.io.Externalizable;
import java.io.IOException;
import java.io.ObjectInput;
import java.io.ObjectOutput;
import java.lang.reflect.Array;
import java.util.Arrays;
import java.util.Collection;
import java.util.Iterator;
import java.util.NoSuchElementException;
import java.util.Set;

import com.gs.collections.api.ByteIterable;
import com.gs.collections.api.LazyByteIterable;
import com.gs.collections.api.LazyIterable;
import com.gs.collections.api.RichIterable;
import com.gs.collections.api.bag.primitive.MutableByteBag;
import com.gs.collections.api.block.function.primitive.ByteFunction;
import com.gs.collections.api.block.function.primitive.ByteFunction0;
import com.gs.collections.api.block.function.primitive.ByteToByteFunction;
import com.gs.collections.api.block.function.primitive.ByteToObjectFunction;
import com.gs.collections.api.block.function.primitive.ObjectByteToObjectFunction;
import com.gs.collections.api.block.predicate.primitive.BytePredicate;
import com.gs.collections.api.block.predicate.primitive.ObjectBytePredicate;
import com.gs.collections.api.block.procedure.Procedure;
import com.gs.collections.api.block.procedure.Procedure2;
import com.gs.collections.api.block.procedure.primitive.ByteProcedure;
import com.gs.collections.api.block.procedure.primitive.ObjectByteProcedure;
import com.gs.collections.api.collection.MutableCollection;
import com.gs.collections.api.collection.primitive.ImmutableByteCollection;
import com.gs.collections.api.collection.primitive.MutableByteCollection;
import com.gs.collections.api.iterator.ByteIterator;
import com.gs.collections.api.iterator.MutableByteIterator;
import com.gs.collections.api.list.MutableList;
import com.gs.collections.api.list.primitive.MutableByteList;
import com.gs.collections.api.map.primitive.ImmutableObjectByteMap;
import com.gs.collections.api.map.primitive.MutableObjectByteMap;
import com.gs.collections.api.map.primitive.ObjectByteMap;
import com.gs.collections.api.set.primitive.ByteSet;
import com.gs.collections.api.set.primitive.MutableByteSet;
import com.gs.collections.api.tuple.primitive.ObjectBytePair;
import com.gs.collections.impl.bag.mutable.primitive.ByteHashBag;
import com.gs.collections.impl.collection.mutable.primitive.SynchronizedByteCollection;
import com.gs.collections.impl.collection.mutable.primitive.UnmodifiableByteCollection;
import com.gs.collections.impl.lazy.AbstractLazyIterable;
import com.gs.collections.impl.factory.primitive.ByteLists;
import com.gs.collections.impl.factory.primitive.ObjectByteMaps;
import com.gs.collections.impl.lazy.primitive.LazyByteIterableAdapter;
import com.gs.collections.impl.list.mutable.FastList;
import com.gs.collections.impl.list.mutable.primitive.ByteArrayList;
import com.gs.collections.impl.set.mutable.primitive.ByteHashSet;
import com.gs.collections.api.block.procedure.primitive.ObjectIntProcedure;
import com.gs.collections.impl.tuple.primitive.PrimitiveTuples;
/**
 * This file was automatically generated from template file objectPrimitiveHashMap.stg.
 *
 * @since 3.0.
 */
public class ObjectByteHashMap<K> implements MutableObjectByteMap<K>, Externalizable
{
    public static final byte EMPTY_VALUE = (byte) 0;

    private static final long serialVersionUID = 1L;
    private static final int OCCUPIED_DATA_RATIO = 2;
    private static final int OCCUPIED_SENTINEL_RATIO = 4;
    private static final int DEFAULT_INITIAL_CAPACITY = 8;

    private static final Object NULL_KEY = new Object()
    {
        @Override
        public boolean equals(Object obj)
        {
            throw new RuntimeException("Possible corruption through unsynchronized concurrent modification.");
        }

        @Override
        public int hashCode()
        {
            throw new RuntimeException("Possible corruption through unsynchronized concurrent modification.");
        }

        @Override
        public String toString()
        {
            return "ObjectByteHashMap.NULL_KEY";
        }
    };

    private static final Object REMOVED_KEY = new Object()
    {
        @Override
        public boolean equals(Object obj)
        {
            throw new RuntimeException("Possible corruption through unsynchronized concurrent modification.");
        }

        @Override
        public int hashCode()
        {
            throw new RuntimeException("Possible corruption through unsynchronized concurrent modification.");
        }

        @Override
        public String toString()
        {
            return "ObjectByteHashMap.REMOVED_KEY";
        }
    };

    private Object[] keys;
    private byte[] values;

    private int occupiedWithData;
    private int occupiedWithSentinels;

    public ObjectByteHashMap()
    {
        this.allocateTable(DEFAULT_INITIAL_CAPACITY << 1);
    }

    public ObjectByteHashMap(int initialCapacity)
    {
        if (initialCapacity < 0)
        {
            throw new IllegalArgumentException("initial capacity cannot be less than 0");
        }
        int capacity = this.smallestPowerOfTwoGreaterThan(this.fastCeil(initialCapacity * OCCUPIED_DATA_RATIO));
        this.allocateTable(capacity);
    }

    public ObjectByteHashMap(ObjectByteMap<? extends K> map)
    {
        this(Math.max(map.size(), DEFAULT_INITIAL_CAPACITY));
        this.putAll(map);
    }

    public static <K> ObjectByteHashMap<K> newMap()
    {
        return new ObjectByteHashMap<K>();
    }

    public static <K> ObjectByteHashMap<K> newWithKeysValues(K key1, byte value1)
    {
        return new ObjectByteHashMap<K>(1).withKeyValue(key1, value1);
    }

    public static <K> ObjectByteHashMap<K> newWithKeysValues(K key1, byte value1, K key2, byte value2)
    {
        return new ObjectByteHashMap<K>(2).withKeysValues(key1, value1, key2, value2);
    }

    public static <K> ObjectByteHashMap<K> newWithKeysValues(K key1, byte value1, K key2, byte value2, K key3, byte value3)
    {
        return new ObjectByteHashMap<K>(3).withKeysValues(key1, value1, key2, value2, key3, value3);
    }

    public static <K> ObjectByteHashMap<K> newWithKeysValues(K key1, byte value1, K key2, byte value2, K key3, byte value3, K key4, byte value4)
    {
        return new ObjectByteHashMap<K>(4).withKeysValues(key1, value1, key2, value2, key3, value3, key4, value4);
    }

    private int smallestPowerOfTwoGreaterThan(int n)
    {
        return n > 1 ? Integer.highestOneBit(n - 1) << 1 : 1;
    }

    private int fastCeil(float v)
    {
        int possibleResult = (int) v;
        if (v - possibleResult > 0.0F)
        {
            possibleResult++;
        }
        return possibleResult;
    }

    @Override
    public boolean equals(Object obj)
    {
        if (this == obj)
        {
            return true;
        }

        if (!(obj instanceof ObjectByteMap))
        {
            return false;
        }

        ObjectByteMap<K> other = (ObjectByteMap<K>) obj;

        if (this.size() != other.size())
        {
            return false;
        }

        for (int i = 0; i < this.keys.length; i++)
        {
            if (isNonSentinel(this.keys[i]) && (!other.containsKey(this.toNonSentinel(this.keys[i])) || this.values[i] != other.getOrThrow(this.toNonSentinel(this.keys[i]))))
            {
                return false;
            }
        }
        return true;
    }

    @Override
    public int hashCode()
    {
        int result = 0;

        for (int i = 0; i < this.keys.length; i++)
        {
            if (isNonSentinel(this.keys[i]))
            {
                result += (this.toNonSentinel(this.keys[i]) == null ? 0 : this.keys[i].hashCode()) ^ (int) this.values[i];
            }
        }
        return result;
    }

    @Override
    public String toString()
    {
        StringBuilder appendable = new StringBuilder();

        appendable.append("{");

        boolean first = true;

        for (int i = 0; i < this.keys.length; i++)
        {
            Object key = this.keys[i];
            if (isNonSentinel(key))
            {
                if (!first)
                {
                    appendable.append(", ");
                }
                appendable.append(this.toNonSentinel(key)).append("=").append(this.values[i]);
                first = false;
            }
        }
        appendable.append("}");

        return appendable.toString();
    }

    public int size()
    {
        return this.occupiedWithData;
    }

    public boolean isEmpty()
    {
        return this.size() == 0;
    }

    public boolean notEmpty()
    {
        return this.size() != 0;
    }

    public String makeString()
    {
        return this.makeString(", ");
    }

    public String makeString(String separator)
    {
        return this.makeString("", separator, "");
    }

    public String makeString(String start, String separator, String end)
    {
        Appendable stringBuilder = new StringBuilder();
        this.appendString(stringBuilder, start, separator, end);
        return stringBuilder.toString();
    }

    public void appendString(Appendable appendable)
    {
        this.appendString(appendable, ", ");
    }

    public void appendString(Appendable appendable, String separator)
    {
        this.appendString(appendable, "", separator, "");
    }

    public void appendString(Appendable appendable, String start, String separator, String end)
    {
        try
        {
            appendable.append(start);

            boolean first = true;

            for (int i = 0; i < this.keys.length; i++)
            {
                Object key = this.keys[i];
                if (isNonSentinel(key))
                {
                    if (!first)
                    {
                        appendable.append(separator);
                    }
                    appendable.append(String.valueOf(String.valueOf(this.values[i])));
                    first = false;
                }
            }
            appendable.append(end);
        }
        catch (IOException e)
        {
            throw new RuntimeException(e);
        }
    }

    public MutableByteIterator byteIterator()
    {
        return new InternalByteIterator();
    }

    public byte[] toArray()
    {
        byte[] result = new byte[this.size()];
        int index = 0;

        for (int i = 0; i < this.keys.length; i++)
        {
            if (isNonSentinel(this.keys[i]))
            {
                result[index] = this.values[i];
                index++;
            }
        }
        return result;
    }

    public boolean contains(byte value)
    {
        return this.containsValue(value);
    }

    public boolean containsAll(byte... source)
    {
        for (byte item : source)
        {
            if (!this.containsValue(item))
            {
                return false;
            }
        }
        return true;
    }

    public boolean containsAll(ByteIterable source)
    {
        return this.containsAll(source.toArray());
    }

    public void clear()
    {
        this.occupiedWithData = 0;
        this.occupiedWithSentinels = 0;
        Arrays.fill(this.keys, null);
        Arrays.fill(this.values, EMPTY_VALUE);
    }

    public void put(K key, byte value)
    {
        int index = this.probe(key);

        if (isNonSentinel(this.keys[index]) && nullSafeEquals(this.toNonSentinel(this.keys[index]), key))
        {
            // key already present in map
            this.values[index] = value;
            return;
        }

        this.addKeyValueAtIndex(key, value, index);
    }

    public void putAll(ObjectByteMap<? extends K> map)
    {
        map.forEachKeyValue(new ObjectByteProcedure<K>()
        {
            public void value(K key, byte value)
            {
                ObjectByteHashMap.this.put(key, value);
            }
        });
    }

    public void removeKey(K key)
    {
        int index = this.probe(key);
        this.removeKeyAtIndex(key, index);
    }

    private void removeKeyAtIndex(K key, int index)
    {
        if (isNonSentinel(this.keys[index]) && nullSafeEquals(this.toNonSentinel(this.keys[index]), key))
        {
            this.keys[index] = REMOVED_KEY;
            this.values[index] = EMPTY_VALUE;
            this.occupiedWithData--;
            this.occupiedWithSentinels++;
        }
    }

    public void remove(Object key)
    {
        this.removeKey((K) key);
    }

    public byte removeKeyIfAbsent(K key, byte value)
    {
        int index = this.probe(key);
        if (isNonSentinel(this.keys[index]) && nullSafeEquals(this.toNonSentinel(this.keys[index]), key))
        {
            this.keys[index] = REMOVED_KEY;
            byte oldValue = this.values[index];
            this.values[index] = EMPTY_VALUE;
            this.occupiedWithData--;
            this.occupiedWithSentinels++;

            return oldValue;
        }
        return value;
    }

    public byte getIfAbsentPut(K key, byte value)
    {
        int index = this.probe(key);
        if (isNonSentinel(this.keys[index]) && nullSafeEquals(this.toNonSentinel(this.keys[index]), key))
        {
            return this.values[index];
        }
        this.addKeyValueAtIndex(key, value, index);
        return value;
    }

    public byte getIfAbsentPut(K key, ByteFunction0 function)
    {
        int index = this.probe(key);
        if (isNonSentinel(this.keys[index]) && nullSafeEquals(this.toNonSentinel(this.keys[index]), key))
        {
            return this.values[index];
        }
        byte value = function.value();
        this.addKeyValueAtIndex(key, value, index);
        return value;
    }

    public <P> byte getIfAbsentPutWith(K key, ByteFunction<? super P> function, P parameter)
    {
        int index = this.probe(key);
        if (isNonSentinel(this.keys[index]) && nullSafeEquals(this.toNonSentinel(this.keys[index]), key))
        {
            return this.values[index];
        }
        byte value = function.byteValueOf(parameter);
        this.addKeyValueAtIndex(key, value, index);
        return value;
    }

    public byte getIfAbsentPutWithKey(K key, ByteFunction<? super K> function)
    {
        int index = this.probe(key);
        if (isNonSentinel(this.keys[index]) && nullSafeEquals(this.toNonSentinel(this.keys[index]), key))
        {
            return this.values[index];
        }
        byte value = function.byteValueOf(key);
        this.addKeyValueAtIndex(key, value, index);
        return value;
    }

    public byte updateValue(K key, byte initialValueIfAbsent, ByteToByteFunction function)
    {
        int index = this.probe(key);
        if (isNonSentinel(this.keys[index]) && nullSafeEquals(this.toNonSentinel(this.keys[index]), key))
        {
            this.values[index] = function.valueOf(this.values[index]);
            return this.values[index];
        }
        byte value = function.valueOf(initialValueIfAbsent);
        this.addKeyValueAtIndex(key, value, index);
        return value;
    }

    private void addKeyValueAtIndex(K key, byte value, int index)
    {
        if (this.keys[index] == REMOVED_KEY)
        {
            --this.occupiedWithSentinels;
        }
        this.keys[index] = toSentinelIfNull(key);
        this.values[index] = value;
        ++this.occupiedWithData;
        if (this.occupiedWithData + this.occupiedWithSentinels > this.maxOccupiedWithData())
        {
            this.rehashAndGrow();
        }
    }

    public byte addToValue(K key, byte toBeAdded)
    {
        int index = this.probe(key);
        if (isNonSentinel(this.keys[index]) && nullSafeEquals(this.toNonSentinel(this.keys[index]), key))
        {
            this.values[index] += toBeAdded;
            return this.values[index];
        }
        this.addKeyValueAtIndex(key, toBeAdded, index);
        return this.values[index];
    }

    public ObjectByteHashMap<K> withKeyValue(K key1, byte value1)
    {
        this.put(key1, value1);
        return this;
    }

    public ObjectByteHashMap<K> withKeysValues(K key1, byte value1, K key2, byte value2)
    {
        this.put(key1, value1);
        this.put(key2, value2);
        return this;
    }

    public ObjectByteHashMap<K> withKeysValues(K key1, byte value1, K key2, byte value2, K key3, byte value3)
    {
        this.put(key1, value1);
        this.put(key2, value2);
        this.put(key3, value3);
        return this;
    }

    public ObjectByteHashMap<K> withKeysValues(K key1, byte value1, K key2, byte value2, K key3, byte value3, K key4, byte value4)
    {
        this.put(key1, value1);
        this.put(key2, value2);
        this.put(key3, value3);
        this.put(key4, value4);
        return this;
    }

    public ObjectByteHashMap<K> withoutKey(K key)
    {
        this.removeKey(key);
        return this;
    }

    public ObjectByteHashMap<K> withoutAllKeys(Iterable<? extends K> keys)
    {
        for (K key : keys)
        {
            this.removeKey(key);
        }
        return this;
    }

    public MutableObjectByteMap<K> asUnmodifiable()
    {
        return new UnmodifiableObjectByteMap<K>(this);
    }

    public MutableObjectByteMap<K> asSynchronized()
    {
        return new SynchronizedObjectByteMap<K>(this);
    }

    public ImmutableObjectByteMap<K> toImmutable()
    {
        return ObjectByteMaps.immutable.withAll(this);
    }

    public byte get(Object key)
    {
        return this.getIfAbsent(key, EMPTY_VALUE);
    }

    public byte getOrThrow(Object key)
    {
        int index = this.probe(key);
        if (isNonSentinel(this.keys[index]))
        {
            return this.values[index];
        }
        throw new IllegalStateException("Key " + key + " not present.");
    }

    public byte getIfAbsent(Object key, byte ifAbsent)
    {
        int index = this.probe(key);
        if (isNonSentinel(this.keys[index]) && nullSafeEquals(this.toNonSentinel(this.keys[index]), key))
        {
            return this.values[index];
        }
        return ifAbsent;
    }

    public boolean containsKey(Object key)
    {
        int index = this.probe(key);
        return isNonSentinel(this.keys[index]) && nullSafeEquals(this.toNonSentinel(this.keys[index]), key);
    }

    public boolean containsValue(byte value)
    {
        for (int i = 0; i < this.values.length; i++)
        {
            if (isNonSentinel(this.keys[i]) && this.values[i] == value)
            {
                return true;
            }
        }
        return false;
    }

    public void forEach(ByteProcedure procedure)
    {
        this.each(procedure);
    }

    /**
     * @since 7.0.
     */
    public void each(ByteProcedure procedure)
    {
        this.forEachValue(procedure);
    }

    public void forEachValue(ByteProcedure procedure)
    {
        for (int i = 0; i < this.keys.length; i++)
        {
            if (isNonSentinel(this.keys[i]))
            {
                procedure.value(this.values[i]);
            }
        }
    }

    public void forEachKey(Procedure<? super K> procedure)
    {
        for (int i = 0; i < this.keys.length; i++)
        {
            if (isNonSentinel(this.keys[i]))
            {
                procedure.value(this.toNonSentinel(this.keys[i]));
            }
        }
    }

    public void forEachKeyValue(ObjectByteProcedure<? super K> procedure)
    {
        for (int i = 0; i < this.keys.length; i++)
        {
            if (isNonSentinel(this.keys[i]))
            {
                procedure.value(this.toNonSentinel(this.keys[i]), this.values[i]);
            }
        }
    }

    public ObjectByteHashMap<K> select(ObjectBytePredicate<? super K> predicate)
    {
        ObjectByteHashMap<K> result = ObjectByteHashMap.newMap();

        for (int i = 0; i < this.keys.length; i++)
        {
            if (isNonSentinel(this.keys[i]) && predicate.accept(this.toNonSentinel(this.keys[i]), this.values[i]))
            {
                result.put(this.toNonSentinel(this.keys[i]), this.values[i]);
            }
        }
        return result;
    }

    public ObjectByteHashMap<K> reject(ObjectBytePredicate<? super K> predicate)
    {
        ObjectByteHashMap<K> result = ObjectByteHashMap.newMap();

        for (int i = 0; i < this.keys.length; i++)
        {
            if (isNonSentinel(this.keys[i]) && !predicate.accept(this.toNonSentinel(this.keys[i]), this.values[i]))
            {
                result.put(this.toNonSentinel(this.keys[i]), this.values[i]);
            }
        }
        return result;
    }

    public MutableByteCollection select(BytePredicate predicate)
    {
        ByteArrayList result = new ByteArrayList();

        for (int i = 0; i < this.keys.length; i++)
        {
            if (isNonSentinel(this.keys[i]) && predicate.accept(this.values[i]))
            {
                result.add(this.values[i]);
            }
        }
        return result;
    }

    public MutableByteCollection reject(BytePredicate predicate)
    {
        ByteArrayList result = new ByteArrayList();

        for (int i = 0; i < this.keys.length; i++)
        {
            if (isNonSentinel(this.keys[i]) && !predicate.accept(this.values[i]))
            {
                result.add(this.values[i]);
            }
        }
        return result;
    }

    public byte detectIfNone(BytePredicate predicate, byte ifNone)
    {
        for (int i = 0; i < this.keys.length; i++)
        {
            if (isNonSentinel(this.keys[i]) && predicate.accept(this.values[i]))
            {
                return this.values[i];
            }
        }
        return ifNone;
    }

    public <V> MutableCollection<V> collect(ByteToObjectFunction<? extends V> function)
    {
        MutableList<V> result = FastList.newList(this.size());
        for (int i = 0; i < this.keys.length; i++)
        {
            if (isNonSentinel(this.keys[i]))
            {
                result.add(function.valueOf(this.values[i]));
            }
        }
        return result;
    }

    public int count(BytePredicate predicate)
    {
        int count = 0;

        for (int i = 0; i < this.keys.length; i++)
        {
            if (isNonSentinel(this.keys[i]) && predicate.accept(this.values[i]))
            {
                count++;
            }
        }
        return count;
    }

    public boolean anySatisfy(BytePredicate predicate)
    {
        for (int i = 0; i < this.keys.length; i++)
        {
            if (isNonSentinel(this.keys[i]) && predicate.accept(this.values[i]))
            {
                return true;
            }
        }
        return false;
    }

    public boolean allSatisfy(BytePredicate predicate)
    {
        for (int i = 0; i < this.keys.length; i++)
        {
            if (isNonSentinel(this.keys[i]) && !predicate.accept(this.values[i]))
            {
                return false;
            }
        }
        return true;
    }

    public boolean noneSatisfy(BytePredicate predicate)
    {
        for (int i = 0; i < this.keys.length; i++)
        {
            if (isNonSentinel(this.keys[i]) && predicate.accept(this.values[i]))
            {
                return false;
            }
        }
        return true;
    }

    public <V> V injectInto(V injectedValue, ObjectByteToObjectFunction<? super V, ? extends V> function)
    {
        V result = injectedValue;

        for (int i = 0; i < this.keys.length; i++)
        {
            if (isNonSentinel(this.keys[i]))
            {
                result = function.valueOf(result, this.values[i]);
            }
        }

        return result;
    }

    public long sum()
    {
        long result = 0L;

        for (int i = 0; i < this.keys.length; i++)
        {
            if (isNonSentinel(this.keys[i]))
            {
                result += this.values[i];
            }
        }
        return result;
    }

    public byte max()
    {
        if (this.isEmpty())
        {
            throw new NoSuchElementException();
        }
        byte max = (byte) 0;
        boolean isMaxSet = false;

        for (int i = 0; i < this.keys.length; i++)
        {
            if (isNonSentinel(this.keys[i]) && (!isMaxSet || max < this.values[i]))
            {
                max = this.values[i];
                isMaxSet = true;
            }
        }
        return max;
    }

    public byte min()
    {
        if (this.isEmpty())
        {
            throw new NoSuchElementException();
        }
        byte min = (byte) 0;
        boolean isMinSet = false;

        for (int i = 0; i < this.keys.length; i++)
        {
            if (isNonSentinel(this.keys[i]) && (!isMinSet || this.values[i] < min))
            {
                min = this.values[i];
                isMinSet = true;
            }
        }
        return min;
    }

    public byte maxIfEmpty(byte defaultValue)
    {
        if (this.isEmpty())
        {
            return defaultValue;
        }
        byte max = (byte) 0;
        boolean isMaxSet = false;

        for (int i = 0; i < this.keys.length; i++)
        {
            if (isNonSentinel(this.keys[i]) && (!isMaxSet || max < this.values[i]))
            {
                max = this.values[i];
                isMaxSet = true;
            }
        }
        return max;
    }

    public byte minIfEmpty(byte defaultValue)
    {
        if (this.isEmpty())
        {
            return defaultValue;
        }
        byte min = (byte) 0;
        boolean isMinSet = false;

        for (int i = 0; i < this.keys.length; i++)
        {
            if (isNonSentinel(this.keys[i]) && (!isMinSet || this.values[i] < min))
            {
                min = this.values[i];
                isMinSet = true;
            }
        }
        return min;
    }

    public double average()
    {
        if (this.isEmpty())
        {
            throw new ArithmeticException();
        }
        return (double) this.sum() / (double) this.size();
    }

    public double median()
    {
        if (this.isEmpty())
        {
            throw new ArithmeticException();
        }
        byte[] sortedArray = this.toSortedArray();
        int middleIndex = sortedArray.length >> 1;
        if (sortedArray.length > 1 && (sortedArray.length & 1) == 0)
        {
            byte first = sortedArray[middleIndex];
            byte second = sortedArray[middleIndex - 1];
            return ((double) first + (double) second) / 2.0;
        }
        return (double) sortedArray[middleIndex];
    }

    public MutableByteList toList()
    {
        MutableByteList result = new ByteArrayList(this.size());

        for (int i = 0; i < this.keys.length; i++)
        {
            if (isNonSentinel(this.keys[i]))
            {
                result.add(this.values[i]);
            }
        }
        return result;
    }

    public MutableByteSet toSet()
    {
        MutableByteSet result = new ByteHashSet(this.size());

        for (int i = 0; i < this.keys.length; i++)
        {
            if (isNonSentinel(this.keys[i]))
            {
                result.add(this.values[i]);
            }
        }
        return result;
    }

    public MutableByteBag toBag()
    {
        MutableByteBag result = new ByteHashBag(this.size());

        for (int i = 0; i < this.keys.length; i++)
        {
            if (isNonSentinel(this.keys[i]))
            {
                result.add(this.values[i]);
            }
        }
        return result;
    }

    public LazyByteIterable asLazy()
    {
        return new LazyByteIterableAdapter(this);
    }

    public byte[] toSortedArray()
    {
        byte[] array = this.toArray();
        Arrays.sort(array);
        return array;
    }

    public MutableByteList toSortedList()
    {
        return this.toList().sortThis();
    }

    public void writeExternal(ObjectOutput out) throws IOException
    {
        out.writeInt(this.size());
        for (int i = 0; i < this.keys.length; i++)
        {
            if (isNonSentinel(this.keys[i]))
            {
                out.writeObject(this.toNonSentinel(this.keys[i]));
                out.writeByte(this.values[i]);
            }
        }
    }

    public void readExternal(ObjectInput in) throws IOException, ClassNotFoundException
    {
        int size = in.readInt();
        int capacity = this.smallestPowerOfTwoGreaterThan(this.fastCeil(size * OCCUPIED_DATA_RATIO));
        this.allocateTable(capacity);

        for (int i = 0; i < size; i++)
        {
            this.put((K) in.readObject(), in.readByte());
        }
    }

    public LazyIterable<K> keysView()
    {
        return new KeysView();
    }

    public RichIterable<ObjectBytePair<K>> keyValuesView()
    {
        return new KeyValuesView();
    }

    /**
     * Rehashes every element in the set into a new backing table of the smallest possible size and eliminating removed sentinels.
     */
    public void compact()
    {
        this.rehash(this.smallestPowerOfTwoGreaterThan(this.size()));
    }

    private void rehashAndGrow()
    {
        this.rehash(this.keys.length << 1);
    }

    private void rehash(int newCapacity)
    {
        int oldLength = this.keys.length;
        Object[] old = this.keys;
        byte[] oldValues = this.values;
        this.allocateTable(newCapacity);
        this.occupiedWithData = 0;
        this.occupiedWithSentinels = 0;

        for (int i = 0; i < oldLength; i++)
        {
            if (isNonSentinel(old[i]))
            {
                this.put(this.toNonSentinel(old[i]), oldValues[i]);
            }
        }
    }

    // exposed for testing
    int probe(Object element)
    {
        int index = this.spread(element);

        int removedIndex = -1;
        if (isRemovedKey(this.keys[index]))
        {
            removedIndex = index;
        }

        else if (this.keys[index] == null || nullSafeEquals(this.toNonSentinel(this.keys[index]), element))
        {
            return index;
        }

        int nextIndex = index;
        int probe = 17;

        // loop until an empty slot is reached
        while (true)
        {
            // Probe algorithm: 17*n*(n+1)/2 where n = no. of collisions
            nextIndex += probe;
            probe += 17;
            nextIndex &= this.keys.length - 1;

            if (isRemovedKey(this.keys[nextIndex]))
            {
                if (removedIndex == -1)
                {
                    removedIndex = nextIndex;
                }
            }
            else if (nullSafeEquals(this.toNonSentinel(this.keys[nextIndex]), element))
            {
                return nextIndex;
            }
            else if (this.keys[nextIndex] == null)
            {
                return removedIndex == -1 ? nextIndex : removedIndex;
            }
        }
    }

    // exposed for testing
    int spread(Object element)
    {
        // This function ensures that hashCodes that differ only by
        // constant multiples at each bit position have a bounded
        // number of collisions (approximately 8 at default load factor).
        int h = element == null ? 0 : element.hashCode();
        h ^= h >>> 20 ^ h >>> 12;
        h ^= h >>> 7 ^ h >>> 4;
        return h & (this.keys.length - 1);
    }

    private static boolean nullSafeEquals(Object value, Object other)
    {
        if (value == null)
        {
            if (other == null)
            {
                return true;
            }
        }
        else if (other == value || value.equals(other))
        {
            return true;
        }
        return false;
    }

    private void allocateTable(int sizeToAllocate)
    {
        this.keys = new Object[sizeToAllocate];
        this.values = new byte[sizeToAllocate];
    }

    private static boolean isRemovedKey(Object key)
    {
        return key == REMOVED_KEY;
    }

    private static <K> boolean isNonSentinel(K key)
    {
        return key != null && !isRemovedKey(key);
    }

    private K toNonSentinel(Object key)
    {
        return key == NULL_KEY ? null : (K) key;
    }

    private static Object toSentinelIfNull(Object key)
    {
        return key == null ? NULL_KEY : key;
    }

    private int maxOccupiedWithData()
    {
        int capacity = this.keys.length;
        // need at least one free slot for open addressing
        return Math.min(capacity - 1, capacity / OCCUPIED_DATA_RATIO);
    }

    private int maxOccupiedWithSentinels()
    {
        return this.keys.length / OCCUPIED_SENTINEL_RATIO;
    }

    private class InternalByteIterator implements MutableByteIterator
    {
        private int count;
        private int position;

        public boolean hasNext()
        {
            return this.count != ObjectByteHashMap.this.size();
        }

        public byte next()
        {
            if (!this.hasNext())
            {
                throw new NoSuchElementException();
            }

            Object[] keys = ObjectByteHashMap.this.keys;
            while (!isNonSentinel(keys[this.position]))
            {
                this.position++;
            }
            byte result = ObjectByteHashMap.this.values[this.position];
            this.count++;
            this.position++;
            return result;
        }

        public void remove()
        {
            if (this.position == 0 || !isNonSentinel(ObjectByteHashMap.this.keys[this.position - 1]))
            {
                throw new IllegalStateException();
            }
            ObjectByteHashMap.this.remove(ObjectByteHashMap.this.keys[this.position - 1]);
            this.count--;
        }
    }

    public Set<K> keySet()
    {
        return new KeySet();
    }

    public MutableByteCollection values()
    {
        return new ValuesCollection();
    }

    private class KeySet implements Set<K>
    {
        @Override
        public boolean equals(Object obj)
        {
            if (obj instanceof Set)
            {
                Set<?> other = (Set<?>) obj;
                if (other.size() == this.size())
                {
                    return this.containsAll(other);
                }
            }
            return false;
        }

        @Override
        public int hashCode()
        {
            int hashCode = 0;
            Object[] table = ObjectByteHashMap.this.keys;
            for (int i = 0; i < table.length; i++)
            {
                Object key = table[i];
                if (ObjectByteHashMap.isNonSentinel(key))
                {
                    K nonSentinelKey = ObjectByteHashMap.this.toNonSentinel(key);
                    hashCode += nonSentinelKey == null ? 0 : nonSentinelKey.hashCode();
                }
            }
            return hashCode;
        }

        public int size()
        {
            return ObjectByteHashMap.this.size();
        }

        public boolean isEmpty()
        {
            return ObjectByteHashMap.this.isEmpty();
        }

        public boolean contains(Object o)
        {
            return ObjectByteHashMap.this.containsKey(o);
        }

        public Object[] toArray()
        {
            int size = ObjectByteHashMap.this.size();
            Object[] result = new Object[size];
            this.copyKeys(result);
            return result;
        }

        public <T> T[] toArray(T[] result)
        {
            int size = ObjectByteHashMap.this.size();
            if (result.length < size)
            {
                result = (T[]) Array.newInstance(result.getClass().getComponentType(), size);
            }
            this.copyKeys(result);
            if (size < result.length)
            {
                result[size] = null;
            }
            return result;
        }

        public boolean add(K key)
        {
            throw new UnsupportedOperationException("Cannot call add() on " + this.getClass().getSimpleName());
        }

        public boolean remove(Object key)
        {
            int oldSize = ObjectByteHashMap.this.size();
            ObjectByteHashMap.this.removeKey((K) key);
            return oldSize != ObjectByteHashMap.this.size();
        }

        public boolean containsAll(Collection<?> collection)
        {
            for (Object aCollection : collection)
            {
                if (!ObjectByteHashMap.this.containsKey(aCollection))
                {
                    return false;
                }
            }
            return true;
        }

        public boolean addAll(Collection<? extends K> collection)
        {
            throw new UnsupportedOperationException("Cannot call addAll() on " + this.getClass().getSimpleName());
        }

        public boolean retainAll(Collection<?> collection)
        {
            int oldSize = ObjectByteHashMap.this.size();
            Iterator<K> iterator = this.iterator();
            while (iterator.hasNext())
            {
                K next = iterator.next();
                if (!collection.contains(next))
                {
                    iterator.remove();
                }
            }
            return oldSize != ObjectByteHashMap.this.size();
        }

        public boolean removeAll(Collection<?> collection)
        {
            int oldSize = ObjectByteHashMap.this.size();
            for (Object object : collection)
            {
                ObjectByteHashMap.this.removeKey((K) object);
            }
            return oldSize != ObjectByteHashMap.this.size();
        }

        public void clear()
        {
            ObjectByteHashMap.this.clear();
        }

        public Iterator<K> iterator()
        {
            return new KeySetIterator();
        }

        private void copyKeys(Object[] result)
        {
            int count = 0;
            for (int i = 0; i < ObjectByteHashMap.this.keys.length; i++)
            {
                Object key = ObjectByteHashMap.this.keys[i];
                if (ObjectByteHashMap.isNonSentinel(key))
                {
                    result[count++] = ObjectByteHashMap.this.keys[i];
                }
            }
        }
    }

    private class KeySetIterator implements Iterator<K>
    {
        private int count;
        private int position;
        private K currentKey;
        private boolean isCurrentKeySet;

        public boolean hasNext()
        {
            return this.count < ObjectByteHashMap.this.size();
        }

        public K next()
        {
            if (!this.hasNext())
            {
                throw new NoSuchElementException();
            }
            this.count++;
            Object[] keys = ObjectByteHashMap.this.keys;
            while (!isNonSentinel(keys[this.position]))
            {
                this.position++;
            }
            this.currentKey = (K) ObjectByteHashMap.this.keys[this.position];
            this.isCurrentKeySet = true;
            this.position++;
            return ObjectByteHashMap.this.toNonSentinel(this.currentKey);
        }

        public void remove()
        {
            if (!this.isCurrentKeySet)
            {
                throw new IllegalStateException();
            }

            this.isCurrentKeySet = false;
            this.count--;

            if (isNonSentinel(this.currentKey))
            {
                int index = this.position - 1;
                ObjectByteHashMap.this.removeKeyAtIndex(toNonSentinel(this.currentKey), index);
            }
            else
            {
                ObjectByteHashMap.this.removeKey(this.currentKey);
            }
        }
    }

    private class ValuesCollection implements MutableByteCollection
    {
        public int size()
        {
            return ObjectByteHashMap.this.size();
        }

        public boolean isEmpty()
        {
            return ObjectByteHashMap.this.isEmpty();
        }

        public boolean notEmpty()
        {
            return ObjectByteHashMap.this.notEmpty();
        }

        public String makeString()
        {
            return this.makeString(", ");
        }

        public String makeString(String separator)
        {
            return this.makeString("", separator, "");
        }

        public String makeString(String start, String separator, String end)
        {
            Appendable stringBuilder = new StringBuilder();
            this.appendString(stringBuilder, start, separator, end);
            return stringBuilder.toString();
        }

        public void appendString(Appendable appendable)
        {
            this.appendString(appendable, ", ");
        }

        public void appendString(Appendable appendable, String separator)
        {
            this.appendString(appendable, "", separator, "");
        }

        public void appendString(Appendable appendable, String start, String separator, String end)
        {
            try
            {
                appendable.append(start);

                boolean first = true;

                for (int i = 0; i < ObjectByteHashMap.this.keys.length; i++)
                {
                    Object key = ObjectByteHashMap.this.keys[i];
                    if (isNonSentinel(key))
                    {
                        if (!first)
                        {
                            appendable.append(separator);
                        }
                        appendable.append(String.valueOf(ObjectByteHashMap.this.values[i]));
                        first = false;
                    }
                }
                appendable.append(end);
            }
            catch (IOException e)
            {
                throw new RuntimeException(e);
            }
        }

        public boolean add(byte element)
        {
            throw new UnsupportedOperationException("Cannot call add() on " + this.getClass().getSimpleName());
        }

        public boolean addAll(byte... source)
        {
            throw new UnsupportedOperationException("Cannot call addAll() on " + this.getClass().getSimpleName());
        }

        public boolean addAll(ByteIterable source)
        {
            throw new UnsupportedOperationException("Cannot call addAll() on " + this.getClass().getSimpleName());
        }

        public boolean remove(byte item)
        {
            int oldSize = ObjectByteHashMap.this.size();

            for (int i = 0; i < ObjectByteHashMap.this.keys.length; i++)
            {
                if (isNonSentinel(ObjectByteHashMap.this.keys[i]) && item == ObjectByteHashMap.this.values[i])
                {
                    ObjectByteHashMap.this.removeKey((K) ObjectByteHashMap.this.keys[i]);
                }
            }
            return oldSize != ObjectByteHashMap.this.size();
        }

        public boolean removeAll(ByteIterable source)
        {
            int oldSize = ObjectByteHashMap.this.size();

            ByteIterator iterator = source.byteIterator();
            while (iterator.hasNext())
            {
                this.remove(iterator.next());
            }
            return oldSize != ObjectByteHashMap.this.size();
        }

        public boolean removeAll(byte... source)
        {
            int oldSize = ObjectByteHashMap.this.size();

            for (byte item : source)
            {
                this.remove(item);
            }
            return oldSize != ObjectByteHashMap.this.size();
        }

        public boolean retainAll(ByteIterable source)
        {
            int oldSize = ObjectByteHashMap.this.size();
            final ByteSet sourceSet = source instanceof ByteSet ? (ByteSet) source : source.toSet();
            ObjectByteHashMap<K> retained = ObjectByteHashMap.this.select(new ObjectBytePredicate<K>()
            {
                public boolean accept(K object, byte value)
                {
                    return sourceSet.contains(value);
                }
            });
            if (retained.size() != oldSize)
            {
                ObjectByteHashMap.this.keys = retained.keys;
                ObjectByteHashMap.this.values = retained.values;
                ObjectByteHashMap.this.occupiedWithData = retained.occupiedWithData;
                ObjectByteHashMap.this.occupiedWithSentinels = retained.occupiedWithSentinels;
                return true;
            }
            return false;
        }

        public boolean retainAll(byte... source)
        {
            return this.retainAll(ByteHashSet.newSetWith(source));
        }

        public void clear()
        {
            ObjectByteHashMap.this.clear();
        }

        public MutableByteCollection with(byte element)
        {
            throw new UnsupportedOperationException("Cannot call with() on " + this.getClass().getSimpleName());
        }

        public MutableByteCollection without(byte element)
        {
            throw new UnsupportedOperationException("Cannot call without() on " + this.getClass().getSimpleName());
        }

        public MutableByteCollection withAll(ByteIterable elements)
        {
            throw new UnsupportedOperationException("Cannot call withAll() on " + this.getClass().getSimpleName());
        }

        public MutableByteCollection withoutAll(ByteIterable elements)
        {
            throw new UnsupportedOperationException("Cannot call withoutAll() on " + this.getClass().getSimpleName());
        }

        public MutableByteCollection asUnmodifiable()
        {
            return UnmodifiableByteCollection.of(this);
        }

        public MutableByteCollection asSynchronized()
        {
            return SynchronizedByteCollection.of(this);
        }

        public ImmutableByteCollection toImmutable()
        {
            return ByteLists.immutable.withAll(this);
        }

        public MutableByteIterator byteIterator()
        {
            return ObjectByteHashMap.this.byteIterator();
        }

        public byte[] toArray()
        {
            return ObjectByteHashMap.this.toArray();
        }

        public boolean contains(byte value)
        {
            return ObjectByteHashMap.this.containsValue(value);
        }

        public boolean containsAll(byte... source)
        {
            return ObjectByteHashMap.this.containsAll(source);
        }

        public boolean containsAll(ByteIterable source)
        {
            return ObjectByteHashMap.this.containsAll(source);
        }

        public void forEach(ByteProcedure procedure)
        {
            this.each(procedure);
        }

        public void each(ByteProcedure procedure)
        {
            ObjectByteHashMap.this.forEach(procedure);
        }

        public MutableByteCollection select(BytePredicate predicate)
        {
            return ObjectByteHashMap.this.select(predicate);
        }

        public MutableByteCollection reject(BytePredicate predicate)
        {
            return ObjectByteHashMap.this.reject(predicate);
        }

        public <V> MutableCollection<V> collect(ByteToObjectFunction<? extends V> function)
        {
            return ObjectByteHashMap.this.collect(function);
        }

        public <T> T injectInto(T injectedValue, ObjectByteToObjectFunction<? super T, ? extends T> function)
        {
            return ObjectByteHashMap.this.injectInto(injectedValue, function);
        }

        public byte detectIfNone(BytePredicate predicate, byte ifNone)
        {
            return ObjectByteHashMap.this.detectIfNone(predicate, ifNone);
        }

        public int count(BytePredicate predicate)
        {
            return ObjectByteHashMap.this.count(predicate);
        }

        public boolean anySatisfy(BytePredicate predicate)
        {
            return ObjectByteHashMap.this.anySatisfy(predicate);
        }

        public boolean allSatisfy(BytePredicate predicate)
        {
            return ObjectByteHashMap.this.allSatisfy(predicate);
        }

        public boolean noneSatisfy(BytePredicate predicate)
        {
            return ObjectByteHashMap.this.noneSatisfy(predicate);
        }

        public MutableByteList toList()
        {
            return ObjectByteHashMap.this.toList();
        }

        public MutableByteSet toSet()
        {
            return ObjectByteHashMap.this.toSet();
        }

        public MutableByteBag toBag()
        {
            return ObjectByteHashMap.this.toBag();
        }

        public LazyByteIterable asLazy()
        {
            return new LazyByteIterableAdapter(this);
        }

        public byte[] toSortedArray()
        {
            return ObjectByteHashMap.this.toSortedArray();
        }

        public MutableByteList toSortedList()
        {
            return ObjectByteHashMap.this.toSortedList();
        }

        public long sum()
        {
            return ObjectByteHashMap.this.sum();
        }

        public byte max()
        {
            return ObjectByteHashMap.this.max();
        }

        public byte maxIfEmpty(byte defaultValue)
        {
            return ObjectByteHashMap.this.maxIfEmpty(defaultValue);
        }

        public byte min()
        {
            return ObjectByteHashMap.this.min();
        }

        public byte minIfEmpty(byte defaultValue)
        {
            return ObjectByteHashMap.this.minIfEmpty(defaultValue);
        }

        public double average()
        {
            return ObjectByteHashMap.this.average();
        }

        public double median()
        {
            return ObjectByteHashMap.this.median();
        }
    }

    private class KeysView extends AbstractLazyIterable<K>
    {
        public void each(Procedure<? super K> procedure)
        {
            ObjectByteHashMap.this.forEachKey(procedure);
        }

        public void forEachWithIndex(ObjectIntProcedure<? super K> objectByteProcedure)
        {
            int index = 0;
            for (int i = 0; i < ObjectByteHashMap.this.keys.length; i++)
            {
                if (ObjectByteHashMap.isNonSentinel(ObjectByteHashMap.this.keys[i]))
                {
                    objectByteProcedure.value(ObjectByteHashMap.this.toNonSentinel(ObjectByteHashMap.this.keys[i]), index);
                    index++;
                }
            }
        }

        public <P> void forEachWith(Procedure2<? super K, ? super P> procedure, P parameter)
        {
            for (int i = 0; i < ObjectByteHashMap.this.keys.length; i++)
            {
                if (ObjectByteHashMap.isNonSentinel(ObjectByteHashMap.this.keys[i]))
                {
                    procedure.value(ObjectByteHashMap.this.toNonSentinel(ObjectByteHashMap.this.keys[i]), parameter);
                }
            }
        }

        public Iterator<K> iterator()
        {
            return new InternalKeysViewIterator();
        }

        public class InternalKeysViewIterator implements Iterator<K>
        {
            private int count;
            private int position;

            public K next()
            {
                if (!this.hasNext())
                {
                    throw new NoSuchElementException();
                }

                Object[] keys = ObjectByteHashMap.this.keys;
                while (!isNonSentinel(keys[this.position]))
                {
                    this.position++;
                }
                K result = ObjectByteHashMap.this.toNonSentinel(ObjectByteHashMap.this.keys[this.position]);
                this.count++;
                this.position++;
                return result;
            }

            public void remove()
            {
                throw new UnsupportedOperationException("Cannot call remove() on " + this.getClass().getSimpleName());
            }

            public boolean hasNext()
            {
                return this.count != ObjectByteHashMap.this.size();
            }
        }
    }

    private class KeyValuesView extends AbstractLazyIterable<ObjectBytePair<K>>
    {
        public void each(Procedure<? super ObjectBytePair<K>> procedure)
        {
            for (int i = 0; i < ObjectByteHashMap.this.keys.length; i++)
            {
                if (ObjectByteHashMap.isNonSentinel(ObjectByteHashMap.this.keys[i]))
                {
                    procedure.value(PrimitiveTuples.pair(ObjectByteHashMap.this.toNonSentinel(ObjectByteHashMap.this.keys[i]), ObjectByteHashMap.this.values[i]));
                }
            }
        }

        public void forEachWithIndex(ObjectIntProcedure<? super ObjectBytePair<K>> objectIntProcedure)
        {
            int index = 0;
            for (int i = 0; i < ObjectByteHashMap.this.keys.length; i++)
            {
                if (ObjectByteHashMap.isNonSentinel(ObjectByteHashMap.this.keys[i]))
                {
                    objectIntProcedure.value(PrimitiveTuples.pair(ObjectByteHashMap.this.toNonSentinel(ObjectByteHashMap.this.keys[i]), ObjectByteHashMap.this.values[i]), index);
                    index++;
                }
            }
        }

        public <P> void forEachWith(Procedure2<? super ObjectBytePair<K>, ? super P> procedure, P parameter)
        {
            for (int i = 0; i < ObjectByteHashMap.this.keys.length; i++)
            {
                if (ObjectByteHashMap.isNonSentinel(ObjectByteHashMap.this.keys[i]))
                {
                    procedure.value(PrimitiveTuples.pair(ObjectByteHashMap.this.toNonSentinel(ObjectByteHashMap.this.keys[i]), ObjectByteHashMap.this.values[i]), parameter);
                }
            }
        }

        public Iterator<ObjectBytePair<K>> iterator()
        {
            return new InternalKeyValuesIterator();
        }

        public class InternalKeyValuesIterator implements Iterator<ObjectBytePair<K>>
        {
            private int count;
            private int position;

            public ObjectBytePair<K> next()
            {
                if (!this.hasNext())
                {
                    throw new NoSuchElementException();
                }

                Object[] keys = ObjectByteHashMap.this.keys;
                while (!isNonSentinel(keys[this.position]))
                {
                    this.position++;
                }
                ObjectBytePair<K> result = PrimitiveTuples.pair(ObjectByteHashMap.this.toNonSentinel(ObjectByteHashMap.this.keys[this.position]), ObjectByteHashMap.this.values[this.position]);
                this.count++;
                this.position++;
                return result;
            }

            public void remove()
            {
                throw new UnsupportedOperationException("Cannot call remove() on " + this.getClass().getSimpleName());
            }

            public boolean hasNext()
            {
                return this.count != ObjectByteHashMap.this.size();
            }
        }
    }
}
