/*
 * Copyright 2014 Goldman Sachs.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.gs.collections.impl.bag.immutable.primitive;

import java.io.IOException;
import java.io.Serializable;

import com.gs.collections.api.DoubleIterable;
import com.gs.collections.api.LazyDoubleIterable;
import com.gs.collections.api.bag.ImmutableBag;
import com.gs.collections.api.bag.primitive.ImmutableDoubleBag;
import com.gs.collections.api.bag.primitive.DoubleBag;
import com.gs.collections.api.bag.primitive.MutableDoubleBag;
import com.gs.collections.api.block.function.primitive.DoubleToObjectFunction;
import com.gs.collections.api.block.function.primitive.ObjectDoubleToObjectFunction;
import com.gs.collections.api.block.predicate.primitive.DoublePredicate;
import com.gs.collections.api.block.procedure.primitive.DoubleIntProcedure;
import com.gs.collections.api.block.procedure.primitive.DoubleProcedure;
import com.gs.collections.api.iterator.DoubleIterator;
import com.gs.collections.api.list.primitive.MutableDoubleList;
import com.gs.collections.api.set.primitive.MutableDoubleSet;
import com.gs.collections.impl.bag.mutable.HashBag;
import com.gs.collections.impl.bag.mutable.primitive.DoubleHashBag;
import com.gs.collections.impl.factory.primitive.DoubleBags;
import com.gs.collections.impl.iterator.UnmodifiableDoubleIterator;
import com.gs.collections.impl.lazy.primitive.LazyDoubleIterableAdapter;
import com.gs.collections.impl.list.mutable.primitive.DoubleArrayList;
import com.gs.collections.impl.set.mutable.primitive.DoubleHashSet;
import net.jcip.annotations.Immutable;

/**
 * ImmutableDoubleSingletonBag is an optimization for {@link ImmutableDoubleBag} of size 1.
 * This file was automatically generated from template file immutablePrimitiveSingletonBag.stg.
 *
 * @since 4.0.
 */
@Immutable
final class ImmutableDoubleSingletonBag implements ImmutableDoubleBag, Serializable
{
    private static final long serialVersionUID = 1L;

    private final double element1;

    ImmutableDoubleSingletonBag(double element)
    {
        this.element1 = element;
    }

    public ImmutableDoubleBag newWith(double element)
    {
        return DoubleBags.immutable.with(this.element1, element);
    }

    public ImmutableDoubleBag newWithout(double element)
    {
        return Double.compare(this.element1, element) == 0 ? DoubleBags.immutable.with() : this;
    }

    public ImmutableDoubleBag newWithAll(DoubleIterable elements)
    {
        return DoubleHashBag.newBag(elements).with(this.element1).toImmutable();
    }

    public ImmutableDoubleBag newWithoutAll(DoubleIterable elements)
    {
        return elements.contains(this.element1) ? DoubleBags.immutable.with() : this;
    }

    public int size()
    {
        return 1;
    }

    public boolean isEmpty()
    {
        return false;
    }

    public boolean notEmpty()
    {
        return true;
    }

    public boolean contains(double value)
    {
        return Double.compare(this.element1, value) == 0;
    }

    public boolean containsAll(DoubleIterable source)
    {
        for (DoubleIterator iterator = source.doubleIterator(); iterator.hasNext(); )
        {
            if (Double.compare(this.element1, iterator.next()) != 0)
            {
                return false;
            }
        }
        return true;
    }

    public boolean containsAll(double... source)
    {
        for (double value : source)
        {
            if (Double.compare(this.element1, value) != 0)
            {
                return false;
            }
        }
        return true;
    }

    public void forEach(DoubleProcedure procedure)
    {
        this.each(procedure);
    }

    /**
     * @since 7.0.
     */
    public void each(DoubleProcedure procedure)
    {
        procedure.value(this.element1);
    }

    public ImmutableDoubleBag select(DoublePredicate predicate)
    {
        return predicate.accept(this.element1) ? DoubleHashBag.newBagWith(this.element1).toImmutable()
                : new DoubleHashBag().toImmutable();
    }

    public ImmutableDoubleBag reject(DoublePredicate predicate)
    {
        return predicate.accept(this.element1) ? new DoubleHashBag().toImmutable()
                : DoubleHashBag.newBagWith(this.element1).toImmutable();
    }

    public <V> ImmutableBag<V> collect(DoubleToObjectFunction<? extends V> function)
    {
        return HashBag.newBagWith(function.valueOf(this.element1)).toImmutable();
    }

    public MutableDoubleList toList()
    {
        return DoubleArrayList.newListWith(this.element1);
    }

    public int sizeDistinct()
    {
        return 1;
    }

    public int occurrencesOf(double item)
    {
        return Double.compare(this.element1, item) == 0 ? 1 : 0;
    }

    public void forEachWithOccurrences(DoubleIntProcedure doubleIntProcedure)
    {
        doubleIntProcedure.value(this.element1, 1);
    }

    public double detectIfNone(DoublePredicate predicate, double ifNone)
    {
        return predicate.accept(this.element1) ? this.element1 : ifNone;
    }

    public int count(DoublePredicate predicate)
    {
        return predicate.accept(this.element1) ? 1 : 0;
    }

    public boolean anySatisfy(DoublePredicate predicate)
    {
        return predicate.accept(this.element1);
    }

    public double sum()
    {
        return this.element1;
    }

    public double min()
    {
        return this.element1;
    }

    public double max()
    {
        return this.element1;
    }

    public double maxIfEmpty(double defaultValue)
    {
        return this.element1;
    }

    public double minIfEmpty(double defaultValue)
    {
        return this.element1;
    }

    public double average()
    {
        return this.element1;
    }

    public double median()
    {
        return this.element1;
    }

    public double[] toSortedArray()
    {
        return new double[]{this.element1};
    }

    public MutableDoubleList toSortedList()
    {
        return DoubleArrayList.newListWith(this.element1);
    }

    public boolean noneSatisfy(DoublePredicate predicate)
    {
        return !predicate.accept(this.element1);
    }

    public boolean allSatisfy(DoublePredicate predicate)
    {
        return predicate.accept(this.element1);
    }

    public <T> T injectInto(T injectedValue, ObjectDoubleToObjectFunction<? super T, ? extends T> function)
    {
        return function.valueOf(injectedValue, this.element1);
    }

    @Override
    public boolean equals(Object obj)
    {
        if (obj == this)
        {
            return true;
        }
        if (!(obj instanceof DoubleBag))
        {
            return false;
        }
        DoubleBag bag = (DoubleBag) obj;
        if (bag.size() != 1)
        {
            return false;
        }
        return this.occurrencesOf(this.element1) == bag.occurrencesOf(this.element1);
    }

    @Override
    public int hashCode()
    {
        return (int) (Double.doubleToLongBits(this.element1) ^ Double.doubleToLongBits(this.element1) >>> 32) ^ 1;
    }

    public MutableDoubleSet toSet()
    {
        return DoubleHashSet.newSetWith(this.element1);
    }

    public MutableDoubleBag toBag()
    {
        return DoubleHashBag.newBagWith(this.element1);
    }

    public ImmutableDoubleBag toImmutable()
    {
        return this;
    }

    public LazyDoubleIterable asLazy()
    {
        return new LazyDoubleIterableAdapter(this);
    }

    public double[] toArray()
    {
        return new double[]{this.element1};
    }

    @Override
    public String toString()
    {
        return '[' + this.makeString() + ']';
    }

    public String makeString()
    {
        return this.makeString(", ");
    }

    public String makeString(String separator)
    {
        return this.makeString("", separator, "");
    }

    public String makeString(String start, String separator, String end)
    {
        Appendable stringBuilder = new StringBuilder();
        this.appendString(stringBuilder, start, separator, end);
        return stringBuilder.toString();
    }

    public void appendString(Appendable appendable)
    {
        this.appendString(appendable, ", ");
    }

    public void appendString(Appendable appendable, String separator)
    {
        this.appendString(appendable, "", separator, "");
    }

    public void appendString(Appendable appendable, String start, String separator, String end)
    {
        try
        {
            appendable.append(start);
            appendable.append(String.valueOf(this.element1));
            appendable.append(end);
        }
        catch (IOException e)
        {
            throw new RuntimeException(e);
        }
    }

    public DoubleIterator doubleIterator()
    {
        return new UnmodifiableDoubleIterator(DoubleHashBag.newBagWith(this.element1).doubleIterator());
    }
}

