/*
 * Copyright 2014 Goldman Sachs.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.gs.collections.impl.list.mutable.primitive;

import java.io.Externalizable;
import java.io.IOException;
import java.io.ObjectInput;
import java.io.ObjectOutput;
import java.util.Arrays;
import java.util.NoSuchElementException;

import com.gs.collections.api.FloatIterable;
import com.gs.collections.api.LazyFloatIterable;
import com.gs.collections.api.block.function.primitive.ObjectFloatIntToObjectFunction;
import com.gs.collections.api.block.function.primitive.ObjectFloatToObjectFunction;
import com.gs.collections.api.block.function.primitive.FloatToObjectFunction;
import com.gs.collections.api.block.predicate.primitive.FloatPredicate;
import com.gs.collections.api.block.procedure.primitive.FloatIntProcedure;
import com.gs.collections.api.block.procedure.primitive.FloatProcedure;
import com.gs.collections.api.iterator.MutableFloatIterator;
import com.gs.collections.api.list.MutableList;
import com.gs.collections.api.list.primitive.FloatList;
import com.gs.collections.api.list.primitive.ImmutableFloatList;
import com.gs.collections.api.list.primitive.MutableFloatList;
import com.gs.collections.api.set.primitive.FloatSet;
import com.gs.collections.api.set.primitive.MutableFloatSet;
import com.gs.collections.impl.factory.primitive.FloatLists;
import com.gs.collections.impl.lazy.primitive.ReverseFloatIterable;
import com.gs.collections.impl.list.mutable.FastList;
import com.gs.collections.impl.primitive.AbstractFloatIterable;
import com.gs.collections.impl.set.mutable.primitive.FloatHashSet;
import net.jcip.annotations.NotThreadSafe;

/**
 * FloatArrayList is similar to {@link FastList}, and is memory-optimized for float primitives.
 * This file was automatically generated from template file primitiveArrayList.stg.
 *
 * @since 3.0.
 */
@NotThreadSafe
public class FloatArrayList extends AbstractFloatIterable
        implements MutableFloatList, Externalizable
{
    private static final long serialVersionUID = 1L;
    private static final float[] DEFAULT_SIZED_EMPTY_ARRAY = {};
    private static final float[] ZERO_SIZED_ARRAY = {};
    private static final int MAXIMUM_ARRAY_SIZE = Integer.MAX_VALUE - 8;

    protected int size;
    protected transient float[] items = DEFAULT_SIZED_EMPTY_ARRAY;

    public FloatArrayList()
    {
    }

    public FloatArrayList(int initialCapacity)
    {
        this.items = initialCapacity == 0 ? ZERO_SIZED_ARRAY : new float[initialCapacity];
    }

    public FloatArrayList(float... array)
    {
        this.size = array.length;
        this.items = array;
    }

    /**
     * Creates a new list using the passed {@code elements} argument as the backing store.
     * <p>
     * !!! WARNING: This method uses the passed in array, so can be very unsafe if the original
     * array is held onto anywhere else. !!!
     */
    public static FloatArrayList newListWith(float... elements)
    {
        return new FloatArrayList(elements);
    }

    public static FloatArrayList newList(FloatIterable source)
    {
        return FloatArrayList.newListWith(source.toArray());
    }

    public static FloatArrayList newWithNValues(int size, float value)
    {
        FloatArrayList newList = new FloatArrayList(size);
        for (int i = 0; i < size; i++)
        {
            newList.add(value);
        }
        return newList;
    }

    public int size()
    {
        return this.size;
    }

    public void clear()
    {
        Arrays.fill(this.items, 0.0f);
        this.size = 0;
    }

    public boolean contains(float value)
    {
        for (int i = 0; i < this.size; i++)
        {
            if (Float.compare(this.items[i], value) == 0)
            {
                return true;
            }
        }
        return false;
    }

    public float get(int index)
    {
        if (index < this.size)
        {
            return this.items[index];
        }
        throw this.newIndexOutOfBoundsException(index);
    }

    private IndexOutOfBoundsException newIndexOutOfBoundsException(int index)
    {
        return new IndexOutOfBoundsException("Index: " + index + " Size: " + this.size);
    }

    public float getFirst()
    {
        this.checkEmpty();
        return this.items[0];
    }

    public float getLast()
    {
        this.checkEmpty();
        return this.items[this.size() - 1];
    }

    private void checkEmpty()
    {
        if (this.isEmpty())
        {
            throw this.newIndexOutOfBoundsException(0);
        }
    }

    public int indexOf(float value)
    {
        for (int i = 0; i < this.size; i++)
        {
            if (Float.compare(this.items[i], value) == 0)
            {
                return i;
            }
        }
        return -1;
    }

    public int lastIndexOf(float value)
    {
        for (int i = this.size - 1; i >= 0; i--)
        {
            if (Float.compare(this.items[i], value) == 0)
            {
                return i;
            }
        }
        return -1;
    }

    public void trimToSize()
    {
        if (this.size < this.items.length)
        {
            this.transferItemsToNewArrayWithCapacity(this.size);
        }
    }

    private void transferItemsToNewArrayWithCapacity(int newCapacity)
    {
        this.items = this.copyItemsWithNewCapacity(newCapacity);
    }

    private float[] copyItemsWithNewCapacity(int newCapacity)
    {
        float[] newItems = new float[newCapacity];
        System.arraycopy(this.items, 0, newItems, 0, Math.min(this.size, newCapacity));
        return newItems;
    }

    private int sizePlusFiftyPercent(int oldSize)
    {
        int result = oldSize + (oldSize >> 1) + 1;
        return result < oldSize ? MAXIMUM_ARRAY_SIZE : result;
    }

    public void ensureCapacity(int minCapacity)
    {
        int oldCapacity = this.items.length;
        if (minCapacity > oldCapacity)
        {
            int newCapacity = Math.max(this.sizePlusFiftyPercent(oldCapacity), minCapacity);
            this.transferItemsToNewArrayWithCapacity(newCapacity);
        }
    }

    private void ensureCapacityForAdd()
    {
        if (this.items == DEFAULT_SIZED_EMPTY_ARRAY)
        {
            this.items = new float[10];
        }
        else
        {
            this.transferItemsToNewArrayWithCapacity(this.sizePlusFiftyPercent(this.size));
        }
    }

    public boolean add(float newItem)
    {
        if (this.items.length == this.size)
        {
            this.ensureCapacityForAdd();
        }
        this.items[this.size] = newItem;
        this.size++;
        return true;
    }

    public boolean addAll(float... source)
    {
        if (source.length < 1)
        {
            return false;
        }
        this.copyItems(source.length, source);
        return true;
    }

    public boolean addAll(FloatIterable source)
    {
        if (source instanceof FloatArrayList)
        {
            if (source.isEmpty())
            {
                return false;
            }
            FloatArrayList other = (FloatArrayList) source;
            this.copyItems(other.size(), other.items);
            return true;
        }
        return this.addAll(source.toArray());
    }

    private void copyItems(int sourceSize, float[] source)
    {
        int newSize = this.size + sourceSize;
        this.ensureCapacity(newSize);
        System.arraycopy(source, 0, this.items, this.size, sourceSize);
        this.size = newSize;
    }

    private void throwOutOfBounds(int index)
    {
        throw this.newIndexOutOfBoundsException(index);
    }

    public void addAtIndex(int index, float element)
    {
        if (index > -1 && index < this.size)
        {
            this.addAtIndexLessThanSize(index, element);
        }
        else if (index == this.size)
        {
            this.add(element);
        }
        else
        {
            this.throwOutOfBounds(index);
        }
    }

    private void addAtIndexLessThanSize(int index, float element)
    {
        int oldSize = this.size;
        this.size++;
        if (this.items.length == oldSize)
        {
            float[] newItems = new float[this.sizePlusFiftyPercent(oldSize)];
            if (index > 0)
            {
                System.arraycopy(this.items, 0, newItems, 0, index);
            }
            System.arraycopy(this.items, index, newItems, index + 1, oldSize - index);
            this.items = newItems;
        }
        else
        {
            System.arraycopy(this.items, index, this.items, index + 1, oldSize - index);
        }
        this.items[index] = element;
    }

    public boolean addAllAtIndex(int index, float... source)
    {
        if (index > this.size || index < 0)
        {
            this.throwOutOfBounds(index);
        }
        if (source.length == 0)
        {
            return false;
        }
        int sourceSize = source.length;
        int newSize = this.size + sourceSize;
        this.ensureCapacity(newSize);
        this.shiftElementsAtIndex(index, sourceSize);
        System.arraycopy(source, 0, this.items, index, sourceSize);
        this.size = newSize;
        return true;
    }

    public boolean addAllAtIndex(int index, FloatIterable source)
    {
        return this.addAllAtIndex(index, source.toArray());
    }

    private void shiftElementsAtIndex(int index, int sourceSize)
    {
        int numberToMove = this.size - index;
        if (numberToMove > 0)
        {
            System.arraycopy(this.items, index, this.items, index + sourceSize, numberToMove);
        }
    }

    public boolean remove(float value)
    {
        int index = this.indexOf(value);
        if (index >= 0)
        {
            this.removeAtIndex(index);
            return true;
        }
        return false;
    }

    public boolean removeAll(FloatIterable source)
    {
        boolean modified = false;
        for (int index = 0; index < this.size; index++)
        {
            if (source.contains(this.get(index)))
            {
                this.removeAtIndex(index);
                index--;
                modified = true;
            }
        }
        return modified;
    }

    public boolean removeAll(float... source)
    {
        FloatHashSet set = FloatHashSet.newSetWith(source);
        float[] newItems = new float[this.size];
        int count = 0;
        int oldSize = this.size;
        for (int index = 0; index < this.size; index++)
        {
            if (!set.contains(this.items[index]))
            {
                newItems[count] = this.items[index];
                count++;
            }
        }
        this.items = newItems;
        this.size = count;
        return oldSize != this.size;
    }

    public boolean retainAll(FloatIterable source)
    {
        int oldSize = this.size();
        final FloatSet sourceSet = source instanceof FloatSet ? (FloatSet) source : source.toSet();
        FloatArrayList retained = this.select(new FloatPredicate()
        {
            public boolean accept(float value)
            {
                return sourceSet.contains(value);
            }
        });
        this.size = retained.size;
        this.items = retained.items;
        return oldSize != this.size();
    }

    public boolean retainAll(float... source)
    {
        return this.retainAll(FloatHashSet.newSetWith(source));
    }

    public float removeAtIndex(int index)
    {
        float previous = this.get(index);
        int totalOffset = this.size - index - 1;
        if (totalOffset > 0)
        {
            System.arraycopy(this.items, index + 1, this.items, index, totalOffset);
        }
        --this.size;
        this.items[this.size] = 0.0f;
        return previous;
    }

    public float set(int index, float element)
    {
        float previous = this.get(index);
        this.items[index] = element;
        return previous;
    }

    public FloatArrayList with(float element)
    {
        this.add(element);
        return this;
    }

    public FloatArrayList without(float element)
    {
        this.remove(element);
        return this;
    }

    public FloatArrayList withAll(FloatIterable elements)
    {
        this.addAll(elements.toArray());
        return this;
    }

    public FloatArrayList withoutAll(FloatIterable elements)
    {
        this.removeAll(elements);
        return this;
    }

    public FloatArrayList with(float element1, float element2)
    {
        this.add(element1);
        this.add(element2);
        return this;
    }

    public FloatArrayList with(float element1, float element2, float element3)
    {
        this.add(element1);
        this.add(element2);
        this.add(element3);
        return this;
    }

    public FloatArrayList with(float element1, float element2, float element3, float... elements)
    {
        this.add(element1);
        this.add(element2);
        this.add(element3);
        return this.withArrayCopy(elements, 0, elements.length);
    }

    private FloatArrayList withArrayCopy(float[] elements, int begin, int length)
    {
        this.ensureCapacity(this.size + length);
        System.arraycopy(elements, begin, this.items, this.size, length);
        this.size += length;
        return this;
    }

    public MutableFloatIterator floatIterator()
    {
        return new InternalFloatIterator();
    }

    public void forEach(FloatProcedure procedure)
    {
        this.each(procedure);
    }

    /**
     * @since 7.0.
     */
    public void each(FloatProcedure procedure)
    {
        for (int i = 0; i < this.size; i++)
        {
            procedure.value(this.items[i]);
        }
    }

    public void forEachWithIndex(FloatIntProcedure procedure)
    {
        for (int i = 0; i < this.size; i++)
        {
            procedure.value(this.items[i], i);
        }
    }

    public <T> T injectInto(T injectedValue, ObjectFloatToObjectFunction<? super T, ? extends T> function)
    {
        T result = injectedValue;
        for (int i = 0; i < this.size; i++)
        {
            result = function.valueOf(result, this.items[i]);
        }
        return result;
    }

    public <T> T injectIntoWithIndex(T injectedValue, ObjectFloatIntToObjectFunction<? super T, ? extends T> function)
    {
        T result = injectedValue;
        for (int i = 0; i < this.size; i++)
        {
            result = function.valueOf(result, this.items[i], i);
        }
        return result;
    }

    public int count(FloatPredicate predicate)
    {
        int count = 0;
        for (int i = 0; i < this.size; i++)
        {
            if (predicate.accept(this.items[i]))
            {
                count++;
            }
        }
        return count;
    }

    public boolean anySatisfy(FloatPredicate predicate)
    {
        for (int i = 0; i < this.size; i++)
        {
            if (predicate.accept(this.items[i]))
            {
                return true;
            }
        }
        return false;
    }

    public boolean allSatisfy(FloatPredicate predicate)
    {
        for (int i = 0; i < this.size; i++)
        {
            if (!predicate.accept(this.items[i]))
            {
                return false;
            }
        }
        return true;
    }

    public boolean noneSatisfy(FloatPredicate predicate)
    {
        for (int i = 0; i < this.size; i++)
        {
            if (predicate.accept(this.items[i]))
            {
                return false;
            }
        }
        return true;
    }

    public FloatArrayList select(FloatPredicate predicate)
    {
        FloatArrayList result = new FloatArrayList();
        for (int i = 0; i < this.size; i++)
        {
            float item = this.items[i];
            if (predicate.accept(item))
            {
                result.add(item);
            }
        }
        return result;
    }

    public FloatArrayList reject(FloatPredicate predicate)
    {
        FloatArrayList result = new FloatArrayList();
        for (int i = 0; i < this.size; i++)
        {
            float item = this.items[i];
            if (!predicate.accept(item))
            {
                result.add(item);
            }
        }
        return result;
    }

    public float detectIfNone(FloatPredicate predicate, float ifNone)
    {
        for (int i = 0; i < this.size; i++)
        {
            float item = this.items[i];
            if (predicate.accept(item))
            {
                return item;
            }
        }
        return ifNone;
    }

    public <V> MutableList<V> collect(FloatToObjectFunction<? extends V> function)
    {
        FastList<V> target = FastList.newList(this.size);
        for (int i = 0; i < this.size; i++)
        {
            target.add(function.valueOf(this.items[i]));
        }
        return target;
    }

    public float max()
    {
        if (this.isEmpty())
        {
            throw new NoSuchElementException();
        }
        float max = this.items[0];
        for (int i = 1; i < this.size; i++)
        {
            float value = this.items[i];
            if (Float.compare(max, value) < 0)
            {
                max = value;
            }
        }
        return max;
    }

    public float min()
    {
        if (this.isEmpty())
        {
            throw new NoSuchElementException();
        }
        float min = this.items[0];
        for (int i = 1; i < this.size; i++)
        {
            float value = this.items[i];
            if (Float.compare(value, min) < 0)
            {
                min = value;
            }
        }
        return min;
    }

    public double sum()
    {
        double result = 0.0;
        double compensation = 0.0;
        for (int i = 0; i < this.size; i++)
        {
            double adjustedValue = this.items[i] - compensation;
            double nextSum = result + adjustedValue;
            compensation = nextSum - result - adjustedValue;
            result = nextSum;
        }
        return result;
    }

    public double dotProduct(FloatList list)
    {
        if (this.size != list.size())
        {
            throw new IllegalArgumentException("Lists used in dotProduct must be the same size");
        }
        double sum = 0.0;
        for (int i = 0; i < this.size; i++)
        {
            sum += (double) this.items[i] * list.get(i);
        }
        return sum;
    }

    public float[] toArray()
    {
        float[] newItems = new float[this.size];
        System.arraycopy(this.items, 0, newItems, 0, this.size);
        return newItems;
    }

    @Override
    public boolean equals(Object otherList)
    {
        if (otherList == this)
        {
            return true;
        }
        if (!(otherList instanceof FloatList))
        {
            return false;
        }
        FloatList list = (FloatList) otherList;
        if (this.size != list.size())
        {
            return false;
        }
        for (int i = 0; i < this.size; i++)
        {
            if (Float.compare(this.items[i], list.get(i)) != 0)
            {
                return false;
            }
        }
        return true;
    }

    @Override
    public int hashCode()
    {
        int hashCode = 1;
        for (int i = 0; i < this.size; i++)
        {
            float item = this.items[i];
            hashCode = 31 * hashCode + Float.floatToIntBits(item);
        }
        return hashCode;
    }

    public void appendString(
            Appendable appendable,
            String start,
            String separator,
            String end)
    {
        try
        {
            appendable.append(start);
            for (int i = 0; i < this.size; i++)
            {
                if (i > 0)
                {
                    appendable.append(separator);
                }
                float value = this.items[i];
                appendable.append(String.valueOf(value));
            }
            appendable.append(end);
        }
        catch (IOException e)
        {
            throw new RuntimeException(e);
        }
    }

    public MutableFloatList asUnmodifiable()
    {
        return new UnmodifiableFloatList(this);
    }

    public MutableFloatList asSynchronized()
    {
        return new SynchronizedFloatList(this);
    }

    public ImmutableFloatList toImmutable()
    {
        if (this.size == 0)
        {
            return FloatLists.immutable.empty();
        }
        if (this.size == 1)
        {
            return FloatLists.immutable.with(this.items[0]);
        }
        return FloatLists.immutable.with(this.toArray());
    }

    public void writeExternal(ObjectOutput out) throws IOException
    {
        out.writeInt(this.size);
        for (int i = 0; i < this.size; i++)
        {
            out.writeFloat(this.items[i]);
        }
    }

    public void readExternal(ObjectInput in) throws IOException
    {
        this.size = in.readInt();
        this.items = new float[this.size];
        for (int i = 0; i < this.size; i++)
        {
            this.items[i] = in.readFloat();
        }
    }

    public LazyFloatIterable asReversed()
    {
        return ReverseFloatIterable.adapt(this);
    }

    public FloatArrayList reverseThis()
    {
        int endIndex = this.size - 1;
        for (int i = 0; i < this.size / 2; i++)
        {
            float tempSwapValue = this.items[i];
            this.items[i] = this.items[endIndex - i];
            this.items[endIndex - i] = tempSwapValue;
        }
        return this;
    }

    public FloatArrayList sortThis()
    {
        Arrays.sort(this.items, 0, this.size);
        return this;
    }

    public FloatArrayList toReversed()
    {
        return FloatArrayList.newList(this.asReversed());
    }

    public int binarySearch(float value)
    {
        return Arrays.binarySearch(this.items, 0, this.size, value);
    }

    public MutableFloatList distinct()
    {
        FloatArrayList target = new FloatArrayList();
        MutableFloatSet seenSoFar = new FloatHashSet(this.size());

        for (int i = 0; i < this.size; i++)
        {
            float each = this.items[i];
            if (seenSoFar.add(each))
            {
                target.add(each);
            }
        }
        return target;
    }

    public MutableFloatList subList(int fromIndex, int toIndex)
    {
        throw new UnsupportedOperationException("subList not yet implemented!");
    }

    private class InternalFloatIterator implements MutableFloatIterator
    {
        /**
         * Index of element to be returned by subsequent call to next.
         */
        private int currentIndex;
        private int lastIndex = -1;

        public boolean hasNext()
        {
            return this.currentIndex != FloatArrayList.this.size();
        }

        public float next()
        {
            if (!this.hasNext())
            {
                throw new NoSuchElementException();
            }
            float next = FloatArrayList.this.items[this.currentIndex];
            this.lastIndex = this.currentIndex++;
            return next;
        }

        public void remove()
        {
            if (this.lastIndex == -1)
            {
                throw new IllegalStateException();
            }
            FloatArrayList.this.removeAtIndex(this.lastIndex);
            this.currentIndex--;
            this.lastIndex = -1;
        }
    }
}
