/*
 * Copyright 2014 Goldman Sachs.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.gs.collections.impl.list.mutable.primitive;

import java.util.Collection;
import java.util.Collections;

import com.gs.collections.api.FloatIterable;
import com.gs.collections.api.LazyFloatIterable;
import com.gs.collections.api.block.function.primitive.FloatToObjectFunction;
import com.gs.collections.api.block.function.primitive.ObjectFloatIntToObjectFunction;
import com.gs.collections.api.block.predicate.primitive.FloatPredicate;
import com.gs.collections.api.block.procedure.primitive.FloatIntProcedure;
import com.gs.collections.api.iterator.FloatIterator;
import com.gs.collections.api.list.MutableList;
import com.gs.collections.api.list.primitive.FloatList;
import com.gs.collections.api.list.primitive.ImmutableFloatList;
import com.gs.collections.api.list.primitive.MutableFloatList;
import com.gs.collections.impl.collection.mutable.primitive.AbstractSynchronizedFloatCollection;
import com.gs.collections.impl.factory.primitive.FloatLists;
import com.gs.collections.impl.lazy.primitive.LazyFloatIterableAdapter;
import com.gs.collections.impl.lazy.primitive.ReverseFloatIterable;
import net.jcip.annotations.GuardedBy;
import net.jcip.annotations.ThreadSafe;

/**
 * A synchronized view of a {@link MutableFloatList}. It is imperative that the user manually synchronize on the collection when iterating over it using the
 * {@link FloatIterator}, as per {@link Collections#synchronizedCollection(Collection)}.
 * <p>
 * This file was automatically generated from template file synchronizedPrimitiveList.stg.
 *
 * @see MutableFloatList#asSynchronized()
 * @see MutableList#asSynchronized()
 * @since 3.1.
 */
@ThreadSafe
public final class SynchronizedFloatList
        extends AbstractSynchronizedFloatCollection
        implements MutableFloatList
{
    private static final long serialVersionUID = 1L;

    SynchronizedFloatList(MutableFloatList list)
    {
        super(list);
    }

    SynchronizedFloatList(MutableFloatList list, Object newLock)
    {
        super(list, newLock);
    }

    @GuardedBy("getLock()")
    private MutableFloatList getMutableFloatList()
    {
        return (MutableFloatList) this.getFloatCollection();
    }

    public float get(int index)
    {
        synchronized (this.getLock())
        {
            return this.getMutableFloatList().get(index);
        }
    }

    public float getFirst()
    {
        synchronized (this.getLock())
        {
            return this.getMutableFloatList().getFirst();
        }
    }

    public float getLast()
    {
        synchronized (this.getLock())
        {
            return this.getMutableFloatList().getLast();
        }
    }

    public int indexOf(float value)
    {
        synchronized (this.getLock())
        {
            return this.getMutableFloatList().indexOf(value);
        }
    }

    public int lastIndexOf(float value)
    {
        synchronized (this.getLock())
        {
            return this.getMutableFloatList().lastIndexOf(value);
        }
    }

    public void addAtIndex(int index, float element)
    {
        synchronized (this.getLock())
        {
            this.getMutableFloatList().addAtIndex(index, element);
        }
    }

    public boolean addAllAtIndex(int index, float... source)
    {
        synchronized (this.getLock())
        {
            return this.getMutableFloatList().addAllAtIndex(index, source);
        }
    }

    public boolean addAllAtIndex(int index, FloatIterable source)
    {
        synchronized (this.getLock())
        {
            return this.getMutableFloatList().addAllAtIndex(index, source);
        }
    }

    public float removeAtIndex(int index)
    {
        synchronized (this.getLock())
        {
            return this.getMutableFloatList().removeAtIndex(index);
        }
    }

    public float set(int index, float element)
    {
        synchronized (this.getLock())
        {
            return this.getMutableFloatList().set(index, element);
        }
    }

    @Override
    public SynchronizedFloatList with(float element)
    {
        synchronized (this.getLock())
        {
            this.getMutableFloatList().add(element);
        }
        return this;
    }

    @Override
    public SynchronizedFloatList without(float element)
    {
        synchronized (this.getLock())
        {
            this.getMutableFloatList().remove(element);
        }
        return this;
    }

    @Override
    public SynchronizedFloatList withAll(FloatIterable elements)
    {
        synchronized (this.getLock())
        {
            this.getMutableFloatList().addAll(elements.toArray());
        }
        return this;
    }

    @Override
    public SynchronizedFloatList withoutAll(FloatIterable elements)
    {
        synchronized (this.getLock())
        {
            this.getMutableFloatList().removeAll(elements);
        }
        return this;
    }

    @Override
    public MutableFloatList select(FloatPredicate predicate)
    {
        synchronized (this.getLock())
        {
            return this.getMutableFloatList().select(predicate);
        }
    }

    @Override
    public MutableFloatList reject(FloatPredicate predicate)
    {
        synchronized (this.getLock())
        {
            return this.getMutableFloatList().reject(predicate);
        }
    }

    @Override
    public <V> MutableList<V> collect(FloatToObjectFunction<? extends V> function)
    {
        synchronized (this.getLock())
        {
            return this.getMutableFloatList().collect(function);
        }
    }

    public MutableFloatList sortThis()
    {
        synchronized (this.getLock())
        {
            this.getMutableFloatList().sortThis();
        }
        return this;
    }

    public int binarySearch(float value)
    {
        synchronized (this.getLock())
        {
            return this.getMutableFloatList().binarySearch(value);
        }
    }

    public double dotProduct(FloatList list)
    {
        synchronized (this.getLock())
        {
            return this.getMutableFloatList().dotProduct(list);
        }
    }

    @Override
    public boolean equals(Object otherList)
    {
        synchronized (this.getLock())
        {
            return this.getMutableFloatList().equals(otherList);
        }
    }

    @Override
    public int hashCode()
    {
        synchronized (this.getLock())
        {
            return this.getMutableFloatList().hashCode();
        }
    }

    @Override
    public LazyFloatIterable asLazy()
    {
        synchronized (this.getLock())
        {
            return new LazyFloatIterableAdapter(this);
        }
    }

    @Override
    public MutableFloatList asUnmodifiable()
    {
        return new UnmodifiableFloatList(this);
    }

    @Override
    public MutableFloatList asSynchronized()
    {
        return this;
    }

    @Override
    public ImmutableFloatList toImmutable()
    {
        synchronized (this.getLock())
        {
            int size = this.size();
            if (size == 0)
            {
                return FloatLists.immutable.with();
            }
            if (size == 1)
            {
                return FloatLists.immutable.with(this.getFirst());
            }
            return FloatLists.immutable.with(this.toArray());
        }
    }

    public MutableFloatList reverseThis()
    {
        synchronized (this.getLock())
        {
            this.getMutableFloatList().reverseThis();
        }
        return this;
    }

    public MutableFloatList toReversed()
    {
        synchronized (this.getLock())
        {
            return this.getMutableFloatList().toReversed();
        }
    }

    public LazyFloatIterable asReversed()
    {
        return ReverseFloatIterable.adapt(this);
    }

    public void forEachWithIndex(FloatIntProcedure procedure)
    {
        synchronized (this.getLock())
        {
            this.getMutableFloatList().forEachWithIndex(procedure);
        }
    }

    public <T> T injectIntoWithIndex(T injectedValue, ObjectFloatIntToObjectFunction<? super T, ? extends T> function)
    {
        synchronized (this.getLock())
        {
            return this.getMutableFloatList().injectIntoWithIndex(injectedValue, function);
        }
    }

    /**
     * @since 6.0.
     */
    public MutableFloatList distinct()
    {
        synchronized (this.getLock())
        {
            return this.getMutableFloatList().distinct();
        }
    }

    public MutableFloatList subList(int fromIndex, int toIndex)
    {
        throw new UnsupportedOperationException("subList not yet implemented!");
    }
}
