/*
 * Copyright 2020 The Android Open Source Project
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      https://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

@file:Suppress("NOTHING_TO_INLINE", "unused", "DEPRECATION")

package com.google.accompanist.insets

import androidx.compose.foundation.layout.PaddingValues
import androidx.compose.foundation.layout.padding
import androidx.compose.runtime.Composable
import androidx.compose.runtime.Stable
import androidx.compose.runtime.getValue
import androidx.compose.runtime.mutableStateOf
import androidx.compose.runtime.remember
import androidx.compose.runtime.setValue
import androidx.compose.ui.Modifier
import androidx.compose.ui.composed
import androidx.compose.ui.platform.LocalDensity
import androidx.compose.ui.unit.Density
import androidx.compose.ui.unit.Dp
import androidx.compose.ui.unit.LayoutDirection
import androidx.compose.ui.unit.dp

/**
 * Selectively apply additional space which matches the width/height of any system bars present
 * on the respective edges of the screen.
 *
 * @param enabled Whether to apply padding using the system bars dimensions on the respective edges.
 * Defaults to `true`.
 */
@Deprecated(
"""
accompanist/insets is deprecated.
For more migration information, please visit https://google.github.io/accompanist/insets/#migration
""",
    replaceWith = ReplaceWith(
        "systemBarsPadding()",
        "androidx.compose.foundation.layout.systemBarsPadding"
    )
)
inline fun Modifier.systemBarsPadding(enabled: Boolean = true): Modifier = composed {
    padding(
        rememberInsetsPaddingValues(
            insets = LocalWindowInsets.current.systemBars,
            applyStart = enabled,
            applyTop = enabled,
            applyEnd = enabled,
            applyBottom = enabled
        )
    )
}

/**
 * Selectively apply additional space which matches the width/height of any system bars present
 * on the respective edges of the screen.
 *
 * @param start Whether to apply padding to the start edge, which matches the system bars width
 * (if present) on the start edge of the screen. Defaults to `true`.
 * @param top Whether to apply padding to the top edge, which matches the system bars height
 * (if present) at the top edge of the screen. Defaults to `true`.
 * @param end Whether to apply padding to the end edge, which matches the system bars width
 * (if present) on the end edge of the screen. Defaults to `true`.
 * @param bottom Whether to apply padding to the bottom edge, which matches the system bars
 * height (if present) at the bottom edge of the screen. Defaults to `true`.
 */
@Deprecated(
"""
accompanist/insets is deprecated.
For more migration information, please visit https://google.github.io/accompanist/insets/#migration
""",
    replaceWith = ReplaceWith(
        "systemBarsPadding()",
        "androidx.compose.foundation.layout.systemBarsPadding"
    )
)
inline fun Modifier.systemBarsPadding(
    start: Boolean = true,
    top: Boolean = true,
    end: Boolean = true,
    bottom: Boolean = true,
): Modifier = composed {
    padding(
        rememberInsetsPaddingValues(
            insets = LocalWindowInsets.current.systemBars,
            applyStart = start,
            applyTop = top,
            applyEnd = end,
            applyBottom = bottom
        )
    )
}

/**
 * Apply additional space which matches the height of the status bars height along the top edge
 * of the content.
 */
@Deprecated(
"""
accompanist/insets is deprecated.
For more migration information, please visit https://google.github.io/accompanist/insets/#migration
""",
    replaceWith = ReplaceWith(
        "statusBarsPadding()",
        "androidx.compose.foundation.layout.statusBarsPadding"
    )
)
inline fun Modifier.statusBarsPadding(): Modifier = composed {
    padding(
        rememberInsetsPaddingValues(
            insets = LocalWindowInsets.current.statusBars,
            applyTop = true
        )
    )
}

/**
 * Apply additional space which matches the height of the navigation bars height
 * along the [bottom] edge of the content, and additional space which matches the width of
 * the navigation bars on the respective [start] and [end] edges.
 *
 * @param bottom Whether to apply padding to the bottom edge, which matches the navigation bars
 * height (if present) at the bottom edge of the screen. Defaults to `true`.
 * @param start Whether to apply padding to the start edge, which matches the navigation bars width
 * (if present) on the start edge of the screen. Defaults to `true`.
 * @param end Whether to apply padding to the end edge, which matches the navigation bars width
 * (if present) on the end edge of the screen. Defaults to `true`.
 */
@Deprecated(
"""
accompanist/insets is deprecated.
For more migration information, please visit https://google.github.io/accompanist/insets/#migration
""",
    replaceWith = ReplaceWith(
        "navigationBarsPadding()",
        "androidx.compose.foundation.layout.navigationBarsPadding"
    )
)
inline fun Modifier.navigationBarsPadding(
    bottom: Boolean = true,
    start: Boolean = true,
    end: Boolean = true,
): Modifier = composed {
    padding(
        rememberInsetsPaddingValues(
            insets = LocalWindowInsets.current.navigationBars,
            applyStart = start,
            applyEnd = end,
            applyBottom = bottom
        )
    )
}

/**
 * Apply additional space which matches the height of the [WindowInsets.ime] (on-screen keyboard)
 * height along the bottom edge of the content.
 *
 * This method has no special handling for the [WindowInsets.navigationBars], which usually
 * intersect the [WindowInsets.ime]. Most apps will usually want to use the
 * [Modifier.navigationBarsWithImePadding] modifier.
 */
@Deprecated(
"""
accompanist/insets is deprecated.
For more migration information, please visit https://google.github.io/accompanist/insets/#migration
""",
    replaceWith = ReplaceWith(
        "imePadding()",
        "androidx.compose.foundation.layout.imePadding"
    )
)
inline fun Modifier.imePadding(): Modifier = composed {
    padding(
        rememberInsetsPaddingValues(
            insets = LocalWindowInsets.current.ime,
            applyStart = true,
            applyEnd = true,
            applyBottom = true
        )
    )
}

/**
 * Selectively apply additional space which matches the width/height of any display cutout present
 * on the respective edges of the screen.
 *
 * @param start Whether to apply padding to the start edge, which matches the display cutout width
 * (if present) on the start edge of the screen. Defaults to `true`.
 * @param top Whether to apply padding to the top edge, which matches the display cutout height
 * (if present) at the top edge of the screen. Defaults to `true`.
 * @param end Whether to apply padding to the end edge, which matches the display cutout width
 * (if present) on the end edge of the screen. Defaults to `true`.
 * @param bottom Whether to apply padding to the bottom edge, which matches the display cutout
 * height (if present) at the bottom edge of the screen. Defaults to `true`.
 */
@Deprecated(
"""
accompanist/insets is deprecated.
For more migration information, please visit https://google.github.io/accompanist/insets/#migration
""",
    replaceWith = ReplaceWith(
        "cutoutPadding()",
        "androidx.compose.foundation.layout.cutoutPadding"
    )
)
inline fun Modifier.cutoutPadding(
    start: Boolean = true,
    top: Boolean = true,
    end: Boolean = true,
    bottom: Boolean = true,
): Modifier = composed {
    padding(
        rememberInsetsPaddingValues(
            insets = LocalWindowInsets.current.displayCutout,
            applyStart = start,
            applyTop = top,
            applyEnd = end,
            applyBottom = bottom
        )
    )
}

/**
 * Apply additional space which matches the height of the [WindowInsets.ime] (on-screen keyboard)
 * height and [WindowInsets.navigationBars]. This is what apps should use to handle any insets
 * at the bottom of the screen.
 */
@Deprecated(
"""
accompanist/insets is deprecated.
For more migration information, please visit https://google.github.io/accompanist/insets/#migration
""",
    replaceWith = ReplaceWith(
        "navigationBarsPadding().imePadding()",
        "androidx.compose.foundation.layout.navigationBarsPadding",
        "androidx.compose.foundation.layout.imePadding"
    )
)
inline fun Modifier.navigationBarsWithImePadding(): Modifier = composed {
    val ime = LocalWindowInsets.current.ime
    val navBars = LocalWindowInsets.current.navigationBars
    val insets = remember(ime, navBars) { derivedWindowInsetsTypeOf(ime, navBars) }
    padding(
        rememberInsetsPaddingValues(
            insets = insets,
            applyStart = true,
            applyEnd = true,
            applyBottom = true
        )
    )
}

/**
 * Returns the current insets converted into a [PaddingValues].
 *
 * @param start Whether to apply the inset on the start dimension.
 * @param top Whether to apply the inset on the top dimension.
 * @param end Whether to apply the inset on the end dimension.
 * @param bottom Whether to apply the inset on the bottom dimension.
 * @param additionalHorizontal Value to add to the start and end dimensions.
 * @param additionalVertical Value to add to the top and bottom dimensions.
 */
@Deprecated(
"""
accompanist/insets is deprecated.
For more migration information, please visit https://google.github.io/accompanist/insets/#migration
""",
    ReplaceWith(
        """rememberInsetsPaddingValues(
            insets = this,
            applyStart = start,
            applyTop = top,
            applyEnd = end,
            applyBottom = bottom,
            additionalStart = additionalHorizontal,
            additionalTop = additionalVertical,
            additionalEnd = additionalHorizontal,
            additionalBottom = additionalVertical
        )""",
        "com.google.accompanist.insets.rememberInsetsPaddingValues"
    )
)
@Composable
inline fun WindowInsets.Type.toPaddingValues(
    start: Boolean = true,
    top: Boolean = true,
    end: Boolean = true,
    bottom: Boolean = true,
    additionalHorizontal: Dp = 0.dp,
    additionalVertical: Dp = 0.dp,
): PaddingValues = rememberInsetsPaddingValues(
    insets = this,
    applyStart = start,
    applyTop = top,
    applyEnd = end,
    applyBottom = bottom,
    additionalStart = additionalHorizontal,
    additionalTop = additionalVertical,
    additionalEnd = additionalHorizontal,
    additionalBottom = additionalVertical
)

/**
 * Returns the current insets converted into a [PaddingValues].
 *
 * @param start Whether to apply the inset on the start dimension.
 * @param top Whether to apply the inset on the top dimension.
 * @param end Whether to apply the inset on the end dimension.
 * @param bottom Whether to apply the inset on the bottom dimension.
 * @param additionalStart Value to add to the start dimension.
 * @param additionalTop Value to add to the top dimension.
 * @param additionalEnd Value to add to the end dimension.
 * @param additionalBottom Value to add to the bottom dimension.
 */
@Deprecated(
"""
accompanist/insets is deprecated.
For more migration information, please visit https://google.github.io/accompanist/insets/#migration
""",
    ReplaceWith(
        """rememberInsetsPaddingValues(
            insets = this,
            applyStart = start,
            applyTop = top,
            applyEnd = end,
            applyBottom = bottom,
            additionalStart = additionalStart,
            additionalTop = additionalTop,
            additionalEnd = additionalEnd,
            additionalBottom = additionalBottom
        )""",
        "com.google.accompanist.insets.rememberInsetsPaddingValues"
    )
)
@Composable
inline fun WindowInsets.Type.toPaddingValues(
    start: Boolean = true,
    top: Boolean = true,
    end: Boolean = true,
    bottom: Boolean = true,
    additionalStart: Dp = 0.dp,
    additionalTop: Dp = 0.dp,
    additionalEnd: Dp = 0.dp,
    additionalBottom: Dp = 0.dp,
): PaddingValues = rememberInsetsPaddingValues(
    insets = this,
    applyStart = start,
    applyTop = top,
    applyEnd = end,
    applyBottom = bottom,
    additionalStart = additionalStart,
    additionalTop = additionalTop,
    additionalEnd = additionalEnd,
    additionalBottom = additionalBottom
)

/**
 * Returns the provided insets [insets] in the form of a [PaddingValues] instance.
 *
 * The returned instance is stable, meaning that any changes to the inset values will result in
 * composition being notified, and these padding values being re-read.
 *
 * @param insets The [WindowInsets.Type] to read from.
 * @param applyStart Whether to apply the inset on the start dimension. Defaults to true.
 * @param applyTop Whether to apply the inset on the top dimension. Defaults to true.
 * @param applyEnd Whether to apply the inset on the end dimension. Defaults to true.
 * @param applyBottom Whether to apply the inset on the bottom dimension. Defaults to true.
 * @param additionalStart Value to add to the start dimension.
 * @param additionalTop Value to add to the top dimension.
 * @param additionalEnd Value to add to the end dimension.
 * @param additionalBottom Value to add to the bottom dimension.
 */
@Composable
@Deprecated(
"""
accompanist/insets is deprecated.
For more migration information, please visit https://google.github.io/accompanist/insets/#migration
""",
    replaceWith = ReplaceWith(
        "insets.asPaddingValues()",
        "androidx.compose.foundation.layout.asPaddingValues"
    )
)
fun rememberInsetsPaddingValues(
    insets: Insets,
    applyStart: Boolean = true,
    applyTop: Boolean = true,
    applyEnd: Boolean = true,
    applyBottom: Boolean = true,
    additionalStart: Dp = 0.dp,
    additionalTop: Dp = 0.dp,
    additionalEnd: Dp = 0.dp,
    additionalBottom: Dp = 0.dp,
): PaddingValues {
    val density = LocalDensity.current

    return remember(density, insets) {
        InsetsPaddingValues(insets = insets, density = density)
    }.apply {
        this.applyStart = applyStart
        this.applyTop = applyTop
        this.applyEnd = applyEnd
        this.applyBottom = applyBottom

        this.additionalStart = additionalStart
        this.additionalTop = additionalTop
        this.additionalEnd = additionalEnd
        this.additionalBottom = additionalBottom
    }
}

/**
 * Our [PaddingValues] implementation which is state-backed, reading values from [insets] as
 * appropriate. Can be created via [rememberInsetsPaddingValues].
 */
@Stable
internal class InsetsPaddingValues(
    private val insets: Insets,
    private val density: Density,
) : PaddingValues {
    var applyStart: Boolean by mutableStateOf(false)
    var applyTop: Boolean by mutableStateOf(false)
    var applyEnd: Boolean by mutableStateOf(false)
    var applyBottom: Boolean by mutableStateOf(false)

    var additionalStart: Dp by mutableStateOf(0.dp)
    var additionalTop: Dp by mutableStateOf(0.dp)
    var additionalEnd: Dp by mutableStateOf(0.dp)
    var additionalBottom: Dp by mutableStateOf(0.dp)

    override fun calculateLeftPadding(layoutDirection: LayoutDirection): Dp {
        return when (layoutDirection) {
            LayoutDirection.Ltr -> {
                additionalStart + if (applyStart) with(density) { insets.left.toDp() } else 0.dp
            }
            LayoutDirection.Rtl -> {
                additionalEnd + if (applyEnd) with(density) { insets.left.toDp() } else 0.dp
            }
        }
    }

    override fun calculateTopPadding(): Dp = additionalTop + when {
        applyTop -> with(density) { insets.top.toDp() }
        else -> 0.dp
    }

    override fun calculateRightPadding(layoutDirection: LayoutDirection): Dp {
        return when (layoutDirection) {
            LayoutDirection.Ltr -> {
                additionalEnd + if (applyEnd) with(density) { insets.right.toDp() } else 0.dp
            }
            LayoutDirection.Rtl -> {
                additionalStart + if (applyStart) with(density) { insets.right.toDp() } else 0.dp
            }
        }
    }

    override fun calculateBottomPadding(): Dp = additionalBottom + when {
        applyBottom -> with(density) { insets.bottom.toDp() }
        else -> 0.dp
    }
}
