// Copyright 2020 Google LLC
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//     http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.

package com.google.ads.googleads.lib.stubs.annotations;

import com.google.ads.googleads.lib.stubs.exceptions.BaseGoogleAdsException;
import java.lang.annotation.ElementType;
import java.lang.annotation.Retention;
import java.lang.annotation.RetentionPolicy;
import java.lang.annotation.Target;

/**
 * Specifies that a class is a factory of service clients for a version of the Google Ads API. All
 * methods in the class should be annotated with @ServiceClientDescriptor.
 */
@Retention(RetentionPolicy.RUNTIME)
@Target({ElementType.TYPE, ElementType.METHOD})
public @interface VersionDescriptor {

  /** Specifies the name of the version, as in v[0-9]+ */
  String versionName();

  /** Specifies the factory to use for creating GoogleAdsException instances for this version. */
  Class<? extends BaseGoogleAdsException.Factory> googleAdsExceptionFactory();

  /**
   * Specifies a virtual namespace for the files generated by the annotation processor. The
   * catalogName will be appended to each of the generated files (e.g. GeneratedCatalog_v1_v2 for a
   * catalogName equal to "v1_v2") unless the catalogName is an empty string, in which case, nothing
   * will be appended to the generated files.
   *
   * <p>This is necessary to write tests against the files generated by the annotation processor
   * that include versionName parameters that are independent of the existing release of the
   * library.
   *
   * <p>Should be formatted as a list of the versions included in that given catalog, ordered from
   * least to greatest and joined by an underscore. For example, a catalog that includes only v1
   * should equal "v1". A catalog that includes v1 and v2 should equal v1_v2. A catalog that
   * includes v1, v2, and v3 should equal v1_v2_v3. CatalogNames need not be formatted in this way
   * and can be any arbitrary string. However, formatting in this manner is the convention adopted
   * by this library.
   */
  String catalogName() default "";
}
