/*
 * Copyright 2021 Google LLC
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     https://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.google.appengine.api.datastore;

import com.google.appengine.api.datastore.Query.FilterPredicate;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;

/**
 * This class determines the manner in which multiple queries are generated.
 *
 * @see MultiQueryBuilder
 */
class MultiQueryComponent implements Comparable<MultiQueryComponent> {
  public enum Order {
    SERIAL,
    PARALLEL
  }

  private final Order order;
  private final List<List<FilterPredicate>> filters;

  public MultiQueryComponent(Order order) {
    this.order = order;
    this.filters = new ArrayList<List<FilterPredicate>>();
  }

  public MultiQueryComponent(Order order, List<List<FilterPredicate>> filters) {
    this.order = order;
    this.filters = filters;
  }

  public Order getOrder() {
    return order;
  }

  /**
   * Adds a set of filters that are then applied to {@link Query}s generated by {@link
   * MultiQueryBuilder} (in order).
   */
  public void addFilters(FilterPredicate... filters) {
    this.filters.add(Arrays.asList(filters));
  }

  public List<List<FilterPredicate>> getFilters() {
    return filters;
  }

  @Override
  public int compareTo(MultiQueryComponent o) {
    return order.compareTo(o.order);
  }

  @Override
  public String toString() {
    return "MultiQueryComponent [filters=" + filters + ", order=" + order + "]";
  }
}
