/*
 * Decompiled with CFR 0.152.
 */
package com.google.appengine.api.search.dev;

import com.google.appengine.api.search.GeoPoint;
import com.google.apphosting.api.search.DocumentPb;
import java.text.DateFormat;
import java.text.NumberFormat;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.Locale;
import java.util.TimeZone;
import java.util.logging.Level;
import java.util.logging.Logger;
import org.apache.lucene.document.AbstractField;
import org.apache.lucene.document.Document;
import org.apache.lucene.document.Field;
import org.apache.lucene.document.Fieldable;
import org.apache.lucene.document.NumericField;
import org.apache.lucene.index.Term;
import org.apache.lucene.search.Query;
import org.apache.lucene.search.TermQuery;
import org.jsoup.Jsoup;

final class LuceneUtils {
    public static final String FIELDLESS_FIELD_NAME = "_GLOBAL";
    static final String DOCID_FIELD_NAME = "_DOCID";
    private static final String ALLDOCS_FIELD_NAME = "_ALLDOC";
    private static final String LOCALE_FIELD_NAME = "_LOCALE";
    private static final String ALLDOCS_FIELD_VALUE = "X";
    private static final String ALLDOCS_FIELD_MISSING_VALUE = "Y";
    static final String ORDER_ID_FIELD_NAME = "_ORDERID";
    private static final Logger LOG = Logger.getLogger(LuceneUtils.class.getName());
    private static final String CONVERTED_HTML_TYPE = "HTML2TEXT";
    private static final ThreadLocal<TimeZone> UTC_TZ = new ThreadLocal<TimeZone>(){

        @Override
        protected TimeZone initialValue() {
            return TimeZone.getTimeZone("UTC");
        }
    };
    private static final ThreadLocal<DateFormat> ISO8601_SIMPLE = new ThreadLocal<DateFormat>(){

        @Override
        protected DateFormat initialValue() {
            return LuceneUtils.getDateFormat("yyyy-MM-dd");
        }
    };

    LuceneUtils() {
    }

    private static DateFormat getDateFormat(String formatString) {
        SimpleDateFormat format = new SimpleDateFormat(formatString, Locale.US);
        format.setTimeZone(UTC_TZ.get());
        return format;
    }

    public static String makeLuceneFieldName(DocumentPb.Field field) {
        return LuceneUtils.makeLuceneFieldName(field.getName(), field.getValue().getType());
    }

    public static String makeLuceneFieldName(String name, DocumentPb.FieldValue.ContentType type) {
        return type + "@" + name;
    }

    public static String makeLuceneFieldNameWithExtractedText(String name, DocumentPb.FieldValue.ContentType type) {
        if (type == DocumentPb.FieldValue.ContentType.HTML) {
            return "HTML2TEXT@" + name;
        }
        return LuceneUtils.makeLuceneFieldName(name, type);
    }

    public static String makeLuceneFieldNameWithExtractedText(DocumentPb.Field field) {
        return LuceneUtils.makeLuceneFieldNameWithExtractedText(field.getName(), field.getValue().getType());
    }

    public static String extractTextFromHtml(String html) {
        return Jsoup.parse(html).body().text();
    }

    public static double stringValueToDouble(String value) {
        try {
            return NumberFormat.getNumberInstance().parse(value).doubleValue();
        }
        catch (ParseException e) {
            throw new RuntimeException(e);
        }
    }

    public static LuceneFieldName splitLuceneFieldName(String field) {
        DocumentPb.FieldValue.ContentType typeEnum;
        int colon = field.indexOf("@");
        if (colon == -1) {
            LOG.severe("Misformed field name encountered " + field);
            return null;
        }
        String type = field.substring(0, colon);
        if (CONVERTED_HTML_TYPE.equals(type)) {
            return null;
        }
        String name = field.substring(colon + 1);
        try {
            typeEnum = DocumentPb.FieldValue.ContentType.valueOf((String)type);
        }
        catch (IllegalArgumentException e) {
            LOG.log(Level.SEVERE, "Failed to convert '" + name + "' extracted from '" + field + "' to ContentType", e);
            typeEnum = DocumentPb.FieldValue.ContentType.TEXT;
        }
        return new LuceneFieldName(name, typeEnum);
    }

    public static Document toLuceneDocument(String docId, DocumentPb.Document input) {
        Document output = new Document();
        if (input.hasLanguage()) {
            output.add(new Field(LOCALE_FIELD_NAME, input.getLanguage(), Field.Store.YES, Field.Index.NO));
        }
        for (DocumentPb.Field field : input.getFieldList()) {
            DocumentPb.FieldValue fieldValue = field.getValue();
            Field.Index indexStrategy = Field.Index.ANALYZED;
            String fieldName = LuceneUtils.makeLuceneFieldName(field);
            String value = fieldValue.getStringValue();
            AbstractField luceneField = null;
            switch (fieldValue.getType()) {
                case HTML: {
                    output.add(new Field(fieldName, value, Field.Store.YES, Field.Index.NOT_ANALYZED));
                    value = LuceneUtils.extractTextFromHtml(value);
                    luceneField = new Field(LuceneUtils.makeLuceneFieldNameWithExtractedText(field), value, Field.Store.YES, Field.Index.ANALYZED);
                    break;
                }
                case TEXT: {
                    break;
                }
                case ATOM: {
                    indexStrategy = Field.Index.NOT_ANALYZED;
                    value = value.toLowerCase();
                    break;
                }
                case DATE: {
                    NumericField dateField = new NumericField(fieldName, Field.Store.YES, true);
                    try {
                        dateField.setLongValue(LuceneUtils.dateStringToLong(value));
                    }
                    catch (ParseException e) {
                        LOG.warning("Failed to parse date for " + fieldName + ": " + value);
                        dateField.setLongValue(0L);
                    }
                    luceneField = dateField;
                    indexStrategy = Field.Index.NOT_ANALYZED;
                    break;
                }
                case NUMBER: {
                    NumericField numericField = new NumericField(fieldName, Field.Store.YES, true);
                    numericField.setDoubleValue(LuceneUtils.stringValueToDouble(value));
                    luceneField = numericField;
                    indexStrategy = Field.Index.NOT_ANALYZED;
                    break;
                }
                case GEO: {
                    LOG.warning("GeoPoints are not supported on dev server");
                    break;
                }
                default: {
                    throw new IllegalArgumentException("Field type " + fieldValue.getType() + " not handled");
                }
            }
            if (luceneField == null) {
                luceneField = new Field(fieldName, value, Field.Store.YES, indexStrategy);
            }
            output.add(luceneField);
            output.add(new Field(FIELDLESS_FIELD_NAME, value, Field.Store.NO, indexStrategy));
        }
        output.add(new Field(ALLDOCS_FIELD_NAME, ALLDOCS_FIELD_VALUE, Field.Store.NO, Field.Index.NOT_ANALYZED));
        output.add(new Field(DOCID_FIELD_NAME, docId, Field.Store.YES, Field.Index.NOT_ANALYZED));
        output.add(new Field(ORDER_ID_FIELD_NAME, Integer.toString(input.getOrderId()), Field.Store.YES, Field.Index.NOT_ANALYZED));
        return output;
    }

    public static Long dateStringToLong(String value) throws ParseException {
        try {
            return Long.parseLong(value);
        }
        catch (IllegalArgumentException exception) {
            return ISO8601_SIMPLE.get().parse(value).getTime();
        }
    }

    public static boolean isDateString(String value) {
        try {
            ISO8601_SIMPLE.get().parse(value);
            return true;
        }
        catch (ParseException e) {
            return false;
        }
    }

    public static double numericFieldToDouble(Fieldable f) {
        if (f instanceof NumericField) {
            NumericField numericField = (NumericField)f;
            return numericField.getNumericValue().doubleValue();
        }
        return Double.parseDouble(f.stringValue());
    }

    public static Object luceneFieldToValue(Fieldable f, DocumentPb.FieldValue.ContentType type) {
        switch (type) {
            case HTML: 
            case TEXT: 
            case ATOM: {
                return ((Field)f).stringValue();
            }
            case DATE: {
                long value = f instanceof NumericField ? ((NumericField)f).getNumericValue().longValue() : Long.parseLong(f.stringValue());
                return Long.toString(value);
            }
            case NUMBER: {
                if (f instanceof NumericField) {
                    NumericField numericField = (NumericField)f;
                    return Double.toString(numericField.getNumericValue().doubleValue());
                }
                return f.stringValue();
            }
            case GEO: {
                LOG.warning("GeoPoints are not implemented on dev server.");
                return "";
            }
        }
        throw new IllegalArgumentException("Failed to correctly handle type " + type);
    }

    private static DocumentPb.Document.Builder toAppengineDocumentIdBuilder(Document d) {
        String docId = ((Field)d.getFieldable(DOCID_FIELD_NAME)).stringValue();
        DocumentPb.Document.Builder docBuilder = DocumentPb.Document.newBuilder();
        docBuilder.setId(docId);
        return docBuilder;
    }

    public static DocumentPb.Document toAppengineDocumentId(Document d) {
        return LuceneUtils.toAppengineDocumentIdBuilder(d).build();
    }

    public static DocumentPb.Document toAppengineDocument(Document d) {
        DocumentPb.Document.Builder docBuilder = LuceneUtils.toAppengineDocumentIdBuilder(d);
        String orderIdStr = ((Field)d.getFieldable(ORDER_ID_FIELD_NAME)).stringValue();
        docBuilder.setOrderId(Integer.parseInt(orderIdStr));
        Field localeField = (Field)d.getFieldable(LOCALE_FIELD_NAME);
        if (localeField != null) {
            docBuilder.setLanguage(localeField.stringValue());
        }
        for (Fieldable f : d.getFields()) {
            if (!LuceneUtils.isPlainField(f.name())) continue;
            DocumentPb.FieldValue.Builder valueBuilder = DocumentPb.FieldValue.newBuilder();
            LuceneFieldName fieldName = LuceneUtils.splitLuceneFieldName(f.name());
            if (fieldName == null) {
                LOG.warning("Unable to find type for " + f.name() + "; ignored");
                continue;
            }
            Object value = LuceneUtils.luceneFieldToValue(f, fieldName.type);
            if (value instanceof String) {
                valueBuilder.setStringValue(String.valueOf(value));
            } else if (value instanceof GeoPoint) {
                GeoPoint point = (GeoPoint)value;
                valueBuilder.setGeo(DocumentPb.FieldValue.Geo.newBuilder().setLat(point.getLatitude()).setLng(point.getLongitude()));
            }
            valueBuilder.setType(fieldName.type);
            docBuilder.addField(DocumentPb.Field.newBuilder().setName(fieldName.name).setValue(valueBuilder));
        }
        return docBuilder.build();
    }

    private static boolean isPlainField(String fieldName) {
        return !DOCID_FIELD_NAME.equals(fieldName) && !FIELDLESS_FIELD_NAME.equals(fieldName) && !ALLDOCS_FIELD_NAME.equals(fieldName) && !ORDER_ID_FIELD_NAME.equals(fieldName) && !LOCALE_FIELD_NAME.equals(fieldName);
    }

    public static Query getMatchAnyDocumentQuery() {
        return new TermQuery(new Term(ALLDOCS_FIELD_NAME, ALLDOCS_FIELD_VALUE));
    }

    public static Query getMatchNoneQuery() {
        return new TermQuery(new Term(ALLDOCS_FIELD_NAME, ALLDOCS_FIELD_MISSING_VALUE));
    }

    public static Term newDeleteTerm(String docId) {
        return new Term(DOCID_FIELD_NAME, docId);
    }

    public static class LuceneFieldName {
        public final String name;
        public final DocumentPb.FieldValue.ContentType type;

        LuceneFieldName(String name, DocumentPb.FieldValue.ContentType type) {
            this.name = name;
            this.type = type;
        }
    }
}

