/*
 * Decompiled with CFR 0.152.
 */
package com.google.cloud.datastore.core.rep.converter;

import com.google.appengine.repackaged.com.google.common.base.Utf8;
import com.google.appengine.repackaged.com.google.common.io.BaseEncoding;
import com.google.appengine.repackaged.com.google.protobuf.Timestamp;
import com.google.appengine.repackaged.com.google.protobuf.util.Timestamps;
import com.google.cloud.datastore.core.exception.InvalidConversionException;
import com.google.cloud.datastore.core.rep.Value;
import com.google.cloud.datastore.logs.ProblemCode;
import java.nio.charset.StandardCharsets;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import javax.annotation.Nullable;

public final class ConverterHelper {
    private static final int NANOSECONDS_PER_MICROSECOND = 1000;
    private static final String NON_UTF8_BYTES_STRING_PREFIX = "__bytes";
    private static final Pattern NON_UTF8_BYTES_STRING_REGEX = Pattern.compile("__bytes(.*)__", 32);
    private static final String NON_UTF8_BYTES_STRING_FORMAT = "__bytes%s__";

    private ConverterHelper() {
    }

    public static String convertUtf8(String description, String descriptionSuffix, byte[] byteArray) throws InvalidConversionException {
        InvalidConversionException.checkConversion(Utf8.isWellFormed((byte[])byteArray), ProblemCode.PROPERTY_NAME_NOT_UTF8, "%s%s is not valid UTF-8.", description, descriptionSuffix);
        return new String(byteArray, StandardCharsets.UTF_8);
    }

    public static String convertBytesToString(String description, byte[] byteArray) throws InvalidConversionException {
        if (Utf8.isWellFormed((byte[])byteArray)) {
            String string = new String(byteArray, StandardCharsets.UTF_8);
            InvalidConversionException.checkConversion(!string.startsWith(NON_UTF8_BYTES_STRING_PREFIX) || !NON_UTF8_BYTES_STRING_REGEX.matcher(string).matches(), "Invalid %s \"%s\".", description, string);
            return string;
        }
        return String.format(NON_UTF8_BYTES_STRING_FORMAT, BaseEncoding.base64Url().encode(byteArray));
    }

    public static String convertBytesToStringUnchecked(byte[] byteArray) {
        if (Utf8.isWellFormed((byte[])byteArray)) {
            return new String(byteArray, StandardCharsets.UTF_8);
        }
        return String.format(NON_UTF8_BYTES_STRING_FORMAT, BaseEncoding.base64Url().encode(byteArray));
    }

    public static byte[] convertStringToBytes(String description, String string) throws InvalidConversionException {
        byte[] bytes = ConverterHelper.convertStringToBytesIfNonUtf8(description, string);
        if (bytes != null) {
            return bytes;
        }
        return string.getBytes(StandardCharsets.UTF_8);
    }

    public static byte[] convertStringToBytesUnchecked(String description, String string) {
        try {
            return ConverterHelper.convertStringToBytes(description, string);
        }
        catch (InvalidConversionException e) {
            throw new IllegalArgumentException(e);
        }
    }

    @Nullable
    public static byte[] convertStringToBytesIfNonUtf8(String description, String string) throws InvalidConversionException {
        byte[] byteArray;
        if (!string.startsWith(NON_UTF8_BYTES_STRING_PREFIX)) {
            return null;
        }
        Matcher patternMatch = NON_UTF8_BYTES_STRING_REGEX.matcher(string);
        if (!patternMatch.matches()) {
            return null;
        }
        String encodedText = patternMatch.group(1);
        try {
            byteArray = BaseEncoding.base64Url().decode((CharSequence)encodedText);
        }
        catch (IllegalArgumentException exception) {
            throw new InvalidConversionException(String.format("Invalid %s \"%s\": not base64 encoded.", description, string), exception);
        }
        InvalidConversionException.checkConversion(!Utf8.isWellFormed((byte[])byteArray), "Invalid %s \"%s\": valid UTF-8 bytes encoded as non-UTF-8.", description, string);
        return byteArray;
    }

    public static String convertStringLossyUnchecked(String string) {
        try {
            byte[] bytes = ConverterHelper.convertStringToBytesIfNonUtf8("", string);
            if (bytes == null) {
                return string;
            }
            return new String(bytes, StandardCharsets.UTF_8);
        }
        catch (InvalidConversionException e) {
            throw new RuntimeException(e);
        }
    }

    public static Timestamp microsecondsToTimestamp(long microseconds) throws InvalidConversionException {
        long seconds = microseconds / 1000000L;
        int microsecondsRemainder = (int)(microseconds % 1000000L);
        if (microsecondsRemainder < 0) {
            microsecondsRemainder += 1000000;
            --seconds;
        }
        Timestamp timestamp = Timestamp.newBuilder().setSeconds(seconds).setNanos(microsecondsRemainder * 1000).build();
        try {
            Timestamps.checkValid((Timestamp)timestamp);
        }
        catch (IllegalArgumentException exception) {
            throw new InvalidConversionException(exception);
        }
        return timestamp;
    }

    public static long timestampToMicrosecondsLossy(Timestamp timestamp) {
        long nanoseconds = timestamp.getNanos();
        return timestamp.getSeconds() * 1000000L + nanoseconds / 1000L;
    }

    public static long timestampsToMicroseconds(Timestamp timestamp) throws InvalidConversionException {
        InvalidConversionException.checkConversion(Timestamps.isValid((Timestamp)timestamp), "invalid timestamp", new Object[0]);
        InvalidConversionException.checkConversion(timestamp.getNanos() % 1000 == 0, "timestamp cannot have more than microseconds precision", new Object[0]);
        InvalidConversionException.checkConversion(timestamp.getSeconds() >= 0L, "Only timestamps past epoch are supported.", new Object[0]);
        return ConverterHelper.timestampToMicrosecondsLossy(timestamp);
    }

    public static Value valueAddLongMeaning(Value value, long longMeaning) throws InvalidConversionException {
        Value.Meaning meaning = Value.Meaning.createFromOrdinal(longMeaning);
        InvalidConversionException.checkConversion(meaning != null, "Meaning value \"%d\" is not valid.", longMeaning);
        Value resultValue = value.withMeaningIfValid(meaning);
        InvalidConversionException.checkConversion(resultValue != null, "Meaning \"%d\" is not valid for type \"%s\".", new Object[]{longMeaning, value.type()});
        return resultValue;
    }
}

