/*
 * Decompiled with CFR 0.152.
 */
package com.google.common.flags.ext;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.CharMatcher;
import com.google.common.base.Converter;
import com.google.common.base.MoreObjects;
import com.google.common.base.Preconditions;
import com.google.common.base.Predicate;
import com.google.common.base.Verify;
import com.google.common.collect.ImmutableMap;
import com.google.common.flags.Flag;
import com.google.common.flags.InvalidFlagValueException;
import com.google.common.flags.ext.FlagBuilder;
import com.google.common.time.Durations;
import java.time.Duration;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import javax.annotation.Nullable;

public final class DurationFlag
extends Flag<Duration> {
    public static final String FORMAT_DESCRIPTION = "Days (d), hours (h), minutes (m), seconds (s), milliseconds (ms), microseconds (us), and nanoseconds (ns) can be specified using the appropriate suffix (e.g. 1h5m500ms = 1 hour 5 minutes 500 millis).";
    private static final ImmutableMap<String, Duration> ABBREVIATION_TO_DURATION = new ImmutableMap.Builder().put((Object)"d", (Object)Duration.ofDays(1L)).put((Object)"h", (Object)Duration.ofHours(1L)).put((Object)"m", (Object)Duration.ofMinutes(1L)).put((Object)"s", (Object)Duration.ofSeconds(1L)).put((Object)"ms", (Object)Duration.ofMillis(1L)).put((Object)"us", (Object)Durations.ofMicros((long)1L)).put((Object)"ns", (Object)Duration.ofNanos(1L)).buildOrThrow();
    private static final Converter<String, Duration> DURATION_CONVERTER = new Converter<String, Duration>(){

        protected Duration doForward(String durationString) {
            return DurationFlag.parseDuration(durationString);
        }

        protected String doBackward(Duration duration) {
            return DurationFlag.formatDuration(duration);
        }
    };
    private static final String UNIT_PATTERN_STRING = "(?x) (?<whole>[0-9]+)? (?<frac>\\.[0-9]*)? (?<unit>d|h|ms?|s|us|ns)";
    private static final Pattern UNIT_PATTERN = Pattern.compile("(?x) (?<whole>[0-9]+)? (?<frac>\\.[0-9]*)? (?<unit>d|h|ms?|s|us|ns)");
    private static final CharMatcher ASCII_DIGIT = CharMatcher.inRange((char)'0', (char)'9');

    public static Converter<String, Duration> converter() {
        return DURATION_CONVERTER;
    }

    public static Flag<Duration> value(Duration defaultValue) {
        return new DurationFlag((Duration)Preconditions.checkNotNull((Object)defaultValue));
    }

    public static Flag<Duration> nullValue() {
        return new DurationFlag(null);
    }

    public static Flag<Duration> zero() {
        return DurationFlag.value(Duration.ZERO);
    }

    public static Flag<Duration> nonNegativeValue(@Nullable Duration defaultValue) {
        return FlagBuilder.newBuilder(DURATION_CONVERTER).addConstraint((Predicate<Duration>)((Predicate)d -> !d.isNegative()), "duration must not be negative").withDefaultValue(defaultValue).build();
    }

    public static Flag<Duration> positiveValue(@Nullable Duration defaultValue) {
        return FlagBuilder.newBuilder(DURATION_CONVERTER).addConstraint((Predicate<Duration>)((Predicate)d -> !d.isNegative() && !d.isZero()), "duration must be positive").withDefaultValue(defaultValue).build();
    }

    public static Flag<Duration> interval(@Nullable Duration defaultValue, Duration min, Duration max) {
        Preconditions.checkNotNull((Object)min, (Object)"minimum value cannot be null");
        Preconditions.checkNotNull((Object)max, (Object)"maximum value cannot be null");
        Preconditions.checkArgument((min.compareTo(max) <= 0 ? 1 : 0) != 0, (String)"the minimum value (%s) must be <= to the maximum value (%s)", (Object)min, (Object)max);
        return FlagBuilder.newBuilder(DURATION_CONVERTER).addConstraint((Predicate<Duration>)((Predicate)d -> d.compareTo(min) >= 0), "duration cannot be less than " + min).addConstraint((Predicate<Duration>)((Predicate)d -> d.compareTo(max) <= 0), "duration cannot be greater than " + max).withDefaultValue(defaultValue).build();
    }

    public static Flag<Duration> days(long defaultValue) {
        return DurationFlag.value(Duration.ofDays(defaultValue));
    }

    public static Flag<Duration> hours(long defaultValue) {
        return DurationFlag.value(Duration.ofHours(defaultValue));
    }

    public static Flag<Duration> minutes(long defaultValue) {
        return DurationFlag.value(Duration.ofMinutes(defaultValue));
    }

    public static Flag<Duration> seconds(long defaultValue) {
        return DurationFlag.value(Duration.ofSeconds(defaultValue));
    }

    public static Flag<Duration> millis(long defaultValue) {
        return DurationFlag.value(Duration.ofMillis(defaultValue));
    }

    public static Flag<Duration> micros(long defaultValue) {
        return DurationFlag.value(Durations.ofMicros((long)defaultValue));
    }

    public static Flag<Duration> nanos(long defaultValue) {
        return DurationFlag.value(Duration.ofNanos(defaultValue));
    }

    private DurationFlag(@Nullable Duration defaultValue) {
        super(defaultValue);
    }

    @Override
    protected String parsableStringValue(Duration value) {
        return DurationFlag.formatDuration(value);
    }

    @Override
    protected Duration parse(String text) throws InvalidFlagValueException {
        try {
            return DurationFlag.parseDuration(text);
        }
        catch (IllegalArgumentException e) {
            throw new InvalidFlagValueException("Invalid syntax for Duration", e);
        }
    }

    @VisibleForTesting
    static Duration parseDuration(String value) {
        Preconditions.checkArgument((value != null ? 1 : 0) != 0, (Object)"input value cannot be null");
        Preconditions.checkArgument((!value.isEmpty() ? 1 : 0) != 0, (Object)"input value cannot be empty");
        Preconditions.checkArgument((!value.equals("-") ? 1 : 0) != 0, (Object)"input value cannot be '-'");
        Preconditions.checkArgument((!value.equals("+") ? 1 : 0) != 0, (Object)"input value cannot be '+'");
        value = CharMatcher.whitespace().trimFrom((CharSequence)value);
        if ("0".equals(value)) {
            return Duration.ZERO;
        }
        Duration duration = Duration.ZERO;
        boolean negative = value.startsWith("-");
        boolean explicitlyPositive = value.startsWith("+");
        int index = negative || explicitlyPositive ? 1 : 0;
        Matcher matcher = UNIT_PATTERN.matcher(value);
        while (matcher.find(index) && matcher.start() == index) {
            Preconditions.checkArgument((boolean)ASCII_DIGIT.matchesAnyOf((CharSequence)matcher.group(0)));
            try {
                String unit = matcher.group("unit");
                long whole = Long.parseLong((String)MoreObjects.firstNonNull((Object)matcher.group("whole"), (Object)"0"));
                Duration singleUnit = (Duration)ABBREVIATION_TO_DURATION.get((Object)unit);
                Preconditions.checkArgument((singleUnit != null ? 1 : 0) != 0, (String)"invalid unit (%s)", (Object)unit);
                duration = duration.plus(singleUnit.multipliedBy(whole));
                long nanosPerUnit = singleUnit.toNanos();
                double frac = Double.parseDouble("0" + (String)MoreObjects.firstNonNull((Object)matcher.group("frac"), (Object)""));
                duration = duration.plus(Duration.ofNanos((long)((double)nanosPerUnit * frac)));
            }
            catch (ArithmeticException e) {
                throw new IllegalArgumentException(e);
            }
            index = matcher.end();
        }
        if (index < value.length()) {
            throw new IllegalArgumentException("Could not parse entire duration: " + value);
        }
        if (negative) {
            duration = duration.negated();
        }
        return duration;
    }

    static String formatDuration(Duration duration) {
        long nanos;
        long micros;
        long millis;
        long seconds;
        long minutes;
        long hours;
        long days;
        if (duration.isZero()) {
            return "0";
        }
        StringBuilder sb = new StringBuilder();
        if (duration.isNegative()) {
            sb.append('-');
            duration = duration.negated();
        }
        if ((days = duration.toDays()) > 0L) {
            sb.append(days).append("d");
            duration = duration.minusDays(days);
        }
        if ((hours = duration.toHours()) > 0L) {
            sb.append(hours).append("h");
            duration = duration.minusHours(hours);
        }
        if ((minutes = duration.toMinutes()) > 0L) {
            sb.append(minutes).append("m");
            duration = duration.minusMinutes(minutes);
        }
        if ((seconds = duration.getSeconds()) > 0L) {
            sb.append(seconds).append("s");
            duration = duration.minusSeconds(seconds);
        }
        if ((millis = duration.toMillis()) > 0L) {
            sb.append(millis).append("ms");
            duration = duration.minusMillis(millis);
        }
        if ((micros = Durations.toMicros((Duration)duration)) > 0L) {
            sb.append(micros).append("us");
            duration = duration.minus(Durations.ofMicros((long)micros));
        }
        if ((nanos = duration.toNanos()) > 0L) {
            sb.append(nanos).append("ns");
            duration = duration.minusNanos(nanos);
        }
        Verify.verify((boolean)duration.isZero(), (String)"problem formatting duration. partial format: %s; leftovers: %s", (Object)sb, (Object)duration);
        Verify.verify((sb.length() > 0 ? 1 : 0) != 0, (String)"problem formatting duration. leftovers: %s", (Object)duration);
        return sb.toString();
    }

    public static final class DurationFactory {
        public static Duration valueOf(String durationString) {
            return DurationFlag.parseDuration(durationString);
        }

        private DurationFactory() {
        }
    }
}

