/*
 * Copyright (C) 2012 Google Inc.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.google.caliper.runner.worker;

import com.google.common.base.Joiner;
import com.google.common.base.Splitter;

/**
 * An exception created on the runner with the same stack trace as one thrown on the worker that
 * reports the actual exception class and message in its message.
 */
public final class ProxyWorkerException extends RuntimeException {

  private final String exceptionType;
  private final String message;

  public ProxyWorkerException(String exceptionType, String message, String stackTrace) {
    super(formatMessage(stackTrace));
    this.message = message;
    this.exceptionType = exceptionType;
  }

  /** Returns the type of the original exception. */
  public String exceptionType() {
    return exceptionType;
  }

  /** Returns the message from the original exception. */
  public String message() {
    return message;
  }

  private static String formatMessage(String stackTrace) {
    StringBuilder builder =
        new StringBuilder(stackTrace.length() + 512)
            .append(
                "An exception occurred in a worker process. The stack trace is as follows:\n\t");
    Joiner.on("\n\t").appendTo(builder, Splitter.on('\n').split(stackTrace));
    return builder.toString();
  }
}
