/*
 * Decompiled with CFR 0.152.
 */
package com.google.cloud.hadoop.gcsio;

import com.google.api.client.googleapis.services.AbstractGoogleClientRequest;
import com.google.api.client.http.HttpResponse;
import com.google.api.client.util.BackOff;
import com.google.api.client.util.BackOffUtils;
import com.google.api.client.util.ExponentialBackOff;
import com.google.api.client.util.NanoClock;
import com.google.api.client.util.Sleeper;
import com.google.api.services.storage.Storage;
import com.google.api.services.storage.model.StorageObject;
import com.google.cloud.hadoop.gcsio.GoogleCloudStorageExceptions;
import com.google.cloud.hadoop.gcsio.SeekableReadableByteChannel;
import com.google.cloud.hadoop.gcsio.StorageResourceId;
import com.google.cloud.hadoop.util.ApiErrorExtractor;
import com.google.cloud.hadoop.util.ClientRequestHelper;
import com.google.cloud.hadoop.util.LogUtil;
import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Preconditions;
import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.nio.ByteBuffer;
import java.nio.channels.Channels;
import java.nio.channels.ClosedChannelException;
import java.nio.channels.ReadableByteChannel;
import java.util.regex.Pattern;

public class GoogleCloudStorageReadChannel
implements SeekableReadableByteChannel {
    private static LogUtil log = new LogUtil(GoogleCloudStorageReadChannel.class);
    private static final Pattern SLASH = Pattern.compile("/");
    private Storage gcs;
    private String bucketName;
    private String objectName;
    private ReadableByteChannel readChannel;
    private boolean channelIsOpen;
    private long currentPosition = -1L;
    private boolean lazySeekPending;
    private long size = -1L;
    private int maxRetries = 10;
    private final ApiErrorExtractor errorExtractor;
    private final ClientRequestHelper<StorageObject> clientRequestHelper;
    private Sleeper sleeper = Sleeper.DEFAULT;
    private NanoClock clock = NanoClock.SYSTEM;
    private BackOff backOff = null;
    public static final int DEFAULT_BACKOFF_INITIAL_INTERVAL_MILLIS = 200;
    public static final double DEFAULT_BACKOFF_RANDOMIZATION_FACTOR = 0.5;
    public static final double DEFAULT_BACKOFF_MULTIPLIER = 1.5;
    public static final int DEFAULT_BACKOFF_MAX_INTERVAL_MILLIS = 10000;
    public static final int DEFAULT_BACKOFF_MAX_ELAPSED_TIME_MILLIS = 120000;

    GoogleCloudStorageReadChannel(Storage gcs, String bucketName, String objectName, ApiErrorExtractor errorExtractor, ClientRequestHelper<StorageObject> requestHelper) throws IOException {
        this.gcs = gcs;
        this.clientRequestHelper = requestHelper;
        this.bucketName = bucketName;
        this.objectName = objectName;
        this.errorExtractor = errorExtractor;
        this.channelIsOpen = true;
        this.position(0L);
    }

    @VisibleForTesting
    GoogleCloudStorageReadChannel() throws IOException {
        this.clientRequestHelper = null;
        this.errorExtractor = null;
        this.channelIsOpen = true;
        this.position(0L);
    }

    @VisibleForTesting
    void setSleeper(Sleeper sleeper) {
        Preconditions.checkArgument((sleeper != null ? 1 : 0) != 0, (Object)"sleeper must not be null!");
        this.sleeper = sleeper;
    }

    @VisibleForTesting
    void setNanoClock(NanoClock clock) {
        Preconditions.checkArgument((clock != null ? 1 : 0) != 0, (Object)"clock must not be null!");
        this.clock = clock;
    }

    @VisibleForTesting
    void setBackOff(BackOff backOff) {
        this.backOff = backOff;
    }

    @VisibleForTesting
    BackOff getBackOff() {
        return this.backOff;
    }

    private BackOff createBackOff() {
        return new ExponentialBackOff.Builder().setInitialIntervalMillis(200).setRandomizationFactor(0.5).setMultiplier(1.5).setMaxIntervalMillis(10000).setMaxElapsedTimeMillis(120000).setNanoClock(this.clock).build();
    }

    public void setMaxRetries(int maxRetries) {
        this.maxRetries = maxRetries;
    }

    @Override
    public int read(ByteBuffer buffer) throws IOException {
        this.throwIfNotOpen();
        if (buffer.remaining() == 0) {
            return 0;
        }
        this.performLazySeek();
        int totalBytesRead = 0;
        int retriesAttempted = 0;
        do {
            int remainingBeforeRead = buffer.remaining();
            try {
                int numBytesRead = this.readChannel.read(buffer);
                Preconditions.checkState((numBytesRead != 0 ? 1 : 0) != 0, (Object)"Read 0 bytes without blocking!");
                if (numBytesRead < 0) break;
                totalBytesRead += numBytesRead;
                this.currentPosition += (long)numBytesRead;
                retriesAttempted = 0;
            }
            catch (IOException ioe) {
                if (retriesAttempted == this.maxRetries) {
                    log.error("Already attempted max of %d retries while reading '%s'; throwing exception.", new Object[]{this.maxRetries, StorageResourceId.createReadableString(this.bucketName, this.objectName)});
                    throw ioe;
                }
                if (retriesAttempted == 0) {
                    if (this.backOff == null) {
                        this.backOff = this.createBackOff();
                    } else {
                        this.backOff.reset();
                    }
                }
                log.warn("Got exception: %s while reading '%s'; retry # %d. Sleeping...", new Object[]{ioe.getMessage(), StorageResourceId.createReadableString(this.bucketName, this.objectName), ++retriesAttempted});
                try {
                    boolean backOffSuccessful = BackOffUtils.next((Sleeper)this.sleeper, (BackOff)this.backOff);
                    if (!backOffSuccessful) {
                        log.error("BackOff returned false; maximum total elapsed time exhausted. Giving up after %d retries for '%s'", new Object[]{retriesAttempted, StorageResourceId.createReadableString(this.bucketName, this.objectName)});
                        throw ioe;
                    }
                }
                catch (InterruptedException ie) {
                    log.error("Interrupted while sleeping before retry. Giving up after %d retries for '%s'", new Object[]{retriesAttempted, StorageResourceId.createReadableString(this.bucketName, this.objectName)});
                    ioe.addSuppressed(ie);
                    throw ioe;
                }
                log.info("Done sleeping before retry for '%s'; retry # %d.", new Object[]{StorageResourceId.createReadableString(this.bucketName, this.objectName), retriesAttempted});
                if (buffer.remaining() != remainingBeforeRead) {
                    int partialRead = remainingBeforeRead - buffer.remaining();
                    log.info("Despite exception, had partial read of %d bytes; resetting retry count.", new Object[]{partialRead});
                    retriesAttempted = 0;
                    totalBytesRead += partialRead;
                    this.currentPosition += (long)partialRead;
                }
                long newPosition = this.currentPosition;
                this.currentPosition = -1L;
                this.position(newPosition);
                this.performLazySeek();
            }
        } while (buffer.remaining() > 0);
        return totalBytesRead == 0 ? -1 : totalBytesRead;
    }

    @Override
    public boolean isOpen() {
        return this.channelIsOpen;
    }

    @Override
    public void close() throws IOException {
        this.throwIfNotOpen();
        this.channelIsOpen = false;
        if (this.readChannel != null) {
            this.readChannel.close();
        }
    }

    @Override
    public long position() throws IOException {
        this.throwIfNotOpen();
        return this.currentPosition;
    }

    @Override
    public SeekableReadableByteChannel position(long newPosition) throws IOException {
        this.throwIfNotOpen();
        if (newPosition == this.currentPosition) {
            return this;
        }
        this.validatePosition(newPosition);
        this.currentPosition = newPosition;
        this.lazySeekPending = true;
        return this;
    }

    @Override
    public long size() throws IOException {
        this.throwIfNotOpen();
        this.performLazySeek();
        return this.size;
    }

    protected void setSize(long size) {
        this.size = size;
    }

    protected void validatePosition(long newPosition) {
        if (newPosition < 0L) {
            throw new IllegalArgumentException(String.format("Invalid seek offset: position value (%d) must be >= 0", newPosition));
        }
        if (this.size >= 0L && newPosition >= this.size) {
            throw new IllegalArgumentException(String.format("Invalid seek offset: position value (%d) must be between 0 and %d", newPosition, this.size));
        }
    }

    private void performLazySeek() throws IOException {
        if (!this.lazySeekPending) {
            return;
        }
        if (this.readChannel != null) {
            this.readChannel.close();
        }
        InputStream objectContentStream = this.openStreamAndSetSize(this.currentPosition);
        this.readChannel = Channels.newChannel(objectContentStream);
        this.lazySeekPending = false;
    }

    protected InputStream openStreamAndSetSize(long newPosition) throws IOException {
        HttpResponse response;
        this.validatePosition(newPosition);
        Storage.Objects.Get getObject = this.gcs.objects().get(this.bucketName, this.objectName);
        this.clientRequestHelper.getRequestHeaders((AbstractGoogleClientRequest)getObject).setRange(String.format("bytes=%d-", newPosition));
        try {
            response = getObject.executeMedia();
        }
        catch (IOException e) {
            if (this.errorExtractor.itemNotFound(e)) {
                throw GoogleCloudStorageExceptions.getFileNotFoundException(this.bucketName, this.objectName);
            }
            if (this.errorExtractor.rangeNotSatisfiable(e) && newPosition == 0L && this.size == -1L) {
                log.info("Got 'range not satisfiable' for reading %s at position 0; assuming empty.", new Object[]{StorageResourceId.createReadableString(this.bucketName, this.objectName)});
                this.size = 0L;
                return new ByteArrayInputStream(new byte[0]);
            }
            String msg = String.format("Error reading %s at position %d", StorageResourceId.createReadableString(this.bucketName, this.objectName), newPosition);
            throw new IOException(msg, e);
        }
        String contentRange = response.getHeaders().getContentRange();
        if (response.getHeaders().getContentLength() != null) {
            this.size = response.getHeaders().getContentLength() + newPosition;
        } else if (contentRange != null) {
            String sizeStr = SLASH.split(contentRange)[1];
            try {
                this.size = Long.parseLong(sizeStr);
            }
            catch (NumberFormatException e) {
                String string = String.valueOf(contentRange);
                throw new IOException(string.length() != 0 ? "Could not determine size from response from Content-Range: ".concat(string) : new String("Could not determine size from response from Content-Range: "), e);
            }
        } else {
            throw new IOException("Could not determine size of response");
        }
        return response.getContent();
    }

    private void throwIfNotOpen() throws IOException {
        if (!this.isOpen()) {
            throw new ClosedChannelException();
        }
    }
}

