/*
 * Decompiled with CFR 0.152.
 */
package com.google.cloud.hadoop.fs.gcs;

import com.google.cloud.hadoop.fs.gcs.GoogleHadoopFileSystemBase;
import com.google.cloud.hadoop.gcsio.GoogleCloudStorageReadOptions;
import com.google.common.base.Preconditions;
import java.io.IOException;
import java.net.URI;
import java.nio.ByteBuffer;
import java.nio.channels.ClosedChannelException;
import java.nio.channels.SeekableByteChannel;
import org.apache.hadoop.fs.FSInputStream;
import org.apache.hadoop.fs.FileSystem;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

class GoogleHadoopFSInputStream
extends FSInputStream {
    private static final Logger LOG = LoggerFactory.getLogger(GoogleHadoopFSInputStream.class);
    private GoogleHadoopFileSystemBase ghfs;
    private final SeekableByteChannel channel;
    private ByteBuffer buffer;
    private URI gcsPath;
    private long totalBytesRead;
    private final FileSystem.Statistics statistics;
    private long initTime;
    private final byte[] singleReadBuf = new byte[1];

    GoogleHadoopFSInputStream(GoogleHadoopFileSystemBase ghfs, URI gcsPath, int bufferSize, FileSystem.Statistics statistics) throws IOException {
        LOG.debug("GoogleHadoopFSInputStream({}, {})", (Object)gcsPath, (Object)bufferSize);
        this.ghfs = ghfs;
        this.gcsPath = gcsPath;
        this.statistics = statistics;
        this.initTime = System.nanoTime();
        this.totalBytesRead = 0L;
        boolean enableInternalBuffer = ghfs.getConf().getBoolean("fs.gs.inputstream.internalbuffer.enable", false);
        LOG.debug("enableInternalBuffer: {}", (Object)enableInternalBuffer);
        boolean supportContentEncoding = ghfs.getConf().getBoolean("fs.gs.inputstream.support.content.encoding.enable", true);
        LOG.debug("supportContentEncoding: {}", (Object)supportContentEncoding);
        boolean fastFailOnNotFound = ghfs.getConf().getBoolean("fs.gs.inputstream.fast.fail.on.not.found.enable", true);
        LOG.debug("fastFailOnNotFound: {}", (Object)fastFailOnNotFound);
        long inplaceSeekLimit = ghfs.getConf().getLong("fs.gs.inputstream.inplace.seek.limit", 0x800000L);
        LOG.debug("inplaceSeekLimit: {}", (Object)inplaceSeekLimit);
        GoogleCloudStorageReadOptions.Builder readOptions = new GoogleCloudStorageReadOptions.Builder().setSupportContentEncoding(supportContentEncoding).setFastFailOnNotFound(fastFailOnNotFound).setInplaceSeekLimit(inplaceSeekLimit);
        if (enableInternalBuffer) {
            this.buffer = ByteBuffer.allocate(bufferSize);
            this.buffer.limit(0);
            this.buffer.rewind();
            readOptions.setBufferSize(0);
        } else {
            this.buffer = null;
            readOptions.setBufferSize(bufferSize);
        }
        this.channel = ghfs.getGcsFs().open(gcsPath, readOptions.build());
    }

    public synchronized int read() throws IOException {
        byte b;
        long startTime = System.nanoTime();
        if (this.buffer == null) {
            int numRead = this.channel.read(ByteBuffer.wrap(this.singleReadBuf));
            if (numRead == -1) {
                return -1;
            }
            if (numRead != 1) {
                throw new IOException(String.format("Somehow read %d bytes using single-byte buffer for path %s ending in position %d!", numRead, this.gcsPath, this.channel.position()));
            }
            b = this.singleReadBuf[0];
        } else {
            if (!this.buffer.hasRemaining()) {
                this.buffer.clear();
                int numBytesRead = this.channel.read(this.buffer);
                if (numBytesRead <= 0) {
                    this.buffer.limit(0);
                    this.buffer.rewind();
                    return -1;
                }
                this.buffer.flip();
            }
            b = this.buffer.get();
        }
        ++this.totalBytesRead;
        this.statistics.incrementBytesRead(1L);
        long duration = System.nanoTime() - startTime;
        this.ghfs.increment(GoogleHadoopFileSystemBase.Counter.READ1);
        this.ghfs.increment(GoogleHadoopFileSystemBase.Counter.READ1_TIME, duration);
        return b & 0xFF;
    }

    public int read(byte[] buf, int offset, int length) throws IOException {
        long startTime = System.nanoTime();
        Preconditions.checkNotNull((Object)buf, (Object)"buf must not be null");
        if (offset < 0 || length < 0 || length > buf.length - offset) {
            throw new IndexOutOfBoundsException();
        }
        int numRead = 0;
        if (this.buffer == null) {
            numRead = this.channel.read(ByteBuffer.wrap(buf, offset, length));
        } else {
            while (numRead < length) {
                int needToRead = length - numRead;
                if (this.buffer.remaining() >= needToRead) {
                    this.buffer.get(buf, offset + numRead, needToRead);
                    numRead += needToRead;
                    continue;
                }
                if (this.buffer.hasRemaining()) {
                    int singleRead = this.buffer.remaining();
                    this.buffer.get(buf, offset + numRead, singleRead);
                    numRead += singleRead;
                    continue;
                }
                long channelTime = System.nanoTime();
                this.buffer.clear();
                int numNewBytes = this.channel.read(this.buffer);
                long channelDuration = System.nanoTime() - channelTime;
                this.ghfs.increment(GoogleHadoopFileSystemBase.Counter.READ_FROM_CHANNEL);
                this.ghfs.increment(GoogleHadoopFileSystemBase.Counter.READ_FROM_CHANNEL_TIME, channelDuration);
                if (numNewBytes <= 0) {
                    this.buffer.limit(0);
                    this.buffer.rewind();
                    if (numRead != 0) break;
                    numRead = -1;
                    break;
                }
                this.buffer.flip();
            }
        }
        if (numRead > 0) {
            this.statistics.incrementBytesRead((long)numRead);
            this.totalBytesRead += (long)numRead;
        }
        long duration = System.nanoTime() - startTime;
        this.ghfs.increment(GoogleHadoopFileSystemBase.Counter.READ);
        this.ghfs.increment(GoogleHadoopFileSystemBase.Counter.READ_TIME, duration);
        return numRead;
    }

    public int read(long position, byte[] buf, int offset, int length) throws IOException {
        long startTime = System.nanoTime();
        int result = super.read(position, buf, offset, length);
        long duration = System.nanoTime() - startTime;
        this.ghfs.increment(GoogleHadoopFileSystemBase.Counter.READ_POS);
        this.ghfs.increment(GoogleHadoopFileSystemBase.Counter.READ_POS_TIME, duration);
        return result;
    }

    public synchronized long getPos() throws IOException {
        int bufRemaining = this.buffer == null ? 0 : this.buffer.remaining();
        long pos = this.channel.position() - (long)bufRemaining;
        LOG.debug("getPos: {}", (Object)pos);
        return pos;
    }

    public synchronized void seek(long pos) throws IOException {
        long startTime = System.nanoTime();
        LOG.debug("seek: {}", (Object)pos);
        if (this.buffer == null) {
            try {
                this.channel.position(pos);
            }
            catch (IllegalArgumentException e) {
                throw new IOException(e);
            }
        }
        long curPos = this.getPos();
        if (curPos == pos) {
            LOG.debug("Skipping no-op seek.");
        } else if (pos < curPos && curPos - pos <= (long)this.buffer.position()) {
            long skipBack = curPos - pos;
            int newBufferPosition = this.buffer.position() - (int)skipBack;
            LOG.debug("Skipping backward {} bytes in-place from buffer pos {} to new pos {}", new Object[]{skipBack, this.buffer.position(), newBufferPosition});
            this.buffer.position(newBufferPosition);
        } else if (curPos < pos && pos < this.channel.position()) {
            long skipBytes = pos - curPos;
            Preconditions.checkState((skipBytes < (long)this.buffer.remaining() ? 1 : 0) != 0, (String)"skipBytes (%s) must be less than buffer.remaining() (%s)", (Object[])new Object[]{skipBytes, this.buffer.remaining()});
            int newBufferPosition = this.buffer.position() + (int)skipBytes;
            LOG.debug("Skipping {} bytes in-place from buffer pos {} to new pos {}", new Object[]{skipBytes, this.buffer.position(), newBufferPosition});
            this.buffer.position(newBufferPosition);
        } else {
            LOG.debug("New position '{}' out of range of inplace buffer, with curPos ({}), buffer.position() ({}) and buffer.remaining() ({}).", new Object[]{pos, curPos, this.buffer.position(), this.buffer.remaining()});
            try {
                this.channel.position(pos);
            }
            catch (IllegalArgumentException e) {
                throw new IOException(e);
            }
            this.buffer.limit(0);
            this.buffer.rewind();
        }
        long duration = System.nanoTime() - startTime;
        this.ghfs.increment(GoogleHadoopFileSystemBase.Counter.SEEK);
        this.ghfs.increment(GoogleHadoopFileSystemBase.Counter.SEEK_TIME, duration);
    }

    public synchronized boolean seekToNewSource(long targetPos) throws IOException {
        return false;
    }

    public synchronized void close() throws IOException {
        if (this.channel != null) {
            long startTime = System.nanoTime();
            LOG.debug("close: file: {}, totalBytesRead: {}", (Object)this.gcsPath, (Object)this.totalBytesRead);
            this.channel.close();
            long duration = System.nanoTime() - startTime;
            this.ghfs.increment(GoogleHadoopFileSystemBase.Counter.READ_CLOSE);
            this.ghfs.increment(GoogleHadoopFileSystemBase.Counter.READ_CLOSE_TIME, duration);
            long streamDuration = System.nanoTime() - this.initTime;
            this.ghfs.increment(GoogleHadoopFileSystemBase.Counter.INPUT_STREAM);
            this.ghfs.increment(GoogleHadoopFileSystemBase.Counter.INPUT_STREAM_TIME, streamDuration);
        }
    }

    public boolean markSupported() {
        return false;
    }

    public int available() throws IOException {
        if (!this.channel.isOpen()) {
            throw new ClosedChannelException();
        }
        return super.available();
    }
}

