/*
 * Decompiled with CFR 0.152.
 */
package com.google.cloud.hadoop.fs.gcs;

import com.google.api.client.googleapis.json.GoogleJsonResponseException;
import com.google.cloud.hadoop.fs.gcs.GhfsStatistic;
import com.google.cloud.hadoop.fs.gcs.GhfsStorageStatistics;
import com.google.cloud.hadoop.fs.gcs.GoogleHadoopFileSystem;
import com.google.cloud.hadoop.fs.gcs.GoogleHadoopFileSystemBase;
import com.google.cloud.hadoop.fs.gcs.GoogleHadoopFileSystemConfiguration;
import com.google.cloud.hadoop.fs.gcs.GoogleHadoopFileSystemIntegrationHelper;
import com.google.cloud.hadoop.fs.gcs.TestUtils;
import com.google.cloud.hadoop.gcsio.GoogleCloudStorageFileSystemIntegrationHelper;
import com.google.common.truth.Truth;
import java.net.URI;
import java.nio.charset.StandardCharsets;
import java.time.Duration;
import java.util.Arrays;
import java.util.Random;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.fs.FSDataOutputStream;
import org.apache.hadoop.fs.Path;
import org.junit.AfterClass;
import org.junit.Assert;
import org.junit.BeforeClass;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.junit.runners.JUnit4;

@RunWith(value=JUnit4.class)
public class GoogleHadoopSyncableOutputStreamIntegrationTest {
    private static GoogleCloudStorageFileSystemIntegrationHelper gcsFsIHelper;

    @BeforeClass
    public static void beforeClass() throws Exception {
        gcsFsIHelper = GoogleCloudStorageFileSystemIntegrationHelper.create();
        gcsFsIHelper.beforeAllTests();
    }

    @AfterClass
    public static void afterClass() {
        gcsFsIHelper.afterAllTests();
    }

    private static Configuration getTestConfig() {
        Configuration conf = GoogleHadoopFileSystemIntegrationHelper.getTestConfig();
        conf.setEnum(GoogleHadoopFileSystemConfiguration.GCS_OUTPUT_STREAM_TYPE.getKey(), (Enum)GoogleHadoopFileSystemBase.OutputStreamType.SYNCABLE_COMPOSITE);
        return conf;
    }

    @Test
    public void hsync() throws Exception {
        URI path = gcsFsIHelper.getUniqueObjectUri("hsync");
        Path hadoopPath = new Path(path);
        GoogleHadoopFileSystem fs = GoogleHadoopFileSystemIntegrationHelper.createGhfs(path, GoogleHadoopSyncableOutputStreamIntegrationTest.getTestConfig());
        GhfsStorageStatistics stats = TestUtils.getStorageStatistics();
        byte[] expected = new byte[5];
        new Random().nextBytes(expected);
        try (FSDataOutputStream fout = fs.create(hadoopPath);){
            for (int i = 0; i < expected.length; ++i) {
                fout.write((int)expected[i]);
                fout.hsync();
                int composedLength = i + 1;
                Truth.assertThat((Long)fs.getFileStatus(hadoopPath).getLen()).isEqualTo((Object)composedLength);
                Truth.assertThat((byte[])gcsFsIHelper.readFile(path)).isEqualTo((Object)Arrays.copyOf(expected, composedLength));
            }
        }
        Truth.assertThat((Long)fs.getFileStatus(hadoopPath).getLen()).isEqualTo((Object)expected.length);
        Truth.assertThat((byte[])gcsFsIHelper.readFile(path)).isEqualTo((Object)expected);
        TestUtils.verifyDurationMetric(stats, GhfsStatistic.STREAM_WRITE_CLOSE_OPERATIONS, expected.length + 1);
        TestUtils.verifyDurationMetric(stats, GhfsStatistic.STREAM_WRITE_OPERATIONS, expected.length);
        TestUtils.verifyCounter(stats, GhfsStatistic.STREAM_WRITE_BYTES, expected.length);
        TestUtils.verifyDurationMetric(stats, GhfsStatistic.INVOCATION_HSYNC, expected.length);
    }

    @Test
    public void hsync_noBuffer() throws Exception {
        URI path = gcsFsIHelper.getUniqueObjectUri("hsync_noBuffer");
        Path hadoopPath = new Path(path);
        Configuration config = GoogleHadoopSyncableOutputStreamIntegrationTest.getTestConfig();
        config.setInt(GoogleHadoopFileSystemConfiguration.GCS_OUTPUT_STREAM_BUFFER_SIZE.getKey(), 0);
        GoogleHadoopFileSystem fs = GoogleHadoopFileSystemIntegrationHelper.createGhfs(path, config);
        String line1 = "hello\n";
        byte[] line1Bytes = line1.getBytes(StandardCharsets.UTF_8);
        String line2 = "world\n";
        byte[] line2Bytes = line2.getBytes(StandardCharsets.UTF_8);
        String line3 = "foobar\n";
        byte[] line3Bytes = line3.getBytes(StandardCharsets.UTF_8);
        StringBuilder expected = new StringBuilder();
        try (FSDataOutputStream out = fs.create(hadoopPath);){
            for (byte b : line1Bytes) {
                out.write((int)b);
            }
            expected.append(line1);
            out.hsync();
            String readText = gcsFsIHelper.readTextFile(path);
            Truth.assertWithMessage((String)"Expected line1 after first hsync()").that(readText).isEqualTo((Object)expected.toString());
            out.write(line2Bytes, 0, line2Bytes.length);
            expected.append(line2);
            out.hsync();
            readText = gcsFsIHelper.readTextFile(path);
            Truth.assertWithMessage((String)"Expected line1 + line2 after second sync()").that(readText).isEqualTo((Object)expected.toString());
            out.write(line3Bytes, 0, line3Bytes.length);
            expected.append(line3);
        }
        String readText = gcsFsIHelper.readTextFile(path);
        Truth.assertWithMessage((String)"Expected line1 + line2 + line3 after close()").that(readText).isEqualTo((Object)expected.toString());
    }

    @Test
    public void append_shouldAppendNewData() throws Exception {
        URI path = gcsFsIHelper.getUniqueObjectUri("append_shouldAppendNewData");
        Path hadoopPath = new Path(path);
        gcsFsIHelper.writeTextFile(path.getAuthority(), path.getPath(), "original-content");
        GoogleHadoopFileSystem ghfs = GoogleHadoopFileSystemIntegrationHelper.createGhfs(path, GoogleHadoopSyncableOutputStreamIntegrationTest.getTestConfig());
        try (FSDataOutputStream os = ghfs.append(hadoopPath, 20, () -> {});){
            os.write("_append-1".getBytes(StandardCharsets.UTF_8));
            Truth.assertThat((String)gcsFsIHelper.readTextFile(path)).isEqualTo((Object)"original-content");
            os.hsync();
            Truth.assertThat((String)gcsFsIHelper.readTextFile(path)).isEqualTo((Object)"original-content_append-1");
            os.write("_append-2".getBytes(StandardCharsets.UTF_8));
            os.write("_append-3".getBytes(StandardCharsets.UTF_8));
        }
        String expectedContent = "original-content_append-1_append-2_append-3";
        Truth.assertThat((String)gcsFsIHelper.readTextFile(path)).isEqualTo((Object)expectedContent);
        Truth.assertThat((Long)ghfs.getFileStatus(hadoopPath).getLen()).isEqualTo((Object)expectedContent.length());
    }

    @Test
    public void append_shouldFail_whenFileDoesNotExist() throws Exception {
        URI path = gcsFsIHelper.getUniqueObjectUri("append_shouldFail_whenFileDoesNotExist");
        Path hadoopPath = new Path(path);
        GoogleHadoopFileSystem ghfs = GoogleHadoopFileSystemIntegrationHelper.createGhfs(path, GoogleHadoopSyncableOutputStreamIntegrationTest.getTestConfig());
        FSDataOutputStream fsos = ghfs.append(hadoopPath, 20, () -> {});
        fsos.write("_append-1".getBytes(StandardCharsets.UTF_8));
        Assert.assertThrows(GoogleJsonResponseException.class, () -> ((FSDataOutputStream)fsos).hsync());
        Assert.assertThrows(NullPointerException.class, () -> ((FSDataOutputStream)fsos).close());
        Truth.assertThat((Boolean)ghfs.exists(hadoopPath)).isFalse();
    }

    @Test
    public void hflush_syncsEverything() throws Exception {
        URI path = gcsFsIHelper.getUniqueObjectUri("hflush_syncsEverything");
        Path hadoopPath = new Path(path);
        Configuration config = GoogleHadoopSyncableOutputStreamIntegrationTest.getTestConfig();
        config.setEnum(GoogleHadoopFileSystemConfiguration.GCS_OUTPUT_STREAM_TYPE.getKey(), (Enum)GoogleHadoopFileSystemBase.OutputStreamType.FLUSHABLE_COMPOSITE);
        GoogleHadoopFileSystem ghfs = GoogleHadoopFileSystemIntegrationHelper.createGhfs(path, config);
        GhfsStorageStatistics stats = TestUtils.getStorageStatistics();
        byte[] testData = new byte[5];
        new Random().nextBytes(testData);
        try (FSDataOutputStream out = ghfs.create(hadoopPath);){
            for (int i = 0; i < testData.length; ++i) {
                out.write((int)testData[i]);
                out.hflush();
                int composedLength = i + 1;
                Truth.assertThat((Long)ghfs.getFileStatus(hadoopPath).getLen()).isEqualTo((Object)composedLength);
                Truth.assertThat((byte[])gcsFsIHelper.readFile(path)).isEqualTo((Object)Arrays.copyOf(testData, composedLength));
            }
        }
        Truth.assertThat((Long)ghfs.getFileStatus(hadoopPath).getLen()).isEqualTo((Object)testData.length);
        Truth.assertThat((byte[])gcsFsIHelper.readFile(path)).isEqualTo((Object)testData);
        TestUtils.verifyDurationMetric(stats, GhfsStatistic.STREAM_WRITE_CLOSE_OPERATIONS, testData.length + 1);
        TestUtils.verifyDurationMetric(stats, GhfsStatistic.STREAM_WRITE_OPERATIONS, testData.length);
        TestUtils.verifyCounter(stats, GhfsStatistic.STREAM_WRITE_BYTES, testData.length);
        TestUtils.verifyDurationMetric(stats, GhfsStatistic.INVOCATION_HFLUSH, testData.length);
    }

    @Test
    public void hflush_rateLimited_writesEverything() throws Exception {
        URI path = gcsFsIHelper.getUniqueObjectUri("hflush_syncsEverything");
        Path hadoopPath = new Path(path);
        Configuration config = GoogleHadoopSyncableOutputStreamIntegrationTest.getTestConfig();
        config.setEnum(GoogleHadoopFileSystemConfiguration.GCS_OUTPUT_STREAM_TYPE.getKey(), (Enum)GoogleHadoopFileSystemBase.OutputStreamType.FLUSHABLE_COMPOSITE);
        config.setLong(GoogleHadoopFileSystemConfiguration.GCS_OUTPUT_STREAM_SYNC_MIN_INTERVAL_MS.getKey(), Duration.ofDays(1L).toMillis());
        GoogleHadoopFileSystem ghfs = GoogleHadoopFileSystemIntegrationHelper.createGhfs(path, config);
        byte[] testData = new byte[10];
        new Random().nextBytes(testData);
        try (FSDataOutputStream out = ghfs.create(hadoopPath);){
            for (byte testDataByte : testData) {
                out.write((int)testDataByte);
                out.hflush();
                Truth.assertThat((Long)ghfs.getFileStatus(hadoopPath).getLen()).isEqualTo((Object)1);
                Truth.assertThat((byte[])gcsFsIHelper.readFile(path)).isEqualTo((Object)new byte[]{testData[0]});
            }
        }
        Truth.assertThat((Long)ghfs.getFileStatus(hadoopPath).getLen()).isEqualTo((Object)testData.length);
        Truth.assertThat((byte[])gcsFsIHelper.readFile(path)).isEqualTo((Object)testData);
    }
}

