/*
 * Decompiled with CFR 0.152.
 */
package com.google.cloud.hadoop.fs.gcs;

import com.google.cloud.hadoop.fs.gcs.GhfsStatistic;
import com.google.cloud.hadoop.fs.gcs.GhfsStorageStatistics;
import com.google.cloud.hadoop.fs.gcs.GhfsStreamStats;
import com.google.cloud.hadoop.fs.gcs.GoogleHadoopFileSystemBase;
import com.google.cloud.hadoop.gcsio.GoogleCloudStorageReadOptions;
import com.google.cloud.hadoop.util.GoogleCloudStorageEventBus;
import com.google.cloud.hadoop.util.ITraceFactory;
import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Preconditions;
import com.google.common.base.Stopwatch;
import com.google.common.collect.ImmutableSet;
import com.google.common.flogger.GoogleLogger;
import com.google.common.flogger.LazyArgs;
import com.google.gson.Gson;
import java.io.IOException;
import java.net.URI;
import java.nio.ByteBuffer;
import java.nio.channels.ClosedChannelException;
import java.nio.channels.SeekableByteChannel;
import java.util.HashMap;
import java.util.Locale;
import java.util.Map;
import java.util.concurrent.TimeUnit;
import java.util.logging.Level;
import org.apache.hadoop.fs.FSInputStream;
import org.apache.hadoop.fs.FileSystem;

class GoogleHadoopFSInputStream
extends FSInputStream {
    private static final GoogleLogger logger = GoogleLogger.forEnclosingClass();
    @VisibleForTesting
    static final String READ_METHOD = "gcsFSRead";
    @VisibleForTesting
    static final String POSITIONAL_READ_METHOD = "gcsFSReadPositional";
    @VisibleForTesting
    static final String SEEK_METHOD = "gcsFSSeek";
    @VisibleForTesting
    static final String CLOSE_METHOD = "gcsFSClose";
    @VisibleForTesting
    static final String DURATION_NS = "durationNs";
    @VisibleForTesting
    static final String BYTES_READ = "bytesRead";
    @VisibleForTesting
    static final String GCS_PATH = "gcsPath";
    @VisibleForTesting
    static final String METHOD = "method";
    @VisibleForTesting
    static final String POSITION = "position";
    @VisibleForTesting
    static final String LENGTH = "length";
    @VisibleForTesting
    static final String OFFSET = "offset";
    private static final Gson gson = new Gson();
    private final boolean isTraceLoggingEnabled;
    private final SeekableByteChannel channel;
    private final long logThreshold;
    private final ImmutableSet<String> logFilterProperties;
    private final GhfsStorageStatistics storageStatistics;
    private final ITraceFactory traceFactory;
    private URI gcsPath;
    private long totalBytesRead;
    private final FileSystem.Statistics statistics;
    private final byte[] singleReadBuf = new byte[1];
    private final GhfsStreamStats streamStats;
    private final GhfsStreamStats seekStreamStats;

    GoogleHadoopFSInputStream(GoogleHadoopFileSystemBase ghfs, URI gcsPath, GoogleCloudStorageReadOptions readOptions, FileSystem.Statistics statistics) throws IOException {
        ((GoogleLogger.Api)logger.atFiner()).log("GoogleHadoopFSInputStream(gcsPath: %s, readOptions: %s)", (Object)gcsPath, (Object)readOptions);
        this.gcsPath = gcsPath;
        this.statistics = statistics;
        this.totalBytesRead = 0L;
        this.isTraceLoggingEnabled = readOptions.isTraceLogEnabled();
        this.channel = ghfs.getGcsFs().open(gcsPath, readOptions);
        this.logThreshold = readOptions.getTraceLogTimeThreshold();
        this.logFilterProperties = readOptions.getTraceLogExcludeProperties();
        this.storageStatistics = ghfs.getStorageStatistics();
        this.streamStats = new GhfsStreamStats(this.storageStatistics, GhfsStatistic.STREAM_READ_OPERATIONS, gcsPath);
        this.seekStreamStats = new GhfsStreamStats(this.storageStatistics, GhfsStatistic.STREAM_READ_SEEK_OPERATIONS, gcsPath);
        this.traceFactory = ghfs.getTraceFactory();
    }

    public synchronized int read() throws IOException {
        long startTime = System.nanoTime();
        int numRead = this.channel.read(ByteBuffer.wrap(this.singleReadBuf));
        if (numRead == -1) {
            return -1;
        }
        if (numRead != 1) {
            GoogleCloudStorageEventBus.postOnException();
            throw new IOException(String.format("Somehow read %d bytes using single-byte buffer for path %s ending in position %d!", numRead, this.gcsPath, this.channel.position()));
        }
        byte b = this.singleReadBuf[0];
        ++this.totalBytesRead;
        this.statistics.incrementBytesRead(1L);
        this.statistics.incrementReadOps(1);
        this.streamStats.updateReadStreamStats(1, startTime);
        return b & 0xFF;
    }

    public synchronized int read(byte[] buf, int offset, int length) throws IOException {
        long startTimeNs = System.nanoTime();
        Preconditions.checkNotNull((Object)buf, (Object)"buf must not be null");
        if (offset < 0 || length < 0 || length > buf.length - offset) {
            throw new IndexOutOfBoundsException();
        }
        int numRead = this.channel.read(ByteBuffer.wrap(buf, offset, length));
        this.readAPITrace(READ_METHOD, startTimeNs, 0L, offset, length, numRead, Level.INFO);
        if (numRead > 0) {
            this.statistics.incrementBytesRead((long)numRead);
            this.statistics.incrementReadOps(1);
            this.totalBytesRead += (long)numRead;
            this.streamStats.updateReadStreamStats(numRead, startTimeNs);
        }
        this.storageStatistics.streamReadOperationInComplete(length, Math.max(numRead, 0));
        return numRead;
    }

    public synchronized int read(long position, byte[] buf, int offset, int length) throws IOException {
        long startTimeNs = System.nanoTime();
        int result = super.read(position, buf, offset, length);
        this.readAPITrace(POSITIONAL_READ_METHOD, startTimeNs, position, offset, length, result, Level.FINE);
        if (result > 0) {
            this.statistics.incrementBytesRead((long)result);
            this.totalBytesRead += (long)result;
        }
        return result;
    }

    public synchronized long getPos() throws IOException {
        long pos = this.channel.position();
        ((GoogleLogger.Api)logger.atFiner()).log("getPos(): %d", pos);
        return pos;
    }

    public synchronized void seek(long pos) throws IOException {
        ((GoogleLogger.Api)logger.atFiner()).log("seek(%d)", pos);
        long startTimeNs = System.nanoTime();
        long curPos = this.getPos();
        long diff = pos - curPos;
        if (diff > 0L) {
            this.storageStatistics.streamReadSeekForward(diff);
        } else {
            this.storageStatistics.streamReadSeekBackward(diff);
        }
        try {
            this.channel.position(pos);
            this.seekAPITrace(SEEK_METHOD, startTimeNs, pos);
        }
        catch (IllegalArgumentException e) {
            GoogleCloudStorageEventBus.postOnException();
            throw new IOException(e);
        }
        this.seekStreamStats.updateReadStreamSeekStats(startTimeNs);
    }

    public synchronized boolean seekToNewSource(long targetPos) throws IOException {
        return false;
    }

    public synchronized void close() throws IOException {
        GhfsStorageStatistics.trackDuration(this.storageStatistics, GhfsStatistic.STREAM_READ_CLOSE_OPERATIONS, this.gcsPath, this.traceFactory, () -> {
            try {
                ((GoogleLogger.Api)logger.atFiner()).log("close(): %s", (Object)this.gcsPath);
                Stopwatch stopwatch = Stopwatch.createStarted();
                HashMap apiTraces = new HashMap();
                if (this.channel != null) {
                    ((GoogleLogger.Api)logger.atFiner()).log("Closing '%s' file with %d total bytes read", (Object)this.gcsPath, this.totalBytesRead);
                    this.channel.close();
                    this.closeAPITrace(CLOSE_METHOD, stopwatch);
                }
            }
            finally {
                this.streamStats.close();
                this.seekStreamStats.close();
            }
            return null;
        });
    }

    public boolean markSupported() {
        return false;
    }

    public int available() throws IOException {
        if (!this.channel.isOpen()) {
            GoogleCloudStorageEventBus.postOnException();
            throw new ClosedChannelException();
        }
        return super.available();
    }

    private void readAPITrace(String method, long startTimeNs, long position, int offset, int length, int bytesRead, Level logLevel) {
        if (this.shouldLog(startTimeNs)) {
            HashMap<String, Object> jsonMap = new HashMap<String, Object>();
            this.addLogProperty(METHOD, method, jsonMap);
            this.addLogProperty(GCS_PATH, this.gcsPath, jsonMap);
            this.addLogProperty(DURATION_NS, System.nanoTime() - startTimeNs, jsonMap);
            this.addLogProperty(POSITION, position, jsonMap);
            this.addLogProperty(OFFSET, offset, jsonMap);
            this.addLogProperty(LENGTH, length, jsonMap);
            this.addLogProperty(BYTES_READ, bytesRead, jsonMap);
            this.captureAPITraces(jsonMap, logLevel);
        }
    }

    private void seekAPITrace(String method, long startTimeNs, long pos) {
        if (this.isTraceLoggingEnabled) {
            HashMap<String, Object> jsonMap = new HashMap<String, Object>();
            this.addLogProperty(METHOD, method, jsonMap);
            this.addLogProperty(GCS_PATH, this.gcsPath, jsonMap);
            this.addLogProperty(DURATION_NS, System.nanoTime() - startTimeNs, jsonMap);
            this.addLogProperty(POSITION, pos, jsonMap);
            this.captureAPITraces(jsonMap, Level.FINE);
        }
    }

    private void addLogProperty(String propertyName, Object propertyValue, Map<String, Object> jsonMap) {
        if (this.logFilterProperties.contains((Object)propertyName.toLowerCase(Locale.US))) {
            return;
        }
        jsonMap.put(propertyName, propertyValue);
    }

    private boolean shouldLog(Stopwatch stopwatch) {
        return this.shouldLog(stopwatch.elapsed(TimeUnit.NANOSECONDS));
    }

    private boolean shouldLog(long startTime) {
        return this.isTraceLoggingEnabled && GoogleHadoopFSInputStream.getElapsedMillisFromStartTime(startTime) >= this.logThreshold;
    }

    private static long getElapsedMillisFromStartTime(long timeNs) {
        return (System.nanoTime() - timeNs) / 10001000L;
    }

    private void closeAPITrace(String method, Stopwatch stopwatch) {
        if (this.shouldLog(stopwatch)) {
            HashMap<String, Object> jsonMap = new HashMap<String, Object>();
            this.addLogProperty(METHOD, method, jsonMap);
            this.addLogProperty(GCS_PATH, this.gcsPath, jsonMap);
            this.addLogProperty(DURATION_NS, stopwatch.elapsed(TimeUnit.NANOSECONDS), jsonMap);
            this.captureAPITraces(jsonMap, Level.INFO);
        }
    }

    private void captureAPITraces(Map<String, Object> apiTraces, Level logLevel) {
        if (this.isTraceLoggingEnabled) {
            logger.at(logLevel).log("%s", (Object)LazyArgs.lazy(() -> gson.toJson((Object)apiTraces)));
        }
    }
}

