/*
 * Decompiled with CFR 0.152.
 */
package com.google.cloud.hadoop.gcsio;

import com.google.api.client.googleapis.services.AbstractGoogleClientRequest;
import com.google.api.client.http.HttpResponse;
import com.google.api.client.util.BackOff;
import com.google.api.client.util.BackOffUtils;
import com.google.api.client.util.ExponentialBackOff;
import com.google.api.client.util.NanoClock;
import com.google.api.client.util.Sleeper;
import com.google.api.services.storage.Storage;
import com.google.api.services.storage.model.StorageObject;
import com.google.cloud.hadoop.gcsio.GoogleCloudStorageExceptions;
import com.google.cloud.hadoop.gcsio.StorageResourceId;
import com.google.cloud.hadoop.util.ApiErrorExtractor;
import com.google.cloud.hadoop.util.ClientRequestHelper;
import com.google.cloud.hadoop.util.ResilientOperation;
import com.google.cloud.hadoop.util.RetryBoundedBackOff;
import com.google.cloud.hadoop.util.RetryDeterminer;
import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Preconditions;
import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.nio.ByteBuffer;
import java.nio.channels.Channels;
import java.nio.channels.ClosedChannelException;
import java.nio.channels.ReadableByteChannel;
import java.nio.channels.SeekableByteChannel;
import java.util.regex.Pattern;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class GoogleCloudStorageReadChannel
implements SeekableByteChannel {
    private static final Logger LOG = LoggerFactory.getLogger(GoogleCloudStorageReadChannel.class);
    private static final Pattern SLASH = Pattern.compile("/");
    private Storage gcs;
    private String bucketName;
    private String objectName;
    @VisibleForTesting
    ReadableByteChannel readChannel;
    private boolean channelIsOpen;
    private long currentPosition = -1L;
    @VisibleForTesting
    boolean lazySeekPending;
    private long size = -1L;
    private int maxRetries = 10;
    private final ApiErrorExtractor errorExtractor;
    private final ClientRequestHelper<StorageObject> clientRequestHelper;
    private Sleeper sleeper = Sleeper.DEFAULT;
    private NanoClock clock = NanoClock.SYSTEM;
    private BackOff backOff = null;
    public static final int DEFAULT_BACKOFF_INITIAL_INTERVAL_MILLIS = 200;
    public static final double DEFAULT_BACKOFF_RANDOMIZATION_FACTOR = 0.5;
    public static final double DEFAULT_BACKOFF_MULTIPLIER = 1.5;
    public static final int DEFAULT_BACKOFF_MAX_INTERVAL_MILLIS = 10000;
    public static final int DEFAULT_BACKOFF_MAX_ELAPSED_TIME_MILLIS = 120000;
    private FileEncoding fileEncoding = FileEncoding.UNINITIALIZED;

    public GoogleCloudStorageReadChannel(Storage gcs, String bucketName, String objectName, ApiErrorExtractor errorExtractor, ClientRequestHelper<StorageObject> requestHelper) throws IOException {
        this.gcs = gcs;
        this.clientRequestHelper = requestHelper;
        this.bucketName = bucketName;
        this.objectName = objectName;
        this.errorExtractor = errorExtractor;
        this.channelIsOpen = true;
        this.position(0L);
    }

    @VisibleForTesting
    GoogleCloudStorageReadChannel() throws IOException {
        this.clientRequestHelper = null;
        this.errorExtractor = null;
        this.channelIsOpen = true;
        this.position(0L);
    }

    @VisibleForTesting
    void setSleeper(Sleeper sleeper) {
        Preconditions.checkArgument((sleeper != null ? 1 : 0) != 0, (Object)"sleeper must not be null!");
        this.sleeper = sleeper;
    }

    @VisibleForTesting
    void setNanoClock(NanoClock clock) {
        Preconditions.checkArgument((clock != null ? 1 : 0) != 0, (Object)"clock must not be null!");
        this.clock = clock;
    }

    @VisibleForTesting
    void setBackOff(BackOff backOff) {
        this.backOff = backOff;
    }

    @VisibleForTesting
    BackOff getBackOff() {
        return this.backOff;
    }

    private BackOff resetOrCreateBackOff() throws IOException {
        if (this.backOff != null) {
            this.backOff.reset();
        } else {
            this.backOff = new ExponentialBackOff.Builder().setInitialIntervalMillis(200).setRandomizationFactor(0.5).setMultiplier(1.5).setMaxIntervalMillis(10000).setMaxElapsedTimeMillis(120000).setNanoClock(this.clock).build();
        }
        return this.backOff;
    }

    public void setMaxRetries(int maxRetries) {
        this.maxRetries = maxRetries;
    }

    @Override
    public int read(ByteBuffer buffer) throws IOException {
        boolean isEndOfStream;
        this.throwIfNotOpen();
        if (buffer.remaining() == 0) {
            return 0;
        }
        int totalBytesRead = 0;
        int retriesAttempted = 0;
        do {
            this.performLazySeek();
            int remainingBeforeRead = buffer.remaining();
            try {
                int numBytesRead = this.readChannel.read(buffer);
                this.checkIOPrecondition(numBytesRead != 0, "Read 0 bytes without blocking");
                if (numBytesRead < 0) {
                    this.checkIOPrecondition(this.fileEncoding == FileEncoding.GZIPPED || this.currentPosition == this.size, String.format("Received end of stream result before all the file data has been received; totalBytesRead: %s, currentPosition: %s, size: %s", totalBytesRead, this.currentPosition, this.size));
                    break;
                }
                totalBytesRead += numBytesRead;
                this.currentPosition += (long)numBytesRead;
                if (retriesAttempted != 0) {
                    LOG.info("Success after {} retries on reading '{}'", (Object)retriesAttempted, (Object)StorageResourceId.createReadableString(this.bucketName, this.objectName));
                }
                retriesAttempted = 0;
            }
            catch (IOException ioe) {
                if (retriesAttempted == this.maxRetries) {
                    LOG.error("Already attempted max of {} retries while reading '{}'; throwing exception.", (Object)this.maxRetries, (Object)StorageResourceId.createReadableString(this.bucketName, this.objectName));
                    this.closeReadChannel();
                    throw ioe;
                }
                if (retriesAttempted == 0) {
                    this.resetOrCreateBackOff();
                }
                LOG.warn("Got exception: {} while reading '{}'; retry # {}. Sleeping...", new Object[]{ioe.getMessage(), StorageResourceId.createReadableString(this.bucketName, this.objectName), ++retriesAttempted});
                try {
                    boolean backOffSuccessful = BackOffUtils.next((Sleeper)this.sleeper, (BackOff)this.backOff);
                    if (!backOffSuccessful) {
                        LOG.error("BackOff returned false; maximum total elapsed time exhausted. Giving up after {} retries for '{}'", (Object)retriesAttempted, (Object)StorageResourceId.createReadableString(this.bucketName, this.objectName));
                        this.closeReadChannel();
                        throw ioe;
                    }
                }
                catch (InterruptedException ie) {
                    LOG.error("Interrupted while sleeping before retry. Giving up after {} retries for '{}'", (Object)retriesAttempted, (Object)StorageResourceId.createReadableString(this.bucketName, this.objectName));
                    ioe.addSuppressed(ie);
                    this.closeReadChannel();
                    throw ioe;
                }
                LOG.info("Done sleeping before retry for '{}'; retry # {}.", (Object)StorageResourceId.createReadableString(this.bucketName, this.objectName), (Object)retriesAttempted);
                if (buffer.remaining() != remainingBeforeRead) {
                    int partialRead = remainingBeforeRead - buffer.remaining();
                    LOG.info("Despite exception, had partial read of {} bytes; resetting retry count.", (Object)partialRead);
                    retriesAttempted = 0;
                    totalBytesRead += partialRead;
                    this.currentPosition += (long)partialRead;
                }
                this.closeReadChannel();
                this.lazySeekPending = true;
            }
            catch (RuntimeException r) {
                this.closeReadChannel();
                throw r;
            }
        } while (buffer.remaining() > 0);
        boolean bl = isEndOfStream = totalBytesRead == 0;
        if (isEndOfStream) {
            this.checkIOPrecondition(this.fileEncoding == FileEncoding.GZIPPED || this.currentPosition == this.size, String.format("Failed to read any data before all the file data has been received; currentPosition: %s, size: %s", this.currentPosition, this.size));
            return -1;
        }
        return totalBytesRead;
    }

    @Override
    public SeekableByteChannel truncate(long size) throws IOException {
        throw new UnsupportedOperationException("Cannot mutate read-only channel");
    }

    @Override
    public int write(ByteBuffer src) throws IOException {
        throw new UnsupportedOperationException("Cannot mutate read-only channel");
    }

    @Override
    public boolean isOpen() {
        return this.channelIsOpen;
    }

    protected void closeReadChannel() {
        if (this.readChannel != null) {
            try {
                this.readChannel.close();
            }
            catch (Exception e) {
                LOG.debug("Got an exception on readChannel.close(); ignoring it.", (Throwable)e);
            }
            finally {
                this.readChannel = null;
            }
        }
    }

    @Override
    public void close() throws IOException {
        if (!this.channelIsOpen) {
            LOG.warn("Channel for '{}' is not open.", (Object)StorageResourceId.createReadableString(this.bucketName, this.objectName));
            return;
        }
        this.channelIsOpen = false;
        this.closeReadChannel();
    }

    @Override
    public long position() throws IOException {
        this.throwIfNotOpen();
        return this.currentPosition;
    }

    @Override
    public SeekableByteChannel position(long newPosition) throws IOException {
        this.throwIfNotOpen();
        if (newPosition == this.currentPosition) {
            return this;
        }
        this.validatePosition(newPosition);
        this.currentPosition = newPosition;
        this.lazySeekPending = true;
        return this;
    }

    @Override
    public long size() throws IOException {
        this.throwIfNotOpen();
        this.performLazySeek();
        return this.size;
    }

    protected void setSize(long size) {
        this.size = size;
    }

    protected void validatePosition(long newPosition) {
        if (newPosition < 0L) {
            throw new IllegalArgumentException(String.format("Invalid seek offset: position value (%d) must be >= 0", newPosition));
        }
        if (this.size >= 0L && newPosition >= this.size && this.fileEncoding != FileEncoding.GZIPPED) {
            throw new IllegalArgumentException(String.format("Invalid seek offset: position value (%d) must be between 0 and %d", newPosition, this.size));
        }
    }

    @VisibleForTesting
    void performLazySeek() throws IOException {
        if (!this.lazySeekPending) {
            return;
        }
        this.closeReadChannel();
        InputStream objectContentStream = this.openStreamAndSetMetadata(this.currentPosition);
        this.readChannel = Channels.newChannel(objectContentStream);
        this.lazySeekPending = false;
    }

    protected StorageObject getMetadata() throws IOException {
        Storage.Objects.Get getObject = this.gcs.objects().get(this.bucketName, this.objectName);
        try {
            return (StorageObject)ResilientOperation.retry((ResilientOperation.CheckedCallable)ResilientOperation.getGoogleRequestCallable((AbstractGoogleClientRequest)getObject), (BackOff)new RetryBoundedBackOff(3, this.resetOrCreateBackOff()), (RetryDeterminer)RetryDeterminer.SOCKET_ERRORS, IOException.class, (Sleeper)this.sleeper);
        }
        catch (IOException e) {
            if (this.errorExtractor.itemNotFound(e)) {
                throw GoogleCloudStorageExceptions.getFileNotFoundException(this.bucketName, this.objectName);
            }
            String string = String.valueOf(StorageResourceId.createReadableString(this.bucketName, this.objectName));
            String msg = string.length() != 0 ? "Error reading ".concat(string) : new String("Error reading ");
            throw new IOException(msg, e);
        }
        catch (InterruptedException e) {
            throw new IOException("Thread interrupt received.", e);
        }
    }

    protected static FileEncoding getEncoding(StorageObject metadata) {
        String contentEncoding = metadata.getContentEncoding();
        return contentEncoding != null && contentEncoding.contains("gzip") ? FileEncoding.GZIPPED : FileEncoding.OTHER;
    }

    protected void setSize(HttpResponse response, long offset) throws IOException {
        String contentRange = response.getHeaders().getContentRange();
        if (response.getHeaders().getContentLength() != null) {
            this.size = response.getHeaders().getContentLength() + offset;
        } else if (contentRange != null) {
            String sizeStr = SLASH.split(contentRange)[1];
            try {
                this.size = Long.parseLong(sizeStr);
            }
            catch (NumberFormatException e) {
                String string = String.valueOf(contentRange);
                throw new IOException(string.length() != 0 ? "Could not determine size from response from Content-Range: ".concat(string) : new String("Could not determine size from response from Content-Range: "), e);
            }
        } else {
            throw new IOException("Could not determine size of response");
        }
    }

    protected InputStream openStreamAndSetMetadata(long newPosition) throws IOException {
        HttpResponse response;
        if (this.fileEncoding == FileEncoding.UNINITIALIZED) {
            StorageObject metadata = this.getMetadata();
            this.fileEncoding = GoogleCloudStorageReadChannel.getEncoding(metadata);
        }
        this.validatePosition(newPosition);
        Storage.Objects.Get getObject = this.gcs.objects().get(this.bucketName, this.objectName);
        this.clientRequestHelper.getRequestHeaders((AbstractGoogleClientRequest)getObject).setRange(String.format("bytes=%d-", this.fileEncoding == FileEncoding.GZIPPED ? 0L : newPosition));
        try {
            response = getObject.executeMedia();
        }
        catch (IOException e) {
            if (this.errorExtractor.itemNotFound(e)) {
                throw GoogleCloudStorageExceptions.getFileNotFoundException(this.bucketName, this.objectName);
            }
            if (this.errorExtractor.rangeNotSatisfiable(e) && newPosition == 0L && this.size == -1L) {
                LOG.info("Got 'range not satisfiable' for reading {} at position 0; assuming empty.", (Object)StorageResourceId.createReadableString(this.bucketName, this.objectName));
                this.size = 0L;
                return new ByteArrayInputStream(new byte[0]);
            }
            String msg = String.format("Error reading %s at position %d", StorageResourceId.createReadableString(this.bucketName, this.objectName), newPosition);
            throw new IOException(msg, e);
        }
        InputStream content = null;
        try {
            content = response.getContent();
            this.setSize(response, this.fileEncoding == FileEncoding.GZIPPED ? 0L : newPosition);
            if (this.fileEncoding == FileEncoding.GZIPPED) {
                content.skip(newPosition);
            }
        }
        catch (IOException e) {
            try {
                if (content != null) {
                    content.close();
                }
            }
            catch (IOException closeException) {
                LOG.debug("Caught exception on close after IOException thrown.", (Throwable)closeException);
                e.addSuppressed(closeException);
            }
            throw e;
        }
        return content;
    }

    private void throwIfNotOpen() throws IOException {
        if (!this.isOpen()) {
            throw new ClosedChannelException();
        }
    }

    private void checkIOPrecondition(boolean precondition, String errorMessage) throws IOException {
        if (!precondition) {
            throw new IOException(errorMessage);
        }
    }

    private static enum FileEncoding {
        UNINITIALIZED,
        GZIPPED,
        OTHER;

    }
}

