/*
 * Decompiled with CFR 0.152.
 */
package com.google.cloud.hadoop.gcsio;

import com.google.cloud.hadoop.gcsio.CacheEntry;
import com.google.cloud.hadoop.gcsio.DirectoryListCache;
import com.google.cloud.hadoop.gcsio.GoogleCloudStorageStrings;
import com.google.cloud.hadoop.gcsio.StorageResourceId;
import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Function;
import com.google.common.base.Preconditions;
import com.google.common.base.Strings;
import com.google.common.collect.ImmutableList;
import java.io.File;
import java.io.IOException;
import java.nio.file.FileAlreadyExistsException;
import java.nio.file.FileVisitResult;
import java.nio.file.FileVisitor;
import java.nio.file.Files;
import java.nio.file.NoSuchFileException;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.nio.file.SimpleFileVisitor;
import java.nio.file.attribute.BasicFileAttributes;
import java.nio.file.attribute.FileAttribute;
import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Set;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class FileSystemBackedDirectoryListCache
extends DirectoryListCache {
    private static final Logger LOG = LoggerFactory.getLogger(FileSystemBackedDirectoryListCache.class);
    static final int MAX_RETRIES_FOR_CREATE = 5;
    private Path basePath = null;
    private Function<StorageResourceId, Void> createMirrorFileListener = null;

    @VisibleForTesting
    public static FileSystemBackedDirectoryListCache getUninitializedInstanceForTest() {
        return new FileSystemBackedDirectoryListCache();
    }

    public FileSystemBackedDirectoryListCache(String basePathStr) {
        this.setBasePath(basePathStr);
    }

    private FileSystemBackedDirectoryListCache() {
    }

    @Override
    public boolean supportsCacheEntryByReference() {
        return false;
    }

    @Override
    public boolean containsEntriesForImplicitDirectories() {
        return true;
    }

    @VisibleForTesting
    Path getMirrorPath(StorageResourceId resourceId) {
        Preconditions.checkArgument((!Paths.get(resourceId.getBucketName(), new String[0]).isAbsolute() ? 1 : 0) != 0, (String)"Bucket name must not look like an absolute path for resourceId '%s'", (Object)resourceId);
        Path resolvedPath = this.basePath.resolve(resourceId.getBucketName());
        if (resourceId.isStorageObject()) {
            Preconditions.checkArgument((!Paths.get(resourceId.getObjectName(), new String[0]).isAbsolute() ? 1 : 0) != 0, (String)"Object name must not look like an absolute path for resourceId '%s'", (Object)resourceId);
            resolvedPath = resolvedPath.resolve(resourceId.getObjectName());
        }
        Path normalizedPath = resolvedPath.normalize();
        Preconditions.checkArgument((boolean)normalizedPath.toString().equals(resolvedPath.toString()), (String)"Normalized path '%s' doesn't match raw resolved path '%s', resolvedPath; relative components are not allowed.", (Object)normalizedPath, (Object)resolvedPath);
        Preconditions.checkState((boolean)normalizedPath.startsWith(this.basePath), (String)"Somehow got a normalized resolved path '%s' that no longer starts with '%s'!", (Object)normalizedPath, (Object)this.basePath);
        return normalizedPath;
    }

    private CacheEntry getCacheEntryInternal(StorageResourceId resourceId, File mirrorFile, boolean allowDirectoryMismatch) throws IOException {
        boolean resourceIdIsDirectory;
        if (!mirrorFile.exists()) {
            return null;
        }
        boolean mirrorIsDirectory = mirrorFile.isDirectory();
        if (mirrorIsDirectory != (resourceIdIsDirectory = resourceId.isDirectory())) {
            String errorMessage = String.format("Existing mirrorFile and resourceId don't match isDirectory status! '%s' (dir: '%s') vs '%s' (dir: '%s')", mirrorFile, mirrorIsDirectory, resourceId, resourceIdIsDirectory);
            if (allowDirectoryMismatch) {
                LOG.info(errorMessage);
                return null;
            }
            throw new IOException(errorMessage);
        }
        return new CacheEntry(resourceId, mirrorFile.lastModified());
    }

    private StorageResourceId getParentResourceId(StorageResourceId resourceId) {
        Preconditions.checkArgument((boolean)resourceId.isStorageObject(), (String)"resourceId must be a StorageObject, got '%s'", (Object)resourceId);
        Path parentObjectPath = Paths.get(resourceId.getObjectName(), new String[0]).getParent();
        StorageResourceId parentResourceId = parentObjectPath == null ? new StorageResourceId(resourceId.getBucketName()) : new StorageResourceId(resourceId.getBucketName(), StorageResourceId.convertToDirectoryPath(parentObjectPath.toString()));
        return parentResourceId;
    }

    private void ensureParentDirectoriesExist(StorageResourceId resourceId, File mirrorFile) throws IOException {
        File parentFile = mirrorFile.getParentFile();
        if (!parentFile.exists()) {
            Preconditions.checkState((!resourceId.isBucket() ? 1 : 0) != 0, (String)"Parent directory doesn't exist for bucket '%s'", (Object)resourceId);
            StorageResourceId parentResourceId = this.getParentResourceId(resourceId);
            Path parentPathFromFile = parentFile.toPath();
            Path parentPathFromResourceId = this.getMirrorPath(parentResourceId);
            Preconditions.checkState((boolean)parentPathFromFile.equals(parentPathFromResourceId), (String)"parentFile.toPath() '%s' doesn't match getMirrorPath of parentResourceId '%s'", (Object)parentPathFromFile, (Object)parentPathFromResourceId);
            LOG.debug("Caching nonexistent parent resourceId '{}', which is path '{}'", (Object)parentResourceId, (Object)parentFile);
            this.putResourceId(parentResourceId);
        }
        Preconditions.checkState((boolean)parentFile.exists(), (String)"Parent file '%s' still doesn't exist adding resourceId '%s' with file '%s'", (Object)parentFile, (Object)resourceId, (Object)mirrorFile);
        if (!parentFile.isDirectory()) {
            throw new IOException(String.format("Parent file '%s' already exists and isn't a directory, trying to add resourceId '%s' with file '%s'", parentFile, resourceId, mirrorFile));
        }
    }

    /*
     * Unable to fully structure code
     */
    private void createMirrorFileInternal(StorageResourceId resourceId, File mirrorFile) throws IOException {
        if (this.createMirrorFileListener != null) {
            this.createMirrorFileListener.apply((Object)resourceId);
        }
        if (resourceId.isDirectory()) {
            try {
                FileSystemBackedDirectoryListCache.LOG.debug("Creating directory '{}' for resourceId '{}'", (Object)mirrorFile, (Object)resourceId);
                Files.createDirectory(mirrorFile.toPath(), new FileAttribute[0]);
                FileSystemBackedDirectoryListCache.LOG.debug("Successfully created directory '{}' for resourceId '{}'", (Object)mirrorFile, (Object)resourceId);
            }
            catch (FileAlreadyExistsException faee) {
                FileSystemBackedDirectoryListCache.LOG.debug("Directory '{}' already exists, with FileAlreadyExistsException message: '{}'", (Object)mirrorFile, (Object)faee.getMessage());
                if (mirrorFile.isDirectory()) ** GOTO lbl32
                throw new IOException(String.format("Mirror file '%s' for resourceId '%s' already exists and isn't a directory!", new Object[]{mirrorFile, resourceId}), faee);
            }
            catch (Throwable t) {
                FileSystemBackedDirectoryListCache.LOG.debug("Exception thrown during creation", t);
                throw t;
            }
        } else {
            try {
                FileSystemBackedDirectoryListCache.LOG.debug("Creating empty file '{}' for resourceId '{}'", (Object)mirrorFile, (Object)resourceId);
                Files.createFile(mirrorFile.toPath(), new FileAttribute[0]);
                FileSystemBackedDirectoryListCache.LOG.debug("Successfully created empty file '{}' for resourceId '{}'", (Object)mirrorFile, (Object)resourceId);
            }
            catch (FileAlreadyExistsException faee) {
                FileSystemBackedDirectoryListCache.LOG.debug("File '{}' already exists, with FileAlreadyExistsException message: '{}'", (Object)mirrorFile, (Object)faee.getMessage());
                if (mirrorFile.isDirectory()) {
                    throw new IOException(String.format("Mirror file '%s' for resourceId '%s' already exists and is a directory!", new Object[]{mirrorFile, resourceId}), faee);
                }
            }
            catch (Throwable t) {
                FileSystemBackedDirectoryListCache.LOG.debug("Exception thrown during creation", t);
                throw t;
            }
        }
lbl32:
        // 4 sources

        currentTime = this.clock.currentTimeMillis();
        FileSystemBackedDirectoryListCache.LOG.debug("Setting lastModified on '{}' to '{}'", (Object)mirrorFile, (Object)currentTime);
        mirrorFile.setLastModified(currentTime);
    }

    @Override
    public CacheEntry putResourceId(StorageResourceId resourceId) throws IOException {
        LOG.debug("putResourceId({})", (Object)resourceId);
        FileSystemBackedDirectoryListCache.validateResourceId(resourceId);
        Path mirrorPath = this.getMirrorPath(resourceId);
        File mirrorFile = mirrorPath.toFile();
        CacheEntry createdOrExistingEntry = null;
        NoSuchFileException lastException = null;
        for (int retry = 0; createdOrExistingEntry == null && retry < 5; ++retry) {
            try {
                boolean allowDirectoryMismatch = false;
                createdOrExistingEntry = this.getCacheEntryInternal(resourceId, mirrorFile, allowDirectoryMismatch);
                if (createdOrExistingEntry != null) {
                    LOG.debug("Returning immediately with pre-existing file '{}' for resourceId '{}'", (Object)mirrorFile, (Object)resourceId);
                    continue;
                }
                this.ensureParentDirectoriesExist(resourceId, mirrorFile);
                File parentFile = mirrorFile.getParentFile();
                long parentLastModified = parentFile.lastModified();
                this.createMirrorFileInternal(resourceId, mirrorFile);
                if (!this.basePath.equals(parentFile.toPath())) {
                    LOG.debug("Restoring parentLastModified to '{}' for '{}'", (Object)parentLastModified, (Object)parentFile);
                    parentFile.setLastModified(parentLastModified);
                }
                createdOrExistingEntry = new CacheEntry(resourceId, mirrorFile.lastModified());
                continue;
            }
            catch (NoSuchFileException nsfe) {
                LOG.info(String.format("Possible concurrently deleted parent dir for file '%s', id '%s', retrying...", mirrorFile, resourceId), (Throwable)nsfe);
                lastException = nsfe;
            }
        }
        if (createdOrExistingEntry == null) {
            throw new IOException(String.format("Exhausted all retries creating mirrorFile '%s' for resourceId '%s'", mirrorFile, resourceId), lastException);
        }
        return createdOrExistingEntry;
    }

    @Override
    public CacheEntry getCacheEntry(StorageResourceId resourceId) throws IOException {
        LOG.debug("getCacheEntry({})", (Object)resourceId);
        FileSystemBackedDirectoryListCache.validateResourceId(resourceId);
        Path mirrorPath = this.getMirrorPath(resourceId);
        File mirrorFile = mirrorPath.toFile();
        return this.getCacheEntryInternal(resourceId, mirrorFile, true);
    }

    @Override
    public void removeResourceId(StorageResourceId resourceId) throws IOException {
        LOG.debug("removeResourceId({})", (Object)resourceId);
        FileSystemBackedDirectoryListCache.validateResourceId(resourceId);
        Path mirrorPath = this.getMirrorPath(resourceId);
        File mirrorFile = mirrorPath.toFile();
        if (this.getCacheEntry(resourceId) != null) {
            long parentLastModified = mirrorFile.getParentFile().lastModified();
            if (!mirrorFile.delete()) {
                LOG.debug("Failed to delete file '{}' for resourceId '{}'", (Object)mirrorFile, (Object)resourceId);
            }
            if (!this.basePath.equals(mirrorFile.getParentFile().toPath())) {
                LOG.debug("Restoring parentListModified to '{}' for '{}'", (Object)parentLastModified, (Object)mirrorFile.getParentFile());
                mirrorFile.getParentFile().setLastModified(parentLastModified);
            }
        } else {
            LOG.debug("Tried to remove nonexistent file '{}' for resourceId '{}'", (Object)mirrorFile, (Object)resourceId);
        }
    }

    @Override
    public List<CacheEntry> getBucketList() throws IOException {
        LOG.debug("getBucketList()");
        File[] bucketList = this.listBuckets();
        if (bucketList.length == 0) {
            return ImmutableList.of();
        }
        ArrayList<CacheEntry> bucketEntries = new ArrayList<CacheEntry>();
        HashSet<StorageResourceId> expiredBuckets = new HashSet<StorageResourceId>();
        for (File bucket : bucketList) {
            if (!bucket.isDirectory()) {
                LOG.error("Found non-directory file '{}' in bucket listing! Ignoring it.", (Object)bucket);
                continue;
            }
            StorageResourceId bucketId = new StorageResourceId(bucket.getName());
            CacheEntry entry = new CacheEntry(bucketId, bucket.lastModified());
            if (this.isCacheEntryExpired(entry)) {
                expiredBuckets.add(bucketId);
                continue;
            }
            bucketEntries.add(entry);
        }
        for (StorageResourceId expiredBucket : expiredBuckets) {
            LOG.debug("About to delete expired entry for resourceId '{}'", (Object)expiredBucket);
            this.removeResourceId(expiredBucket);
        }
        return bucketEntries;
    }

    @Override
    public List<CacheEntry> getRawBucketList() throws IOException {
        LOG.debug("getRawBucketList()");
        File[] bucketList = this.listBuckets();
        if (bucketList.length == 0) {
            return ImmutableList.of();
        }
        ArrayList<CacheEntry> bucketEntries = new ArrayList<CacheEntry>();
        for (File bucket : bucketList) {
            if (!bucket.isDirectory()) {
                LOG.error("Found non-directory file '{}' in bucket listing! Ignoring it.", (Object)bucket);
                continue;
            }
            StorageResourceId bucketId = new StorageResourceId(bucket.getName());
            CacheEntry entry = new CacheEntry(bucketId, bucket.lastModified());
            bucketEntries.add(entry);
        }
        return bucketEntries;
    }

    private File[] listBuckets() throws IOException {
        File baseFile = this.basePath.toFile();
        File[] bucketList = baseFile.listFiles();
        if (bucketList == null && baseFile.exists()) {
            throw new IOException(String.format("Failed to list buckets: %s [exists: %s, directory: %s]", this.basePath, baseFile.exists(), baseFile.isDirectory()));
        }
        return bucketList == null ? new File[]{} : bucketList;
    }

    /*
     * Enabled aggressive block sorting
     */
    @Override
    public List<CacheEntry> getObjectList(final String bucketName, final String objectNamePrefix, final String delimiter, Set<String> returnedPrefixes) throws IOException {
        StorageResourceId listBase;
        LOG.debug("getObjectList({}, {}, {})", new Object[]{bucketName, objectNamePrefix, delimiter});
        Preconditions.checkArgument((delimiter == null || "/".equals(delimiter) ? 1 : 0) != 0, (String)"Only null or '/' are supported as delimiters for file-backed caching, got '%s'.", (Object)delimiter);
        if (Strings.isNullOrEmpty((String)objectNamePrefix)) {
            listBase = new StorageResourceId(bucketName);
        } else {
            int indexOfDelim = objectNamePrefix.lastIndexOf(47);
            if (indexOfDelim == -1) {
                listBase = new StorageResourceId(bucketName);
            } else {
                String objectToList = objectNamePrefix.substring(0, indexOfDelim + 1);
                listBase = new StorageResourceId(bucketName, objectToList);
            }
        }
        LOG.debug("Using '{}' as the listBase", (Object)listBase);
        FileSystemBackedDirectoryListCache.validateResourceId(listBase);
        final Path listBasePath = this.getMirrorPath(listBase);
        File listBaseFile = listBasePath.toFile();
        if (!listBaseFile.exists()) {
            LOG.debug("listBaseFile '{}' corresponding to lastBase '{}' doesn't exist; returning null", (Object)listBaseFile, (Object)listBase);
            return null;
        }
        final ArrayList<CacheEntry> cacheEntries = new ArrayList<CacheEntry>();
        final Path bucketBasePath = this.getMirrorPath(new StorageResourceId(bucketName));
        final FileCallback fileVisitor = new FileCallback(){

            @Override
            public void run(File candidateFile) throws IOException {
                StorageResourceId objectId;
                CacheEntry entry;
                Path objectNamePath = bucketBasePath.relativize(candidateFile.toPath());
                String objectNamePathString = objectNamePath.toString();
                if (candidateFile.isDirectory()) {
                    objectNamePathString = StorageResourceId.convertToDirectoryPath(objectNamePathString);
                }
                if (FileSystemBackedDirectoryListCache.this.isCacheEntryExpired(entry = new CacheEntry(objectId = new StorageResourceId(bucketName, objectNamePathString), candidateFile.lastModified()))) {
                    LOG.debug("About to delete expired entry for resourceId '{}'", (Object)objectId);
                    FileSystemBackedDirectoryListCache.this.removeResourceId(objectId);
                } else {
                    String objectName = objectId.getObjectName();
                    String matchedName = GoogleCloudStorageStrings.matchListPrefix(objectNamePrefix, delimiter, objectName);
                    if (matchedName != null && objectName.equals(matchedName)) {
                        LOG.debug("Adding matching entry '{}'", (Object)objectId);
                        cacheEntries.add(entry);
                    }
                }
            }
        };
        if (delimiter != null) {
            File[] fileList = listBaseFile.listFiles();
            if (fileList == null) {
                LOG.warn("Got null fileList for listBaseFile '{}' even though exists() was true!", (Object)listBaseFile);
                return null;
            }
            for (File object : fileList) {
                fileVisitor.run(object);
            }
            return cacheEntries;
        } else {
            Files.walkFileTree(listBasePath, (FileVisitor<? super Path>)new SimpleFileVisitor<Path>(){

                @Override
                public FileVisitResult postVisitDirectory(Path dir, IOException ioe) throws IOException {
                    if (!listBasePath.equals(dir)) {
                        fileVisitor.run(dir.toFile());
                    }
                    return FileVisitResult.CONTINUE;
                }

                @Override
                public FileVisitResult visitFile(Path file, BasicFileAttributes attrs) throws IOException {
                    fileVisitor.run(file.toFile());
                    return FileVisitResult.CONTINUE;
                }
            });
        }
        return cacheEntries;
    }

    @Override
    public int getInternalNumBuckets() throws IOException {
        return this.listBuckets().length;
    }

    @Override
    public int getInternalNumObjects() throws IOException {
        final int[] count = new int[1];
        Files.walkFileTree(this.basePath, (FileVisitor<? super Path>)new SimpleFileVisitor<Path>(){

            @Override
            public FileVisitResult postVisitDirectory(Path dir, IOException ioe) throws IOException {
                if (!FileSystemBackedDirectoryListCache.this.basePath.equals(dir) && !FileSystemBackedDirectoryListCache.this.basePath.equals(dir.getParent())) {
                    LOG.debug("Counting '{}'", (Object)dir);
                    count[0] = count[0] + 1;
                }
                return FileVisitResult.CONTINUE;
            }

            @Override
            public FileVisitResult visitFile(Path file, BasicFileAttributes attrs) throws IOException {
                LOG.debug("Counting '{}'", (Object)file);
                count[0] = count[0] + 1;
                return FileVisitResult.CONTINUE;
            }
        });
        return count[0];
    }

    @VisibleForTesting
    void setCreateMirrorFileListener(Function<StorageResourceId, Void> listener) {
        this.createMirrorFileListener = listener;
    }

    @VisibleForTesting
    public void setBasePath(String basePathStr) {
        this.basePath = Paths.get(basePathStr, new String[0]);
        Preconditions.checkArgument((!Strings.isNullOrEmpty((String)basePathStr) ? 1 : 0) != 0, (String)"basePathStr '%s' can't be null/empty!", (Object)basePathStr);
        Preconditions.checkArgument((boolean)this.basePath.isAbsolute(), (String)"basePathStr '%s' must be absolute!", (Object)basePathStr);
        if (!this.basePath.toFile().exists()) {
            LOG.info("Creating '{}' with createDirectories()...", (Object)this.basePath);
            try {
                Files.createDirectories(this.basePath, new FileAttribute[0]);
            }
            catch (IOException ioe) {
                throw new RuntimeException(ioe);
            }
        }
        Preconditions.checkArgument((boolean)this.basePath.toFile().exists(), (String)"basePathStr '%s' must exist", (Object)basePathStr);
        Preconditions.checkArgument((boolean)this.basePath.toFile().isDirectory(), (String)"basePathStr '%s' must be a directory!", (Object)basePathStr);
    }

    private static interface FileCallback {
        public void run(File var1) throws IOException;
    }
}

