/*
 * Decompiled with CFR 0.152.
 */
package com.google.cloud.hadoop.gcsio;

import com.google.api.client.auth.oauth2.Credential;
import com.google.api.client.util.Clock;
import com.google.cloud.hadoop.gcsio.CreateFileOptions;
import com.google.cloud.hadoop.gcsio.CreateObjectOptions;
import com.google.cloud.hadoop.gcsio.FileInfo;
import com.google.cloud.hadoop.gcsio.GoogleCloudStorage;
import com.google.cloud.hadoop.gcsio.GoogleCloudStorageFileSystemOptions;
import com.google.cloud.hadoop.gcsio.GoogleCloudStorageImpl;
import com.google.cloud.hadoop.gcsio.GoogleCloudStorageItemInfo;
import com.google.cloud.hadoop.gcsio.GoogleCloudStorageReadOptions;
import com.google.cloud.hadoop.gcsio.LegacyPathCodec;
import com.google.cloud.hadoop.gcsio.PathCodec;
import com.google.cloud.hadoop.gcsio.PerformanceCachingGoogleCloudStorage;
import com.google.cloud.hadoop.gcsio.StorageResourceId;
import com.google.cloud.hadoop.gcsio.UpdatableItemInfo;
import com.google.cloud.hadoop.gcsio.UriEncodingPathCodec;
import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.CharMatcher;
import com.google.common.base.Function;
import com.google.common.base.Preconditions;
import com.google.common.base.Strings;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.Lists;
import com.google.common.collect.Sets;
import com.google.common.flogger.GoogleLogger;
import com.google.common.util.concurrent.ThreadFactoryBuilder;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.net.URI;
import java.nio.channels.SeekableByteChannel;
import java.nio.channels.WritableByteChannel;
import java.nio.file.DirectoryNotEmptyException;
import java.nio.file.FileAlreadyExistsException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.TreeMap;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Future;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.concurrent.RejectedExecutionException;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.concurrent.TimeUnit;
import java.util.regex.Pattern;
import java.util.stream.Collectors;

public class GoogleCloudStorageFileSystem {
    private static final GoogleLogger logger = GoogleLogger.forEnclosingClass();
    public static final String SCHEME = "gs";
    public static final URI GCS_ROOT = URI.create("gs:/");
    private static final CharMatcher BUCKET_NAME_CHAR_MATCHER = CharMatcher.ascii().and(CharMatcher.inRange((char)'0', (char)'9').or(CharMatcher.inRange((char)'a', (char)'z')).or(CharMatcher.anyOf((CharSequence)"_.-"))).precomputed();
    private GoogleCloudStorage gcs;
    private final PathCodec pathCodec;
    private final GoogleCloudStorageFileSystemOptions options;
    private ExecutorService updateTimestampsExecutor = GoogleCloudStorageFileSystem.createUpdateTimestampsExecutor();
    @VisibleForTesting
    static final Comparator<URI> PATH_COMPARATOR = Comparator.comparing(URI::toString, (as, bs) -> as.length() == bs.length() ? as.compareTo((String)bs) : Integer.compare(as.length(), bs.length()));
    @VisibleForTesting
    static final Comparator<FileInfo> FILE_INFO_PATH_COMPARATOR = Comparator.comparing(FileInfo::getPath, PATH_COMPARATOR);
    private static final Comparator<FileInfo> STRING_LENGTH_COMPARATOR = Comparator.comparingInt(arg -> arg.getPath().toString().length());
    public static final PathCodec LEGACY_PATH_CODEC = new LegacyPathCodec();
    public static final PathCodec URI_ENCODED_PATH_CODEC = new UriEncodingPathCodec();

    public GoogleCloudStorageFileSystem(Credential credential, GoogleCloudStorageFileSystemOptions options) throws IOException {
        ((GoogleLogger.Api)logger.atFine()).log("GCSFS(%s)", (Object)options.getCloudStorageOptions().getAppName());
        options.throwIfNotValid();
        Preconditions.checkArgument((credential != null ? 1 : 0) != 0, (Object)"credential must not be null");
        this.options = options;
        this.gcs = new GoogleCloudStorageImpl(options.getCloudStorageOptions(), credential);
        this.pathCodec = options.getPathCodec();
        if (options.isPerformanceCacheEnabled()) {
            this.gcs = new PerformanceCachingGoogleCloudStorage(this.gcs, options.getPerformanceCacheOptions());
        }
    }

    public GoogleCloudStorageFileSystem(GoogleCloudStorage gcs) throws IOException {
        this(gcs, GoogleCloudStorageFileSystemOptions.newBuilder().setImmutableCloudStorageOptions(gcs.getOptions()).build());
    }

    public GoogleCloudStorageFileSystem(GoogleCloudStorage gcs, GoogleCloudStorageFileSystemOptions options) throws IOException {
        this.gcs = gcs;
        this.options = options;
        this.pathCodec = options.getPathCodec();
    }

    @VisibleForTesting
    void setUpdateTimestampsExecutor(ExecutorService executor) {
        this.updateTimestampsExecutor = executor;
    }

    private static ExecutorService createUpdateTimestampsExecutor() {
        ThreadPoolExecutor service = new ThreadPoolExecutor(2, 2, 5L, TimeUnit.SECONDS, new LinkedBlockingQueue<Runnable>(1000), new ThreadFactoryBuilder().setNameFormat("gcsfs-timestamp-updates-%d").setDaemon(true).build());
        service.allowCoreThreadTimeOut(true);
        return service;
    }

    public GoogleCloudStorageFileSystemOptions getOptions() {
        return this.options;
    }

    public static CreateObjectOptions objectOptionsFromFileOptions(CreateFileOptions options) {
        return new CreateObjectOptions(options.overwriteExisting(), options.getContentType(), options.getAttributes());
    }

    public WritableByteChannel create(URI path) throws IOException {
        ((GoogleLogger.Api)logger.atFine()).log("create(%s)", (Object)path);
        return this.create(path, CreateFileOptions.DEFAULT);
    }

    public WritableByteChannel create(URI path, CreateFileOptions options) throws IOException {
        URI parentPath;
        URI dirPath;
        ((GoogleLogger.Api)logger.atFine()).log("create(%s)", (Object)path);
        Preconditions.checkNotNull((Object)path, (Object)"path could not be null");
        if (FileInfo.isDirectoryPath(path)) {
            throw new IOException(String.format("Cannot create a file whose name looks like a directory. Got '%s'", path));
        }
        if (options.checkNoDirectoryConflict() && this.exists(dirPath = FileInfo.convertToDirectoryPath(this.pathCodec, path))) {
            throw new FileAlreadyExistsException("A directory with that name exists: " + path);
        }
        if (options.ensureParentDirectoriesExist() && (parentPath = this.getParentPath(path)) != null) {
            this.mkdirs(parentPath);
        }
        return this.createInternal(path, options);
    }

    WritableByteChannel createInternal(URI path, CreateFileOptions options) throws IOException {
        StorageResourceId resourceId = this.pathCodec.validatePathAndGetId(path, false);
        if (options.getExistingGenerationId() != -1L) {
            resourceId = new StorageResourceId(resourceId.getBucketName(), resourceId.getObjectName(), options.getExistingGenerationId());
        }
        WritableByteChannel channel = this.gcs.create(resourceId, GoogleCloudStorageFileSystem.objectOptionsFromFileOptions(options));
        this.tryUpdateTimestampsForParentDirectories((List<URI>)ImmutableList.of((Object)path), (List<URI>)ImmutableList.of());
        return channel;
    }

    public SeekableByteChannel open(URI path) throws IOException {
        return this.open(path, GoogleCloudStorageReadOptions.DEFAULT);
    }

    public SeekableByteChannel open(URI path, GoogleCloudStorageReadOptions readOptions) throws IOException {
        ((GoogleLogger.Api)logger.atFine()).log("open(%s, %s)", (Object)path, (Object)readOptions);
        Preconditions.checkNotNull((Object)path);
        Preconditions.checkArgument((!FileInfo.isDirectoryPath(path) ? 1 : 0) != 0, (String)"Cannot open a directory for reading: %s", (Object)path);
        StorageResourceId resourceId = this.pathCodec.validatePathAndGetId(path, false);
        return this.gcs.open(resourceId, readOptions);
    }

    public void delete(URI path, boolean recursive) throws IOException {
        ((GoogleLogger.Api)logger.atFine()).log("delete(%s, %s)", (Object)path, recursive);
        Preconditions.checkNotNull((Object)path);
        Preconditions.checkArgument((!path.equals(GCS_ROOT) ? 1 : 0) != 0, (String)"Cannot delete root path (%s)", (Object)path);
        FileInfo fileInfo = this.getFileInfo(path);
        if (!fileInfo.exists()) {
            throw GoogleCloudStorageFileSystem.getFileNotFoundException(path);
        }
        List<Object> itemsToDelete = new ArrayList<FileInfo>();
        ArrayList<FileInfo> bucketsToDelete = new ArrayList<FileInfo>();
        if (fileInfo.isDirectory()) {
            List<FileInfo> list = itemsToDelete = recursive ? this.listAllFileInfoForPrefix(fileInfo.getPath()) : this.listFileInfo(fileInfo.getPath(), false);
            if (!itemsToDelete.isEmpty() && !recursive) {
                throw new DirectoryNotEmptyException("Cannot delete a non-empty directory.");
            }
        }
        if (fileInfo.getItemInfo().isBucket()) {
            bucketsToDelete.add(fileInfo);
        } else {
            itemsToDelete.add(fileInfo);
        }
        this.deleteInternal(itemsToDelete, bucketsToDelete);
        if (bucketsToDelete.isEmpty()) {
            List itemsToDeleteNames = itemsToDelete.stream().map(FileInfo::getPath).collect(Collectors.toCollection(ArrayList::new));
            this.tryUpdateTimestampsForParentDirectories(itemsToDeleteNames, itemsToDeleteNames);
        }
    }

    private void deleteInternal(List<FileInfo> itemsToDelete, List<FileInfo> bucketsToDelete) throws IOException {
        itemsToDelete.sort(FILE_INFO_PATH_COMPARATOR.reversed());
        if (!itemsToDelete.isEmpty()) {
            ArrayList<StorageResourceId> objectsToDelete = new ArrayList<StorageResourceId>(itemsToDelete.size());
            for (FileInfo fileInfo : itemsToDelete) {
                objectsToDelete.add(new StorageResourceId(fileInfo.getItemInfo().getBucketName(), fileInfo.getItemInfo().getObjectName(), fileInfo.getItemInfo().getContentGeneration()));
            }
            this.gcs.deleteObjects(objectsToDelete);
        }
        if (!bucketsToDelete.isEmpty()) {
            ArrayList<String> bucketNames = new ArrayList<String>(bucketsToDelete.size());
            for (FileInfo bucketInfo : bucketsToDelete) {
                StorageResourceId resourceId = bucketInfo.getItemInfo().getResourceId();
                this.gcs.waitForBucketEmpty(resourceId.getBucketName());
                bucketNames.add(resourceId.getBucketName());
            }
            if (this.options.enableBucketDelete()) {
                this.gcs.deleteBuckets(bucketNames);
            } else {
                ((GoogleLogger.Api)logger.atInfo()).log("Skipping deletion of buckets because enableBucketDelete is false: %s", bucketNames);
            }
        }
    }

    public boolean exists(URI path) throws IOException {
        ((GoogleLogger.Api)logger.atFine()).log("exists(%s)", (Object)path);
        return this.getFileInfo(path).exists();
    }

    public void repairDirs(List<URI> exactDirPaths) throws IOException {
        ((GoogleLogger.Api)logger.atFine()).log("repairDirs(%s)", exactDirPaths);
        ArrayList<StorageResourceId> dirsToCreate = new ArrayList<StorageResourceId>();
        for (URI dirUri : exactDirPaths) {
            StorageResourceId resourceId = this.pathCodec.validatePathAndGetId(dirUri, true);
            if (!resourceId.isStorageObject()) continue;
            resourceId = FileInfo.convertToDirectoryPath(resourceId);
            dirsToCreate.add(resourceId);
        }
        if (dirsToCreate.isEmpty()) {
            return;
        }
        this.gcs.createEmptyObjects(dirsToCreate);
        ((GoogleLogger.Api)logger.atWarning()).log("Successfully repaired %s directories.", dirsToCreate.size());
    }

    public void mkdirs(URI path) throws IOException {
        ((GoogleLogger.Api)logger.atFine()).log("mkdirs(%s)", (Object)path);
        Preconditions.checkNotNull((Object)path);
        if (path.equals(GCS_ROOT)) {
            return;
        }
        StorageResourceId resourceId = this.pathCodec.validatePathAndGetId(path, true);
        if (resourceId.isStorageObject()) {
            resourceId = FileInfo.convertToDirectoryPath(resourceId);
            path = this.pathCodec.getPath(resourceId.getBucketName(), resourceId.getObjectName(), false);
        }
        ArrayList<URI> subDirPaths = new ArrayList<URI>();
        List<String> subdirs = GoogleCloudStorageFileSystem.getSubDirs(resourceId.getObjectName());
        for (String subdir : subdirs) {
            URI subPath = this.pathCodec.getPath(resourceId.getBucketName(), subdir, true);
            subDirPaths.add(subPath);
            ((GoogleLogger.Api)logger.atFine()).log("mkdirs: sub-path: %s", (Object)subPath);
            if (Strings.isNullOrEmpty((String)subdir)) continue;
            URI uRI = this.pathCodec.getPath(resourceId.getBucketName(), subdir.substring(0, subdir.length() - 1), true);
            subDirPaths.add(uRI);
            ((GoogleLogger.Api)logger.atFine()).log("mkdirs: sub-path: %s", (Object)uRI);
        }
        URI bucketPath = this.pathCodec.getPath(resourceId.getBucketName(), null, true);
        subDirPaths.add(bucketPath);
        ((GoogleLogger.Api)logger.atFine()).log("mkdirs: sub-path: %s", (Object)bucketPath);
        List<FileInfo> subDirInfos = this.getFileInfos(subDirPaths);
        for (FileInfo fileInfo : subDirInfos) {
            if (!fileInfo.exists() || fileInfo.isDirectory()) continue;
            throw new FileAlreadyExistsException("Cannot create directories because of existing file: " + fileInfo.getPath());
        }
        Collections.sort(subDirInfos, STRING_LENGTH_COMPARATOR);
        ArrayList<StorageResourceId> dirsToCreate = new ArrayList<StorageResourceId>();
        for (FileInfo fileInfo : subDirInfos) {
            if (!fileInfo.isDirectory() || fileInfo.exists()) continue;
            StorageResourceId dirId = fileInfo.getItemInfo().getResourceId();
            Preconditions.checkArgument((!dirId.isRoot() ? 1 : 0) != 0, (Object)"Cannot create root directory.");
            if (dirId.isBucket()) {
                this.gcs.create(dirId.getBucketName());
                continue;
            }
            dirId = FileInfo.convertToDirectoryPath(dirId);
            dirsToCreate.add(dirId);
        }
        this.gcs.createEmptyObjects(dirsToCreate);
        List list = Lists.transform(dirsToCreate, (Function)new Function<StorageResourceId, URI>(){

            public URI apply(StorageResourceId resourceId) {
                return GoogleCloudStorageFileSystem.this.pathCodec.getPath(resourceId.getBucketName(), resourceId.getObjectName(), false);
            }
        });
        this.tryUpdateTimestampsForParentDirectories(list, list);
    }

    public void rename(URI src, URI dst) throws IOException {
        ((GoogleLogger.Api)logger.atFine()).log("rename(%s, %s)", (Object)src, (Object)dst);
        Preconditions.checkNotNull((Object)src);
        Preconditions.checkNotNull((Object)dst);
        Preconditions.checkArgument((!src.equals(GCS_ROOT) ? 1 : 0) != 0, (Object)"Root path cannot be renamed.");
        String srcItemName = this.getItemName(src);
        URI dstParent = this.getParentPath(dst);
        ArrayList<URI> paths = new ArrayList<URI>();
        paths.add(src);
        paths.add(dst);
        if (dstParent != null) {
            paths.add(dstParent);
        }
        List<FileInfo> fileInfo = this.getFileInfos(paths);
        FileInfo srcInfo = fileInfo.get(0);
        FileInfo dstInfo = fileInfo.get(1);
        FileInfo dstParentInfo = dstParent == null ? null : fileInfo.get(2);
        src = srcInfo.getPath();
        dst = dstInfo.getPath();
        if (!srcInfo.exists()) {
            throw GoogleCloudStorageFileSystem.getFileNotFoundException(src);
        }
        if (!srcInfo.isDirectory() && dst.equals(GCS_ROOT)) {
            throw new IOException("A file cannot be created in root.");
        }
        if (dstInfo.exists() && !dstInfo.isDirectory() && (srcInfo.isDirectory() || !dst.equals(src))) {
            throw new IOException("Cannot overwrite existing file: " + dst);
        }
        if (dstParentInfo != null && !dstParentInfo.exists()) {
            throw new IOException("Cannot rename because path does not exist: " + dstParent);
        }
        if (srcInfo.isDirectory()) {
            if (!dstInfo.isDirectory()) {
                dst = FileInfo.convertToDirectoryPath(this.pathCodec, dst);
                dstInfo = this.getFileInfo(dst);
            }
            if (src.equals(dst)) {
                throw new IOException("Rename dir to self is forbidden");
            }
            URI dstRelativeToSrc = src.relativize(dst);
            if (!dstRelativeToSrc.equals(dst)) {
                throw new IOException("Rename to subdir is forbidden");
            }
            if (dstInfo.exists()) {
                dst = dst.equals(GCS_ROOT) ? this.pathCodec.getPath(srcItemName, null, true) : FileInfo.convertToDirectoryPath(this.pathCodec, dst.resolve(srcItemName));
            }
        } else if (dstInfo.isDirectory()) {
            if (!dstInfo.exists()) {
                throw new IOException("Cannot rename because path does not exist: " + dstInfo.getPath());
            }
            dst = dst.resolve(srcItemName);
        } else {
            URI dstDir = FileInfo.convertToDirectoryPath(this.pathCodec, dst);
            FileInfo dstDirInfo = this.getFileInfo(dstDir);
            if (dstDirInfo.exists()) {
                dst = dstDir.resolve(srcItemName);
            }
        }
        if (src.equals(dst)) {
            return;
        }
        this.renameInternal(srcInfo, dst);
    }

    public void compose(List<URI> sources, URI destination, String contentType) throws IOException {
        StorageResourceId destResource = StorageResourceId.fromObjectName(destination.toString());
        List sourceObjects = Lists.transform(sources, uri -> StorageResourceId.fromObjectName(uri.toString()).getObjectName());
        this.gcs.compose(destResource.getBucketName(), sourceObjects, destResource.getObjectName(), contentType);
    }

    private void renameInternal(FileInfo srcInfo, URI dst) throws IOException {
        if (srcInfo.isDirectory()) {
            this.renameDirectoryInternal(srcInfo, dst);
        } else {
            URI src = srcInfo.getPath();
            StorageResourceId srcResourceId = this.pathCodec.validatePathAndGetId(src, true);
            StorageResourceId dstResourceId = this.pathCodec.validatePathAndGetId(dst, true);
            this.gcs.copy(srcResourceId.getBucketName(), (List<String>)ImmutableList.of((Object)srcResourceId.getObjectName()), dstResourceId.getBucketName(), (List<String>)ImmutableList.of((Object)dstResourceId.getObjectName()));
            this.tryUpdateTimestampsForParentDirectories((List<URI>)ImmutableList.of((Object)dst), (List<URI>)ImmutableList.of());
            this.gcs.deleteObjects((List<StorageResourceId>)ImmutableList.of((Object)new StorageResourceId(srcInfo.getItemInfo().getBucketName(), srcInfo.getItemInfo().getObjectName(), srcInfo.getItemInfo().getContentGeneration())));
            this.tryUpdateTimestampsForParentDirectories((List<URI>)ImmutableList.of((Object)src), (List<URI>)ImmutableList.of());
        }
    }

    private void renameDirectoryInternal(FileInfo srcInfo, URI dst) throws IOException {
        Preconditions.checkArgument((boolean)srcInfo.isDirectory(), (String)"'%s' should be a directory", (Object)srcInfo);
        Pattern markerFilePattern = this.options.getMarkerFilePattern();
        TreeMap<FileInfo, URI> srcToDstItemNames = new TreeMap<FileInfo, URI>(FILE_INFO_PATH_COMPARATOR);
        TreeMap<FileInfo, URI> srcToDstMarkerItemNames = new TreeMap<FileInfo, URI>(FILE_INFO_PATH_COMPARATOR);
        List<FileInfo> srcItemInfos = this.listAllFileInfoForPrefix(srcInfo.getPath());
        dst = FileInfo.convertToDirectoryPath(this.pathCodec, dst);
        this.mkdir(dst);
        String prefix = srcInfo.getPath().toString();
        for (FileInfo srcItemInfo : srcItemInfos) {
            String relativeItemName = srcItemInfo.getPath().toString().substring(prefix.length());
            URI dstItemName = dst.resolve(relativeItemName);
            if (markerFilePattern != null && markerFilePattern.matcher(relativeItemName).matches()) {
                srcToDstMarkerItemNames.put(srcItemInfo, dstItemName);
                continue;
            }
            srcToDstItemNames.put(srcItemInfo, dstItemName);
        }
        this.copyInternal(srcToDstItemNames);
        this.copyInternal(srcToDstMarkerItemNames);
        if (!srcToDstItemNames.isEmpty() || !srcToDstMarkerItemNames.isEmpty()) {
            ArrayList<URI> allDestinationUris = new ArrayList<URI>(srcToDstItemNames.size() + srcToDstMarkerItemNames.size());
            allDestinationUris.addAll(srcToDstItemNames.values());
            allDestinationUris.addAll(srcToDstMarkerItemNames.values());
            this.tryUpdateTimestampsForParentDirectories(allDestinationUris, allDestinationUris);
        }
        ArrayList<FileInfo> bucketsToDelete = new ArrayList<FileInfo>(1);
        ArrayList<FileInfo> srcItemsToDelete = new ArrayList<FileInfo>(srcToDstItemNames.size() + 1);
        srcItemsToDelete.addAll(srcToDstItemNames.keySet());
        if (srcInfo.getItemInfo().isBucket()) {
            bucketsToDelete.add(srcInfo);
        } else {
            srcItemsToDelete.add(srcInfo);
        }
        this.deleteInternal(new ArrayList<FileInfo>(srcToDstMarkerItemNames.keySet()), new ArrayList<FileInfo>());
        this.deleteInternal(srcItemsToDelete, bucketsToDelete);
        if (bucketsToDelete.isEmpty()) {
            List srcItemNames = srcItemInfos.stream().map(FileInfo::getPath).collect(Collectors.toCollection(ArrayList::new));
            this.tryUpdateTimestampsForParentDirectories(srcItemNames, srcItemNames);
        }
    }

    private void copyInternal(Map<FileInfo, URI> srcToDstItemNames) throws IOException {
        if (srcToDstItemNames.isEmpty()) {
            return;
        }
        String srcBucketName = null;
        String dstBucketName = null;
        ArrayList<String> srcObjectNames = new ArrayList<String>(srcToDstItemNames.size());
        ArrayList<String> dstObjectNames = new ArrayList<String>(srcToDstItemNames.size());
        for (Map.Entry<FileInfo, URI> srcToDstItemName : srcToDstItemNames.entrySet()) {
            StorageResourceId srcResourceId = srcToDstItemName.getKey().getItemInfo().getResourceId();
            srcBucketName = srcResourceId.getBucketName();
            String srcObjectName = srcResourceId.getObjectName();
            srcObjectNames.add(srcObjectName);
            StorageResourceId dstResourceId = this.pathCodec.validatePathAndGetId(srcToDstItemName.getValue(), true);
            dstBucketName = dstResourceId.getBucketName();
            String dstObjectName = dstResourceId.getObjectName();
            dstObjectNames.add(dstObjectName);
        }
        this.gcs.copy(srcBucketName, srcObjectNames, dstBucketName, dstObjectNames);
    }

    public List<URI> listFileNames(FileInfo fileInfo) throws IOException {
        return this.listFileNames(fileInfo, false);
    }

    public List<URI> listFileNames(FileInfo fileInfo, boolean recursive) throws IOException {
        Preconditions.checkNotNull((Object)fileInfo);
        URI path = fileInfo.getPath();
        ((GoogleLogger.Api)logger.atFine()).log("listFileNames(%s)", (Object)path);
        ArrayList<URI> paths = new ArrayList<URI>();
        if (fileInfo.isDirectory()) {
            if (fileInfo.exists()) {
                if (fileInfo.isGlobalRoot()) {
                    List<String> childNames = this.gcs.listBucketNames();
                    for (String childName : childNames) {
                        URI childPath = this.pathCodec.getPath(childName, null, true);
                        paths.add(childPath);
                        ((GoogleLogger.Api)logger.atFine()).log("listFileNames: added: %s", (Object)childPath);
                    }
                } else {
                    String delimiter = recursive ? null : "/";
                    GoogleCloudStorageItemInfo itemInfo = fileInfo.getItemInfo();
                    List<String> childNames = this.gcs.listObjectNames(itemInfo.getBucketName(), itemInfo.getObjectName(), delimiter);
                    for (String childName : childNames) {
                        URI childPath = this.pathCodec.getPath(itemInfo.getBucketName(), childName, false);
                        paths.add(childPath);
                        ((GoogleLogger.Api)logger.atFine()).log("listFileNames: added: %s", (Object)childPath);
                    }
                }
            }
        } else {
            paths.add(path);
            ((GoogleLogger.Api)logger.atFine()).log("listFileNames: added single original path since !isDirectory(): %s", (Object)path);
        }
        return paths;
    }

    public boolean repairPossibleImplicitDirectory(URI path) throws IOException {
        ((GoogleLogger.Api)logger.atFine()).log("repairPossibleImplicitDirectory(%s)", (Object)path);
        Preconditions.checkNotNull((Object)path);
        FileInfo pathInfo = this.getFileInfo(path);
        pathInfo = this.repairPossibleImplicitDirectory(pathInfo);
        if (pathInfo.exists()) {
            ((GoogleLogger.Api)logger.atFine()).log("Successfully repaired path '%s'", (Object)path);
            return true;
        }
        ((GoogleLogger.Api)logger.atFine()).log("Repair claimed to succeed, but somehow failed for path '%s'", (Object)path);
        return false;
    }

    private FileInfo repairPossibleImplicitDirectory(FileInfo pathInfo) throws IOException {
        if (pathInfo.exists()) {
            return pathInfo;
        }
        if (pathInfo.isGlobalRoot() || pathInfo.getItemInfo().isBucket() || pathInfo.getItemInfo().getObjectName().equals("/")) {
            return pathInfo;
        }
        try {
            this.gcs.listObjectInfo(pathInfo.getItemInfo().getBucketName(), FileInfo.convertToFilePath(pathInfo.getItemInfo().getObjectName()), "/");
        }
        catch (IOException ioe) {
            ((GoogleLogger.Api)((GoogleLogger.Api)logger.atSevere()).withCause((Throwable)ioe)).log("Got exception trying to listObjectInfo on " + pathInfo, (Object)ioe);
        }
        pathInfo = this.getFileInfo(pathInfo.getPath());
        return pathInfo;
    }

    public List<FileInfo> listAllFileInfoForPrefix(URI prefix) throws IOException {
        ((GoogleLogger.Api)logger.atFine()).log("listAllFileInfoForPrefixPage(%s)", (Object)prefix);
        StorageResourceId prefixId = this.getPrefixId(prefix);
        List<GoogleCloudStorageItemInfo> itemInfos = this.gcs.listObjectInfo(prefixId.getBucketName(), prefixId.getObjectName(), null);
        List<FileInfo> fileInfos = FileInfo.fromItemInfos(this.pathCodec, itemInfos);
        fileInfos.sort(FILE_INFO_PATH_COMPARATOR);
        return fileInfos;
    }

    public GoogleCloudStorage.ListPage<FileInfo> listAllFileInfoForPrefixPage(URI prefix, String pageToken) throws IOException {
        ((GoogleLogger.Api)logger.atFine()).log("listAllFileInfoForPrefixPage(%s, %s)", (Object)prefix, (Object)pageToken);
        StorageResourceId prefixId = this.getPrefixId(prefix);
        GoogleCloudStorage.ListPage<GoogleCloudStorageItemInfo> itemInfosPage = this.gcs.listObjectInfoPage(prefixId.getBucketName(), prefixId.getObjectName(), null, pageToken);
        List<FileInfo> fileInfosPage = FileInfo.fromItemInfos(this.pathCodec, itemInfosPage.getItems());
        fileInfosPage.sort(FILE_INFO_PATH_COMPARATOR);
        return new GoogleCloudStorage.ListPage<FileInfo>(fileInfosPage, itemInfosPage.getNextPageToken());
    }

    private StorageResourceId getPrefixId(URI prefix) {
        Preconditions.checkNotNull((Object)prefix, (Object)"prefix could not be null");
        StorageResourceId prefixId = this.pathCodec.validatePathAndGetId(prefix, true);
        Preconditions.checkArgument((!prefixId.isRoot() ? 1 : 0) != 0, (String)"prefix must not be global root, got '%s'", (Object)prefix);
        return prefixId;
    }

    public List<FileInfo> listFileInfo(URI path, boolean enableAutoRepair) throws IOException {
        ((GoogleLogger.Api)logger.atFine()).log("listFileInfo(%s, %s)", (Object)path, enableAutoRepair);
        Preconditions.checkNotNull((Object)path);
        URI dirPath = FileInfo.convertToDirectoryPath(this.pathCodec, path);
        List<FileInfo> baseAndDirInfos = this.getFileInfosRaw((List<URI>)ImmutableList.of((Object)path, (Object)dirPath));
        Preconditions.checkState((baseAndDirInfos.size() == 2 ? 1 : 0) != 0, (String)"Expected baseAndDirInfos.size() == 2, got %s", (int)baseAndDirInfos.size());
        if (!baseAndDirInfos.get(0).isDirectory() && baseAndDirInfos.get(0).exists()) {
            ArrayList<FileInfo> listedInfo = new ArrayList<FileInfo>();
            listedInfo.add(baseAndDirInfos.get(0));
            return listedInfo;
        }
        FileInfo dirInfo = baseAndDirInfos.get(1);
        if (!dirInfo.exists()) {
            if (enableAutoRepair) {
                dirInfo = this.repairPossibleImplicitDirectory(dirInfo);
            } else if (this.options.getCloudStorageOptions().isInferImplicitDirectoriesEnabled()) {
                StorageResourceId dirId = dirInfo.getItemInfo().getResourceId();
                if (!dirInfo.isDirectory()) {
                    dirId = FileInfo.convertToDirectoryPath(dirId);
                }
                dirInfo = FileInfo.fromItemInfo(this.pathCodec, this.getInferredItemInfo(dirId));
            }
        }
        if (!dirInfo.exists()) {
            throw GoogleCloudStorageFileSystem.getFileNotFoundException(path);
        }
        List<GoogleCloudStorageItemInfo> itemInfos = dirInfo.isGlobalRoot() ? this.gcs.listBucketInfo() : this.gcs.listObjectInfo(dirInfo.getItemInfo().getBucketName(), dirInfo.getItemInfo().getObjectName(), "/");
        List<FileInfo> fileInfos = FileInfo.fromItemInfos(this.pathCodec, itemInfos);
        fileInfos.sort(FILE_INFO_PATH_COMPARATOR);
        return fileInfos;
    }

    public FileInfo getFileInfo(URI path) throws IOException {
        GoogleCloudStorageItemInfo newItemInfo;
        StorageResourceId newResourceId;
        ((GoogleLogger.Api)logger.atFine()).log("getFileInfo(%s)", (Object)path);
        Preconditions.checkArgument((path != null ? 1 : 0) != 0, (Object)"path must not be null");
        StorageResourceId resourceId = this.pathCodec.validatePathAndGetId(path, true);
        GoogleCloudStorageItemInfo itemInfo = this.gcs.getItemInfo(resourceId);
        if (!itemInfo.exists() && !FileInfo.isDirectory(itemInfo)) {
            newResourceId = FileInfo.convertToDirectoryPath(resourceId);
            ((GoogleLogger.Api)logger.atFine()).log("getFileInfo(%s) : not found. trying: %s", (Object)path, (Object)newResourceId);
            newItemInfo = this.gcs.getItemInfo(newResourceId);
            if (newItemInfo.exists()) {
                ((GoogleLogger.Api)logger.atFine()).log("getFileInfo: swapping not-found info: %s for converted info: %s", (Object)itemInfo, (Object)newItemInfo);
                itemInfo = newItemInfo;
                resourceId = newResourceId;
            }
        }
        if (!itemInfo.exists() && this.options.getCloudStorageOptions().isInferImplicitDirectoriesEnabled() && !itemInfo.isRoot() && !itemInfo.isBucket()) {
            newResourceId = resourceId;
            if (!FileInfo.isDirectory(itemInfo)) {
                newResourceId = FileInfo.convertToDirectoryPath(resourceId);
            }
            ((GoogleLogger.Api)logger.atFine()).log("getFileInfo(%s) : still not found, trying inferred: %s", (Object)path, (Object)newResourceId);
            newItemInfo = this.getInferredItemInfo(resourceId);
            if (newItemInfo.exists()) {
                ((GoogleLogger.Api)logger.atFine()).log("getFileInfo: swapping not-found info: %s for inferred info: %s", (Object)itemInfo, (Object)newItemInfo);
                itemInfo = newItemInfo;
                resourceId = newResourceId;
            }
        }
        FileInfo fileInfo = FileInfo.fromItemInfo(this.pathCodec, itemInfo);
        ((GoogleLogger.Api)logger.atFine()).log("getFileInfo: %s", (Object)fileInfo);
        return fileInfo;
    }

    public List<FileInfo> getFileInfos(List<URI> paths) throws IOException {
        ((GoogleLogger.Api)logger.atFine()).log("getFileInfos(list)");
        Preconditions.checkArgument((paths != null ? 1 : 0) != 0, (Object)"paths must not be null");
        ArrayList<StorageResourceId> resourceIdsForPaths = new ArrayList<StorageResourceId>(paths.size());
        for (URI path : paths) {
            resourceIdsForPaths.add(this.pathCodec.validatePathAndGetId(path, true));
        }
        List<GoogleCloudStorageItemInfo> itemInfos = this.gcs.getItemInfos(resourceIdsForPaths);
        HashMap<StorageResourceId, Integer> convertedIdsToIndex = new HashMap<StorageResourceId, Integer>();
        for (int i = 0; i < itemInfos.size(); ++i) {
            if (itemInfos.get(i).exists() || FileInfo.isDirectory(itemInfos.get(i))) continue;
            StorageResourceId convertedId = FileInfo.convertToDirectoryPath(itemInfos.get(i).getResourceId());
            ((GoogleLogger.Api)logger.atFine()).log("getFileInfos(%s) : not found. trying: %s", (Object)itemInfos.get(i).getResourceId(), (Object)convertedId);
            convertedIdsToIndex.put(convertedId, i);
        }
        if (!convertedIdsToIndex.isEmpty()) {
            ArrayList<StorageResourceId> convertedResourceIds = new ArrayList<StorageResourceId>(convertedIdsToIndex.keySet());
            List<GoogleCloudStorageItemInfo> convertedInfos = this.gcs.getItemInfos(convertedResourceIds);
            for (int i = 0; i < convertedResourceIds.size(); ++i) {
                if (!convertedInfos.get(i).exists()) continue;
                int replaceIndex = (Integer)convertedIdsToIndex.get(convertedResourceIds.get(i));
                ((GoogleLogger.Api)logger.atFine()).log("getFileInfos: swapping not-found info: %s for converted info: %s", (Object)itemInfos.get(replaceIndex), (Object)convertedInfos.get(i));
                itemInfos.set(replaceIndex, convertedInfos.get(i));
            }
        }
        if (this.options.getCloudStorageOptions().isInferImplicitDirectoriesEnabled()) {
            HashMap<StorageResourceId, Integer> inferredIdsToIndex = new HashMap<StorageResourceId, Integer>();
            for (int i = 0; i < itemInfos.size(); ++i) {
                if (itemInfos.get(i).exists()) continue;
                StorageResourceId inferredId = itemInfos.get(i).getResourceId();
                if (!FileInfo.isDirectory(itemInfos.get(i))) {
                    inferredId = FileInfo.convertToDirectoryPath(inferredId);
                }
                ((GoogleLogger.Api)logger.atFine()).log("getFileInfos(%s) : still not found, trying inferred: %s", (Object)itemInfos.get(i).getResourceId(), (Object)inferredId);
                inferredIdsToIndex.put(inferredId, i);
            }
            if (!inferredIdsToIndex.isEmpty()) {
                ArrayList<StorageResourceId> inferredResourceIds = new ArrayList<StorageResourceId>(inferredIdsToIndex.keySet());
                List<GoogleCloudStorageItemInfo> inferredInfos = this.getInferredItemInfos(inferredResourceIds);
                for (int i = 0; i < inferredResourceIds.size(); ++i) {
                    if (!inferredInfos.get(i).exists()) continue;
                    int replaceIndex = (Integer)inferredIdsToIndex.get(inferredResourceIds.get(i));
                    ((GoogleLogger.Api)logger.atFine()).log("getFileInfos: swapping not-found info: %s for inferred info: %s", (Object)itemInfos.get(replaceIndex), (Object)inferredInfos.get(i));
                    itemInfos.set(replaceIndex, inferredInfos.get(i));
                }
            }
        }
        return FileInfo.fromItemInfos(this.pathCodec, itemInfos);
    }

    private List<FileInfo> getFileInfosRaw(List<URI> paths) throws IOException {
        ((GoogleLogger.Api)logger.atFine()).log("getFileInfosRaw(%s)", paths);
        Preconditions.checkArgument((paths != null ? 1 : 0) != 0, (Object)"paths must not be null");
        ArrayList<StorageResourceId> resourceIdsForPaths = new ArrayList<StorageResourceId>();
        for (URI path : paths) {
            resourceIdsForPaths.add(this.pathCodec.validatePathAndGetId(path, true));
        }
        List<GoogleCloudStorageItemInfo> itemInfos = this.gcs.getItemInfos(resourceIdsForPaths);
        return FileInfo.fromItemInfos(this.pathCodec, itemInfos);
    }

    private GoogleCloudStorageItemInfo getInferredItemInfo(StorageResourceId dirId) throws IOException {
        String delimiter;
        String objectNamePrefix;
        if (dirId.isRoot() || dirId.isBucket()) {
            return GoogleCloudStorageItemInfo.createNotFound(dirId);
        }
        StorageResourceId bucketId = new StorageResourceId(dirId.getBucketName());
        if (!this.gcs.getItemInfo(bucketId).exists()) {
            return GoogleCloudStorageItemInfo.createNotFound(dirId);
        }
        String bucketName = (dirId = FileInfo.convertToDirectoryPath(dirId)).getBucketName();
        List<String> objectNames = this.gcs.listObjectNames(bucketName, objectNamePrefix = dirId.getObjectName(), delimiter = "/", 1L);
        if (objectNames.size() > 0) {
            return GoogleCloudStorageItemInfo.createInferredDirectory(dirId);
        }
        return GoogleCloudStorageItemInfo.createNotFound(dirId);
    }

    private List<GoogleCloudStorageItemInfo> getInferredItemInfos(List<StorageResourceId> resourceIds) throws IOException {
        ArrayList<GoogleCloudStorageItemInfo> itemInfos = new ArrayList<GoogleCloudStorageItemInfo>();
        for (int i = 0; i < resourceIds.size(); ++i) {
            itemInfos.add(this.getInferredItemInfo(resourceIds.get(i)));
        }
        return itemInfos;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void close() {
        if (this.gcs != null) {
            ((GoogleLogger.Api)logger.atFine()).log("close()");
            try {
                this.gcs.close();
            }
            finally {
                this.gcs = null;
                if (this.updateTimestampsExecutor != null) {
                    this.updateTimestampsExecutor.shutdown();
                    try {
                        if (!this.updateTimestampsExecutor.awaitTermination(10L, TimeUnit.SECONDS)) {
                            ((GoogleLogger.Api)logger.atWarning()).log("Forcibly shutting down timestamp update thread pool.");
                            this.updateTimestampsExecutor.shutdownNow();
                        }
                    }
                    catch (InterruptedException e) {
                        ((GoogleLogger.Api)((GoogleLogger.Api)logger.atFine()).withCause((Throwable)e)).log("Failed to await termination: forcibly shutting down timestamp update thread pool");
                        this.updateTimestampsExecutor.shutdownNow();
                    }
                    finally {
                        this.updateTimestampsExecutor = null;
                    }
                }
            }
        }
    }

    @VisibleForTesting
    public void mkdir(URI path) throws IOException {
        ((GoogleLogger.Api)logger.atFine()).log("mkdir(%s)", (Object)path);
        Preconditions.checkNotNull((Object)path);
        Preconditions.checkArgument((!path.equals(GCS_ROOT) ? 1 : 0) != 0, (Object)"Cannot create root directory.");
        StorageResourceId resourceId = this.pathCodec.validatePathAndGetId(path, true);
        if (resourceId.isBucket()) {
            this.gcs.create(resourceId.getBucketName());
            return;
        }
        resourceId = FileInfo.convertToDirectoryPath(resourceId);
        this.gcs.createEmptyObject(resourceId);
        this.tryUpdateTimestampsForParentDirectories((List<URI>)ImmutableList.of((Object)path), (List<URI>)ImmutableList.of());
    }

    protected void updateTimestampsForParentDirectories(List<URI> modifiedObjects, List<URI> excludedParents) throws IOException {
        ((GoogleLogger.Api)logger.atFine()).log("updateTimestampsForParentDirectories(%s, %s)", modifiedObjects, excludedParents);
        GoogleCloudStorageFileSystemOptions.TimestampUpdatePredicate updatePredicate = this.options.getShouldIncludeInTimestampUpdatesPredicate();
        HashSet<URI> excludedParentPathsSet = new HashSet<URI>(excludedParents);
        HashSet parentUrisToUpdate = Sets.newHashSetWithExpectedSize((int)modifiedObjects.size());
        for (URI modifiedObjectUri : modifiedObjects) {
            URI parentPathUri = this.getParentPath(modifiedObjectUri);
            if (excludedParentPathsSet.contains(parentPathUri) || !updatePredicate.shouldUpdateTimestamp(parentPathUri)) continue;
            parentUrisToUpdate.add(parentPathUri);
        }
        HashMap<String, byte[]> modificationAttributes = new HashMap<String, byte[]>();
        FileInfo.addModificationTimeToAttributes(modificationAttributes, Clock.SYSTEM);
        ArrayList<UpdatableItemInfo> itemUpdates = new ArrayList<UpdatableItemInfo>(parentUrisToUpdate.size());
        for (URI parentUri : parentUrisToUpdate) {
            StorageResourceId resourceId = this.pathCodec.validatePathAndGetId(parentUri, true);
            if (resourceId.isBucket() || resourceId.isRoot()) continue;
            itemUpdates.add(new UpdatableItemInfo(resourceId, modificationAttributes));
        }
        if (!itemUpdates.isEmpty()) {
            this.gcs.updateItems(itemUpdates);
        } else {
            ((GoogleLogger.Api)logger.atFine()).log("All paths were excluded from directory timestamp updating.");
        }
    }

    protected void tryUpdateTimestampsForParentDirectories(List<URI> modifiedObjects, List<URI> excludedParents) {
        ((GoogleLogger.Api)logger.atFine()).log("tryUpdateTimestampsForParentDirectories(%s, %s)", modifiedObjects, excludedParents);
        try {
            Future<?> future = this.updateTimestampsExecutor.submit(() -> {
                try {
                    this.updateTimestampsForParentDirectories(modifiedObjects, excludedParents);
                }
                catch (IOException ioe) {
                    ((GoogleLogger.Api)((GoogleLogger.Api)logger.atFine()).withCause((Throwable)ioe)).log("Exception caught when trying to update parent directory timestamps.");
                }
            });
        }
        catch (RejectedExecutionException ree) {
            ((GoogleLogger.Api)((GoogleLogger.Api)logger.atFine()).withCause((Throwable)ree)).log("Exhausted thread pool and queue space while updating parent timestamps");
        }
    }

    static List<String> getSubDirs(String objectName) {
        ArrayList<String> subdirs = new ArrayList<String>();
        if (!Strings.isNullOrEmpty((String)objectName)) {
            int index;
            int currentIndex = 0;
            while (currentIndex < objectName.length() && (index = objectName.indexOf(47, currentIndex)) >= 0) {
                subdirs.add(objectName.substring(0, index + 1));
                currentIndex = index + 1;
            }
        }
        return subdirs;
    }

    static String validateBucketName(String bucketName) {
        if (Strings.isNullOrEmpty((String)(bucketName = FileInfo.convertToFilePath(bucketName)))) {
            throw new IllegalArgumentException("Google Cloud Storage bucket name cannot be empty.");
        }
        if (!BUCKET_NAME_CHAR_MATCHER.matchesAllOf((CharSequence)bucketName)) {
            throw new IllegalArgumentException(String.format("Invalid bucket name '%s': bucket name must contain only 'a-z0-9_.-' characters.", bucketName));
        }
        return bucketName;
    }

    static String validateObjectName(String objectName, boolean allowEmptyObjectName) {
        ((GoogleLogger.Api)logger.atFine()).log("validateObjectName('%s', %s)", (Object)objectName, allowEmptyObjectName);
        if (Strings.isNullOrEmpty((String)objectName) || objectName.equals("/")) {
            if (allowEmptyObjectName) {
                objectName = "";
            } else {
                throw new IllegalArgumentException("Google Cloud Storage path must include non-empty object name.");
            }
        }
        for (int i = 0; i < objectName.length() - 1; ++i) {
            if (objectName.charAt(i) != '/' || objectName.charAt(i + 1) != '/') continue;
            throw new IllegalArgumentException(String.format("Google Cloud Storage path must not have consecutive '/' characters, got '%s'", objectName));
        }
        if (objectName.startsWith("/")) {
            objectName = objectName.substring(1);
        }
        ((GoogleLogger.Api)logger.atFine()).log("validateObjectName -> '%s'", (Object)objectName);
        return objectName;
    }

    String getItemName(URI path) {
        Preconditions.checkNotNull((Object)path);
        if (path.equals(GCS_ROOT)) {
            return null;
        }
        StorageResourceId resourceId = this.pathCodec.validatePathAndGetId(path, true);
        if (resourceId.isBucket()) {
            return resourceId.getBucketName();
        }
        int index = FileInfo.objectHasDirectoryPath(resourceId.getObjectName()) ? resourceId.getObjectName().lastIndexOf("/", resourceId.getObjectName().length() - 2) : resourceId.getObjectName().lastIndexOf("/");
        if (index < 0) {
            return resourceId.getObjectName();
        }
        return resourceId.getObjectName().substring(index + 1);
    }

    public URI getParentPath(URI path) {
        return GoogleCloudStorageFileSystem.getParentPath(this.getPathCodec(), path);
    }

    static FileNotFoundException getFileNotFoundException(URI path) {
        return new FileNotFoundException(String.format("Item not found: %s", path));
    }

    public GoogleCloudStorage getGcs() {
        return this.gcs;
    }

    public PathCodec getPathCodec() {
        return this.pathCodec;
    }

    @Deprecated
    public static StorageResourceId validatePathAndGetId(URI uri, boolean allowEmptyObjectNames) {
        return LEGACY_PATH_CODEC.validatePathAndGetId(uri, allowEmptyObjectNames);
    }

    @Deprecated
    public static URI getPath(String bucketName, String objectName, boolean allowEmptyObjectName) {
        return LEGACY_PATH_CODEC.getPath(bucketName, objectName, allowEmptyObjectName);
    }

    @Deprecated
    public static URI getPath(String bucketName) {
        return LEGACY_PATH_CODEC.getPath(bucketName, null, true);
    }

    @Deprecated
    public static URI getPath(String bucketName, String objectName) {
        return LEGACY_PATH_CODEC.getPath(bucketName, objectName, false);
    }

    @Deprecated
    public static URI getParentPath(PathCodec pathCodec, URI path) {
        Preconditions.checkNotNull((Object)path);
        if (path.equals(GCS_ROOT)) {
            return null;
        }
        StorageResourceId resourceId = pathCodec.validatePathAndGetId(path, true);
        if (resourceId.isBucket()) {
            return GCS_ROOT;
        }
        int index = FileInfo.objectHasDirectoryPath(resourceId.getObjectName()) ? resourceId.getObjectName().lastIndexOf("/", resourceId.getObjectName().length() - 2) : resourceId.getObjectName().lastIndexOf("/");
        if (index < 0) {
            return pathCodec.getPath(resourceId.getBucketName(), null, true);
        }
        return pathCodec.getPath(resourceId.getBucketName(), resourceId.getObjectName().substring(0, index + 1), false);
    }
}

