/*
 * Decompiled with CFR 0.152.
 */
package com.google.cloud.hadoop.gcsio;

import com.google.api.client.auth.oauth2.Credential;
import com.google.api.client.util.Clock;
import com.google.cloud.hadoop.gcsio.CreateFileOptions;
import com.google.cloud.hadoop.gcsio.CreateObjectOptions;
import com.google.cloud.hadoop.gcsio.FileInfo;
import com.google.cloud.hadoop.gcsio.GoogleCloudStorage;
import com.google.cloud.hadoop.gcsio.GoogleCloudStorageFileSystemOptions;
import com.google.cloud.hadoop.gcsio.GoogleCloudStorageImpl;
import com.google.cloud.hadoop.gcsio.GoogleCloudStorageItemInfo;
import com.google.cloud.hadoop.gcsio.GoogleCloudStorageReadOptions;
import com.google.cloud.hadoop.gcsio.LegacyPathCodec;
import com.google.cloud.hadoop.gcsio.PathCodec;
import com.google.cloud.hadoop.gcsio.PerformanceCachingGoogleCloudStorage;
import com.google.cloud.hadoop.gcsio.StorageResourceId;
import com.google.cloud.hadoop.gcsio.UpdatableItemInfo;
import com.google.cloud.hadoop.gcsio.UriEncodingPathCodec;
import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.CharMatcher;
import com.google.common.base.Preconditions;
import com.google.common.base.Strings;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.Lists;
import com.google.common.collect.Sets;
import com.google.common.flogger.GoogleLogger;
import com.google.common.util.concurrent.Futures;
import com.google.common.util.concurrent.ListeningExecutorService;
import com.google.common.util.concurrent.MoreExecutors;
import com.google.common.util.concurrent.ThreadFactoryBuilder;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.net.URI;
import java.nio.channels.SeekableByteChannel;
import java.nio.channels.WritableByteChannel;
import java.nio.file.DirectoryNotEmptyException;
import java.nio.file.FileAlreadyExistsException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.TreeMap;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.Future;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.concurrent.RejectedExecutionException;
import java.util.concurrent.ThreadFactory;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.concurrent.TimeUnit;
import java.util.regex.Pattern;
import java.util.stream.Collectors;

public class GoogleCloudStorageFileSystem {
    private static final GoogleLogger logger = GoogleLogger.forEnclosingClass();
    private static final ThreadFactory DAEMON_THREAD_FACTORY = new ThreadFactoryBuilder().setDaemon(true).build();
    public static final String SCHEME = "gs";
    public static final URI GCS_ROOT = URI.create("gs:/");
    private static final CharMatcher BUCKET_NAME_CHAR_MATCHER = CharMatcher.ascii().and(CharMatcher.inRange((char)'0', (char)'9').or(CharMatcher.inRange((char)'a', (char)'z')).or(CharMatcher.anyOf((CharSequence)"_.-"))).precomputed();
    private GoogleCloudStorage gcs;
    private final PathCodec pathCodec;
    private final GoogleCloudStorageFileSystemOptions options;
    private ExecutorService updateTimestampsExecutor = GoogleCloudStorageFileSystem.createUpdateTimestampsExecutor();
    @VisibleForTesting
    static final Comparator<URI> PATH_COMPARATOR = Comparator.comparing(URI::toString, (as, bs) -> as.length() == bs.length() ? as.compareTo((String)bs) : Integer.compare(as.length(), bs.length()));
    @VisibleForTesting
    static final Comparator<FileInfo> FILE_INFO_PATH_COMPARATOR = Comparator.comparing(FileInfo::getPath, PATH_COMPARATOR);
    public static final PathCodec LEGACY_PATH_CODEC = new LegacyPathCodec();
    public static final PathCodec URI_ENCODED_PATH_CODEC = new UriEncodingPathCodec();

    public GoogleCloudStorageFileSystem(Credential credential, GoogleCloudStorageFileSystemOptions options) throws IOException {
        ((GoogleLogger.Api)logger.atFine()).log("GCSFS(%s)", (Object)options.getCloudStorageOptions().getAppName());
        options.throwIfNotValid();
        Preconditions.checkArgument((credential != null ? 1 : 0) != 0, (Object)"credential must not be null");
        this.options = options;
        this.gcs = new GoogleCloudStorageImpl(options.getCloudStorageOptions(), credential);
        this.pathCodec = options.getPathCodec();
        if (options.isPerformanceCacheEnabled()) {
            this.gcs = new PerformanceCachingGoogleCloudStorage(this.gcs, options.getPerformanceCacheOptions());
        }
    }

    public GoogleCloudStorageFileSystem(GoogleCloudStorage gcs) throws IOException {
        this(gcs, GoogleCloudStorageFileSystemOptions.newBuilder().setImmutableCloudStorageOptions(gcs.getOptions()).build());
    }

    public GoogleCloudStorageFileSystem(GoogleCloudStorage gcs, GoogleCloudStorageFileSystemOptions options) throws IOException {
        this.gcs = gcs;
        this.options = options;
        this.pathCodec = options.getPathCodec();
    }

    @VisibleForTesting
    void setUpdateTimestampsExecutor(ExecutorService executor) {
        this.updateTimestampsExecutor = executor;
    }

    private static ExecutorService createUpdateTimestampsExecutor() {
        ThreadPoolExecutor service = new ThreadPoolExecutor(2, 2, 5L, TimeUnit.SECONDS, new LinkedBlockingQueue<Runnable>(1000), new ThreadFactoryBuilder().setNameFormat("gcsfs-timestamp-updates-%d").setDaemon(true).build());
        service.allowCoreThreadTimeOut(true);
        return service;
    }

    public GoogleCloudStorageFileSystemOptions getOptions() {
        return this.options;
    }

    public static CreateObjectOptions objectOptionsFromFileOptions(CreateFileOptions options) {
        return new CreateObjectOptions(options.overwriteExisting(), options.getContentType(), options.getAttributes());
    }

    public WritableByteChannel create(URI path) throws IOException {
        ((GoogleLogger.Api)logger.atFine()).log("create(%s)", (Object)path);
        return this.create(path, CreateFileOptions.DEFAULT);
    }

    public WritableByteChannel create(URI path, CreateFileOptions options) throws IOException {
        URI parentPath;
        URI dirPath;
        ((GoogleLogger.Api)logger.atFine()).log("create(%s)", (Object)path);
        Preconditions.checkNotNull((Object)path, (Object)"path could not be null");
        if (FileInfo.isDirectoryPath(path)) {
            throw new IOException(String.format("Cannot create a file whose name looks like a directory. Got '%s'", path));
        }
        if (options.checkNoDirectoryConflict() && this.exists(dirPath = FileInfo.convertToDirectoryPath(this.pathCodec, path))) {
            throw new FileAlreadyExistsException("A directory with that name exists: " + path);
        }
        if (options.ensureParentDirectoriesExist() && (parentPath = this.getParentPath(path)) != null) {
            this.mkdirs(parentPath);
        }
        return this.createInternal(path, options);
    }

    WritableByteChannel createInternal(URI path, CreateFileOptions options) throws IOException {
        StorageResourceId resourceId = this.pathCodec.validatePathAndGetId(path, false);
        if (options.getExistingGenerationId() != -1L) {
            resourceId = new StorageResourceId(resourceId.getBucketName(), resourceId.getObjectName(), options.getExistingGenerationId());
        }
        WritableByteChannel channel = this.gcs.create(resourceId, GoogleCloudStorageFileSystem.objectOptionsFromFileOptions(options));
        this.tryUpdateTimestampsForParentDirectories((List<URI>)ImmutableList.of((Object)path), (List<URI>)ImmutableList.of());
        return channel;
    }

    public SeekableByteChannel open(URI path) throws IOException {
        return this.open(path, GoogleCloudStorageReadOptions.DEFAULT);
    }

    public SeekableByteChannel open(URI path, GoogleCloudStorageReadOptions readOptions) throws IOException {
        ((GoogleLogger.Api)logger.atFine()).log("open(%s, %s)", (Object)path, (Object)readOptions);
        Preconditions.checkNotNull((Object)path);
        Preconditions.checkArgument((!FileInfo.isDirectoryPath(path) ? 1 : 0) != 0, (String)"Cannot open a directory for reading: %s", (Object)path);
        StorageResourceId resourceId = this.pathCodec.validatePathAndGetId(path, false);
        return this.gcs.open(resourceId, readOptions);
    }

    public void delete(URI path, boolean recursive) throws IOException {
        ((GoogleLogger.Api)logger.atFine()).log("delete(%s, %s)", (Object)path, recursive);
        Preconditions.checkNotNull((Object)path);
        Preconditions.checkArgument((!path.equals(GCS_ROOT) ? 1 : 0) != 0, (String)"Cannot delete root path (%s)", (Object)path);
        FileInfo fileInfo = this.getFileInfo(path);
        if (!fileInfo.exists()) {
            throw new FileNotFoundException("Item not found: " + path);
        }
        List<Object> itemsToDelete = new ArrayList<FileInfo>();
        ArrayList<FileInfo> bucketsToDelete = new ArrayList<FileInfo>();
        if (fileInfo.isDirectory()) {
            List<FileInfo> list = itemsToDelete = recursive ? this.listAllFileInfoForPrefix(fileInfo.getPath()) : this.listFileInfo(fileInfo.getPath(), false);
            if (!itemsToDelete.isEmpty() && !recursive) {
                throw new DirectoryNotEmptyException("Cannot delete a non-empty directory.");
            }
        }
        if (fileInfo.getItemInfo().isBucket()) {
            bucketsToDelete.add(fileInfo);
        } else {
            itemsToDelete.add(fileInfo);
        }
        this.deleteInternal(itemsToDelete, bucketsToDelete);
        if (bucketsToDelete.isEmpty()) {
            List itemsToDeleteNames = itemsToDelete.stream().map(FileInfo::getPath).collect(Collectors.toCollection(ArrayList::new));
            this.tryUpdateTimestampsForParentDirectories(itemsToDeleteNames, itemsToDeleteNames);
        }
    }

    private void deleteInternal(List<FileInfo> itemsToDelete, List<FileInfo> bucketsToDelete) throws IOException {
        itemsToDelete.sort(FILE_INFO_PATH_COMPARATOR.reversed());
        if (!itemsToDelete.isEmpty()) {
            ArrayList<StorageResourceId> objectsToDelete = new ArrayList<StorageResourceId>(itemsToDelete.size());
            for (FileInfo fileInfo : itemsToDelete) {
                objectsToDelete.add(new StorageResourceId(fileInfo.getItemInfo().getBucketName(), fileInfo.getItemInfo().getObjectName(), fileInfo.getItemInfo().getContentGeneration()));
            }
            this.gcs.deleteObjects(objectsToDelete);
        }
        if (!bucketsToDelete.isEmpty()) {
            ArrayList<String> bucketNames = new ArrayList<String>(bucketsToDelete.size());
            for (FileInfo bucketInfo : bucketsToDelete) {
                StorageResourceId resourceId = bucketInfo.getItemInfo().getResourceId();
                this.gcs.waitForBucketEmpty(resourceId.getBucketName());
                bucketNames.add(resourceId.getBucketName());
            }
            if (this.options.enableBucketDelete()) {
                this.gcs.deleteBuckets(bucketNames);
            } else {
                ((GoogleLogger.Api)logger.atInfo()).log("Skipping deletion of buckets because enableBucketDelete is false: %s", bucketNames);
            }
        }
    }

    public boolean exists(URI path) throws IOException {
        ((GoogleLogger.Api)logger.atFine()).log("exists(%s)", (Object)path);
        return this.getFileInfo(path).exists();
    }

    public void repairDirs(List<URI> exactDirPaths) throws IOException {
        ((GoogleLogger.Api)logger.atFine()).log("repairDirs(%s)", exactDirPaths);
        ArrayList<StorageResourceId> dirsToCreate = new ArrayList<StorageResourceId>();
        for (URI dirUri : exactDirPaths) {
            StorageResourceId resourceId = this.pathCodec.validatePathAndGetId(dirUri, true);
            if (!resourceId.isStorageObject()) continue;
            resourceId = FileInfo.convertToDirectoryPath(resourceId);
            dirsToCreate.add(resourceId);
        }
        if (dirsToCreate.isEmpty()) {
            return;
        }
        this.gcs.createEmptyObjects(dirsToCreate);
        ((GoogleLogger.Api)logger.atInfo()).log("Successfully repaired %s directories.", dirsToCreate.size());
    }

    public void mkdirs(URI path) throws IOException {
        ((GoogleLogger.Api)logger.atFine()).log("mkdirs(%s)", (Object)path);
        Preconditions.checkNotNull((Object)path);
        if (path.equals(GCS_ROOT)) {
            return;
        }
        StorageResourceId resourceId = this.pathCodec.validatePathAndGetId(path, true);
        resourceId = FileInfo.convertToDirectoryPath(resourceId);
        List<String> subdirs = GoogleCloudStorageFileSystem.getSubDirs(resourceId.getObjectName());
        ArrayList<StorageResourceId> itemIds = new ArrayList<StorageResourceId>(subdirs.size() * 2 + 1);
        for (String subdir : subdirs) {
            itemIds.add(new StorageResourceId(resourceId.getBucketName(), subdir));
            if (Strings.isNullOrEmpty((String)subdir)) continue;
            itemIds.add(new StorageResourceId(resourceId.getBucketName(), FileInfo.convertToFilePath(subdir)));
        }
        itemIds.add(new StorageResourceId(resourceId.getBucketName()));
        ((GoogleLogger.Api)logger.atFine()).log("mkdirs: items: %s", itemIds);
        List<GoogleCloudStorageItemInfo> itemInfos = this.gcs.getItemInfos(itemIds);
        GoogleCloudStorageItemInfo bucketInfo = null;
        ArrayList<StorageResourceId> subdirsToCreate = new ArrayList<StorageResourceId>(subdirs.size());
        for (GoogleCloudStorageItemInfo info : itemInfos) {
            if (info.isBucket()) {
                Preconditions.checkState((bucketInfo == null ? 1 : 0) != 0, (Object)"bucketInfo should be null");
                bucketInfo = info;
                continue;
            }
            if (info.getResourceId().isDirectory() && !info.exists()) {
                subdirsToCreate.add(info.getResourceId());
                continue;
            }
            if (info.getResourceId().isDirectory() || !info.exists()) continue;
            throw new FileAlreadyExistsException("Cannot create directories because of existing file: " + info.getResourceId());
        }
        if (!((GoogleCloudStorageItemInfo)Preconditions.checkNotNull(bucketInfo, (Object)"bucketInfo should not be null")).exists()) {
            this.gcs.create(bucketInfo.getBucketName());
        }
        this.gcs.createEmptyObjects(subdirsToCreate);
        List createdDirectories = (List)subdirsToCreate.stream().map(s -> this.pathCodec.getPath(s.getBucketName(), s.getObjectName(), false)).collect(ImmutableList.toImmutableList());
        this.tryUpdateTimestampsForParentDirectories(createdDirectories, createdDirectories);
    }

    public void rename(URI src, URI dst) throws IOException {
        ((GoogleLogger.Api)logger.atFine()).log("rename(%s, %s)", (Object)src, (Object)dst);
        Preconditions.checkNotNull((Object)src);
        Preconditions.checkNotNull((Object)dst);
        Preconditions.checkArgument((!src.equals(GCS_ROOT) ? 1 : 0) != 0, (Object)"Root path cannot be renamed.");
        String srcItemName = this.getItemName(src);
        URI dstParent = this.getParentPath(dst);
        ArrayList<URI> paths = new ArrayList<URI>();
        paths.add(src);
        paths.add(dst);
        if (dstParent != null) {
            paths.add(dstParent);
        }
        List<FileInfo> fileInfo = this.getFileInfos(paths);
        FileInfo srcInfo = fileInfo.get(0);
        FileInfo dstInfo = fileInfo.get(1);
        FileInfo dstParentInfo = dstParent == null ? null : fileInfo.get(2);
        src = srcInfo.getPath();
        dst = dstInfo.getPath();
        if (!srcInfo.exists()) {
            throw new FileNotFoundException("Item not found: " + src);
        }
        if (!srcInfo.isDirectory() && dst.equals(GCS_ROOT)) {
            throw new IOException("A file cannot be created in root.");
        }
        if (dstInfo.exists() && !dstInfo.isDirectory() && (srcInfo.isDirectory() || !dst.equals(src))) {
            throw new IOException("Cannot overwrite existing file: " + dst);
        }
        if (dstParentInfo != null && !dstParentInfo.exists()) {
            throw new IOException("Cannot rename because path does not exist: " + dstParent);
        }
        if (srcInfo.isDirectory()) {
            if (!dstInfo.isDirectory()) {
                dst = FileInfo.convertToDirectoryPath(this.pathCodec, dst);
                dstInfo = this.getFileInfo(dst);
            }
            if (src.equals(dst)) {
                throw new IOException("Rename dir to self is forbidden");
            }
            URI dstRelativeToSrc = src.relativize(dst);
            if (!dstRelativeToSrc.equals(dst)) {
                throw new IOException("Rename to subdir is forbidden");
            }
            if (dstInfo.exists()) {
                dst = dst.equals(GCS_ROOT) ? this.pathCodec.getPath(srcItemName, null, true) : FileInfo.convertToDirectoryPath(this.pathCodec, dst.resolve(srcItemName));
            }
        } else if (dstInfo.isDirectory()) {
            if (!dstInfo.exists()) {
                throw new IOException("Cannot rename because path does not exist: " + dstInfo.getPath());
            }
            dst = dst.resolve(srcItemName);
        } else {
            URI dstDir = FileInfo.convertToDirectoryPath(this.pathCodec, dst);
            FileInfo dstDirInfo = this.getFileInfo(dstDir);
            if (dstDirInfo.exists()) {
                dst = dstDir.resolve(srcItemName);
            }
        }
        if (src.equals(dst)) {
            return;
        }
        this.renameInternal(srcInfo, dst);
    }

    public void compose(List<URI> sources, URI destination, String contentType) throws IOException {
        StorageResourceId destResource = StorageResourceId.fromObjectName(destination.toString());
        List sourceObjects = Lists.transform(sources, uri -> StorageResourceId.fromObjectName(uri.toString()).getObjectName());
        this.gcs.compose(destResource.getBucketName(), sourceObjects, destResource.getObjectName(), contentType);
    }

    private void renameInternal(FileInfo srcInfo, URI dst) throws IOException {
        if (srcInfo.isDirectory()) {
            this.renameDirectoryInternal(srcInfo, dst);
        } else {
            URI src = srcInfo.getPath();
            StorageResourceId srcResourceId = this.pathCodec.validatePathAndGetId(src, true);
            StorageResourceId dstResourceId = this.pathCodec.validatePathAndGetId(dst, true);
            this.gcs.copy(srcResourceId.getBucketName(), (List<String>)ImmutableList.of((Object)srcResourceId.getObjectName()), dstResourceId.getBucketName(), (List<String>)ImmutableList.of((Object)dstResourceId.getObjectName()));
            this.tryUpdateTimestampsForParentDirectories((List<URI>)ImmutableList.of((Object)dst), (List<URI>)ImmutableList.of());
            this.gcs.deleteObjects((List<StorageResourceId>)ImmutableList.of((Object)new StorageResourceId(srcInfo.getItemInfo().getBucketName(), srcInfo.getItemInfo().getObjectName(), srcInfo.getItemInfo().getContentGeneration())));
            this.tryUpdateTimestampsForParentDirectories((List<URI>)ImmutableList.of((Object)src), (List<URI>)ImmutableList.of());
        }
    }

    private void renameDirectoryInternal(FileInfo srcInfo, URI dst) throws IOException {
        Preconditions.checkArgument((boolean)srcInfo.isDirectory(), (String)"'%s' should be a directory", (Object)srcInfo);
        Pattern markerFilePattern = this.options.getMarkerFilePattern();
        TreeMap<FileInfo, URI> srcToDstItemNames = new TreeMap<FileInfo, URI>(FILE_INFO_PATH_COMPARATOR);
        TreeMap<FileInfo, URI> srcToDstMarkerItemNames = new TreeMap<FileInfo, URI>(FILE_INFO_PATH_COMPARATOR);
        List<FileInfo> srcItemInfos = this.listAllFileInfoForPrefix(srcInfo.getPath());
        dst = FileInfo.convertToDirectoryPath(this.pathCodec, dst);
        this.mkdir(dst);
        String prefix = srcInfo.getPath().toString();
        for (FileInfo srcItemInfo : srcItemInfos) {
            String relativeItemName = srcItemInfo.getPath().toString().substring(prefix.length());
            URI dstItemName = dst.resolve(relativeItemName);
            if (markerFilePattern != null && markerFilePattern.matcher(relativeItemName).matches()) {
                srcToDstMarkerItemNames.put(srcItemInfo, dstItemName);
                continue;
            }
            srcToDstItemNames.put(srcItemInfo, dstItemName);
        }
        this.copyInternal(srcToDstItemNames);
        this.copyInternal(srcToDstMarkerItemNames);
        if (!srcToDstItemNames.isEmpty() || !srcToDstMarkerItemNames.isEmpty()) {
            ArrayList<URI> allDestinationUris = new ArrayList<URI>(srcToDstItemNames.size() + srcToDstMarkerItemNames.size());
            allDestinationUris.addAll(srcToDstItemNames.values());
            allDestinationUris.addAll(srcToDstMarkerItemNames.values());
            this.tryUpdateTimestampsForParentDirectories(allDestinationUris, allDestinationUris);
        }
        ArrayList<FileInfo> bucketsToDelete = new ArrayList<FileInfo>(1);
        ArrayList<FileInfo> srcItemsToDelete = new ArrayList<FileInfo>(srcToDstItemNames.size() + 1);
        srcItemsToDelete.addAll(srcToDstItemNames.keySet());
        if (srcInfo.getItemInfo().isBucket()) {
            bucketsToDelete.add(srcInfo);
        } else {
            srcItemsToDelete.add(srcInfo);
        }
        this.deleteInternal(new ArrayList<FileInfo>(srcToDstMarkerItemNames.keySet()), new ArrayList<FileInfo>());
        this.deleteInternal(srcItemsToDelete, bucketsToDelete);
        if (bucketsToDelete.isEmpty()) {
            List srcItemNames = srcItemInfos.stream().map(FileInfo::getPath).collect(Collectors.toCollection(ArrayList::new));
            this.tryUpdateTimestampsForParentDirectories(srcItemNames, srcItemNames);
        }
    }

    private void copyInternal(Map<FileInfo, URI> srcToDstItemNames) throws IOException {
        if (srcToDstItemNames.isEmpty()) {
            return;
        }
        String srcBucketName = null;
        String dstBucketName = null;
        ArrayList<String> srcObjectNames = new ArrayList<String>(srcToDstItemNames.size());
        ArrayList<String> dstObjectNames = new ArrayList<String>(srcToDstItemNames.size());
        for (Map.Entry<FileInfo, URI> srcToDstItemName : srcToDstItemNames.entrySet()) {
            StorageResourceId srcResourceId = srcToDstItemName.getKey().getItemInfo().getResourceId();
            srcBucketName = srcResourceId.getBucketName();
            String srcObjectName = srcResourceId.getObjectName();
            srcObjectNames.add(srcObjectName);
            StorageResourceId dstResourceId = this.pathCodec.validatePathAndGetId(srcToDstItemName.getValue(), true);
            dstBucketName = dstResourceId.getBucketName();
            String dstObjectName = dstResourceId.getObjectName();
            dstObjectNames.add(dstObjectName);
        }
        this.gcs.copy(srcBucketName, srcObjectNames, dstBucketName, dstObjectNames);
    }

    public List<URI> listFileNames(FileInfo fileInfo) throws IOException {
        return this.listFileNames(fileInfo, false);
    }

    public List<URI> listFileNames(FileInfo fileInfo, boolean recursive) throws IOException {
        Preconditions.checkNotNull((Object)fileInfo);
        URI path = fileInfo.getPath();
        ((GoogleLogger.Api)logger.atFine()).log("listFileNames(%s)", (Object)path);
        ArrayList<URI> paths = new ArrayList<URI>();
        if (fileInfo.isDirectory()) {
            if (fileInfo.exists()) {
                if (fileInfo.isGlobalRoot()) {
                    List<String> childNames = this.gcs.listBucketNames();
                    for (String childName : childNames) {
                        URI childPath = this.pathCodec.getPath(childName, null, true);
                        paths.add(childPath);
                        ((GoogleLogger.Api)logger.atFine()).log("listFileNames: added: %s", (Object)childPath);
                    }
                } else {
                    String delimiter = recursive ? null : "/";
                    GoogleCloudStorageItemInfo itemInfo = fileInfo.getItemInfo();
                    List<String> childNames = this.gcs.listObjectNames(itemInfo.getBucketName(), itemInfo.getObjectName(), delimiter);
                    for (String childName : childNames) {
                        URI childPath = this.pathCodec.getPath(itemInfo.getBucketName(), childName, false);
                        paths.add(childPath);
                        ((GoogleLogger.Api)logger.atFine()).log("listFileNames: added: %s", (Object)childPath);
                    }
                }
            }
        } else {
            paths.add(path);
            ((GoogleLogger.Api)logger.atFine()).log("listFileNames: added single original path since !isDirectory(): %s", (Object)path);
        }
        return paths;
    }

    public boolean repairPossibleImplicitDirectory(URI path) throws IOException {
        ((GoogleLogger.Api)logger.atFine()).log("repairPossibleImplicitDirectory(%s)", (Object)path);
        Preconditions.checkNotNull((Object)path);
        StorageResourceId resourceId = this.pathCodec.validatePathAndGetId(path, true);
        StorageResourceId dirId = FileInfo.convertToDirectoryPath(resourceId);
        if (dirId.isRoot() || dirId.isBucket() || "/".equals(dirId.getObjectName())) {
            return this.gcs.getItemInfo(dirId).exists();
        }
        try {
            this.gcs.listObjectInfo(dirId.getBucketName(), FileInfo.convertToFilePath(dirId.getObjectName()), "/");
        }
        catch (IOException e) {
            ((GoogleLogger.Api)((GoogleLogger.Api)logger.atWarning()).withCause((Throwable)e)).log("Got exception trying to listObjectNames for '%s'", (Object)FileInfo.convertToFilePath(dirId.toString()));
        }
        return this.gcs.getItemInfo(dirId).exists();
    }

    public List<FileInfo> listAllFileInfoForPrefix(URI prefix) throws IOException {
        ((GoogleLogger.Api)logger.atFine()).log("listAllFileInfoForPrefixPage(%s)", (Object)prefix);
        StorageResourceId prefixId = this.getPrefixId(prefix);
        List<GoogleCloudStorageItemInfo> itemInfos = this.gcs.listObjectInfo(prefixId.getBucketName(), prefixId.getObjectName(), null);
        List<FileInfo> fileInfos = FileInfo.fromItemInfos(this.pathCodec, itemInfos);
        fileInfos.sort(FILE_INFO_PATH_COMPARATOR);
        return fileInfos;
    }

    public GoogleCloudStorage.ListPage<FileInfo> listAllFileInfoForPrefixPage(URI prefix, String pageToken) throws IOException {
        ((GoogleLogger.Api)logger.atFine()).log("listAllFileInfoForPrefixPage(%s, %s)", (Object)prefix, (Object)pageToken);
        StorageResourceId prefixId = this.getPrefixId(prefix);
        GoogleCloudStorage.ListPage<GoogleCloudStorageItemInfo> itemInfosPage = this.gcs.listObjectInfoPage(prefixId.getBucketName(), prefixId.getObjectName(), null, pageToken);
        List<FileInfo> fileInfosPage = FileInfo.fromItemInfos(this.pathCodec, itemInfosPage.getItems());
        fileInfosPage.sort(FILE_INFO_PATH_COMPARATOR);
        return new GoogleCloudStorage.ListPage<FileInfo>(fileInfosPage, itemInfosPage.getNextPageToken());
    }

    private StorageResourceId getPrefixId(URI prefix) {
        Preconditions.checkNotNull((Object)prefix, (Object)"prefix could not be null");
        StorageResourceId prefixId = this.pathCodec.validatePathAndGetId(prefix, true);
        Preconditions.checkArgument((!prefixId.isRoot() ? 1 : 0) != 0, (String)"prefix must not be global root, got '%s'", (Object)prefix);
        return prefixId;
    }

    public List<FileInfo> listFileInfo(URI path, boolean enableAutoRepair) throws IOException {
        ((GoogleLogger.Api)logger.atFine()).log("listFileInfo(%s, %s)", (Object)path, enableAutoRepair);
        Preconditions.checkNotNull((Object)path);
        StorageResourceId pathId = this.pathCodec.validatePathAndGetId(path, true);
        StorageResourceId dirId = this.pathCodec.validatePathAndGetId(FileInfo.convertToDirectoryPath(this.pathCodec, path), true);
        ExecutorService dirExecutor = Executors.newFixedThreadPool(2, DAEMON_THREAD_FACTORY);
        try {
            GoogleCloudStorageItemInfo pathInfo;
            Future<GoogleCloudStorageItemInfo> dirFuture = dirExecutor.submit(() -> this.gcs.getItemInfo(dirId));
            Future<List> dirChildrenFutures = dirExecutor.submit(() -> dirId.isRoot() ? this.gcs.listBucketInfo() : this.gcs.listObjectInfo(dirId.getBucketName(), dirId.getObjectName(), "/"));
            dirExecutor.shutdown();
            if (!pathId.isDirectory() && (pathInfo = this.gcs.getItemInfo(pathId)).exists()) {
                ArrayList<FileInfo> listedInfo = new ArrayList<FileInfo>();
                listedInfo.add(FileInfo.fromItemInfo(this.pathCodec, pathInfo));
                ArrayList<FileInfo> arrayList = listedInfo;
                return arrayList;
            }
            GoogleCloudStorageItemInfo dirInfo = dirFuture.get();
            List dirItemInfos = dirChildrenFutures.get();
            if (!dirInfo.exists() && dirItemInfos.isEmpty()) {
                throw new FileNotFoundException("Item not found: " + path);
            }
            if (!dirInfo.exists() && enableAutoRepair) {
                try {
                    this.gcs.createEmptyObject(dirId);
                }
                catch (IOException e) {
                    ((GoogleLogger.Api)((GoogleLogger.Api)logger.atWarning()).withCause((Throwable)e)).log("Failed to repair implicit directory '%s'", (Object)dirId);
                }
            }
            List<FileInfo> fileInfos = FileInfo.fromItemInfos(this.pathCodec, dirItemInfos);
            fileInfos.sort(FILE_INFO_PATH_COMPARATOR);
            List<FileInfo> list = fileInfos;
            return list;
        }
        finally {
            dirExecutor.shutdownNow();
        }
    }

    public FileInfo getFileInfo(URI path) throws IOException {
        ((GoogleLogger.Api)logger.atFine()).log("getFileInfo(%s)", (Object)path);
        Preconditions.checkArgument((path != null ? 1 : 0) != 0, (Object)"path must not be null");
        StorageResourceId resourceId = this.pathCodec.validatePathAndGetId(path, true);
        FileInfo fileInfo = FileInfo.fromItemInfo(this.pathCodec, this.getFileInfoInternal(resourceId));
        ((GoogleLogger.Api)logger.atFine()).log("getFileInfo: %s", (Object)fileInfo);
        return fileInfo;
    }

    private GoogleCloudStorageItemInfo getFileInfoInternal(StorageResourceId resourceId) throws IOException {
        if (resourceId.isRoot() || resourceId.isBucket()) {
            return this.gcs.getItemInfo(resourceId);
        }
        StorageResourceId dirId = FileInfo.convertToDirectoryPath(resourceId);
        ExecutorService dirExecutor = resourceId.isDirectory() ? Executors.newSingleThreadExecutor(DAEMON_THREAD_FACTORY) : Executors.newFixedThreadPool(2, DAEMON_THREAD_FACTORY);
        try {
            GoogleCloudStorageItemInfo itemInfo;
            Future<List> dirChildFuture = dirExecutor.submit(() -> this.gcs.listObjectNames(dirId.getBucketName(), dirId.getObjectName(), "/", 1L));
            Object dirFuture = resourceId.isDirectory() ? Futures.immediateFuture((Object)this.gcs.getItemInfo(resourceId)) : dirExecutor.submit(() -> this.gcs.getItemInfo(dirId));
            dirExecutor.shutdown();
            if (!resourceId.isDirectory() && (itemInfo = this.gcs.getItemInfo(resourceId)).exists()) {
                GoogleCloudStorageItemInfo googleCloudStorageItemInfo = itemInfo;
                return googleCloudStorageItemInfo;
            }
            GoogleCloudStorageItemInfo dirInfo = (GoogleCloudStorageItemInfo)dirFuture.get();
            if (dirInfo.exists()) {
                GoogleCloudStorageItemInfo googleCloudStorageItemInfo = dirInfo;
                return googleCloudStorageItemInfo;
            }
            if (dirChildFuture.get().isEmpty()) {
                GoogleCloudStorageItemInfo googleCloudStorageItemInfo = GoogleCloudStorageItemInfo.createNotFound(resourceId);
                return googleCloudStorageItemInfo;
            }
            GoogleCloudStorageItemInfo googleCloudStorageItemInfo = this.gcs.getOptions().isInferImplicitDirectoriesEnabled() ? GoogleCloudStorageItemInfo.createInferredDirectory(resourceId) : GoogleCloudStorageItemInfo.createNotFound(resourceId);
            return googleCloudStorageItemInfo;
        }
        finally {
            dirExecutor.shutdownNow();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public List<FileInfo> getFileInfos(List<URI> paths) throws IOException {
        Preconditions.checkArgument((paths != null ? 1 : 0) != 0, (Object)"paths must not be null");
        ((GoogleLogger.Api)logger.atFine()).log("getFileInfos(%d paths)", paths.size());
        if (paths.size() == 1) {
            return new ArrayList<FileInfo>(Collections.singleton(this.getFileInfo(paths.get(0))));
        }
        int maxThreads = this.gcs.getOptions().getBatchThreads();
        ListeningExecutorService fileInfoExecutor = maxThreads == 0 ? MoreExecutors.newDirectExecutorService() : Executors.newFixedThreadPool(Math.min(maxThreads, paths.size()), DAEMON_THREAD_FACTORY);
        try {
            ArrayList<Future<FileInfo>> infoFutures = new ArrayList<Future<FileInfo>>(paths.size());
            for (URI path : paths) {
                infoFutures.add(fileInfoExecutor.submit(() -> this.getFileInfo(path)));
            }
            fileInfoExecutor.shutdown();
            ArrayList infos = new ArrayList(paths.size());
            for (Future future : infoFutures) {
                try {
                    infos.add(future.get());
                }
                catch (InterruptedException | ExecutionException e) {
                    if (e instanceof InterruptedException) {
                        Thread.currentThread().interrupt();
                    }
                    throw new IOException(String.format("Failed to getFileInfos for %d paths", paths.size()), e);
                }
            }
            ArrayList arrayList = infos;
            return arrayList;
        }
        finally {
            fileInfoExecutor.shutdownNow();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void close() {
        if (this.gcs != null) {
            ((GoogleLogger.Api)logger.atFine()).log("close()");
            try {
                this.gcs.close();
            }
            finally {
                this.gcs = null;
                if (this.updateTimestampsExecutor != null) {
                    this.updateTimestampsExecutor.shutdown();
                    try {
                        if (!this.updateTimestampsExecutor.awaitTermination(10L, TimeUnit.SECONDS)) {
                            ((GoogleLogger.Api)logger.atWarning()).log("Forcibly shutting down timestamp update thread pool.");
                            this.updateTimestampsExecutor.shutdownNow();
                        }
                    }
                    catch (InterruptedException e) {
                        Thread.currentThread().interrupt();
                        ((GoogleLogger.Api)((GoogleLogger.Api)logger.atFine()).withCause((Throwable)e)).log("Failed to await termination: forcibly shutting down timestamp update thread pool");
                        this.updateTimestampsExecutor.shutdownNow();
                    }
                    finally {
                        this.updateTimestampsExecutor = null;
                    }
                }
            }
        }
    }

    @VisibleForTesting
    public void mkdir(URI path) throws IOException {
        ((GoogleLogger.Api)logger.atFine()).log("mkdir(%s)", (Object)path);
        Preconditions.checkNotNull((Object)path);
        Preconditions.checkArgument((!path.equals(GCS_ROOT) ? 1 : 0) != 0, (Object)"Cannot create root directory.");
        StorageResourceId resourceId = this.pathCodec.validatePathAndGetId(path, true);
        if (resourceId.isBucket()) {
            this.gcs.create(resourceId.getBucketName());
            return;
        }
        resourceId = FileInfo.convertToDirectoryPath(resourceId);
        this.gcs.createEmptyObject(resourceId);
        this.tryUpdateTimestampsForParentDirectories((List<URI>)ImmutableList.of((Object)path), (List<URI>)ImmutableList.of());
    }

    protected void updateTimestampsForParentDirectories(List<URI> modifiedObjects, List<URI> excludedParents) throws IOException {
        ((GoogleLogger.Api)logger.atFine()).log("updateTimestampsForParentDirectories(%s, %s)", modifiedObjects, excludedParents);
        GoogleCloudStorageFileSystemOptions.TimestampUpdatePredicate updatePredicate = this.options.getShouldIncludeInTimestampUpdatesPredicate();
        HashSet<URI> excludedParentPathsSet = new HashSet<URI>(excludedParents);
        HashSet parentUrisToUpdate = Sets.newHashSetWithExpectedSize((int)modifiedObjects.size());
        for (URI modifiedObjectUri : modifiedObjects) {
            URI parentPathUri = this.getParentPath(modifiedObjectUri);
            if (excludedParentPathsSet.contains(parentPathUri) || !updatePredicate.shouldUpdateTimestamp(parentPathUri)) continue;
            parentUrisToUpdate.add(parentPathUri);
        }
        HashMap<String, byte[]> modificationAttributes = new HashMap<String, byte[]>();
        FileInfo.addModificationTimeToAttributes(modificationAttributes, Clock.SYSTEM);
        ArrayList<UpdatableItemInfo> itemUpdates = new ArrayList<UpdatableItemInfo>(parentUrisToUpdate.size());
        for (URI parentUri : parentUrisToUpdate) {
            StorageResourceId resourceId = this.pathCodec.validatePathAndGetId(parentUri, true);
            if (resourceId.isBucket() || resourceId.isRoot()) continue;
            itemUpdates.add(new UpdatableItemInfo(resourceId, modificationAttributes));
        }
        if (!itemUpdates.isEmpty()) {
            this.gcs.updateItems(itemUpdates);
        } else {
            ((GoogleLogger.Api)logger.atFine()).log("All paths were excluded from directory timestamp updating.");
        }
    }

    protected void tryUpdateTimestampsForParentDirectories(List<URI> modifiedObjects, List<URI> excludedParents) {
        ((GoogleLogger.Api)logger.atFine()).log("tryUpdateTimestampsForParentDirectories(%s, %s)", modifiedObjects, excludedParents);
        try {
            Future<?> future = this.updateTimestampsExecutor.submit(() -> {
                try {
                    this.updateTimestampsForParentDirectories(modifiedObjects, excludedParents);
                }
                catch (IOException ioe) {
                    ((GoogleLogger.Api)((GoogleLogger.Api)logger.atFine()).withCause((Throwable)ioe)).log("Exception caught when trying to update parent directory timestamps.");
                }
            });
        }
        catch (RejectedExecutionException ree) {
            ((GoogleLogger.Api)((GoogleLogger.Api)logger.atFine()).withCause((Throwable)ree)).log("Exhausted thread pool and queue space while updating parent timestamps");
        }
    }

    static List<String> getSubDirs(String objectName) {
        ArrayList<String> subdirs = new ArrayList<String>();
        if (!Strings.isNullOrEmpty((String)objectName)) {
            int index;
            int currentIndex = 0;
            while (currentIndex < objectName.length() && (index = objectName.indexOf("/", currentIndex)) >= 0) {
                subdirs.add(objectName.substring(0, index + "/".length()));
                currentIndex = index + "/".length();
            }
        }
        return subdirs;
    }

    static String validateBucketName(String bucketName) {
        if (Strings.isNullOrEmpty((String)(bucketName = FileInfo.convertToFilePath(bucketName)))) {
            throw new IllegalArgumentException("Google Cloud Storage bucket name cannot be empty.");
        }
        if (!BUCKET_NAME_CHAR_MATCHER.matchesAllOf((CharSequence)bucketName)) {
            throw new IllegalArgumentException(String.format("Invalid bucket name '%s': bucket name must contain only 'a-z0-9_.-' characters.", bucketName));
        }
        return bucketName;
    }

    static String validateObjectName(String objectName, boolean allowEmptyObjectName) {
        ((GoogleLogger.Api)logger.atFine()).log("validateObjectName('%s', %s)", (Object)objectName, allowEmptyObjectName);
        if (Strings.isNullOrEmpty((String)objectName) || objectName.equals("/")) {
            if (allowEmptyObjectName) {
                objectName = "";
            } else {
                throw new IllegalArgumentException("Google Cloud Storage path must include non-empty object name.");
            }
        }
        for (int i = 0; i < objectName.length() - 1; ++i) {
            if (objectName.charAt(i) != '/' || objectName.charAt(i + 1) != '/') continue;
            throw new IllegalArgumentException(String.format("Google Cloud Storage path must not have consecutive '/' characters, got '%s'", objectName));
        }
        if (objectName.startsWith("/")) {
            objectName = objectName.substring(1);
        }
        ((GoogleLogger.Api)logger.atFine()).log("validateObjectName -> '%s'", (Object)objectName);
        return objectName;
    }

    String getItemName(URI path) {
        Preconditions.checkNotNull((Object)path);
        if (path.equals(GCS_ROOT)) {
            return null;
        }
        StorageResourceId resourceId = this.pathCodec.validatePathAndGetId(path, true);
        if (resourceId.isBucket()) {
            return resourceId.getBucketName();
        }
        String objectName = resourceId.getObjectName();
        int index = FileInfo.objectHasDirectoryPath(objectName) ? objectName.lastIndexOf("/", objectName.length() - 2) : objectName.lastIndexOf("/");
        return index < 0 ? objectName : objectName.substring(index + 1);
    }

    public URI getParentPath(URI path) {
        return GoogleCloudStorageFileSystem.getParentPath(this.getPathCodec(), path);
    }

    public GoogleCloudStorage getGcs() {
        return this.gcs;
    }

    public PathCodec getPathCodec() {
        return this.pathCodec;
    }

    @Deprecated
    public static StorageResourceId validatePathAndGetId(URI uri, boolean allowEmptyObjectNames) {
        return LEGACY_PATH_CODEC.validatePathAndGetId(uri, allowEmptyObjectNames);
    }

    @Deprecated
    public static URI getPath(String bucketName, String objectName, boolean allowEmptyObjectName) {
        return LEGACY_PATH_CODEC.getPath(bucketName, objectName, allowEmptyObjectName);
    }

    @Deprecated
    public static URI getPath(String bucketName) {
        return LEGACY_PATH_CODEC.getPath(bucketName, null, true);
    }

    @Deprecated
    public static URI getPath(String bucketName, String objectName) {
        return LEGACY_PATH_CODEC.getPath(bucketName, objectName, false);
    }

    @Deprecated
    public static URI getParentPath(PathCodec pathCodec, URI path) {
        Preconditions.checkNotNull((Object)path);
        if (path.equals(GCS_ROOT)) {
            return null;
        }
        StorageResourceId resourceId = pathCodec.validatePathAndGetId(path, true);
        if (resourceId.isBucket()) {
            return GCS_ROOT;
        }
        String objectName = resourceId.getObjectName();
        int index = FileInfo.objectHasDirectoryPath(objectName) ? objectName.lastIndexOf("/", objectName.length() - 2) : objectName.lastIndexOf("/");
        return index < 0 ? pathCodec.getPath(resourceId.getBucketName(), null, true) : pathCodec.getPath(resourceId.getBucketName(), objectName.substring(0, index + 1), false);
    }
}

