/*
 * Decompiled with CFR 0.152.
 */
package com.google.cloud.hadoop.gcsio;

import com.google.api.client.googleapis.services.AbstractGoogleClientRequest;
import com.google.api.client.http.HttpHeaders;
import com.google.api.client.http.HttpResponse;
import com.google.api.client.util.BackOff;
import com.google.api.client.util.BackOffUtils;
import com.google.api.client.util.ExponentialBackOff;
import com.google.api.client.util.NanoClock;
import com.google.api.client.util.Sleeper;
import com.google.api.services.storage.Storage;
import com.google.api.services.storage.model.StorageObject;
import com.google.cloud.hadoop.gcsio.GoogleCloudStorageExceptions;
import com.google.cloud.hadoop.gcsio.GoogleCloudStorageReadOptions;
import com.google.cloud.hadoop.gcsio.StorageResourceId;
import com.google.cloud.hadoop.util.ApiErrorExtractor;
import com.google.cloud.hadoop.util.ClientRequestHelper;
import com.google.cloud.hadoop.util.ResilientOperation;
import com.google.cloud.hadoop.util.RetryBoundedBackOff;
import com.google.cloud.hadoop.util.RetryDeterminer;
import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Preconditions;
import com.google.common.base.Strings;
import com.google.common.base.Supplier;
import com.google.common.base.Suppliers;
import com.google.common.io.ByteStreams;
import java.io.BufferedInputStream;
import java.io.ByteArrayInputStream;
import java.io.EOFException;
import java.io.IOException;
import java.io.InputStream;
import java.nio.ByteBuffer;
import java.nio.channels.Channels;
import java.nio.channels.ClosedChannelException;
import java.nio.channels.ReadableByteChannel;
import java.nio.channels.SeekableByteChannel;
import javax.annotation.Nonnull;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class GoogleCloudStorageReadChannel
implements SeekableByteChannel {
    private static final Logger LOG = LoggerFactory.getLogger(GoogleCloudStorageReadChannel.class);
    @VisibleForTesting
    static final int SKIP_BUFFER_SIZE = 8192;
    private final Storage gcs;
    private final String bucketName;
    private final String objectName;
    private final String resourceIdString;
    @VisibleForTesting
    ReadableByteChannel contentChannel;
    private boolean channelIsOpen = true;
    @VisibleForTesting
    protected long currentPosition = 0L;
    @VisibleForTesting
    long contentChannelPosition = -1L;
    private long size = -1L;
    private long contentChannelEnd = -1L;
    @VisibleForTesting
    boolean randomAccess;
    private int maxRetries = 10;
    private final ApiErrorExtractor errorExtractor;
    private final ClientRequestHelper<StorageObject> clientRequestHelper;
    private final GoogleCloudStorageReadOptions readOptions;
    private Sleeper sleeper = Sleeper.DEFAULT;
    private NanoClock clock = NanoClock.SYSTEM;
    private Supplier<BackOff> backOff = Suppliers.memoize(this::createBackOff);
    private Supplier<BackOff> readBackOff = Suppliers.memoize(this::createBackOff);
    private byte[] skipBuffer = null;
    private boolean gzipEncoded = false;
    private byte[] footerContent;
    @Deprecated
    public static final int DEFAULT_BACKOFF_INITIAL_INTERVAL_MILLIS = 200;
    @Deprecated
    public static final double DEFAULT_BACKOFF_RANDOMIZATION_FACTOR = 0.5;
    @Deprecated
    public static final double DEFAULT_BACKOFF_MULTIPLIER = 1.5;
    @Deprecated
    public static final int DEFAULT_BACKOFF_MAX_INTERVAL_MILLIS = 10000;
    @Deprecated
    public static final int DEFAULT_BACKOFF_MAX_ELAPSED_TIME_MILLIS = 120000;

    public GoogleCloudStorageReadChannel(Storage gcs, String bucketName, String objectName, ApiErrorExtractor errorExtractor, ClientRequestHelper<StorageObject> requestHelper) throws IOException {
        this(gcs, bucketName, objectName, errorExtractor, requestHelper, GoogleCloudStorageReadOptions.DEFAULT);
    }

    public GoogleCloudStorageReadChannel(Storage gcs, String bucketName, String objectName, ApiErrorExtractor errorExtractor, ClientRequestHelper<StorageObject> requestHelper, @Nonnull GoogleCloudStorageReadOptions readOptions) throws IOException {
        this.gcs = gcs;
        this.clientRequestHelper = requestHelper;
        this.bucketName = bucketName;
        this.objectName = objectName;
        this.errorExtractor = errorExtractor;
        this.readOptions = readOptions;
        this.resourceIdString = StorageResourceId.createReadableString(bucketName, objectName);
        if (readOptions.getFadvise() == GoogleCloudStorageReadOptions.Fadvise.SEQUENTIAL || readOptions.getFooterPrefetchSize() == 0) {
            this.initEncodingAndSize();
        } else {
            this.initEncodingAndSizeAndPrefetchFooter();
        }
        this.randomAccess = !this.gzipEncoded && readOptions.getFadvise() == GoogleCloudStorageReadOptions.Fadvise.RANDOM;
        this.checkEncodingAndAccess();
        int prefetchedFooterSize = this.footerContent == null ? 0 : this.footerContent.length;
        LOG.debug("Created and initialized new channel (encoding={}, size={}, randomAccess={}, prefetchedFooterSize={}) for '{}'", new Object[]{this.gzipEncoded ? "gzip" : "other", this.size, this.randomAccess, prefetchedFooterSize, this.resourceIdString});
    }

    @VisibleForTesting
    protected GoogleCloudStorageReadChannel(@Nonnull GoogleCloudStorageReadOptions readOptions) throws IOException {
        this(null, null, null, null, null, readOptions);
    }

    @VisibleForTesting
    void setSleeper(Sleeper sleeper) {
        Preconditions.checkArgument((sleeper != null ? 1 : 0) != 0, (Object)"sleeper must not be null!");
        this.sleeper = sleeper;
    }

    @VisibleForTesting
    void setNanoClock(NanoClock clock) {
        Preconditions.checkArgument((clock != null ? 1 : 0) != 0, (Object)"clock must not be null!");
        this.clock = clock;
    }

    @VisibleForTesting
    void setBackOff(BackOff backOff) {
        this.backOff = Suppliers.ofInstance((Object)Preconditions.checkNotNull((Object)backOff, (Object)"backOff could not be null"));
    }

    void setReadBackOff(BackOff backOff) {
        this.readBackOff = Suppliers.ofInstance((Object)Preconditions.checkNotNull((Object)backOff, (Object)"backOff could not be null"));
    }

    @VisibleForTesting
    BackOff getBackOff() {
        return (BackOff)this.backOff.get();
    }

    @VisibleForTesting
    BackOff getReadBackOff() {
        return (BackOff)this.readBackOff.get();
    }

    @VisibleForTesting
    ExponentialBackOff createBackOff() {
        return new ExponentialBackOff.Builder().setInitialIntervalMillis(this.readOptions.getBackoffInitialIntervalMillis()).setRandomizationFactor(this.readOptions.getBackoffRandomizationFactor()).setMultiplier(this.readOptions.getBackoffMultiplier()).setMaxIntervalMillis(this.readOptions.getBackoffMaxIntervalMillis()).setMaxElapsedTimeMillis(this.readOptions.getBackoffMaxElapsedTimeMillis()).setNanoClock(this.clock).build();
    }

    public void setMaxRetries(int maxRetries) {
        this.maxRetries = maxRetries;
    }

    @Override
    public int read(ByteBuffer buffer) throws IOException {
        boolean isEndOfStream;
        this.throwIfNotOpen();
        Preconditions.checkState((this.size >= 0L ? 1 : 0) != 0, (String)"size should be initialized already, but was %s for '%s'", (long)this.size, (Object)this.resourceIdString);
        if (buffer.remaining() == 0) {
            return 0;
        }
        LOG.debug("Reading {} bytes at {} position from '{}'", new Object[]{buffer.remaining(), this.currentPosition, this.resourceIdString});
        if (this.currentPosition == this.size) {
            return -1;
        }
        int totalBytesRead = 0;
        int retriesAttempted = 0;
        do {
            int remainingBeforeRead = buffer.remaining();
            this.performLazySeek(remainingBeforeRead);
            Preconditions.checkState((this.contentChannelPosition == this.currentPosition ? 1 : 0) != 0, (String)"contentChannelPosition (%s) should be equal to currentPosition (%s) after lazy seek", (long)this.contentChannelPosition, (long)this.currentPosition);
            try {
                int numBytesRead = this.contentChannel.read(buffer);
                this.checkIOPrecondition(numBytesRead != 0, "Read 0 bytes without blocking");
                if (numBytesRead < 0) {
                    if (this.gzipEncoded) {
                        this.size = this.currentPosition;
                        this.contentChannelEnd = this.currentPosition;
                    }
                    this.checkIOPrecondition(this.currentPosition == this.contentChannelEnd || this.currentPosition == this.size, String.format("Received end of stream result before all the file data has been received; totalBytesRead: %d, currentPosition: %d, contentChannelEnd %d, size: %d, object: '%s'", totalBytesRead, this.currentPosition, this.contentChannelEnd, this.size, this.resourceIdString));
                    if (this.contentChannelEnd == this.size || this.currentPosition != this.contentChannelEnd) break;
                    this.closeContentChannel();
                }
                if (numBytesRead > 0) {
                    totalBytesRead += numBytesRead;
                    this.currentPosition += (long)numBytesRead;
                    this.contentChannelPosition += (long)numBytesRead;
                    Preconditions.checkState((this.contentChannelPosition == this.currentPosition ? 1 : 0) != 0, (String)"contentChannelPosition (%s) should be equal to currentPosition (%s) after successful read", (long)this.contentChannelPosition, (long)this.currentPosition);
                }
                if (retriesAttempted != 0) {
                    LOG.info("Success after {} retries on reading '{}'", (Object)retriesAttempted, (Object)this.resourceIdString);
                }
                retriesAttempted = 0;
            }
            catch (IOException ioe) {
                if (retriesAttempted == this.maxRetries) {
                    LOG.error("Already attempted max of {} retries while reading '{}'; throwing exception.", (Object)this.maxRetries, (Object)this.resourceIdString);
                    this.closeContentChannel();
                    throw ioe;
                }
                if (retriesAttempted == 0) {
                    ((BackOff)this.readBackOff.get()).reset();
                }
                LOG.warn("Got exception: {} while reading '{}'; retry #{}. Sleeping...", new Object[]{ioe.getMessage(), this.resourceIdString, ++retriesAttempted});
                try {
                    boolean backOffSuccessful = BackOffUtils.next((Sleeper)this.sleeper, (BackOff)((BackOff)this.readBackOff.get()));
                    if (!backOffSuccessful) {
                        LOG.error("BackOff returned false; maximum total elapsed time exhausted. Giving up after {} retries for '{}'", (Object)retriesAttempted, (Object)this.resourceIdString);
                        this.closeContentChannel();
                        throw ioe;
                    }
                }
                catch (InterruptedException ie) {
                    LOG.error("Interrupted while sleeping before retry. Giving up after {} retries for '{}'", (Object)retriesAttempted, (Object)this.resourceIdString);
                    ioe.addSuppressed(ie);
                    this.closeContentChannel();
                    throw ioe;
                }
                LOG.info("Done sleeping before retry for '{}'; retry #{}", (Object)this.resourceIdString, (Object)retriesAttempted);
                if (buffer.remaining() != remainingBeforeRead) {
                    int partialRead = remainingBeforeRead - buffer.remaining();
                    LOG.info("Despite exception, had partial read of {} bytes from '{}'; resetting retry count.", (Object)partialRead, (Object)this.resourceIdString);
                    retriesAttempted = 0;
                    totalBytesRead += partialRead;
                    this.currentPosition += (long)partialRead;
                }
                this.closeContentChannel();
            }
            catch (RuntimeException r) {
                this.closeContentChannel();
                throw r;
            }
        } while (buffer.remaining() > 0);
        boolean bl = isEndOfStream = totalBytesRead == 0;
        if (isEndOfStream) {
            this.checkIOPrecondition(this.currentPosition == this.size, String.format("Failed to read any data before all the file data has been received; currentPosition: %d, size: %d, object '%s'", this.currentPosition, this.size, this.resourceIdString));
            return -1;
        }
        return totalBytesRead;
    }

    @Override
    public SeekableByteChannel truncate(long size) throws IOException {
        throw new UnsupportedOperationException("Cannot mutate read-only channel");
    }

    @Override
    public int write(ByteBuffer src) throws IOException {
        throw new UnsupportedOperationException("Cannot mutate read-only channel");
    }

    @Override
    public boolean isOpen() {
        return this.channelIsOpen;
    }

    protected void closeContentChannel() {
        if (this.contentChannel != null) {
            LOG.debug("Closing internal contentChannel for '{}'", (Object)this.resourceIdString);
            try {
                this.contentChannel.close();
            }
            catch (Exception e) {
                LOG.debug("Got an exception on contentChannel.close() for '{}'; ignoring it.", (Object)this.resourceIdString, (Object)e);
            }
            finally {
                this.contentChannel = null;
                this.contentChannelPosition = -1L;
                this.contentChannelEnd = -1L;
            }
        }
    }

    @Override
    public void close() throws IOException {
        if (!this.channelIsOpen) {
            LOG.warn("Channel for '{}' is not open.", (Object)this.resourceIdString);
            return;
        }
        LOG.debug("Closing channel for '{}'", (Object)this.resourceIdString);
        this.channelIsOpen = false;
        this.closeContentChannel();
    }

    @Override
    public long position() throws IOException {
        this.throwIfNotOpen();
        return this.currentPosition;
    }

    @Override
    public SeekableByteChannel position(long newPosition) throws IOException {
        this.throwIfNotOpen();
        if (newPosition == this.currentPosition) {
            return this;
        }
        this.validatePosition(newPosition);
        LOG.debug("Seek from {} to {} position for '{}'", new Object[]{this.currentPosition, newPosition, this.resourceIdString});
        this.currentPosition = newPosition;
        return this;
    }

    private boolean isRandomAccessPattern(long oldPosition) {
        if (!this.shouldDetectRandomAccess()) {
            return false;
        }
        if (this.currentPosition < oldPosition) {
            LOG.debug("Detected backward read from {} to {} position, switching to random IO for '{}'", new Object[]{oldPosition, this.currentPosition, this.resourceIdString});
            return true;
        }
        if (oldPosition >= 0L && oldPosition + this.readOptions.getInplaceSeekLimit() < this.currentPosition) {
            LOG.debug("Detected forward read from {} to {} position over {} threshold, switching to random IO for '{}'", new Object[]{oldPosition, this.currentPosition, this.readOptions.getInplaceSeekLimit(), this.resourceIdString});
            return true;
        }
        return false;
    }

    private boolean shouldDetectRandomAccess() {
        return !this.gzipEncoded && !this.randomAccess && this.readOptions.getFadvise() == GoogleCloudStorageReadOptions.Fadvise.AUTO;
    }

    private void setRandomAccess() {
        this.randomAccess = true;
        this.checkEncodingAndAccess();
    }

    private void skipInPlace(long seekDistance) {
        if (this.skipBuffer == null) {
            this.skipBuffer = new byte[8192];
        }
        while (seekDistance > 0L && this.contentChannel != null) {
            try {
                int bufferSize = Math.toIntExact(Math.min((long)this.skipBuffer.length, seekDistance));
                int bytesRead = this.contentChannel.read(ByteBuffer.wrap(this.skipBuffer, 0, bufferSize));
                if (bytesRead < 0) {
                    LOG.info("Somehow read {} bytes trying to skip {} bytes to seek to position {}, size: {}", new Object[]{bytesRead, seekDistance, this.currentPosition, this.size});
                    this.closeContentChannel();
                    continue;
                }
                seekDistance -= (long)bytesRead;
                this.contentChannelPosition += (long)bytesRead;
            }
            catch (IOException e) {
                LOG.info("Got an IO exception on contentChannel.read(), a lazy-seek will be pending for '{}'", (Object)this.resourceIdString, (Object)e);
                this.closeContentChannel();
            }
        }
        Preconditions.checkState((this.contentChannel == null || this.contentChannelPosition == this.currentPosition ? 1 : 0) != 0, (String)"contentChannelPosition (%s) should be equal to currentPosition (%s) after successful in-place skip", (long)this.contentChannelPosition, (long)this.currentPosition);
    }

    @Override
    public long size() throws IOException {
        this.throwIfNotOpen();
        return this.size;
    }

    @VisibleForTesting
    protected void setSize(long size) {
        this.size = size;
    }

    private void checkEncodingAndAccess() {
        Preconditions.checkState((!this.gzipEncoded || !this.randomAccess ? 1 : 0) != 0, (String)"gzipEncoded and randomAccess should not be true at the same time for '%s'", (Object)this.resourceIdString);
    }

    protected void validatePosition(long position) throws IOException {
        if (position < 0L) {
            throw new EOFException(String.format("Invalid seek offset: position value (%d) must be >= 0 for '%s'", position, this.resourceIdString));
        }
        if (position >= this.size) {
            throw new EOFException(String.format("Invalid seek offset: position value (%d) must be between 0 and %d for '%s'", position, this.size, this.resourceIdString));
        }
    }

    @VisibleForTesting
    void performLazySeek(long bytesToRead) throws IOException {
        this.throwIfNotOpen();
        if (this.currentPosition == this.contentChannelPosition && this.contentChannel != null) {
            return;
        }
        LOG.debug("Performing lazySeek from {} to {} position with {} bytesToRead for '{}'", new Object[]{this.contentChannelPosition, this.currentPosition, bytesToRead, this.resourceIdString});
        long oldPosition = this.contentChannelPosition;
        long seekDistance = this.currentPosition - this.contentChannelPosition;
        if (this.contentChannel != null && seekDistance > 0L && (this.gzipEncoded || seekDistance <= this.readOptions.getInplaceSeekLimit()) && this.currentPosition < this.contentChannelEnd) {
            LOG.debug("Seeking forward {} bytes (inplaceSeekLimit: {}) in-place to position {} for '{}'", new Object[]{seekDistance, this.readOptions.getInplaceSeekLimit(), this.currentPosition, this.resourceIdString});
            this.skipInPlace(seekDistance);
        } else {
            this.closeContentChannel();
        }
        if (this.contentChannel == null) {
            if (this.isRandomAccessPattern(oldPosition)) {
                this.setRandomAccess();
            }
            this.openContentChannel(bytesToRead);
        }
    }

    private void openContentChannel(long bytesToRead) throws IOException {
        Preconditions.checkState((this.contentChannel == null ? 1 : 0) != 0, (Object)"contentChannel should be null, before opening new");
        InputStream objectContentStream = this.footerContent != null && this.currentPosition >= this.size - (long)this.footerContent.length ? this.openFooterStream() : this.openStream(bytesToRead);
        this.contentChannel = Channels.newChannel(objectContentStream);
        this.contentChannelPosition = this.currentPosition;
    }

    @VisibleForTesting
    protected void initEncodingAndSize() throws IOException {
        Preconditions.checkState((this.size < 0L ? 1 : 0) != 0, (String)"size should be not initialized yet, but was %s for '%s'", (long)this.size, (Object)this.resourceIdString);
        StorageObject metadata = this.getMetadata();
        this.gzipEncoded = Strings.nullToEmpty((String)metadata.getContentEncoding()).contains("gzip");
        this.size = this.gzipEncoded ? Long.MAX_VALUE : metadata.getSize().longValue();
    }

    protected StorageObject getMetadata() throws IOException {
        Storage.Objects.Get getObject = this.createRequest();
        return (StorageObject)this.retry(ResilientOperation.getGoogleRequestCallable((AbstractGoogleClientRequest)getObject));
    }

    protected void initEncodingAndSizeAndPrefetchFooter() throws IOException {
        final Storage.Objects.Get getObject = this.createDataRequest("bytes=-" + this.readOptions.getFooterPrefetchSize());
        HttpResponse response = this.retry(new ResilientOperation.CheckedCallable<HttpResponse, IOException>(){

            public HttpResponse call() throws IOException {
                return getObject.executeMedia();
            }
        });
        HttpHeaders responseHeaders = response.getHeaders();
        this.gzipEncoded = Strings.nullToEmpty((String)responseHeaders.getContentEncoding()).contains("gzip");
        if (this.gzipEncoded) {
            this.size = Long.MAX_VALUE;
            return;
        }
        String range = responseHeaders.getContentRange();
        this.size = Long.parseLong(range.substring(range.lastIndexOf(47) + 1));
        int contentSize = Math.toIntExact(responseHeaders.getContentLength());
        if (this.size > 0L) {
            try (BufferedInputStream footerStream = new BufferedInputStream(response.getContent(), contentSize);){
                this.footerContent = ByteStreams.toByteArray((InputStream)footerStream);
            }
            catch (IOException e) {
                LOG.info("Failed to prefetch footer for '{}'", (Object)this.resourceIdString, (Object)e);
                this.footerContent = null;
            }
        }
        Preconditions.checkState((this.footerContent == null || this.footerContent.length == contentSize ? 1 : 0) != 0, (String)"prefetched footer size (%s) should equal to content length header (%s)", this.footerContent == null ? null : Integer.valueOf(this.footerContent.length), (int)contentSize);
    }

    private <T> T retry(ResilientOperation.CheckedCallable<T, IOException> callableToRetry) throws IOException {
        ((BackOff)this.backOff.get()).reset();
        try {
            return (T)ResilientOperation.retry(callableToRetry, (BackOff)new RetryBoundedBackOff(this.maxRetries, (BackOff)this.backOff.get()), (RetryDeterminer)RetryDeterminer.SOCKET_ERRORS, IOException.class, (Sleeper)this.sleeper);
        }
        catch (IOException e) {
            if (this.errorExtractor.itemNotFound(e)) {
                throw GoogleCloudStorageExceptions.getFileNotFoundException(this.bucketName, this.objectName);
            }
            throw new IOException("Error reading " + this.resourceIdString, e);
        }
        catch (InterruptedException e) {
            throw new IOException("Thread interrupt received.", e);
        }
    }

    private InputStream openFooterStream() {
        int offset = Math.toIntExact(this.currentPosition - (this.size - (long)this.footerContent.length));
        int length = this.footerContent.length - offset;
        LOG.debug("Opened stream (prefetched footer) from {} position for '{}'", (Object)this.currentPosition, (Object)this.resourceIdString);
        return new ByteArrayInputStream(this.footerContent, offset, length);
    }

    protected InputStream openStream(long bytesToRead) throws IOException {
        HttpResponse response;
        String rangeHeader;
        Preconditions.checkArgument((bytesToRead > 0L ? 1 : 0) != 0, (String)"bytesToRead should be greater than 0, but was %s", (long)bytesToRead);
        Preconditions.checkState((this.contentChannel == null && this.contentChannelEnd < 0L ? 1 : 0) != 0, (String)"contentChannel and contentChannelEnd should be not initialized yet for '%s'", (Object)this.resourceIdString);
        if (this.size == 0L) {
            return new ByteArrayInputStream(new byte[0]);
        }
        if (this.gzipEncoded) {
            rangeHeader = null;
            this.contentChannelEnd = this.size;
        } else {
            long rangeSize = this.size - this.currentPosition;
            if (this.randomAccess) {
                long randomRangeSize = Math.max(bytesToRead, (long)this.readOptions.getMinRangeRequestSize());
                rangeSize = Math.min(randomRangeSize, rangeSize);
            }
            this.contentChannelEnd = this.currentPosition + rangeSize;
            if (this.footerContent != null) {
                this.contentChannelEnd = Math.min(this.contentChannelEnd, this.size - (long)this.footerContent.length);
            }
            Preconditions.checkState((this.currentPosition < this.contentChannelEnd ? 1 : 0) != 0, (String)"currentPosition (%s) should be less than contentChannelEnd (%s) for '%s'", (Object)this.currentPosition, (Object)this.contentChannelEnd, (Object)this.resourceIdString);
            rangeHeader = "bytes=" + this.currentPosition + "-";
            if (this.randomAccess || this.contentChannelEnd != this.size) {
                rangeHeader = rangeHeader + (this.contentChannelEnd - 1L);
            }
        }
        Preconditions.checkState((this.contentChannelEnd > 0L ? 1 : 0) != 0, (String)"contentChannelEnd should be initialized already for '%s'", (Object)this.resourceIdString);
        Storage.Objects.Get getObject = this.createDataRequest(rangeHeader);
        try {
            response = getObject.executeMedia();
        }
        catch (IOException e) {
            if (this.errorExtractor.itemNotFound(e)) {
                throw GoogleCloudStorageExceptions.getFileNotFoundException(this.bucketName, this.objectName);
            }
            String msg = String.format("Error reading '%s' at position %d", this.resourceIdString, this.currentPosition);
            if (this.errorExtractor.rangeNotSatisfiable(e)) {
                throw (EOFException)new EOFException(msg).initCause(e);
            }
            throw new IOException(msg, e);
        }
        try {
            InputStream contentStream = response.getContent();
            if (this.readOptions.getBufferSize() > 0) {
                int bufferSize = this.readOptions.getBufferSize();
                bufferSize = Math.toIntExact(Math.min((long)bufferSize, this.contentChannelEnd - this.currentPosition));
                LOG.debug("Opened stream from {} position with {} range, {} bytesToRead and {} bytes buffer for '{}'", new Object[]{this.currentPosition, rangeHeader, bytesToRead, bufferSize, this.resourceIdString});
                contentStream = new BufferedInputStream(contentStream, bufferSize);
            } else {
                LOG.debug("Opened stream from {} position with {} range and {} bytesToRead for '{}'", new Object[]{this.currentPosition, rangeHeader, bytesToRead, this.resourceIdString});
            }
            if (this.gzipEncoded) {
                contentStream.skip(this.currentPosition);
            }
            return contentStream;
        }
        catch (IOException e) {
            try {
                response.disconnect();
            }
            catch (IOException closeException) {
                e.addSuppressed(closeException);
            }
            throw e;
        }
    }

    private Storage.Objects.Get createDataRequest(String rangeHeader) throws IOException {
        Storage.Objects.Get getObject = this.createRequest();
        HttpHeaders requestHeaders = this.clientRequestHelper.getRequestHeaders((AbstractGoogleClientRequest)getObject);
        requestHeaders.setAcceptEncoding("gzip");
        requestHeaders.setRange(rangeHeader);
        return getObject;
    }

    protected Storage.Objects.Get createRequest() throws IOException {
        return this.gcs.objects().get(this.bucketName, this.objectName);
    }

    private void throwIfNotOpen() throws IOException {
        if (!this.isOpen()) {
            throw new ClosedChannelException();
        }
    }

    private void checkIOPrecondition(boolean precondition, String errorMessage) throws IOException {
        if (!precondition) {
            throw new IOException(errorMessage);
        }
    }
}

