/*
 * Decompiled with CFR 0.152.
 */
package com.google.cloud.hadoop.gcsio.testing;

import com.google.api.client.util.Clock;
import com.google.cloud.hadoop.gcsio.CreateBucketOptions;
import com.google.cloud.hadoop.gcsio.CreateObjectOptions;
import com.google.cloud.hadoop.gcsio.GoogleCloudStorage;
import com.google.cloud.hadoop.gcsio.GoogleCloudStorageExceptions;
import com.google.cloud.hadoop.gcsio.GoogleCloudStorageImpl;
import com.google.cloud.hadoop.gcsio.GoogleCloudStorageItemInfo;
import com.google.cloud.hadoop.gcsio.GoogleCloudStorageOptions;
import com.google.cloud.hadoop.gcsio.GoogleCloudStorageReadOptions;
import com.google.cloud.hadoop.gcsio.GoogleCloudStorageStrings;
import com.google.cloud.hadoop.gcsio.StorageResourceId;
import com.google.cloud.hadoop.gcsio.UpdatableItemInfo;
import com.google.cloud.hadoop.gcsio.testing.InMemoryBucketEntry;
import com.google.cloud.hadoop.gcsio.testing.InMemoryObjectEntry;
import com.google.common.base.Function;
import com.google.common.base.Preconditions;
import com.google.common.base.Strings;
import com.google.common.collect.Lists;
import java.io.ByteArrayOutputStream;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.nio.ByteBuffer;
import java.nio.channels.SeekableByteChannel;
import java.nio.channels.WritableByteChannel;
import java.nio.file.FileAlreadyExistsException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;

public class InMemoryGoogleCloudStorage
implements GoogleCloudStorage {
    private final Map<String, InMemoryBucketEntry> bucketLookup = new HashMap<String, InMemoryBucketEntry>();
    private final GoogleCloudStorageOptions storageOptions;
    private final Clock clock;

    public InMemoryGoogleCloudStorage() {
        this.storageOptions = GoogleCloudStorageOptions.newBuilder().setAppName("in-memory").build();
        this.clock = Clock.SYSTEM;
    }

    public InMemoryGoogleCloudStorage(GoogleCloudStorageOptions options) {
        this.storageOptions = options;
        this.clock = Clock.SYSTEM;
    }

    public InMemoryGoogleCloudStorage(GoogleCloudStorageOptions storageOptions, Clock clock) {
        this.storageOptions = storageOptions;
        this.clock = clock;
    }

    @Override
    public GoogleCloudStorageOptions getOptions() {
        return this.storageOptions;
    }

    private boolean validateBucketName(String bucketName) {
        if (Strings.isNullOrEmpty((String)bucketName)) {
            return false;
        }
        if (bucketName.length() < 3) {
            return false;
        }
        if (!bucketName.matches("^[a-z0-9][a-z0-9_.-]*[a-z0-9]$")) {
            return false;
        }
        return bucketName.length() <= 63;
    }

    private boolean validateObjectName(String objectName) {
        return objectName.length() <= 1024 && objectName.indexOf(10) <= -1 && objectName.indexOf(13) <= -1;
    }

    @Override
    public synchronized WritableByteChannel create(StorageResourceId resourceId) throws IOException {
        return this.create(resourceId, CreateObjectOptions.DEFAULT);
    }

    @Override
    public synchronized WritableByteChannel create(StorageResourceId resourceId, CreateObjectOptions options) throws IOException {
        GoogleCloudStorageItemInfo itemInfo;
        if (!this.bucketLookup.containsKey(resourceId.getBucketName())) {
            throw new IOException(String.format("Tried to insert object '%s' into nonexistent bucket '%s'", resourceId.getObjectName(), resourceId.getBucketName()));
        }
        if (!this.validateObjectName(resourceId.getObjectName())) {
            throw new IOException("Error creating object. Invalid name: " + resourceId.getObjectName());
        }
        if (resourceId.hasGenerationId() && resourceId.getGenerationId() != 0L && (itemInfo = this.getItemInfo(resourceId)).getContentGeneration() != resourceId.getGenerationId()) {
            throw new IOException(String.format("Required generationId '%d' doesn't match existing '%d' for '%s'", resourceId.getGenerationId(), itemInfo.getContentGeneration(), resourceId));
        }
        if ((!options.overwriteExisting() || resourceId.getGenerationId() == 0L) && this.getItemInfo(resourceId).exists()) {
            throw new FileAlreadyExistsException(String.format("%s exists.", resourceId));
        }
        InMemoryObjectEntry entry = new InMemoryObjectEntry(resourceId.getBucketName(), resourceId.getObjectName(), this.clock.currentTimeMillis(), options.getContentType(), options.getMetadata());
        this.bucketLookup.get(resourceId.getBucketName()).add(entry);
        return entry.getWriteChannel();
    }

    @Override
    public synchronized void createEmptyObject(StorageResourceId resourceId) throws IOException {
        this.createEmptyObject(resourceId, CreateObjectOptions.DEFAULT);
    }

    @Override
    public synchronized void createEmptyObject(StorageResourceId resourceId, CreateObjectOptions options) throws IOException {
        this.create(resourceId, options).close();
    }

    @Override
    public synchronized void createEmptyObjects(List<StorageResourceId> resourceIds) throws IOException {
        this.createEmptyObjects(resourceIds, CreateObjectOptions.DEFAULT);
    }

    @Override
    public synchronized void createEmptyObjects(List<StorageResourceId> resourceIds, CreateObjectOptions options) throws IOException {
        for (StorageResourceId resourceId : resourceIds) {
            this.createEmptyObject(resourceId, options);
        }
    }

    @Override
    public synchronized SeekableByteChannel open(StorageResourceId resourceId) throws IOException {
        return this.open(resourceId, GoogleCloudStorageReadOptions.DEFAULT);
    }

    @Override
    public SeekableByteChannel open(StorageResourceId resourceId, GoogleCloudStorageReadOptions readOptions) throws IOException {
        if (!this.getItemInfo(resourceId).exists()) {
            throw GoogleCloudStorageExceptions.getFileNotFoundException(resourceId.getBucketName(), resourceId.getObjectName());
        }
        return this.bucketLookup.get(resourceId.getBucketName()).get(resourceId.getObjectName()).getReadChannel(readOptions);
    }

    @Override
    public synchronized void create(String bucketName) throws IOException {
        this.create(bucketName, CreateBucketOptions.DEFAULT);
    }

    @Override
    public synchronized void create(String bucketName, CreateBucketOptions options) throws IOException {
        if (!this.validateBucketName(bucketName)) {
            throw new IOException("Error creating bucket. Invalid name: " + bucketName);
        }
        if (this.bucketLookup.containsKey(bucketName)) {
            throw new IOException("Bucket '" + bucketName + "'already exists");
        }
        this.bucketLookup.put(bucketName, new InMemoryBucketEntry(bucketName, this.clock.currentTimeMillis(), options));
    }

    @Override
    public synchronized void deleteBuckets(List<String> bucketNames) throws IOException {
        boolean hasError = false;
        for (String bucketName : bucketNames) {
            if (this.bucketLookup.containsKey(bucketName)) {
                this.bucketLookup.remove(bucketName);
            } else {
                hasError = true;
            }
            hasError = hasError || !this.validateBucketName(bucketName);
        }
        if (hasError) {
            throw new IOException("Error deleting");
        }
    }

    @Override
    public synchronized void deleteObjects(List<StorageResourceId> fullObjectNames) throws IOException {
        for (StorageResourceId resourceId : fullObjectNames) {
            if (this.validateObjectName(resourceId.getObjectName())) continue;
            throw new IOException("Error deleting object. Invalid name: " + resourceId.getObjectName());
        }
        for (StorageResourceId fullObjectName : fullObjectNames) {
            GoogleCloudStorageItemInfo existingInfo;
            String bucketName = fullObjectName.getBucketName();
            String objectName = fullObjectName.getObjectName();
            if (fullObjectName.hasGenerationId() && (existingInfo = this.getItemInfo(fullObjectName)).getContentGeneration() != fullObjectName.getGenerationId()) {
                throw new IOException(String.format("Required generationId '%d' doesn't match existing '%d' for '%s'", fullObjectName.getGenerationId(), existingInfo.getContentGeneration(), fullObjectName));
            }
            this.bucketLookup.get(bucketName).remove(objectName);
        }
    }

    @Override
    public synchronized void copy(String srcBucketName, List<String> srcObjectNames, String dstBucketName, List<String> dstObjectNames) throws IOException {
        GoogleCloudStorageImpl.validateCopyArguments(srcBucketName, srcObjectNames, dstBucketName, dstObjectNames, this);
        ArrayList<IOException> innerExceptions = new ArrayList<IOException>();
        for (int i = 0; i < srcObjectNames.size(); ++i) {
            if (!this.getItemInfo(new StorageResourceId(srcBucketName, srcObjectNames.get(i))).exists()) {
                innerExceptions.add(GoogleCloudStorageExceptions.getFileNotFoundException(srcBucketName, srcObjectNames.get(i)));
                continue;
            }
            InMemoryObjectEntry srcObject = this.bucketLookup.get(srcBucketName).get(srcObjectNames.get(i));
            this.bucketLookup.get(dstBucketName).add(srcObject.getShallowCopy(dstBucketName, dstObjectNames.get(i)));
        }
        if (innerExceptions.size() > 0) {
            throw GoogleCloudStorageExceptions.createCompositeException(innerExceptions);
        }
    }

    @Override
    public synchronized List<String> listBucketNames() throws IOException {
        return new ArrayList<String>(this.bucketLookup.keySet());
    }

    @Override
    public synchronized List<GoogleCloudStorageItemInfo> listBucketInfo() throws IOException {
        ArrayList<GoogleCloudStorageItemInfo> bucketInfos = new ArrayList<GoogleCloudStorageItemInfo>();
        for (InMemoryBucketEntry entry : this.bucketLookup.values()) {
            bucketInfos.add(entry.getInfo());
        }
        return bucketInfos;
    }

    @Override
    public synchronized List<String> listObjectNames(String bucketName, String objectNamePrefix, String delimiter) throws IOException {
        return this.listObjectNames(bucketName, objectNamePrefix, delimiter, -1L);
    }

    @Override
    public synchronized List<String> listObjectNames(String bucketName, String objectNamePrefix, String delimiter, long maxResults) throws IOException {
        InMemoryBucketEntry bucketEntry = this.bucketLookup.get(bucketName);
        if (bucketEntry == null) {
            throw new FileNotFoundException("Bucket not found: " + bucketName);
        }
        HashSet<String> uniqueNames = new HashSet<String>();
        for (String objectName : bucketEntry.getObjectNames()) {
            String processedName = GoogleCloudStorageStrings.matchListPrefix(objectNamePrefix, delimiter, objectName);
            if (processedName != null) {
                uniqueNames.add(processedName);
            }
            if (maxResults <= 0L || (long)uniqueNames.size() < maxResults) continue;
            break;
        }
        return new ArrayList<String>(uniqueNames);
    }

    @Override
    public synchronized List<GoogleCloudStorageItemInfo> listObjectInfo(String bucketName, String objectNamePrefix, String delimiter) throws IOException {
        return this.listObjectInfo(bucketName, objectNamePrefix, delimiter, -1L);
    }

    @Override
    public synchronized List<GoogleCloudStorageItemInfo> listObjectInfo(String bucketName, String objectNamePrefix, String delimiter, long maxResults) throws IOException {
        List<String> listedNames = this.listObjectNames(bucketName, objectNamePrefix, delimiter, -1L);
        ArrayList<GoogleCloudStorageItemInfo> listedInfo = new ArrayList<GoogleCloudStorageItemInfo>();
        for (String objectName : listedNames) {
            GoogleCloudStorageItemInfo itemInfo = this.getItemInfo(new StorageResourceId(bucketName, objectName));
            if (itemInfo.exists()) {
                listedInfo.add(itemInfo);
            } else if (itemInfo.getResourceId().isStorageObject() && this.storageOptions.isAutoRepairImplicitDirectoriesEnabled()) {
                this.create(itemInfo.getResourceId()).close();
                GoogleCloudStorageItemInfo newInfo = this.getItemInfo(itemInfo.getResourceId());
                if (newInfo.exists()) {
                    listedInfo.add(newInfo);
                } else if (this.storageOptions.isInferImplicitDirectoriesEnabled()) {
                    listedInfo.add(GoogleCloudStorageItemInfo.createInferredDirectory(itemInfo.getResourceId()));
                }
            } else if (itemInfo.getResourceId().isStorageObject() && this.storageOptions.isInferImplicitDirectoriesEnabled()) {
                listedInfo.add(GoogleCloudStorageItemInfo.createInferredDirectory(itemInfo.getResourceId()));
            }
            if (maxResults <= 0L || (long)listedInfo.size() < maxResults) continue;
            break;
        }
        return listedInfo;
    }

    @Override
    public synchronized GoogleCloudStorageItemInfo getItemInfo(StorageResourceId resourceId) throws IOException {
        if (resourceId.isRoot()) {
            return GoogleCloudStorageItemInfo.ROOT_INFO;
        }
        if (resourceId.isBucket()) {
            if (this.bucketLookup.containsKey(resourceId.getBucketName())) {
                return this.bucketLookup.get(resourceId.getBucketName()).getInfo();
            }
        } else {
            if (!this.validateObjectName(resourceId.getObjectName())) {
                throw new IOException("Error accessing");
            }
            if (this.bucketLookup.containsKey(resourceId.getBucketName()) && this.bucketLookup.get(resourceId.getBucketName()).get(resourceId.getObjectName()) != null) {
                return this.bucketLookup.get(resourceId.getBucketName()).get(resourceId.getObjectName()).getInfo();
            }
        }
        GoogleCloudStorageItemInfo notFoundItemInfo = new GoogleCloudStorageItemInfo(resourceId, 0L, -1L, null, null);
        return notFoundItemInfo;
    }

    @Override
    public synchronized List<GoogleCloudStorageItemInfo> getItemInfos(List<StorageResourceId> resourceIds) throws IOException {
        ArrayList<GoogleCloudStorageItemInfo> itemInfos = new ArrayList<GoogleCloudStorageItemInfo>();
        for (StorageResourceId resourceId : resourceIds) {
            try {
                itemInfos.add(this.getItemInfo(resourceId));
            }
            catch (IOException ioe) {
                throw new IOException("Error getting StorageObject", ioe);
            }
        }
        return itemInfos;
    }

    @Override
    public List<GoogleCloudStorageItemInfo> updateItems(List<UpdatableItemInfo> itemInfoList) throws IOException {
        ArrayList<GoogleCloudStorageItemInfo> itemInfos = new ArrayList<GoogleCloudStorageItemInfo>();
        for (UpdatableItemInfo updatableItemInfo : itemInfoList) {
            StorageResourceId resourceId = updatableItemInfo.getStorageResourceId();
            Preconditions.checkArgument((!resourceId.isRoot() && !resourceId.isBucket() ? 1 : 0) != 0, (Object)"Can't update item on GCS Root or bucket resources");
            if (!this.validateObjectName(resourceId.getObjectName())) {
                throw new IOException("Error accessing");
            }
            if (this.bucketLookup.containsKey(resourceId.getBucketName()) && this.bucketLookup.get(resourceId.getBucketName()).get(resourceId.getObjectName()) != null) {
                InMemoryObjectEntry objectEntry = this.bucketLookup.get(resourceId.getBucketName()).get(resourceId.getObjectName());
                objectEntry.patchMetadata(updatableItemInfo.getMetadata());
                itemInfos.add(this.getItemInfo(resourceId));
                continue;
            }
            throw new IOException(String.format("Error getting StorageObject %s", resourceId.toString()));
        }
        return itemInfos;
    }

    @Override
    public void close() {
    }

    @Override
    public void waitForBucketEmpty(String bucketName) throws IOException {
    }

    @Override
    public void compose(final String bucketName, List<String> sources, String destination, String contentType) throws IOException {
        List sourceResourcesIds = Lists.transform(sources, (Function)new Function<String, StorageResourceId>(){

            public StorageResourceId apply(String s) {
                return new StorageResourceId(bucketName, s);
            }
        });
        StorageResourceId destinationId = new StorageResourceId(bucketName, destination);
        CreateObjectOptions options = new CreateObjectOptions(true, contentType, (Map<String, byte[]>)CreateObjectOptions.EMPTY_METADATA);
        this.composeObjects(sourceResourcesIds, destinationId, options);
    }

    @Override
    public GoogleCloudStorageItemInfo composeObjects(List<StorageResourceId> sources, StorageResourceId destination, CreateObjectOptions options) throws IOException {
        ByteArrayOutputStream tempOutput = new ByteArrayOutputStream();
        for (StorageResourceId sourceId : sources) {
            SeekableByteChannel sourceChannel = this.open(sourceId);
            Throwable throwable = null;
            try {
                byte[] buf = new byte[(int)sourceChannel.size()];
                ByteBuffer reader = ByteBuffer.wrap(buf);
                sourceChannel.read(reader);
                tempOutput.write(buf, 0, buf.length);
            }
            catch (Throwable throwable2) {
                throwable = throwable2;
                throw throwable2;
            }
            finally {
                if (sourceChannel == null) continue;
                if (throwable != null) {
                    try {
                        sourceChannel.close();
                    }
                    catch (Throwable throwable3) {
                        throwable.addSuppressed(throwable3);
                    }
                    continue;
                }
                sourceChannel.close();
            }
        }
        WritableByteChannel destChannel = this.create(destination, options);
        destChannel.write(ByteBuffer.wrap(tempOutput.toByteArray()));
        destChannel.close();
        return this.getItemInfo(destination);
    }
}

