/*
 * Decompiled with CFR 0.152.
 */
package com.google.cloud.dataflow.sdk.util;

import com.google.api.client.googleapis.services.AbstractGoogleClientRequest;
import com.google.api.client.util.BackOff;
import com.google.api.client.util.BackOffUtils;
import com.google.api.client.util.ClassInfo;
import com.google.api.client.util.Data;
import com.google.api.client.util.Sleeper;
import com.google.api.services.bigquery.Bigquery;
import com.google.api.services.bigquery.model.Dataset;
import com.google.api.services.bigquery.model.DatasetReference;
import com.google.api.services.bigquery.model.ErrorProto;
import com.google.api.services.bigquery.model.Job;
import com.google.api.services.bigquery.model.JobConfiguration;
import com.google.api.services.bigquery.model.JobConfigurationQuery;
import com.google.api.services.bigquery.model.JobReference;
import com.google.api.services.bigquery.model.JobStatistics;
import com.google.api.services.bigquery.model.JobStatus;
import com.google.api.services.bigquery.model.Table;
import com.google.api.services.bigquery.model.TableDataList;
import com.google.api.services.bigquery.model.TableFieldSchema;
import com.google.api.services.bigquery.model.TableReference;
import com.google.api.services.bigquery.model.TableRow;
import com.google.api.services.bigquery.model.TableSchema;
import com.google.cloud.dataflow.sdk.repackaged.com.google.common.base.MoreObjects;
import com.google.cloud.dataflow.sdk.repackaged.com.google.common.base.Preconditions;
import com.google.cloud.dataflow.sdk.repackaged.com.google.common.collect.ImmutableList;
import com.google.cloud.dataflow.sdk.repackaged.com.google.common.util.concurrent.Uninterruptibles;
import com.google.cloud.dataflow.sdk.util.AvroUtils;
import com.google.cloud.dataflow.sdk.util.FluentBackoff;
import java.io.IOException;
import java.util.Collection;
import java.util.Collections;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.NoSuchElementException;
import java.util.Objects;
import java.util.Random;
import java.util.concurrent.TimeUnit;
import javax.annotation.Nullable;
import org.joda.time.Duration;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class BigQueryTableRowIterator
implements AutoCloseable {
    private static final Logger LOG = LoggerFactory.getLogger(BigQueryTableRowIterator.class);
    @Nullable
    private TableReference ref;
    @Nullable
    private final String projectId;
    @Nullable
    private TableSchema schema;
    private final Bigquery client;
    private String pageToken;
    private Iterator<TableRow> iteratorOverCurrentBatch;
    private TableRow current;
    private boolean lastPage = false;
    private static final int MAX_RETRIES = 3;
    private static final Duration INITIAL_BACKOFF_TIME = Duration.standardSeconds((long)1L);
    private static final Duration QUERY_COMPLETION_POLL_TIME = Duration.standardSeconds((long)1L);
    private final String query;
    private final boolean flattenResults;
    private final boolean useLegacySql;
    private String temporaryDatasetId = null;
    private String temporaryTableId = null;
    private static final Collection<String> RESERVED_FIELD_NAMES = ClassInfo.of(TableRow.class).getNames();

    private BigQueryTableRowIterator(@Nullable TableReference ref, @Nullable String query, @Nullable String projectId, Bigquery client, boolean flattenResults, boolean useLegacySql) {
        this.ref = ref;
        this.query = query;
        this.projectId = projectId;
        this.client = Preconditions.checkNotNull(client, "client");
        this.flattenResults = flattenResults;
        this.useLegacySql = useLegacySql;
    }

    public static BigQueryTableRowIterator fromTable(TableReference ref, Bigquery client) {
        Preconditions.checkNotNull(ref, "ref");
        Preconditions.checkNotNull(client, "client");
        return new BigQueryTableRowIterator(ref, null, ref.getProjectId(), client, true, true);
    }

    @Deprecated
    public static BigQueryTableRowIterator fromQuery(String query, String projectId, Bigquery client, @Nullable Boolean flattenResults) {
        return BigQueryTableRowIterator.fromQuery(query, projectId, client, flattenResults, null);
    }

    public static BigQueryTableRowIterator fromQuery(String query, String projectId, Bigquery client, @Nullable Boolean flattenResults, @Nullable Boolean useLegacySql) {
        Preconditions.checkNotNull(query, "query");
        Preconditions.checkNotNull(projectId, "projectId");
        Preconditions.checkNotNull(client, "client");
        return new BigQueryTableRowIterator(null, query, projectId, client, MoreObjects.firstNonNull(flattenResults, Boolean.TRUE), MoreObjects.firstNonNull(useLegacySql, Boolean.TRUE));
    }

    public void open() throws IOException, InterruptedException {
        if (this.query != null) {
            this.ref = this.executeQueryAndWaitForCompletion();
        }
        this.schema = this.getTable(this.ref).getSchema();
    }

    public boolean advance() throws IOException, InterruptedException {
        while (true) {
            if (this.iteratorOverCurrentBatch != null && this.iteratorOverCurrentBatch.hasNext()) {
                this.current = this.getTypedTableRow(this.schema.getFields(), (Map)this.iteratorOverCurrentBatch.next());
                return true;
            }
            if (this.lastPage) {
                return false;
            }
            Bigquery.Tabledata.List list = this.client.tabledata().list(this.ref.getProjectId(), this.ref.getDatasetId(), this.ref.getTableId());
            if (this.pageToken != null) {
                list.setPageToken(this.pageToken);
            }
            TableDataList result = (TableDataList)BigQueryTableRowIterator.executeWithBackOff(list, String.format("Error reading from BigQuery table %s of dataset %s.", this.ref.getTableId(), this.ref.getDatasetId()));
            this.pageToken = result.getPageToken();
            Iterator<Object> iterator = this.iteratorOverCurrentBatch = result.getRows() != null ? result.getRows().iterator() : Collections.emptyIterator();
            if (this.pageToken != null && (result.getTotalRows() == null || result.getTotalRows() != 0L)) continue;
            this.lastPage = true;
        }
    }

    public TableRow getCurrent() {
        if (this.current == null) {
            throw new NoSuchElementException();
        }
        return this.current;
    }

    @Nullable
    private Object getTypedCellValue(TableFieldSchema fieldSchema, Object v) {
        if (Data.isNull((Object)v)) {
            return null;
        }
        if (Objects.equals(fieldSchema.getMode(), "REPEATED")) {
            TableFieldSchema elementSchema = fieldSchema.clone().setMode("REQUIRED");
            List rawCells = (List)v;
            ImmutableList.Builder values = ImmutableList.builder();
            for (Map element : rawCells) {
                values.add(this.getTypedCellValue(elementSchema, element.get("v")));
            }
            return values.build();
        }
        if (fieldSchema.getType().equals("RECORD")) {
            Map typedV = (Map)v;
            return this.getTypedTableRow(fieldSchema.getFields(), typedV);
        }
        if (fieldSchema.getType().equals("FLOAT")) {
            return Double.parseDouble((String)v);
        }
        if (fieldSchema.getType().equals("BOOLEAN")) {
            return Boolean.parseBoolean((String)v);
        }
        if (fieldSchema.getType().equals("TIMESTAMP")) {
            return AvroUtils.formatTimestamp((String)v);
        }
        return v;
    }

    private TableRow getTypedTableRow(List<TableFieldSchema> fields, Map<String, Object> rawRow) {
        List cells;
        TableRow row;
        if (rawRow instanceof TableRow) {
            row = (TableRow)rawRow;
            cells = row.getF();
            row.setF(null);
        } else {
            List rawCells;
            row = new TableRow();
            cells = rawCells = (List)rawRow.get("f");
        }
        Preconditions.checkState(cells.size() == fields.size(), "Expected that the row has the same number of cells %s as fields in the schema %s", cells.size(), fields.size());
        Iterator cellIt = cells.iterator();
        Iterator<TableFieldSchema> fieldIt = fields.iterator();
        while (cellIt.hasNext()) {
            Map cell = (Map)cellIt.next();
            TableFieldSchema fieldSchema = fieldIt.next();
            Object convertedValue = this.getTypedCellValue(fieldSchema, cell.get("v"));
            String fieldName = fieldSchema.getName();
            Preconditions.checkArgument(!RESERVED_FIELD_NAMES.contains(fieldName), "BigQueryIO does not support records with columns named %s", fieldName);
            if (convertedValue == null) continue;
            row.set(fieldName, convertedValue);
        }
        return row;
    }

    private Table getTable(TableReference ref) throws IOException, InterruptedException {
        Bigquery.Tables.Get get = this.client.tables().get(ref.getProjectId(), ref.getDatasetId(), ref.getTableId());
        return (Table)BigQueryTableRowIterator.executeWithBackOff(get, String.format("Error opening BigQuery table %s of dataset %s.", ref.getTableId(), ref.getDatasetId()));
    }

    private void createDataset(String datasetId, @Nullable String location) throws IOException, InterruptedException {
        Dataset dataset = new Dataset();
        DatasetReference reference = new DatasetReference();
        reference.setProjectId(this.projectId);
        reference.setDatasetId(datasetId);
        dataset.setDatasetReference(reference);
        if (location != null) {
            dataset.setLocation(location);
        }
        BigQueryTableRowIterator.executeWithBackOff(this.client.datasets().insert(this.projectId, dataset), String.format("Error when trying to create the temporary dataset %s in project %s.", datasetId, this.projectId));
    }

    private void deleteTable(String datasetId, String tableId) throws IOException, InterruptedException {
        BigQueryTableRowIterator.executeWithBackOff(this.client.tables().delete(this.projectId, datasetId, tableId), String.format("Error when trying to delete the temporary table %s in dataset %s of project %s. Manual deletion may be required.", tableId, datasetId, this.projectId));
    }

    private void deleteDataset(String datasetId) throws IOException, InterruptedException {
        BigQueryTableRowIterator.executeWithBackOff(this.client.datasets().delete(this.projectId, datasetId), String.format("Error when trying to delete the temporary dataset %s in project %s. Manual deletion may be required.", datasetId, this.projectId));
    }

    private TableReference executeQueryAndWaitForCompletion() throws IOException, InterruptedException {
        Job dryRunJob = new Job().setConfiguration(new JobConfiguration().setQuery(new JobConfigurationQuery().setQuery(this.query)).setDryRun(Boolean.valueOf(true)));
        JobStatistics jobStats = ((Job)BigQueryTableRowIterator.executeWithBackOff(this.client.jobs().insert(this.projectId, dryRunJob), String.format("Error when trying to dry run query %s.", this.query))).getStatistics();
        String location = null;
        List tables = jobStats.getQuery().getReferencedTables();
        if (tables != null && !tables.isEmpty()) {
            Table table = this.getTable((TableReference)tables.get(0));
            location = table.getLocation();
        }
        Random rnd = new Random(System.currentTimeMillis());
        this.temporaryDatasetId = "_dataflow_temporary_dataset_" + rnd.nextInt(1000000);
        this.temporaryTableId = "dataflow_temporary_table_" + rnd.nextInt(1000000);
        this.createDataset(this.temporaryDatasetId, location);
        Job job = new Job();
        JobConfiguration config = new JobConfiguration();
        JobConfigurationQuery queryConfig = new JobConfigurationQuery();
        config.setQuery(queryConfig);
        job.setConfiguration(config);
        queryConfig.setQuery(this.query);
        queryConfig.setAllowLargeResults(Boolean.valueOf(true));
        queryConfig.setFlattenResults(Boolean.valueOf(this.flattenResults));
        queryConfig.setUseLegacySql(Boolean.valueOf(this.useLegacySql));
        TableReference destinationTable = new TableReference();
        destinationTable.setProjectId(this.projectId);
        destinationTable.setDatasetId(this.temporaryDatasetId);
        destinationTable.setTableId(this.temporaryTableId);
        queryConfig.setDestinationTable(destinationTable);
        Job queryJob = (Job)BigQueryTableRowIterator.executeWithBackOff(this.client.jobs().insert(this.projectId, job), String.format("Error when trying to execute the job for query %s.", this.query));
        JobReference jobId = queryJob.getJobReference();
        while (true) {
            Job pollJob;
            JobStatus status;
            if ((status = (pollJob = (Job)BigQueryTableRowIterator.executeWithBackOff(this.client.jobs().get(this.projectId, jobId.getJobId()), String.format("Error when trying to get status of the job for query %s.", this.query))).getStatus()).getState().equals("DONE")) {
                ErrorProto error = status.getErrorResult();
                if (error == null) {
                    return pollJob.getConfiguration().getQuery().getDestinationTable();
                }
                this.temporaryTableId = null;
                throw new IOException("Executing query " + this.query + " failed: " + error.getMessage());
            }
            Uninterruptibles.sleepUninterruptibly(QUERY_COMPLETION_POLL_TIME.getMillis(), TimeUnit.MILLISECONDS);
        }
    }

    @Deprecated
    public static <T> T executeWithBackOff(AbstractGoogleClientRequest<T> client, String error, Object ... errorArgs) throws IOException, InterruptedException {
        return BigQueryTableRowIterator.executeWithBackOff(client, String.format(error, errorArgs));
    }

    public static <T> T executeWithBackOff(AbstractGoogleClientRequest<T> client, String error) throws IOException, InterruptedException {
        Sleeper sleeper = Sleeper.DEFAULT;
        BackOff backOff = FluentBackoff.DEFAULT.withMaxRetries(3).withInitialBackoff(INITIAL_BACKOFF_TIME).backoff();
        Object result = null;
        while (true) {
            try {
                result = client.execute();
            }
            catch (IOException e) {
                LOG.error("{}", (Object)error, (Object)e);
                if (BackOffUtils.next((Sleeper)sleeper, (BackOff)backOff)) continue;
                String errorMessage = String.format("%s Failing to execute job after %d attempts.", error, 4);
                LOG.error("{}", (Object)errorMessage, (Object)e);
                throw new IOException(errorMessage, e);
            }
            break;
        }
        return (T)result;
    }

    @Override
    public void close() {
        this.lastPage = true;
        try {
            if (this.temporaryDatasetId != null) {
                if (this.temporaryTableId != null) {
                    this.deleteTable(this.temporaryDatasetId, this.temporaryTableId);
                }
                this.deleteDataset(this.temporaryDatasetId);
            }
        }
        catch (IOException | InterruptedException e) {
            if (e instanceof InterruptedException) {
                Thread.currentThread().interrupt();
            }
            throw new RuntimeException(e);
        }
    }
}

