/*
 * Decompiled with CFR 0.152.
 */
package com.google.cloud.dataflow.sdk.util;

import com.google.api.client.googleapis.batch.BatchRequest;
import com.google.api.client.googleapis.batch.json.JsonBatchCallback;
import com.google.api.client.googleapis.json.GoogleJsonError;
import com.google.api.client.googleapis.json.GoogleJsonResponseException;
import com.google.api.client.googleapis.services.AbstractGoogleClientRequest;
import com.google.api.client.http.HttpHeaders;
import com.google.api.client.util.BackOff;
import com.google.api.client.util.Sleeper;
import com.google.api.services.storage.Storage;
import com.google.api.services.storage.model.Objects;
import com.google.api.services.storage.model.StorageObject;
import com.google.cloud.dataflow.sdk.options.DefaultValueFactory;
import com.google.cloud.dataflow.sdk.options.GcsOptions;
import com.google.cloud.dataflow.sdk.options.PipelineOptions;
import com.google.cloud.dataflow.sdk.repackaged.com.google.common.annotations.VisibleForTesting;
import com.google.cloud.dataflow.sdk.repackaged.com.google.common.base.Preconditions;
import com.google.cloud.dataflow.sdk.repackaged.com.google.common.collect.ImmutableList;
import com.google.cloud.dataflow.sdk.repackaged.com.google.common.collect.Lists;
import com.google.cloud.dataflow.sdk.repackaged.com.google.common.util.concurrent.Futures;
import com.google.cloud.dataflow.sdk.repackaged.com.google.common.util.concurrent.ListenableFuture;
import com.google.cloud.dataflow.sdk.repackaged.com.google.common.util.concurrent.ListeningExecutorService;
import com.google.cloud.dataflow.sdk.repackaged.com.google.common.util.concurrent.MoreExecutors;
import com.google.cloud.dataflow.sdk.util.FluentBackoff;
import com.google.cloud.dataflow.sdk.util.Transport;
import com.google.cloud.dataflow.sdk.util.gcsfs.GcsPath;
import com.google.cloud.hadoop.gcsio.GoogleCloudStorageReadChannel;
import com.google.cloud.hadoop.gcsio.GoogleCloudStorageWriteChannel;
import com.google.cloud.hadoop.gcsio.ObjectWriteConditions;
import com.google.cloud.hadoop.util.ApiErrorExtractor;
import com.google.cloud.hadoop.util.AsyncWriteChannelOptions;
import com.google.cloud.hadoop.util.ClientRequestHelper;
import com.google.cloud.hadoop.util.ResilientOperation;
import com.google.cloud.hadoop.util.RetryDeterminer;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.nio.channels.SeekableByteChannel;
import java.nio.channels.WritableByteChannel;
import java.util.Collection;
import java.util.Collections;
import java.util.LinkedList;
import java.util.List;
import java.util.concurrent.Callable;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.concurrent.TimeUnit;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import javax.annotation.Nullable;
import org.joda.time.Duration;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class GcsUtil {
    private static final Logger LOG = LoggerFactory.getLogger(GcsUtil.class);
    private static final long MAX_LIST_ITEMS_PER_CALL = 1024L;
    private static final Pattern GLOB_PREFIX = Pattern.compile("(?<PREFIX>[^\\[*?]*)[\\[*?].*");
    private static final String RECURSIVE_WILDCARD = "[*]{2}";
    private static final Pattern RECURSIVE_GCS_PATTERN = Pattern.compile(".*[*]{2}.*");
    private static final int MAX_REQUESTS_PER_BATCH = 100;
    private static final int MAX_CONCURRENT_BATCHES = 256;
    private static final FluentBackoff BACKOFF_FACTORY = FluentBackoff.DEFAULT.withMaxRetries(3).withInitialBackoff(Duration.millis((long)200L));
    private Storage storageClient;
    @Nullable
    private final Integer uploadBufferSizeBytes;
    private final ApiErrorExtractor errorExtractor = new ApiErrorExtractor();
    final ExecutorService executorService;

    public boolean isGcsPatternSupported(String gcsPattern) {
        if (RECURSIVE_GCS_PATTERN.matcher(gcsPattern).matches()) {
            throw new IllegalArgumentException("Unsupported wildcard usage in \"" + gcsPattern + "\": " + " recursive wildcards are not supported.");
        }
        return true;
    }

    private GcsUtil(Storage storageClient, ExecutorService executorService, @Nullable Integer uploadBufferSizeBytes) {
        this.storageClient = storageClient;
        this.uploadBufferSizeBytes = uploadBufferSizeBytes;
        this.executorService = executorService;
    }

    protected void setStorageClient(Storage storageClient) {
        this.storageClient = storageClient;
    }

    public List<GcsPath> expand(GcsPath gcsPattern) throws IOException {
        Objects objects;
        Preconditions.checkArgument(this.isGcsPatternSupported(gcsPattern.getObject()));
        Matcher m = GLOB_PREFIX.matcher(gcsPattern.getObject());
        Pattern p = null;
        String prefix = null;
        if (!m.matches()) {
            Storage.Objects.Get getObject = this.storageClient.objects().get(gcsPattern.getBucket(), gcsPattern.getObject());
            try {
                ResilientOperation.retry((ResilientOperation.CheckedCallable)ResilientOperation.getGoogleRequestCallable((AbstractGoogleClientRequest)getObject), (BackOff)BACKOFF_FACTORY.backoff(), (RetryDeterminer)RetryDeterminer.SOCKET_ERRORS, IOException.class);
                return ImmutableList.of(gcsPattern);
            }
            catch (IOException | InterruptedException e) {
                if (e instanceof InterruptedException) {
                    Thread.currentThread().interrupt();
                }
                if (e instanceof IOException && this.errorExtractor.itemNotFound((IOException)e)) {
                    return ImmutableList.of();
                }
                throw new IOException("Unable to match files for pattern " + gcsPattern, e);
            }
        }
        prefix = m.group("PREFIX");
        p = Pattern.compile(GcsUtil.globToRegexp(gcsPattern.getObject()));
        LOG.debug("matching files in bucket {}, prefix {} against pattern {}", new Object[]{gcsPattern.getBucket(), prefix, p.toString()});
        Storage.Objects.List listObject = this.storageClient.objects().list(gcsPattern.getBucket());
        listObject.setMaxResults(Long.valueOf(1024L));
        listObject.setPrefix(prefix);
        String pageToken = null;
        LinkedList<GcsPath> results = new LinkedList<GcsPath>();
        do {
            if (pageToken != null) {
                listObject.setPageToken(pageToken);
            }
            try {
                objects = (Objects)ResilientOperation.retry((ResilientOperation.CheckedCallable)ResilientOperation.getGoogleRequestCallable((AbstractGoogleClientRequest)listObject), (BackOff)BACKOFF_FACTORY.backoff(), (RetryDeterminer)RetryDeterminer.SOCKET_ERRORS, IOException.class);
            }
            catch (Exception e) {
                throw new IOException("Unable to match files in bucket " + gcsPattern.getBucket() + ", prefix " + prefix + " against pattern " + p.toString(), e);
            }
            Preconditions.checkNotNull(objects);
            if (objects.getItems() == null) break;
            for (StorageObject o : objects.getItems()) {
                String name = o.getName();
                if (!p.matcher(name).matches() || name.endsWith("/")) continue;
                LOG.debug("Matched object: {}", (Object)name);
                results.add(GcsPath.fromObject(o));
            }
        } while ((pageToken = objects.getNextPageToken()) != null);
        return results;
    }

    @Nullable
    @VisibleForTesting
    Integer getUploadBufferSizeBytes() {
        return this.uploadBufferSizeBytes;
    }

    public long fileSize(GcsPath path) throws IOException {
        return this.fileSize(path, BACKOFF_FACTORY.backoff(), Sleeper.DEFAULT);
    }

    @VisibleForTesting
    long fileSize(GcsPath path, BackOff backoff, Sleeper sleeper) throws IOException {
        Storage.Objects.Get getObject = this.storageClient.objects().get(path.getBucket(), path.getObject());
        try {
            StorageObject object = (StorageObject)ResilientOperation.retry((ResilientOperation.CheckedCallable)ResilientOperation.getGoogleRequestCallable((AbstractGoogleClientRequest)getObject), (BackOff)backoff, (RetryDeterminer)RetryDeterminer.SOCKET_ERRORS, IOException.class, (Sleeper)sleeper);
            return object.getSize().longValue();
        }
        catch (Exception e) {
            if (e instanceof IOException && this.errorExtractor.itemNotFound((IOException)e)) {
                throw new FileNotFoundException(path.toString());
            }
            throw new IOException("Unable to get file size", e);
        }
    }

    public SeekableByteChannel open(GcsPath path) throws IOException {
        return new GoogleCloudStorageReadChannel(this.storageClient, path.getBucket(), path.getObject(), this.errorExtractor, new ClientRequestHelper());
    }

    public WritableByteChannel create(GcsPath path, String type) throws IOException {
        GoogleCloudStorageWriteChannel channel = new GoogleCloudStorageWriteChannel(this.executorService, this.storageClient, new ClientRequestHelper(), path.getBucket(), path.getObject(), AsyncWriteChannelOptions.newBuilder().build(), new ObjectWriteConditions(), Collections.emptyMap(), type);
        if (this.uploadBufferSizeBytes != null) {
            channel.setUploadBufferSize(this.uploadBufferSizeBytes.intValue());
        }
        channel.initialize();
        return channel;
    }

    public boolean bucketExists(GcsPath path) throws IOException {
        return this.bucketExists(path, BACKOFF_FACTORY.backoff(), Sleeper.DEFAULT);
    }

    @VisibleForTesting
    boolean bucketExists(GcsPath path, BackOff backoff, Sleeper sleeper) throws IOException {
        Storage.Buckets.Get getBucket = this.storageClient.buckets().get(path.getBucket());
        try {
            ResilientOperation.retry((ResilientOperation.CheckedCallable)ResilientOperation.getGoogleRequestCallable((AbstractGoogleClientRequest)getBucket), (BackOff)backoff, (RetryDeterminer)new RetryDeterminer<IOException>(){

                public boolean shouldRetry(IOException e) {
                    if (GcsUtil.this.errorExtractor.itemNotFound(e) || GcsUtil.this.errorExtractor.accessDenied(e)) {
                        return false;
                    }
                    return RetryDeterminer.SOCKET_ERRORS.shouldRetry((Exception)e);
                }
            }, IOException.class, (Sleeper)sleeper);
            return true;
        }
        catch (GoogleJsonResponseException e) {
            if (this.errorExtractor.itemNotFound((IOException)((Object)e)) || this.errorExtractor.accessDenied((IOException)((Object)e))) {
                return false;
            }
            throw e;
        }
        catch (InterruptedException e) {
            Thread.currentThread().interrupt();
            throw new IOException(String.format("Error while attempting to verify existence of bucket gs://%s", path.getBucket()), e);
        }
    }

    private static void executeBatches(List<BatchRequest> batches) throws IOException {
        ListeningExecutorService executor = MoreExecutors.listeningDecorator(MoreExecutors.getExitingExecutorService(new ThreadPoolExecutor(256, 256, 0L, TimeUnit.MILLISECONDS, new LinkedBlockingQueue<Runnable>())));
        LinkedList<ListenableFuture<Void>> futures = new LinkedList<ListenableFuture<Void>>();
        for (final BatchRequest batch : batches) {
            futures.add(executor.submit(new Callable<Void>(){

                @Override
                public Void call() throws IOException {
                    batch.execute();
                    return null;
                }
            }));
        }
        try {
            Futures.allAsList(futures).get();
        }
        catch (InterruptedException e) {
            Thread.currentThread().interrupt();
            throw new IOException("Interrupted while executing batch GCS request", e);
        }
        catch (ExecutionException e) {
            throw new IOException("Error executing batch GCS request", e);
        }
        finally {
            executor.shutdown();
        }
    }

    public void copy(List<String> srcFilenames, List<String> destFilenames) throws IOException {
        GcsUtil.executeBatches(this.makeCopyBatches(srcFilenames, destFilenames));
    }

    List<BatchRequest> makeCopyBatches(List<String> srcFilenames, List<String> destFilenames) throws IOException {
        Preconditions.checkArgument(srcFilenames.size() == destFilenames.size(), "Number of source files %s must equal number of destination files %s", srcFilenames.size(), destFilenames.size());
        LinkedList<BatchRequest> batches = new LinkedList<BatchRequest>();
        BatchRequest batch = this.storageClient.batch();
        for (int i = 0; i < srcFilenames.size(); ++i) {
            GcsPath sourcePath = GcsPath.fromUri(srcFilenames.get(i));
            GcsPath destPath = GcsPath.fromUri(destFilenames.get(i));
            this.enqueueCopy(sourcePath, destPath, batch);
            if (batch.size() < 100) continue;
            batches.add(batch);
            batch = this.storageClient.batch();
        }
        if (batch.size() > 0) {
            batches.add(batch);
        }
        return batches;
    }

    List<BatchRequest> makeRemoveBatches(Collection<String> filenames) throws IOException {
        LinkedList<BatchRequest> batches = new LinkedList<BatchRequest>();
        for (List<String> filesToDelete : Lists.partition(Lists.newArrayList(filenames), 100)) {
            BatchRequest batch = this.storageClient.batch();
            for (String file : filesToDelete) {
                this.enqueueDelete(GcsPath.fromUri(file), batch);
            }
            batches.add(batch);
        }
        return batches;
    }

    public void remove(Collection<String> filenames) throws IOException {
        GcsUtil.executeBatches(this.makeRemoveBatches(filenames));
    }

    private void enqueueCopy(final GcsPath from, final GcsPath to, BatchRequest batch) throws IOException {
        Storage.Objects.Copy copyRequest = this.storageClient.objects().copy(from.getBucket(), from.getObject(), to.getBucket(), to.getObject(), null);
        copyRequest.queue(batch, (JsonBatchCallback)new JsonBatchCallback<StorageObject>(){

            public void onSuccess(StorageObject obj, HttpHeaders responseHeaders) {
                LOG.debug("Successfully copied {} to {}", (Object)from, (Object)to);
            }

            public void onFailure(GoogleJsonError e, HttpHeaders responseHeaders) throws IOException {
                if (GcsUtil.this.errorExtractor.itemNotFound(e)) {
                    LOG.debug("{} does not exist, assuming this is a retry after deletion.", (Object)from);
                    return;
                }
                throw new IOException(String.format("Error trying to copy %s to %s: %s", from, to, e));
            }
        });
    }

    private void enqueueDelete(final GcsPath file, BatchRequest batch) throws IOException {
        Storage.Objects.Delete deleteRequest = this.storageClient.objects().delete(file.getBucket(), file.getObject());
        deleteRequest.queue(batch, (JsonBatchCallback)new JsonBatchCallback<Void>(){

            public void onSuccess(Void obj, HttpHeaders responseHeaders) {
                LOG.debug("Successfully deleted {}", (Object)file);
            }

            public void onFailure(GoogleJsonError e, HttpHeaders responseHeaders) throws IOException {
                if (GcsUtil.this.errorExtractor.itemNotFound(e)) {
                    LOG.debug("{} does not exist.", (Object)file);
                    return;
                }
                throw new IOException(String.format("Error trying to delete %s: %s", file, e));
            }
        });
    }

    static String globToRegexp(String globExp) {
        StringBuilder dst = new StringBuilder();
        char[] src = globExp.toCharArray();
        int i = 0;
        block6: while (i < src.length) {
            char c = src[i++];
            switch (c) {
                case '*': {
                    dst.append("[^/]*");
                    continue block6;
                }
                case '?': {
                    dst.append("[^/]");
                    continue block6;
                }
                case '$': 
                case '(': 
                case ')': 
                case '+': 
                case '.': 
                case '^': 
                case '{': 
                case '|': 
                case '}': {
                    dst.append('\\').append(c);
                    continue block6;
                }
                case '\\': {
                    i = GcsUtil.doubleSlashes(dst, src, i);
                    continue block6;
                }
            }
            dst.append(c);
        }
        return dst.toString();
    }

    private static int doubleSlashes(StringBuilder dst, char[] src, int i) {
        dst.append('\\');
        if (i - 1 != src.length) {
            dst.append(src[i]);
            ++i;
        } else {
            dst.append('\\');
        }
        return i;
    }

    public static class GcsUtilFactory
    implements DefaultValueFactory<GcsUtil> {
        @Override
        public GcsUtil create(PipelineOptions options) {
            LOG.debug("Creating new GcsUtil");
            GcsOptions gcsOptions = options.as(GcsOptions.class);
            return new GcsUtil(Transport.newStorageClient(gcsOptions).build(), gcsOptions.getExecutorService(), gcsOptions.getGcsUploadBufferSizeBytes());
        }
    }
}

