/*
 * Decompiled with CFR 0.152.
 */
package com.google.cloud.flink.bigquery.sink.serializer;

import com.google.api.client.util.Preconditions;
import com.google.cloud.Timestamp;
import com.google.cloud.bigquery.storage.v1.BigDecimalByteStringEncoder;
import com.google.cloud.flink.bigquery.sink.exceptions.BigQuerySerializationException;
import com.google.cloud.flink.bigquery.sink.serializer.BigQueryProtoSerializer;
import com.google.cloud.flink.bigquery.sink.serializer.BigQuerySchemaProvider;
import com.google.protobuf.ByteString;
import com.google.protobuf.Descriptors;
import com.google.protobuf.DynamicMessage;
import java.math.BigDecimal;
import java.math.BigInteger;
import java.nio.ByteBuffer;
import java.time.LocalDate;
import java.time.LocalDateTime;
import java.time.LocalTime;
import java.time.ZoneOffset;
import java.time.format.DateTimeFormatter;
import java.time.format.DateTimeParseException;
import java.time.temporal.ChronoUnit;
import java.util.HashMap;
import java.util.List;
import java.util.UUID;
import java.util.concurrent.TimeUnit;
import java.util.function.UnaryOperator;
import java.util.stream.Collectors;
import java.util.stream.StreamSupport;
import org.apache.avro.LogicalTypes;
import org.apache.avro.Schema;
import org.apache.avro.generic.GenericData;
import org.apache.avro.generic.GenericRecord;
import org.apache.avro.util.Utf8;
import org.apache.commons.lang3.tuple.ImmutablePair;
import org.apache.flink.annotation.VisibleForTesting;
import org.joda.time.Days;
import org.joda.time.Instant;
import org.joda.time.ReadableInstant;
import org.json.JSONException;
import org.json.JSONObject;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class AvroToProtoSerializer
extends BigQueryProtoSerializer<GenericRecord> {
    private Descriptors.Descriptor descriptor;

    @Override
    public void init(BigQuerySchemaProvider bigQuerySchemaProvider) {
        Preconditions.checkNotNull((Object)bigQuerySchemaProvider, (Object)"BigQuerySchemaProvider not found while initializing AvroToProtoSerializer");
        Descriptors.Descriptor derivedDescriptor = bigQuerySchemaProvider.getDescriptor();
        Preconditions.checkNotNull((Object)derivedDescriptor, (Object)"Destination BigQuery table's Proto Schema could not be found.");
        this.descriptor = derivedDescriptor;
    }

    @Override
    public ByteString serialize(GenericRecord record) throws BigQuerySerializationException {
        try {
            return AvroToProtoSerializer.getDynamicMessageFromGenericRecord(record, this.descriptor).toByteString();
        }
        catch (Exception e) {
            throw new BigQuerySerializationException(e.getMessage());
        }
    }

    public static DynamicMessage getDynamicMessageFromGenericRecord(GenericRecord element, Descriptors.Descriptor descriptor) {
        Schema recordSchema = element.getSchema();
        DynamicMessage.Builder builder = DynamicMessage.newBuilder((Descriptors.Descriptor)descriptor);
        for (Schema.Field field : recordSchema.getFields()) {
            Descriptors.FieldDescriptor fieldDescriptor = (Descriptors.FieldDescriptor)Preconditions.checkNotNull((Object)descriptor.findFieldByName(field.name().toLowerCase()));
            Object value = element.get(field.name());
            if (value == null) {
                if (!fieldDescriptor.isRequired()) continue;
                throw new IllegalArgumentException("Received null value for non-nullable field " + fieldDescriptor.getName());
            }
            value = AvroToProtoSerializer.toProtoValue(fieldDescriptor, field.schema(), value);
            builder.setField(fieldDescriptor, value);
        }
        return builder.build();
    }

    private static Object toProtoValue(Descriptors.FieldDescriptor fieldDescriptor, Schema avroSchema, Object value) {
        switch (avroSchema.getType()) {
            case RECORD: {
                return AvroToProtoSerializer.getDynamicMessageFromGenericRecord((GenericRecord)value, fieldDescriptor.getMessageType());
            }
            case ARRAY: {
                return AvroSchemaHandler.handleArraySchema(fieldDescriptor, avroSchema, value);
            }
            case UNION: {
                Schema type = (Schema)AvroSchemaHandler.handleUnionSchema(avroSchema).getLeft();
                return AvroToProtoSerializer.toProtoValue(fieldDescriptor, type, value);
            }
            case MAP: {
                throw new UnsupportedOperationException("MAP type not supported yet");
            }
            case STRING: {
                Object convertedValue = AvroSchemaHandler.handleLogicalTypeSchema(avroSchema, value);
                if (convertedValue != value) {
                    return convertedValue;
                }
                return value.toString();
            }
            case LONG: {
                Object convertedValue = AvroSchemaHandler.handleLogicalTypeSchema(avroSchema, value);
                if (convertedValue != value) {
                    return convertedValue;
                }
                return Long.parseLong(value.toString());
            }
            case INT: {
                Object convertedValue = AvroSchemaHandler.handleLogicalTypeSchema(avroSchema, value);
                if (convertedValue != value) {
                    return convertedValue;
                }
                return Integer.parseInt(value.toString());
            }
            case BYTES: {
                Object convertedValue = AvroSchemaHandler.handleLogicalTypeSchema(avroSchema, value);
                if (convertedValue != value) {
                    return convertedValue;
                }
                return ByteString.copyFrom((byte[])((ByteBuffer)value).array());
            }
            case ENUM: {
                return value.toString();
            }
            case FIXED: {
                return ByteString.copyFrom((byte[])((GenericData.Fixed)value).bytes());
            }
            case BOOLEAN: {
                return (boolean)((Boolean)value);
            }
            case FLOAT: {
                return Float.valueOf(Float.parseFloat(String.valueOf(((Float)value).floatValue())));
            }
            case DOUBLE: {
                return (double)((Double)value);
            }
            case NULL: {
                throw new IllegalArgumentException("Null Type Field not supported in BigQuery!");
            }
        }
        throw new IllegalArgumentException("Unexpected Avro type" + avroSchema);
    }

    static class AvroSchemaHandler {
        private static final Logger LOG = LoggerFactory.getLogger(AvroSchemaHandler.class);

        private AvroSchemaHandler() {
        }

        public static List<Object> handleArraySchema(Descriptors.FieldDescriptor fieldDescriptor, Schema avroSchema, Object value) {
            if (!(value instanceof Iterable)) {
                LOG.error(AvroSchemaHandler.getLogErrorMessage("Iterable", "ARRAY", value.getClass().toString()));
                throw new IllegalArgumentException("Expecting the value as Iterable type for type ARRAY.");
            }
            Iterable iterable = (Iterable)value;
            Schema arrayElementType = avroSchema.getElementType();
            if (arrayElementType.isNullable()) {
                throw new IllegalArgumentException("Array cannot have NULLABLE datatype");
            }
            if (arrayElementType.isUnion()) {
                throw new IllegalArgumentException("ARRAY cannot have multiple datatypes in BigQuery.");
            }
            return StreamSupport.stream(iterable.spliterator(), false).map(v -> AvroToProtoSerializer.toProtoValue(fieldDescriptor, arrayElementType, v)).collect(Collectors.toList());
        }

        public static ImmutablePair<Schema, Boolean> handleUnionSchema(Schema schema) throws IllegalArgumentException {
            List types = schema.getTypes();
            if (types.size() == 1) {
                if (((Schema)types.get(0)).getType() != Schema.Type.NULL) {
                    return new ImmutablePair(types.get(0), (Object)false);
                }
            } else if (types.size() == 2) {
                if (((Schema)types.get(0)).getType() != Schema.Type.NULL && ((Schema)types.get(1)).getType() == Schema.Type.NULL) {
                    return new ImmutablePair(types.get(0), (Object)true);
                }
                if (((Schema)types.get(0)).getType() == Schema.Type.NULL && ((Schema)types.get(1)).getType() != Schema.Type.NULL) {
                    return new ImmutablePair(types.get(1), (Object)true);
                }
            }
            LOG.error(AvroSchemaHandler.getLogErrorMessage("['datatype'] or ['null', 'datatype']", "UNION", "Multiple not-null types: " + types));
            throw new IllegalArgumentException("Multiple non-null union types are not supported.");
        }

        private static Object handleLogicalTypeSchema(Schema fieldSchema, Object value) {
            UnaryOperator<Object> encoder;
            String logicalTypeString = fieldSchema.getProp("logicalType");
            if (logicalTypeString != null && (encoder = AvroSchemaHandler.getLogicalEncoder(fieldSchema, logicalTypeString)) != null) {
                return encoder.apply(value);
            }
            return value;
        }

        private static UnaryOperator<Object> getLogicalEncoder(Schema fieldSchema, String logicalTypeString) {
            HashMap<String, UnaryOperator> mapping = new HashMap<String, UnaryOperator>();
            mapping.put(LogicalTypes.date().getName(), AvroSchemaHandler::convertDate);
            mapping.put(LogicalTypes.decimal((int)1).getName(), value -> AvroSchemaHandler.convertBigDecimal(value, fieldSchema));
            mapping.put(LogicalTypes.timestampMicros().getName(), value -> AvroSchemaHandler.convertTimestamp(value, true, "Timestamp(micros/millis)"));
            mapping.put(LogicalTypes.timestampMillis().getName(), value -> AvroSchemaHandler.convertTimestamp(value, false, "Timestamp(micros/millis)"));
            mapping.put(LogicalTypes.uuid().getName(), AvroSchemaHandler::convertUUID);
            mapping.put(LogicalTypes.timeMillis().getName(), value -> AvroSchemaHandler.convertTime(value, false));
            mapping.put(LogicalTypes.timeMicros().getName(), value -> AvroSchemaHandler.convertTime(value, true));
            mapping.put(LogicalTypes.localTimestampMillis().getName(), value -> AvroSchemaHandler.convertDateTime(value, false));
            mapping.put(LogicalTypes.localTimestampMicros().getName(), value -> AvroSchemaHandler.convertDateTime(value, true));
            mapping.put("geography_wkt", AvroSchemaHandler::convertGeography);
            mapping.put("Json", AvroSchemaHandler::convertJson);
            return (UnaryOperator)mapping.get(logicalTypeString);
        }

        @VisibleForTesting
        static String convertUUID(Object value) {
            if (value instanceof UUID) {
                return ((UUID)value).toString();
            }
            if (value instanceof String) {
                UUID.fromString((String)value);
                return (String)value;
            }
            LOG.error(AvroSchemaHandler.getLogErrorMessage("String/UUID", "UUID", value.getClass().toString()));
            throw new IllegalArgumentException(AvroSchemaHandler.getErrorMessage("String/UUID", "UUID"));
        }

        private static void validateTimestamp(long timestamp) {
            try {
                Timestamp.ofTimeMicroseconds((long)timestamp);
            }
            catch (IllegalArgumentException e) {
                LOG.error(String.format("Invalid Timestamp '%s' Provided.\nShould be a long value indicating microseconds since Epoch between %s and %s", timestamp, Timestamp.MIN_VALUE, Timestamp.MAX_VALUE));
                throw new IllegalArgumentException(String.format("Invalid Timestamp '%s' Provided.", timestamp));
            }
        }

        static Long convertTimestamp(Object value, boolean micros, String type) {
            long timestamp;
            if (value instanceof ReadableInstant) {
                timestamp = TimeUnit.MILLISECONDS.toMicros(((ReadableInstant)value).getMillis());
            } else if (value instanceof Long) {
                timestamp = micros ? ((Long)value).longValue() : TimeUnit.MILLISECONDS.toMicros((Long)value);
            } else {
                LOG.error(AvroSchemaHandler.getLogErrorMessage("LONG/ReadableInstant", "TIMESTAMP (" + type + ")", value.getClass().toString()));
                throw new IllegalArgumentException(AvroSchemaHandler.getErrorMessage("LONG/ReadableInstant", "TIMESTAMP"));
            }
            AvroSchemaHandler.validateTimestamp(timestamp);
            return timestamp;
        }

        private static void validateDate(Integer date) {
            int maxDateValue = 2932896;
            int minDateValue = -719162;
            if (date > maxDateValue || date < minDateValue) {
                LOG.error(String.format("Invalid Date '%s' Provided.\nShould be a Integer value indicating days since Epoch (1970-01-01 00:00:00) between %s and %s", LocalDate.ofEpochDay(date.intValue()), "0001-01-01", "9999-12-31"));
                throw new IllegalArgumentException("Invalid date Provided.");
            }
        }

        @VisibleForTesting
        static Integer convertDate(Object value) {
            int date;
            if (value instanceof ReadableInstant) {
                date = Days.daysBetween((ReadableInstant)Instant.EPOCH, (ReadableInstant)((ReadableInstant)value)).getDays();
            } else if (value instanceof Integer) {
                date = (Integer)value;
            } else {
                LOG.error(AvroSchemaHandler.getLogErrorMessage("ReadableInstant/Integer", "Days(micros/millis)", value.getClass().toString()));
                throw new IllegalArgumentException(AvroSchemaHandler.getErrorMessage("Integer", "DATE"));
            }
            AvroSchemaHandler.validateDate(date);
            return date;
        }

        static String convertDateTime(Object value, boolean micros) {
            String obtainedValue;
            if (value instanceof Long) {
                Long convertedTs = AvroSchemaHandler.convertTimestamp(value, micros, "Local Timestamp(millis/micros)");
                Timestamp convertedTimestamp = Timestamp.ofTimeMicroseconds((long)convertedTs);
                return LocalDateTime.ofEpochSecond(convertedTimestamp.getSeconds(), convertedTimestamp.getNanos(), ZoneOffset.UTC).toString();
            }
            if (value instanceof String) {
                obtainedValue = (String)value;
            } else if (value instanceof Utf8) {
                obtainedValue = ((Utf8)value).toString();
            } else {
                LOG.error(AvroSchemaHandler.getLogErrorMessage("String/Long/UTF-8", "Local-Timestamp(micros/millis)", value.getClass().toString()));
                throw new IllegalArgumentException(AvroSchemaHandler.getErrorMessage("String/LONG/UTF-8", "Local-Timestamp(micros/millis)"));
            }
            try {
                return LocalDateTime.parse(obtainedValue, DateTimeFormatter.ofPattern("yyyy-M[M]-d[d][[' ']['T']['t']H[H]':'m[m]':'s[s]['.'SSSSSS]['.'SSSSS]['.'SSSS]['.'SSS]['.'SS]['.'S]]")).toString();
            }
            catch (DateTimeParseException e) {
                throw new IllegalArgumentException(String.format("The datetime string obtained %s, is of invalid format.", (String)value));
            }
        }

        private static void validateTime(long value) {
            long minTimeMicros = ChronoUnit.MICROS.between(LocalTime.MIDNIGHT, LocalTime.MIN);
            long maxTimeMicros = ChronoUnit.MICROS.between(LocalTime.MIDNIGHT, LocalTime.MAX);
            if (value < minTimeMicros || value > maxTimeMicros) {
                LOG.error(String.format("Input Time should be between %s and %s.%n Found %s instead.", LocalTime.MIN, LocalTime.MIN, LocalTime.MIDNIGHT.plusNanos(TimeUnit.MICROSECONDS.toNanos(value))));
                throw new IllegalArgumentException("Invalid time value obtained.");
            }
        }

        /*
         * Enabled aggressive block sorting
         * Enabled unnecessary exception pruning
         * Enabled aggressive exception aggregation
         */
        @VisibleForTesting
        static String convertTime(Object value, boolean micros) {
            if (value instanceof String) {
                try {
                    return LocalTime.parse((String)value, DateTimeFormatter.ofPattern("H[H]':'m[m]':'s[s]['.'SSSSSS]['.'SSSSS]['.'SSSS]['.'SSS]['.'SS]['.'S]")).toString();
                }
                catch (DateTimeParseException e) {
                    throw new IllegalArgumentException(String.format("The datetime string obtained %s, is of invalid format.", (String)value));
                }
            }
            Long microSecondsSinceMidnight = null;
            if (micros) {
                if (!(value instanceof Long)) {
                    LOG.error(AvroSchemaHandler.getLogErrorMessage("LONG", "Time(micros)", value.getClass().toString()));
                    throw new IllegalArgumentException(AvroSchemaHandler.getErrorMessage("LONG", "Time(micros)"));
                }
                microSecondsSinceMidnight = (long)((Long)value);
            } else {
                if (!(value instanceof Integer)) {
                    LOG.error(AvroSchemaHandler.getLogErrorMessage("INTEGER", "Time(millis)", value.getClass().toString()));
                    throw new IllegalArgumentException(AvroSchemaHandler.getErrorMessage("INTEGER", "Time(millis)"));
                }
                microSecondsSinceMidnight = TimeUnit.MILLISECONDS.toMicros(((Integer)value).intValue());
            }
            AvroSchemaHandler.validateTime(microSecondsSinceMidnight);
            LocalTime time = LocalTime.MIDNIGHT.plusNanos(TimeUnit.MICROSECONDS.toNanos(microSecondsSinceMidnight));
            return time.toString();
        }

        @VisibleForTesting
        static ByteString convertBigDecimal(Object value, Schema fieldSchema) {
            if (!(value instanceof ByteBuffer)) {
                LOG.error(AvroSchemaHandler.getLogErrorMessage("ByteBuffer", "BigDecimal", value.getClass().toString()));
                throw new IllegalArgumentException(AvroSchemaHandler.getErrorMessage("ByteBuffer", "BigDecimal"));
            }
            byte[] byteArray = ((ByteBuffer)value).array();
            Preconditions.checkNotNull((Object)fieldSchema.getLogicalType(), (Object)"Invalid decimal type obtained");
            LogicalTypes.Decimal decimalFieldSchema = (LogicalTypes.Decimal)fieldSchema.getLogicalType();
            int scale = decimalFieldSchema.getScale();
            BigInteger scaledValue = new BigInteger(byteArray);
            BigDecimal decimalValue = new BigDecimal(scaledValue, scale);
            try {
                if (fieldSchema.getObjectProp("isNumeric") != null && ((Boolean)fieldSchema.getObjectProp("isNumeric")).booleanValue()) {
                    return BigDecimalByteStringEncoder.encodeToNumericByteString((BigDecimal)decimalValue);
                }
                return BigDecimalByteStringEncoder.encodeToBigNumericByteString((BigDecimal)decimalValue);
            }
            catch (IllegalArgumentException e) {
                throw new IllegalArgumentException(e);
            }
        }

        @VisibleForTesting
        static String convertGeography(Object value) {
            if (value instanceof Utf8) {
                return ((Utf8)value).toString();
            }
            if (value instanceof String) {
                return (String)value;
            }
            LOG.error(AvroSchemaHandler.getLogErrorMessage("STRING/UTF-8", "geography_wkt or geojson format", value.getClass().toString()));
            throw new IllegalArgumentException(AvroSchemaHandler.getErrorMessage("STRING/UTF-8", "GEOGRAPHY"));
        }

        @VisibleForTesting
        static String convertJson(Object value) {
            String jsonString;
            if (value instanceof Utf8) {
                jsonString = ((Utf8)value).toString();
            } else if (value instanceof String) {
                jsonString = (String)value;
            } else {
                LOG.error(AvroSchemaHandler.getLogErrorMessage("UTF-8/STRING", "JSON", value.getClass().toString()));
                throw new IllegalArgumentException(AvroSchemaHandler.getErrorMessage("UTF-8/STRING", "JSON"));
            }
            try {
                new JSONObject(jsonString);
            }
            catch (JSONException e) {
                throw new IllegalArgumentException(String.format("The input string %s is not in valid JSON Format.", jsonString));
            }
            return jsonString;
        }

        private static String getLogErrorMessage(String expectedType, String actualType, String foundInstead) {
            return String.format("Expecting the value as %s type for %s.%nFound %s instead.", expectedType, actualType, foundInstead);
        }

        private static String getErrorMessage(String expectedType, String actualType) {
            return String.format("Expecting the value as %s type for type %s.", expectedType, actualType);
        }
    }
}

