/*
 * Decompiled with CFR 0.152.
 */
package com.google.cloud.flink.bigquery.source.reader.deserializer;

import com.google.api.client.util.Preconditions;
import com.google.cloud.Timestamp;
import com.google.cloud.flink.bigquery.sink.serializer.AvroSchemaConvertor;
import java.io.Serializable;
import java.lang.reflect.Array;
import java.nio.ByteBuffer;
import java.time.Instant;
import java.time.LocalDate;
import java.time.LocalDateTime;
import java.time.LocalTime;
import java.time.format.DateTimeFormatter;
import java.time.format.DateTimeParseException;
import java.time.temporal.ChronoField;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.TimeUnit;
import org.apache.avro.generic.GenericFixed;
import org.apache.avro.generic.IndexedRecord;
import org.apache.avro.util.Utf8;
import org.apache.flink.annotation.Internal;
import org.apache.flink.table.api.DataTypes;
import org.apache.flink.table.data.DecimalData;
import org.apache.flink.table.data.GenericArrayData;
import org.apache.flink.table.data.GenericMapData;
import org.apache.flink.table.data.GenericRowData;
import org.apache.flink.table.data.StringData;
import org.apache.flink.table.data.TimestampData;
import org.apache.flink.table.types.logical.ArrayType;
import org.apache.flink.table.types.logical.DecimalType;
import org.apache.flink.table.types.logical.LocalZonedTimestampType;
import org.apache.flink.table.types.logical.LogicalType;
import org.apache.flink.table.types.logical.RowType;
import org.apache.flink.table.types.logical.TimeType;
import org.apache.flink.table.types.logical.TimestampType;
import org.apache.flink.table.types.logical.utils.LogicalTypeUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@Internal
public class AvroToRowDataConverters {
    private static final Logger LOG = LoggerFactory.getLogger(AvroToRowDataConverters.class);
    private static final int MILLIS_PRECISION = 3;
    private static final int MICROS_PRECISION = 6;

    public static AvroToRowDataConverter createRowConverter(RowType rowType) {
        AvroToRowDataConverter[] fieldConverters = (AvroToRowDataConverter[])rowType.getFields().stream().map(RowType.RowField::getType).map(AvroToRowDataConverters::createNullableConverter).toArray(AvroToRowDataConverter[]::new);
        int arity = rowType.getFieldCount();
        return avroObject -> {
            IndexedRecord record = (IndexedRecord)avroObject;
            GenericRowData row = new GenericRowData(arity);
            for (int i = 0; i < arity; ++i) {
                row.setField(i, fieldConverters[i].convert(record.get(i)));
            }
            return row;
        };
    }

    private static AvroToRowDataConverter createNullableConverter(LogicalType type) {
        AvroToRowDataConverter converter = AvroToRowDataConverters.createConverter(type);
        return avroObject -> {
            if (avroObject == null) {
                return null;
            }
            return converter.convert(avroObject);
        };
    }

    private static AvroToRowDataConverter createConverter(LogicalType type) {
        switch (type.getTypeRoot()) {
            case NULL: {
                return avroObject -> null;
            }
            case TINYINT: {
                return avroObject -> ((Integer)avroObject).byteValue();
            }
            case SMALLINT: {
                return avroObject -> ((Integer)avroObject).shortValue();
            }
            case BOOLEAN: 
            case INTEGER: 
            case INTERVAL_YEAR_MONTH: 
            case BIGINT: 
            case INTERVAL_DAY_TIME: 
            case FLOAT: 
            case DOUBLE: {
                return avroObject -> avroObject;
            }
            case DATE: {
                return AvroToRowDataConverters::convertToDate;
            }
            case TIME_WITHOUT_TIME_ZONE: {
                int timePrecision = ((TimeType)type).getPrecision();
                if (timePrecision <= 3) {
                    return avroObject -> AvroToRowDataConverters.convertToMillisTime(avroObject, 3);
                }
                if (timePrecision <= 6) {
                    return avroObject -> AvroToRowDataConverters.convertToMicrosTime(avroObject, 6);
                }
                throw new UnsupportedOperationException(String.format("Avro to RowData Conversion Error:%nThe TIME type within the Avro schema uses a precision of '%d', which is higher than the maximum supported TIME precision %d.", timePrecision, 6));
            }
            case TIMESTAMP_WITH_LOCAL_TIME_ZONE: {
                int localTsPrecision = ((LocalZonedTimestampType)type).getPrecision();
                return AvroToRowDataConverters.createTimeDatetimeConvertor(localTsPrecision);
            }
            case TIMESTAMP_WITHOUT_TIME_ZONE: {
                int tsPrecision = ((TimestampType)type).getPrecision();
                return AvroToRowDataConverters.createTimeDatetimeConvertor(tsPrecision);
            }
            case CHAR: 
            case VARCHAR: {
                return avroObject -> StringData.fromString((String)avroObject.toString());
            }
            case BINARY: 
            case VARBINARY: {
                return AvroToRowDataConverters::convertToBytes;
            }
            case DECIMAL: {
                return AvroToRowDataConverters.createDecimalConverter((DecimalType)type);
            }
            case ARRAY: {
                return AvroToRowDataConverters.createArrayConverter((ArrayType)type);
            }
            case ROW: {
                return AvroToRowDataConverters.createRowConverter((RowType)type);
            }
            case MAP: 
            case MULTISET: {
                return AvroToRowDataConverters.createMapConverter(type);
            }
        }
        throw new UnsupportedOperationException(String.format("Avro to RowData Conversion Error:%nInput Logical Type %s is not supported.", type.asSummaryString()));
    }

    private static AvroToRowDataConverter createTimeDatetimeConvertor(int precision) {
        if (precision <= 3) {
            return avroObject -> AvroToRowDataConverters.convertToTimestamp(avroObject, 3);
        }
        if (precision <= 6) {
            return avroObject -> AvroToRowDataConverters.convertToTimestamp(avroObject, 6);
        }
        String invalidPrecisionError = String.format("Avro to RowData Conversion Error:%nThe TIMESTAMP/DATETIME type within the Avro schema uses a precision of '%d', which is higher than the maximum supported TIMESTAMP/DATETIME precision %d.", precision, 6);
        throw new UnsupportedOperationException(invalidPrecisionError);
    }

    private static AvroToRowDataConverter createDecimalConverter(DecimalType decimalType) {
        int precision = decimalType.getPrecision();
        int scale = decimalType.getScale();
        return avroObject -> {
            byte[] bytes;
            if (avroObject instanceof GenericFixed) {
                bytes = ((GenericFixed)avroObject).bytes();
            } else if (avroObject instanceof ByteBuffer) {
                ByteBuffer byteBuffer = (ByteBuffer)avroObject;
                bytes = new byte[byteBuffer.remaining()];
                byteBuffer.get(bytes);
            } else if (avroObject instanceof byte[]) {
                bytes = (byte[])avroObject;
            } else {
                String invalidFormatError = AvroToRowDataConverters.getErrorMessage(avroObject, "GenericFixed, byte[] and java.nio.ByteBuffer", "DECIMAL");
                throw new IllegalArgumentException(invalidFormatError);
            }
            return DecimalData.fromUnscaledBytes((byte[])bytes, (int)precision, (int)scale);
        };
    }

    private static String getErrorMessage(Object avroObject, String expectedType, String convertingType) {
        return String.format("Avro to RowData Conversion Error:%nUnexpected Avro object %s of type '%s' input for conversion to Flink's '%s' logical type. Supported type(s): '%s'", avroObject, avroObject.getClass(), convertingType, expectedType);
    }

    private static AvroToRowDataConverter createArrayConverter(ArrayType arrayType) {
        AvroToRowDataConverter elementConverter = AvroToRowDataConverters.createNullableConverter(arrayType.getElementType());
        Class elementClass = LogicalTypeUtils.toInternalConversionClass((LogicalType)arrayType.getElementType());
        return avroObject -> {
            Preconditions.checkArgument((boolean)(avroObject instanceof List), (Object)AvroToRowDataConverters.getErrorMessage(avroObject, "LIST", "ARRAY"));
            List list = (List)avroObject;
            int length = list.size();
            Object[] array = (Object[])Array.newInstance(elementClass, length);
            for (int i = 0; i < length; ++i) {
                array[i] = elementConverter.convert(list.get(i));
            }
            return new GenericArrayData(array);
        };
    }

    private static AvroToRowDataConverter createMapConverter(LogicalType type) {
        AvroToRowDataConverter keyConverter = AvroToRowDataConverters.createConverter(DataTypes.STRING().getLogicalType());
        AvroToRowDataConverter valueConverter = AvroToRowDataConverters.createNullableConverter(AvroSchemaConvertor.extractValueTypeToAvroMap(type));
        return avroObject -> {
            Preconditions.checkArgument((boolean)(avroObject instanceof Map), (Object)AvroToRowDataConverters.getErrorMessage(avroObject, "MAP", "MAP"));
            Map map = (Map)avroObject;
            HashMap<Object, Object> result = new HashMap<Object, Object>();
            for (Map.Entry entry : map.entrySet()) {
                Object key = keyConverter.convert(entry.getKey());
                Object value = valueConverter.convert(entry.getValue());
                result.put(key, value);
            }
            return new GenericMapData(result);
        };
    }

    static TimestampData convertToTimestamp(Object object, int precision) {
        long tempMicros;
        if (object instanceof Long) {
            tempMicros = (Long)object;
            if (precision == 3) {
                return TimestampData.fromEpochMillis((long)tempMicros);
            }
        } else {
            if (object instanceof Instant) {
                return TimestampData.fromInstant((Instant)((Instant)object));
            }
            if (object instanceof String || object instanceof Utf8) {
                String tsValue = object instanceof String ? (String)object : ((Utf8)object).toString();
                try {
                    return TimestampData.fromLocalDateTime((LocalDateTime)LocalDateTime.parse(tsValue, DateTimeFormatter.ofPattern("yyyy-M[M]-d[d][[' ']['T']['t']H[H]':'m[m]':'s[s]['.'SSSSSS]['.'SSSSS]['.'SSSS]['.'SSS]['.'SS]['.'S]]")));
                }
                catch (DateTimeParseException e) {
                    String invalidFormatError = String.format("Avro to RowData Conversion Error:%nThe Avro datetime string input for conversion to Flink's 'TIMESTAMP_LTZ' Logical Type %s, is of invalid format.", object);
                    String supportedFormat = "civil_date_part[time_part]\ncivil_date_part: YYYY-[M]M-[D]D\ntime_part: { |T|t}[H]H:[M]M:[S]S[.F]\nYYYY: Four-digit year.\n[M]M: One or two digit month.\n[D]D: One or two digit day.\n{ |T|t}: A space or a T or t separator. The T and t separators are flags for time.\n[H]H: One or two digit hour (valid values from 00 to 23).\n[M]M: One or two digit minutes (valid values from 00 to 59).\n[S]S: One or two digit seconds (valid values from 00 to 60).\n[.F]: Up to six fractional digits (microsecond precision).";
                    LOG.error(String.format("%s%nSupported Format:%n%s", invalidFormatError, supportedFormat));
                    throw new IllegalArgumentException(invalidFormatError);
                }
            }
            if (object instanceof Timestamp) {
                Timestamp localDateTime = (Timestamp)object;
                tempMicros = TimeUnit.SECONDS.toMicros(localDateTime.getSeconds()) + TimeUnit.NANOSECONDS.toMicros(localDateTime.getNanos());
            } else {
                String invalidFormatError = AvroToRowDataConverters.getErrorMessage(object, "LONG, STRING/UTF-8, com.google.cloud.Timestamp and java.time.Instant", "TIMESTAMP/DATETIME");
                throw new IllegalArgumentException(invalidFormatError);
            }
        }
        long micros = tempMicros;
        long millis = TimeUnit.MICROSECONDS.toMillis(micros);
        long nanos = micros % 1000L;
        nanos = TimeUnit.MICROSECONDS.toNanos(nanos);
        return TimestampData.fromEpochMillis((long)millis, (int)((int)nanos));
    }

    private static int convertToDate(Object object) {
        if (object instanceof Integer) {
            return (Integer)object;
        }
        if (object instanceof LocalDate) {
            return (int)((LocalDate)object).toEpochDay();
        }
        if (object instanceof org.joda.time.LocalDate) {
            org.joda.time.LocalDate value = (org.joda.time.LocalDate)object;
            return (int)value.toDate().getTime();
        }
        String invalidFormatError = AvroToRowDataConverters.getErrorMessage(object, "INT, org.joda.time.LocalDate and java.time.LocalDate", "DATE");
        throw new IllegalArgumentException(invalidFormatError);
    }

    private static int convertToMillisTime(Object object, int precision) {
        Preconditions.checkArgument((precision == 3 ? 1 : 0) != 0, (Object)String.format("Avro to RowData Conversion Error:%nThe millisecond-precision TIME type within the Avro schema uses a precision of '%d',Supported '%d'", precision, 3));
        if (object instanceof Integer) {
            return (Integer)object;
        }
        if (object instanceof LocalTime) {
            return ((LocalTime)object).get(ChronoField.MILLI_OF_DAY);
        }
        String invalidFormatError = AvroToRowDataConverters.getErrorMessage(object, "INT and java.time.LocalTime", "MILLIS-TIME");
        throw new IllegalArgumentException(invalidFormatError);
    }

    private static long convertToMicrosTime(Object object, int precision) {
        Preconditions.checkArgument((precision == 6 ? 1 : 0) != 0, (Object)String.format("Avro to RowData Conversion Error:%nThe microsecond-precision TIME type within the Avro schema uses a precision of '%d',Supported '%d'", precision, 6));
        if (object instanceof Long) {
            return (Long)object;
        }
        if (object instanceof LocalTime) {
            return TimeUnit.NANOSECONDS.toMicros(((LocalTime)object).toNanoOfDay());
        }
        String invalidFormatError = AvroToRowDataConverters.getErrorMessage(object, "LONG and java.time.LocalTime", "MICROS-TIME");
        throw new IllegalArgumentException(invalidFormatError);
    }

    private static byte[] convertToBytes(Object object) {
        if (object instanceof GenericFixed) {
            return ((GenericFixed)object).bytes();
        }
        if (object instanceof ByteBuffer) {
            ByteBuffer byteBuffer = (ByteBuffer)object;
            byte[] bytes = new byte[byteBuffer.remaining()];
            byteBuffer.get(bytes);
            return bytes;
        }
        if (object instanceof byte[]) {
            return (byte[])object;
        }
        String invalidFormatError = AvroToRowDataConverters.getErrorMessage(object, "GenericFixed, byte[] and java.nio.ByteBuffer", "BYTE");
        throw new IllegalArgumentException(invalidFormatError);
    }

    @FunctionalInterface
    public static interface AvroToRowDataConverter
    extends Serializable {
        public Object convert(Object var1);
    }
}

