/*
 * Decompiled with CFR 0.152.
 */
package com.google.cloud.flink.bigquery.common.utils;

import com.google.api.services.bigquery.model.TableSchema;
import com.google.cloud.bigquery.StandardSQLTypeName;
import com.google.cloud.flink.bigquery.common.utils.SchemaTransform;
import com.google.cloud.flink.bigquery.services.PartitionIdWithInfo;
import com.google.cloud.flink.bigquery.services.PartitionIdWithInfoAndStatus;
import com.google.cloud.flink.bigquery.services.TablePartitionInfo;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.time.Instant;
import java.time.LocalDateTime;
import java.time.LocalTime;
import java.time.ZoneId;
import java.time.ZoneOffset;
import java.time.ZonedDateTime;
import java.time.format.DateTimeFormatter;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Date;
import java.util.List;
import java.util.Optional;
import java.util.TimeZone;
import java.util.function.Function;
import java.util.stream.Collectors;
import org.apache.commons.lang3.StringUtils;
import org.apache.flink.annotation.Internal;
import org.apache.flink.util.Preconditions;

@Internal
public class BigQueryPartitionUtils {
    static final Integer HOUR_SECONDS = 3600;
    static final Integer DAY_SECONDS = 86400;
    static final Integer MONTH_SECONDS = 2678400;
    static final Integer YEAR_SECONDS = 31622400;
    private static final ZoneId UTC_ZONE = ZoneId.of("UTC");
    private static final TimeZone UTC_TIME_ZONE = TimeZone.getTimeZone(UTC_ZONE);
    private static final String BQPARTITION_HOUR_FORMAT_STRING = "yyyyMMddHH";
    private static final String BQPARTITION_DAY_FORMAT_STRING = "yyyyMMdd";
    private static final String BQPARTITION_MONTH_FORMAT_STRING = "yyyyMM";
    private static final String BQPARTITION_YEAR_FORMAT_STRING = "yyyy";
    private static final String SQL_HOUR_FORMAT_STRING = "yyyy-MM-dd HH:00:00";
    private static final String SQL_DAY_FORMAT_STRING = "yyyy-MM-dd";
    private static final String SQL_MONTH_FORMAT_STRING = "yyyy-MM-01";
    private static final String SQL_YEAR_FORMAT_STRING = "yyyy-01-01";
    private static final SimpleDateFormat BQPARTITION_HOUR_FORMAT = new SimpleDateFormat("yyyyMMddHH");
    private static final SimpleDateFormat BQPARTITION_DAY_FORMAT = new SimpleDateFormat("yyyyMMdd");
    private static final SimpleDateFormat BQPARTITION_MONTH_FORMAT = new SimpleDateFormat("yyyyMM");
    private static final SimpleDateFormat BQPARTITION_YEAR_FORMAT = new SimpleDateFormat("yyyy");
    private static final SimpleDateFormat SQL_HOUR_FORMAT = new SimpleDateFormat("yyyy-MM-dd HH:00:00");
    private static final SimpleDateFormat SQL_DAY_FORMAT = new SimpleDateFormat("yyyy-MM-dd");
    private static final SimpleDateFormat SQL_MONTH_FORMAT = new SimpleDateFormat("yyyy-MM-01");
    private static final SimpleDateFormat SQL_YEAR_FORMAT = new SimpleDateFormat("yyyy-01-01");

    private BigQueryPartitionUtils() {
    }

    public static StandardSQLTypeName retrievePartitionColumnType(TableSchema schema, String partitionColumn) {
        return SchemaTransform.bigQueryTableFieldSchemaTypeToSQLType(schema.getFields().stream().filter(tfs -> tfs.getName().equals(partitionColumn)).map(tfs -> tfs.getType()).findAny().orElseThrow(() -> new IllegalStateException(String.format("The retrieved partition column provided %s does not correlate with a first level column in the schema %s.", partitionColumn, schema.toString()))));
    }

    static List<String> partitionIdToDateFormat(List<String> partitions, SimpleDateFormat parseFormat, SimpleDateFormat printFormat) {
        return partitions.stream().map(id -> {
            try {
                return parseFormat.parse((String)id);
            }
            catch (ParseException ex) {
                throw new RuntimeException("Problems parsing the temporal value: " + id, ex);
            }
        }).map(date -> printFormat.format((Date)date)).collect(Collectors.toList());
    }

    public static String partitionValueToValueGivenType(String partitionValue, StandardSQLTypeName dataType) {
        switch (dataType) {
            case INT64: {
                return partitionValue;
            }
            case DATE: 
            case DATETIME: 
            case TIMESTAMP: {
                return String.format("'%s'", partitionValue);
            }
        }
        throw new IllegalArgumentException(String.format("The provided SQL type name (%s) is not supported as a partition column.", dataType.name()));
    }

    static String dateRestrictionFromPartitionType(PartitionType partitionType, String columnName, String valueFromSQL) {
        String temporalFormat = "%s >= '%s' AND %s < '%s'";
        try {
            switch (partitionType) {
                case DAY: {
                    Date day = SQL_DAY_FORMAT.parse(valueFromSQL);
                    return String.format(temporalFormat, columnName, SQL_DAY_FORMAT.format(day), columnName, SQL_DAY_FORMAT.format(Date.from(day.toInstant().plusSeconds(DAY_SECONDS.intValue()))));
                }
                case MONTH: {
                    Date day = SQL_MONTH_FORMAT.parse(valueFromSQL);
                    return String.format(temporalFormat, columnName, SQL_MONTH_FORMAT.format(day), columnName, DateTimeFormatter.ofPattern(SQL_DAY_FORMAT_STRING).withZone(UTC_ZONE).format(day.toInstant().atZone(UTC_ZONE).toLocalDate().plusMonths(1L).atTime(LocalTime.MIDNIGHT).toInstant(ZoneOffset.UTC)));
                }
                case YEAR: {
                    Date day = SQL_YEAR_FORMAT.parse(valueFromSQL);
                    return String.format(temporalFormat, columnName, SQL_YEAR_FORMAT.format(day), columnName, DateTimeFormatter.ofPattern(SQL_YEAR_FORMAT_STRING).withZone(UTC_ZONE).format(day.toInstant().atZone(UTC_ZONE).toLocalDate().plusYears(1L).atTime(LocalTime.MIDNIGHT).toInstant(ZoneOffset.UTC)));
                }
            }
            throw new IllegalArgumentException(String.format("The provided partition type %s is not supported as a temporal based partition for the column %s.", new Object[]{partitionType, columnName}));
        }
        catch (ParseException ex) {
            throw new IllegalArgumentException("Problems while manipulating the temporal argument: " + valueFromSQL, ex);
        }
    }

    static String timestampRestrictionFromPartitionType(PartitionType partitionType, String columnName, String valueFromSQL) {
        ZonedDateTime parsedDateTime = LocalDateTime.parse(valueFromSQL, DateTimeFormatter.ofPattern("yyyy-MM-dd HH:mm:ss").withZone(UTC_ZONE)).atZone(UTC_ZONE);
        String temporalFormat = "%s >= '%s' AND %s < '%s'";
        switch (partitionType) {
            case HOUR: {
                DateTimeFormatter hourFormatter = DateTimeFormatter.ofPattern(SQL_HOUR_FORMAT_STRING).withZone(UTC_ZONE);
                return String.format(temporalFormat, columnName, parsedDateTime.format(hourFormatter), columnName, parsedDateTime.plusHours(1L).format(hourFormatter));
            }
            case DAY: {
                DateTimeFormatter dayFormatter = DateTimeFormatter.ofPattern("yyyy-MM-dd 00:00:00").withZone(UTC_ZONE);
                return String.format(temporalFormat, columnName, parsedDateTime.format(dayFormatter), columnName, parsedDateTime.plusDays(1L).format(dayFormatter));
            }
            case MONTH: {
                DateTimeFormatter monthFormatter = DateTimeFormatter.ofPattern("yyyy-MM-01 00:00:00").withZone(UTC_ZONE);
                return String.format(temporalFormat, columnName, parsedDateTime.format(monthFormatter), columnName, parsedDateTime.plusMonths(1L).format(monthFormatter));
            }
            case YEAR: {
                DateTimeFormatter yearFormatter = DateTimeFormatter.ofPattern("yyyy-01-01 00:00:00").withZone(UTC_ZONE);
                return String.format(temporalFormat, columnName, parsedDateTime.format(yearFormatter), columnName, parsedDateTime.plusYears(1L).format(yearFormatter));
            }
        }
        throw new IllegalArgumentException(String.format("The provided partition type %s is not supported as a temporal based partition for the column %s.", new Object[]{partitionType, columnName}));
    }

    public static String formatPartitionRestrictionBasedOnInfo(Optional<TablePartitionInfo> tablePartitionInfo, String columnNameFromSQL, String valueFromSQL) {
        return tablePartitionInfo.map(info -> {
            switch (info.getColumnType()) {
                case INT64: {
                    return String.format("%s = %s", info.getColumnName(), valueFromSQL);
                }
                case DATE: {
                    return BigQueryPartitionUtils.dateRestrictionFromPartitionType(info.getPartitionType(), columnNameFromSQL, valueFromSQL);
                }
                case DATETIME: 
                case TIMESTAMP: {
                    return BigQueryPartitionUtils.timestampRestrictionFromPartitionType(info.getPartitionType(), columnNameFromSQL, valueFromSQL);
                }
            }
            throw new IllegalArgumentException(String.format("The provided SQL type name (%s) is not supported as a partition column in BigQuery.", info.getColumnType()));
        }).orElse(String.format("%s = %s", columnNameFromSQL, valueFromSQL));
    }

    static PartitionIdWithInfoAndStatus retrievePartitionInfoWithStatus(PartitionIdWithInfo partition, Function<String, Long> parseAndManipulatePartitionTS) {
        return BigQueryPartitionUtils.partitionValuesFromIdAndDataType(Arrays.asList(partition.getPartitionId()), partition.getInfo().getColumnType()).stream().map(parseAndManipulatePartitionTS).filter(nextPartitionTs -> partition.getInfo().getStreamingBufferOldestEntryTime().isAfter(Instant.ofEpochSecond(nextPartitionTs))).map(i -> new PartitionIdWithInfoAndStatus(partition.getPartitionId(), partition.getInfo(), PartitionStatus.COMPLETED)).findFirst().orElse(new PartitionIdWithInfoAndStatus(partition.getPartitionId(), partition.getInfo(), PartitionStatus.IN_PROGRESS));
    }

    static Instant retrieveEpochSecondsFromParsedTemporal(SimpleDateFormat sdf, String tsString) {
        try {
            return sdf.parse(tsString).toInstant();
        }
        catch (ParseException ex) {
            throw new RuntimeException("Problems while parsing temporal info from: " + tsString, ex);
        }
    }

    public static PartitionIdWithInfoAndStatus checkPartitionCompleted(PartitionIdWithInfo partition) {
        switch (partition.getInfo().getPartitionType()) {
            case HOUR: {
                return BigQueryPartitionUtils.retrievePartitionInfoWithStatus(partition, tsString -> BigQueryPartitionUtils.retrieveEpochSecondsFromParsedTemporal(SQL_HOUR_FORMAT, tsString).plusSeconds(HOUR_SECONDS.intValue()).getEpochSecond());
            }
            case DAY: {
                return BigQueryPartitionUtils.retrievePartitionInfoWithStatus(partition, tsString -> BigQueryPartitionUtils.retrieveEpochSecondsFromParsedTemporal(SQL_DAY_FORMAT, tsString).plusSeconds(DAY_SECONDS.intValue()).getEpochSecond());
            }
            case MONTH: {
                return BigQueryPartitionUtils.retrievePartitionInfoWithStatus(partition, tsString -> BigQueryPartitionUtils.retrieveEpochSecondsFromParsedTemporal(SQL_MONTH_FORMAT, tsString).plusSeconds(MONTH_SECONDS.intValue()).getEpochSecond());
            }
            case YEAR: {
                return BigQueryPartitionUtils.retrievePartitionInfoWithStatus(partition, tsString -> BigQueryPartitionUtils.retrieveEpochSecondsFromParsedTemporal(SQL_YEAR_FORMAT, tsString).plusSeconds(YEAR_SECONDS.intValue()).getEpochSecond());
            }
            case INT_RANGE: {
                return new PartitionIdWithInfoAndStatus(partition.getPartitionId(), partition.getInfo(), PartitionStatus.COMPLETED);
            }
        }
        throw new IllegalArgumentException("Partition type not supported: " + (Object)((Object)partition.getInfo().getPartitionType()));
    }

    public static List<String> partitionValuesFromIdAndDataType(List<String> partitionIds, StandardSQLTypeName dataType) {
        ArrayList<String> partitionValues = new ArrayList<String>();
        block0 : switch (dataType) {
            case INT64: {
                partitionValues.addAll(partitionIds);
                break;
            }
            case DATE: 
            case DATETIME: 
            case TIMESTAMP: {
                String firstId = partitionIds.get(0);
                Preconditions.checkState((boolean)partitionIds.stream().allMatch(pid -> pid.length() == firstId.length() && StringUtils.isNumeric((CharSequence)pid)), (Object)("Some elements in the partition id list have a different length: " + partitionIds.toString()));
                switch (firstId.length()) {
                    case 4: {
                        partitionValues.addAll(BigQueryPartitionUtils.partitionIdToDateFormat(partitionIds, BQPARTITION_YEAR_FORMAT, SQL_YEAR_FORMAT));
                        break block0;
                    }
                    case 6: {
                        partitionValues.addAll(BigQueryPartitionUtils.partitionIdToDateFormat(partitionIds, BQPARTITION_MONTH_FORMAT, SQL_MONTH_FORMAT));
                        break block0;
                    }
                    case 8: {
                        partitionValues.addAll(BigQueryPartitionUtils.partitionIdToDateFormat(partitionIds, BQPARTITION_DAY_FORMAT, SQL_DAY_FORMAT));
                        break block0;
                    }
                    case 10: {
                        partitionValues.addAll(BigQueryPartitionUtils.partitionIdToDateFormat(partitionIds, BQPARTITION_HOUR_FORMAT, SQL_HOUR_FORMAT));
                        break block0;
                    }
                }
                throw new IllegalArgumentException("The lenght of the partition id is not one of the expected ones: " + firstId);
            }
            default: {
                throw new IllegalArgumentException(String.format("The provided SQL type name (%s) is not supported as a partition column.", dataType.name()));
            }
        }
        return partitionValues;
    }

    static {
        BQPARTITION_HOUR_FORMAT.setTimeZone(UTC_TIME_ZONE);
        BQPARTITION_DAY_FORMAT.setTimeZone(UTC_TIME_ZONE);
        BQPARTITION_MONTH_FORMAT.setTimeZone(UTC_TIME_ZONE);
        BQPARTITION_YEAR_FORMAT.setTimeZone(UTC_TIME_ZONE);
        SQL_HOUR_FORMAT.setTimeZone(UTC_TIME_ZONE);
        SQL_DAY_FORMAT.setTimeZone(UTC_TIME_ZONE);
        SQL_MONTH_FORMAT.setTimeZone(UTC_TIME_ZONE);
        SQL_YEAR_FORMAT.setTimeZone(UTC_TIME_ZONE);
    }

    public static enum PartitionStatus {
        IN_PROGRESS,
        COMPLETED;

    }

    public static enum PartitionType {
        HOUR,
        DAY,
        MONTH,
        YEAR,
        INT_RANGE;

    }
}

