/*
 * Decompiled with CFR 0.152.
 */
package com.google.cloud.bigquery.storage.v1beta1.it;

import com.google.api.gax.core.CredentialsProvider;
import com.google.api.gax.core.FixedCredentialsProvider;
import com.google.api.gax.rpc.ServerStream;
import com.google.api.gax.rpc.UnauthenticatedException;
import com.google.auth.Credentials;
import com.google.auth.oauth2.GoogleCredentials;
import com.google.cloud.RetryOption;
import com.google.cloud.ServiceOptions;
import com.google.cloud.bigquery.BigQuery;
import com.google.cloud.bigquery.DatasetInfo;
import com.google.cloud.bigquery.Field;
import com.google.cloud.bigquery.Job;
import com.google.cloud.bigquery.JobConfiguration;
import com.google.cloud.bigquery.JobInfo;
import com.google.cloud.bigquery.LegacySQLTypeName;
import com.google.cloud.bigquery.QueryJobConfiguration;
import com.google.cloud.bigquery.Schema;
import com.google.cloud.bigquery.StandardTableDefinition;
import com.google.cloud.bigquery.TableDefinition;
import com.google.cloud.bigquery.TableId;
import com.google.cloud.bigquery.TableInfo;
import com.google.cloud.bigquery.TimePartitioning;
import com.google.cloud.bigquery.storage.v1beta1.BigQueryStorageClient;
import com.google.cloud.bigquery.storage.v1beta1.BigQueryStorageSettings;
import com.google.cloud.bigquery.storage.v1beta1.ReadOptions;
import com.google.cloud.bigquery.storage.v1beta1.Storage;
import com.google.cloud.bigquery.storage.v1beta1.TableReferenceProto;
import com.google.cloud.bigquery.storage.v1beta1.it.SimpleRowReader;
import com.google.cloud.bigquery.testing.RemoteBigQueryHelper;
import com.google.common.base.Preconditions;
import com.google.common.truth.Truth;
import com.google.protobuf.MessageOrBuilder;
import com.google.protobuf.TextFormat;
import com.google.protobuf.Timestamp;
import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.math.BigDecimal;
import java.nio.ByteBuffer;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.logging.Logger;
import org.apache.avro.Conversions;
import org.apache.avro.LogicalTypes;
import org.apache.avro.Schema;
import org.apache.avro.generic.GenericData;
import org.apache.avro.generic.GenericRecordBuilder;
import org.apache.avro.util.Utf8;
import org.junit.AfterClass;
import org.junit.Assert;
import org.junit.BeforeClass;
import org.junit.Test;
import org.threeten.bp.Duration;
import org.threeten.bp.Instant;
import org.threeten.bp.LocalDate;
import org.threeten.bp.LocalTime;
import org.threeten.bp.ZoneId;
import org.threeten.bp.ZoneOffset;
import org.threeten.bp.ZonedDateTime;
import org.threeten.bp.format.DateTimeFormatter;

public class ITBigQueryStorageTest {
    private static final Logger LOG = Logger.getLogger(ITBigQueryStorageTest.class.getName());
    private static final String DATASET = RemoteBigQueryHelper.generateDatasetName();
    private static final String DESCRIPTION = "BigQuery Storage Java client test dataset";
    private static BigQueryStorageClient client;
    private static String parentProjectId;
    private static BigQuery bigquery;
    private static final String FAKE_JSON_CRED_WITH_GOOGLE_DOMAIN = "{\n  \"private_key_id\": \"somekeyid\",\n  \"private_key\": \"-----BEGIN PRIVATE KEY-----\\nMIIEvgIBADANBgkqhkiG9w0BAQEFAASCBKgwggSkAgEAAoIBAQC+K2hSuFpAdrJI\\nnCgcDz2M7t7bjdlsadsasad+fvRSW6TjNQZ3p5LLQY1kSZRqBqylRkzteMOyHgaR\\n0Pmxh3ILCND5men43j3h4eDbrhQBuxfEMalkG92sL+PNQSETY2tnvXryOvmBRwa/\\nQP/9dJfIkIDJ9Fw9N4Bhhhp6mCcRpdQjV38H7JsyJ7lih/oNjECgYAt\\nknddadwkwewcVxHFhcZJO+XWf6ofLUXpRwiTZakGMn8EE1uVa2LgczOjwWHGi99MFjxSer5m9\\n1tCa3/KEGKiS/YL71JvjwX3mb+cewlkcmweBKZHM2JPTk0ZednFSpVZMtycjkbLa\\ndYOS8V85AgMBewECggEBAKksaldajfDZDV6nGqbFjMiizAKJolr/M3OQw16K6o3/\\n0S31xIe3sSlgW0+UbYlF4U8KifhManD1apVSC3csafaspP4RZUHFhtBywLO9pR5c\\nr6S5aLp+gPWFyIp1pfXbWGvc5VY/v9x7ya1VEa6rXvLsKupSeWAW4tMj3eo/64ge\\nsdaceaLYw52KeBYiT6+vpsnYrEkAHO1fF/LavbLLOFJmFTMxmsNaG0tuiJHgjshB\\n82DpMCbXG9YcCgI/DbzuIjsdj2JC1cascSP//3PmefWysucBQe7Jryb6NQtASmnv\\nCdDw/0jmZTEjpe4S1lxfHplAhHFtdgYTvyYtaLZiVVkCgYEA8eVpof2rceecw/I6\\n5ng1q3Hl2usdWV/4mZMvR0fOemacLLfocX6IYxT1zA1FFJlbXSRsJMf/Qq39mOR2\\nSpW+hr4jCoHeRVYLgsbggtrevGmILAlNoqCMpGZ6vDmJpq6ECV9olliDvpPgWOP+\\nmYPDreFBGxWvQrADNbRt2dmGsrsCgYEAyUHqB2wvJHFqdmeBsaacewzV8x9WgmeX\\ngUIi9REwXlGDW0Mz50dxpxcKCAYn65+7TCnY5O/jmL0VRxU1J2mSWyWTo1C+17L0\\n3fUqjxL1pkefwecxwecvC+gFFYdJ4CQ/MHHXU81Lwl1iWdFCd2UoGddYaOF+KNeM\\nHC7cmqra+JsCgYEAlUNywzq8nUg7282E+uICfCB0LfwejuymR93CtsFgb7cRd6ak\\nECR8FGfCpH8ruWJINllbQfcHVCX47ndLZwqv3oVFKh6pAS/vVI4dpOepP8++7y1u\\ncoOvtreXCX6XqfrWDtKIvv0vjlHBhhhp6mCcRpdQjV38H7JsyJ7lih/oNjECgYAt\\nkndj5uNl5SiuVxHFhcZJO+XWf6ofLUregtevZakGMn8EE1uVa2AY7eafmoU/nZPT\\n00YB0TBATdCbn/nBSuKDESkhSg9s2GEKQZG5hBmL5uCMfo09z3SfxZIhJdlerreP\\nJ7gSidI12N+EZxYd4xIJh/HFDgp7RRO87f+WJkofMQKBgGTnClK1VMaCRbJZPriw\\nEfeFCoOX75MxKwXs6xgrw4W//AYGGUjDt83lD6AZP6tws7gJ2IwY/qP7+lyhjEqN\\nHtfPZRGFkGZsdaksdlaksd323423d+15/UvrlRSFPNj1tWQmNKkXyRDW4IG1Oa2p\\nrALStNBx5Y9t0/LQnFI4w3aG\\n-----END PRIVATE KEY-----\\n\",\n  \"project_id\": \"someprojectid\",\n  \"client_email\": \"someclientid@developer.gserviceaccount.com\",\n  \"client_id\": \"someclientid.apps.googleusercontent.com\",\n  \"type\": \"service_account\",\n  \"universe_domain\": \"googleapis.com\"\n}";
    private static final String FAKE_JSON_CRED_WITH_INVALID_DOMAIN = "{\n  \"private_key_id\": \"somekeyid\",\n  \"private_key\": \"-----BEGIN PRIVATE KEY-----\\nMIIEvgIBADANBgkqhkiG9w0BAQEFAASCBKgwggSkAgEAAoIBAQC+K2hSuFpAdrJI\\nnCgcDz2M7t7bjdlsadsasad+fvRSW6TjNQZ3p5LLQY1kSZRqBqylRkzteMOyHgaR\\n0Pmxh3ILCND5men43j3h4eDbrhQBuxfEMalkG92sL+PNQSETY2tnvXryOvmBRwa/\\nQP/9dJfIkIDJ9Fw9N4Bhhhp6mCcRpdQjV38H7JsyJ7lih/oNjECgYAt\\nknddadwkwewcVxHFhcZJO+XWf6ofLUXpRwiTZakGMn8EE1uVa2LgczOjwWHGi99MFjxSer5m9\\n1tCa3/KEGKiS/YL71JvjwX3mb+cewlkcmweBKZHM2JPTk0ZednFSpVZMtycjkbLa\\ndYOS8V85AgMBewECggEBAKksaldajfDZDV6nGqbFjMiizAKJolr/M3OQw16K6o3/\\n0S31xIe3sSlgW0+UbYlF4U8KifhManD1apVSC3csafaspP4RZUHFhtBywLO9pR5c\\nr6S5aLp+gPWFyIp1pfXbWGvc5VY/v9x7ya1VEa6rXvLsKupSeWAW4tMj3eo/64ge\\nsdaceaLYw52KeBYiT6+vpsnYrEkAHO1fF/LavbLLOFJmFTMxmsNaG0tuiJHgjshB\\n82DpMCbXG9YcCgI/DbzuIjsdj2JC1cascSP//3PmefWysucBQe7Jryb6NQtASmnv\\nCdDw/0jmZTEjpe4S1lxfHplAhHFtdgYTvyYtaLZiVVkCgYEA8eVpof2rceecw/I6\\n5ng1q3Hl2usdWV/4mZMvR0fOemacLLfocX6IYxT1zA1FFJlbXSRsJMf/Qq39mOR2\\nSpW+hr4jCoHeRVYLgsbggtrevGmILAlNoqCMpGZ6vDmJpq6ECV9olliDvpPgWOP+\\nmYPDreFBGxWvQrADNbRt2dmGsrsCgYEAyUHqB2wvJHFqdmeBsaacewzV8x9WgmeX\\ngUIi9REwXlGDW0Mz50dxpxcKCAYn65+7TCnY5O/jmL0VRxU1J2mSWyWTo1C+17L0\\n3fUqjxL1pkefwecxwecvC+gFFYdJ4CQ/MHHXU81Lwl1iWdFCd2UoGddYaOF+KNeM\\nHC7cmqra+JsCgYEAlUNywzq8nUg7282E+uICfCB0LfwejuymR93CtsFgb7cRd6ak\\nECR8FGfCpH8ruWJINllbQfcHVCX47ndLZwqv3oVFKh6pAS/vVI4dpOepP8++7y1u\\ncoOvtreXCX6XqfrWDtKIvv0vjlHBhhhp6mCcRpdQjV38H7JsyJ7lih/oNjECgYAt\\nkndj5uNl5SiuVxHFhcZJO+XWf6ofLUregtevZakGMn8EE1uVa2AY7eafmoU/nZPT\\n00YB0TBATdCbn/nBSuKDESkhSg9s2GEKQZG5hBmL5uCMfo09z3SfxZIhJdlerreP\\nJ7gSidI12N+EZxYd4xIJh/HFDgp7RRO87f+WJkofMQKBgGTnClK1VMaCRbJZPriw\\nEfeFCoOX75MxKwXs6xgrw4W//AYGGUjDt83lD6AZP6tws7gJ2IwY/qP7+lyhjEqN\\nHtfPZRGFkGZsdaksdlaksd323423d+15/UvrlRSFPNj1tWQmNKkXyRDW4IG1Oa2p\\nrALStNBx5Y9t0/LQnFI4w3aG\\n-----END PRIVATE KEY-----\\n\",\n  \"project_id\": \"someprojectid\",\n  \"client_email\": \"someclientid@developer.gserviceaccount.com\",\n  \"client_id\": \"someclientid.apps.googleusercontent.com\",\n  \"type\": \"service_account\",\n  \"universe_domain\": \"fake.domain\"\n}";

    @BeforeClass
    public static void beforeClass() throws IOException {
        client = BigQueryStorageClient.create();
        parentProjectId = String.format("projects/%s", ServiceOptions.getDefaultProjectId());
        LOG.info(String.format("%s tests running with parent project: %s", ITBigQueryStorageTest.class.getSimpleName(), parentProjectId));
        RemoteBigQueryHelper bigqueryHelper = RemoteBigQueryHelper.create();
        bigquery = (BigQuery)bigqueryHelper.getOptions().getService();
        DatasetInfo datasetInfo = DatasetInfo.newBuilder((String)DATASET).setDescription(DESCRIPTION).build();
        bigquery.create(datasetInfo, new BigQuery.DatasetOption[0]);
        LOG.info("Created test dataset: " + DATASET);
    }

    @AfterClass
    public static void afterClass() {
        if (client != null) {
            client.close();
        }
        if (bigquery != null) {
            RemoteBigQueryHelper.forceDelete((BigQuery)bigquery, (String)DATASET);
            LOG.info("Deleted test dataset: " + DATASET);
        }
    }

    @Test
    public void testSimpleRead() {
        TableReferenceProto.TableReference tableReference = TableReferenceProto.TableReference.newBuilder().setProjectId("bigquery-public-data").setDatasetId("samples").setTableId("shakespeare").build();
        Storage.ReadSession session = client.createReadSession(tableReference, parentProjectId, 1);
        Assert.assertEquals((String)String.format("Did not receive expected number of streams for table reference '%s' CreateReadSession response:%n%s", TextFormat.shortDebugString((MessageOrBuilder)tableReference), session.toString()), (long)1L, (long)session.getStreamsCount());
        Storage.StreamPosition readPosition = Storage.StreamPosition.newBuilder().setStream(session.getStreams(0)).build();
        Storage.ReadRowsRequest readRowsRequest = Storage.ReadRowsRequest.newBuilder().setReadPosition(readPosition).build();
        long rowCount = 0L;
        ServerStream stream = client.readRowsCallable().call((Object)readRowsRequest);
        for (Storage.ReadRowsResponse response : stream) {
            rowCount += response.getRowCount();
        }
        Assert.assertEquals((long)164656L, (long)rowCount);
    }

    @Test
    public void testSimpleReadAndResume() {
        TableReferenceProto.TableReference tableReference = TableReferenceProto.TableReference.newBuilder().setProjectId("bigquery-public-data").setDatasetId("samples").setTableId("shakespeare").build();
        Storage.ReadSession session = client.createReadSession(tableReference, parentProjectId, 1);
        Assert.assertEquals((String)String.format("Did not receive expected number of streams for table reference '%s' CreateReadSession response:%n%s", TextFormat.shortDebugString((MessageOrBuilder)tableReference), session.toString()), (long)1L, (long)session.getStreamsCount());
        long rowCount = this.ReadStreamToOffset(session.getStreams(0), 34846L);
        Storage.StreamPosition readPosition = Storage.StreamPosition.newBuilder().setStream(session.getStreams(0)).setOffset(rowCount).build();
        Storage.ReadRowsRequest readRowsRequest = Storage.ReadRowsRequest.newBuilder().setReadPosition(readPosition).build();
        ServerStream stream = client.readRowsCallable().call((Object)readRowsRequest);
        for (Storage.ReadRowsResponse response : stream) {
            rowCount += response.getRowCount();
        }
        Assert.assertEquals((long)164656L, (long)rowCount);
    }

    @Test
    public void testFilter() throws IOException {
        TableReferenceProto.TableReference tableReference = TableReferenceProto.TableReference.newBuilder().setProjectId("bigquery-public-data").setDatasetId("samples").setTableId("shakespeare").build();
        ReadOptions.TableReadOptions options = ReadOptions.TableReadOptions.newBuilder().setRowRestriction("word_count > 100").build();
        Storage.CreateReadSessionRequest request = Storage.CreateReadSessionRequest.newBuilder().setParent(parentProjectId).setRequestedStreams(1).setTableReference(tableReference).setReadOptions(options).setFormat(Storage.DataFormat.AVRO).build();
        Storage.ReadSession session = client.createReadSession(request);
        Assert.assertEquals((String)String.format("Did not receive expected number of streams for table reference '%s' CreateReadSession response:%n%s", TextFormat.shortDebugString((MessageOrBuilder)tableReference), session.toString()), (long)1L, (long)session.getStreamsCount());
        Storage.StreamPosition readPosition = Storage.StreamPosition.newBuilder().setStream(session.getStreams(0)).build();
        Storage.ReadRowsRequest readRowsRequest = Storage.ReadRowsRequest.newBuilder().setReadPosition(readPosition).build();
        SimpleRowReader reader = new SimpleRowReader(new Schema.Parser().parse(session.getAvroSchema().getSchema()));
        long rowCount = 0L;
        ServerStream stream = client.readRowsCallable().call((Object)readRowsRequest);
        for (Storage.ReadRowsResponse response : stream) {
            rowCount += response.getRowCount();
            reader.processRows(response.getAvroRows(), new SimpleRowReader.AvroRowConsumer(){

                @Override
                public void accept(GenericData.Record record) {
                    Long wordCount = (Long)record.get("word_count");
                    Truth.assertWithMessage((String)"Row not matching expectations: %s", (Object[])new Object[]{record.toString()}).that(wordCount).isGreaterThan((Comparable)Long.valueOf(100L));
                }
            });
        }
        Assert.assertEquals((long)1333L, (long)rowCount);
    }

    @Test
    public void testColumnSelection() throws IOException {
        TableReferenceProto.TableReference tableReference = TableReferenceProto.TableReference.newBuilder().setProjectId("bigquery-public-data").setDatasetId("samples").setTableId("shakespeare").build();
        ReadOptions.TableReadOptions options = ReadOptions.TableReadOptions.newBuilder().addSelectedFields("word").addSelectedFields("word_count").setRowRestriction("word_count > 100").build();
        Storage.CreateReadSessionRequest request = Storage.CreateReadSessionRequest.newBuilder().setParent(parentProjectId).setRequestedStreams(1).setTableReference(tableReference).setReadOptions(options).setFormat(Storage.DataFormat.AVRO).build();
        Storage.ReadSession session = client.createReadSession(request);
        Assert.assertEquals((String)String.format("Did not receive expected number of streams for table reference '%s' CreateReadSession response:%n%s", TextFormat.shortDebugString((MessageOrBuilder)tableReference), session.toString()), (long)1L, (long)session.getStreamsCount());
        Storage.StreamPosition readPosition = Storage.StreamPosition.newBuilder().setStream(session.getStreams(0)).build();
        Storage.ReadRowsRequest readRowsRequest = Storage.ReadRowsRequest.newBuilder().setReadPosition(readPosition).build();
        org.apache.avro.Schema avroSchema = new Schema.Parser().parse(session.getAvroSchema().getSchema());
        String actualSchemaMessage = String.format("Unexpected schema. Actual schema:%n%s", avroSchema.toString(true));
        Assert.assertEquals((String)actualSchemaMessage, (Object)Schema.Type.RECORD, (Object)avroSchema.getType());
        Assert.assertEquals((String)actualSchemaMessage, (Object)"__root__", (Object)avroSchema.getName());
        Assert.assertEquals((String)actualSchemaMessage, (long)2L, (long)avroSchema.getFields().size());
        Assert.assertEquals((String)actualSchemaMessage, (Object)Schema.Type.STRING, (Object)avroSchema.getField("word").schema().getType());
        Assert.assertEquals((String)actualSchemaMessage, (Object)Schema.Type.LONG, (Object)avroSchema.getField("word_count").schema().getType());
        SimpleRowReader reader = new SimpleRowReader(avroSchema);
        long rowCount = 0L;
        ServerStream stream = client.readRowsCallable().call((Object)readRowsRequest);
        for (Storage.ReadRowsResponse response : stream) {
            rowCount += response.getRowCount();
            reader.processRows(response.getAvroRows(), new SimpleRowReader.AvroRowConsumer(){

                @Override
                public void accept(GenericData.Record record) {
                    String rowAssertMessage = String.format("Row not matching expectations: %s", record.toString());
                    Long wordCount = (Long)record.get("word_count");
                    Truth.assertWithMessage((String)rowAssertMessage).that(wordCount).isGreaterThan((Comparable)Long.valueOf(100L));
                    Utf8 word = (Utf8)record.get("word");
                    Truth.assertWithMessage((String)rowAssertMessage).that(Integer.valueOf(word.length())).isGreaterThan((Comparable)Integer.valueOf(0));
                }
            });
        }
        Assert.assertEquals((long)1333L, (long)rowCount);
    }

    @Test
    public void testReadAtSnapshot() throws InterruptedException, IOException {
        Field intFieldSchema = Field.newBuilder((String)"col", (LegacySQLTypeName)LegacySQLTypeName.INTEGER, (Field[])new Field[0]).setMode(Field.Mode.REQUIRED).setDescription("IntegerDescription").build();
        Schema tableSchema = Schema.of((Field[])new Field[]{intFieldSchema});
        TableId testTableId = TableId.of((String)DATASET, (String)"test_read_snapshot");
        bigquery.create(TableInfo.of((TableId)testTableId, (TableDefinition)StandardTableDefinition.of((Schema)tableSchema)), new BigQuery.TableOption[0]);
        TableReferenceProto.TableReference tableReference = TableReferenceProto.TableReference.newBuilder().setTableId(testTableId.getTable()).setDatasetId(DATASET).setProjectId(ServiceOptions.getDefaultProjectId()).build();
        Job firstJob = this.RunQueryAppendJobAndExpectSuccess(testTableId, "SELECT 1 AS col");
        Job secondJob = this.RunQueryAppendJobAndExpectSuccess(testTableId, "SELECT 2 AS col");
        final ArrayList rowsAfterFirstSnapshot = new ArrayList();
        this.ProcessRowsAtSnapshot(tableReference, firstJob.getStatistics().getEndTime(), null, new SimpleRowReader.AvroRowConsumer(){

            @Override
            public void accept(GenericData.Record record) {
                rowsAfterFirstSnapshot.add((Long)record.get("col"));
            }
        });
        Assert.assertEquals(Arrays.asList(1L), rowsAfterFirstSnapshot);
        final ArrayList rowsAfterSecondSnapshot = new ArrayList();
        this.ProcessRowsAtSnapshot(tableReference, secondJob.getStatistics().getEndTime(), null, new SimpleRowReader.AvroRowConsumer(){

            @Override
            public void accept(GenericData.Record record) {
                rowsAfterSecondSnapshot.add((Long)record.get("col"));
            }
        });
        Collections.sort(rowsAfterSecondSnapshot);
        Assert.assertEquals(Arrays.asList(1L, 2L), rowsAfterSecondSnapshot);
    }

    @Test
    public void testColumnPartitionedTableByDateField() throws InterruptedException, IOException {
        String partitionedTableName = "test_column_partition_table_by_date";
        String createTableStatement = String.format(" CREATE TABLE %s.%s (num_field INT64, date_field DATE)  PARTITION BY date_field  OPTIONS(    description=\"a table partitioned by date_field\"  ) AS    SELECT 1, CAST(\"2019-01-01\" AS DATE)   UNION ALL   SELECT 2, CAST(\"2019-01-02\" AS DATE)   UNION ALL   SELECT 3, CAST(\"2019-01-03\" AS DATE)", DATASET, partitionedTableName);
        this.RunQueryJobAndExpectSuccess(QueryJobConfiguration.newBuilder((String)createTableStatement).build());
        TableReferenceProto.TableReference tableReference = TableReferenceProto.TableReference.newBuilder().setTableId(partitionedTableName).setDatasetId(DATASET).setProjectId(ServiceOptions.getDefaultProjectId()).build();
        List<GenericData.Record> unfilteredRows = this.ReadAllRows(tableReference, null);
        Assert.assertEquals((String)("Actual rows read: " + unfilteredRows.toString()), (long)3L, (long)unfilteredRows.size());
        List<GenericData.Record> partitionFilteredRows = this.ReadAllRows(tableReference, "date_field = CAST(\"2019-01-02\" AS DATE)");
        Assert.assertEquals((String)("Actual rows read: " + partitionFilteredRows.toString()), (long)1L, (long)partitionFilteredRows.size());
        Assert.assertEquals((Object)2L, (Object)partitionFilteredRows.get(0).get("num_field"));
    }

    @Test
    public void testIngestionTimePartitionedTable() throws InterruptedException, IOException {
        Field intFieldSchema = Field.newBuilder((String)"num_field", (LegacySQLTypeName)LegacySQLTypeName.INTEGER, (Field[])new Field[0]).setMode(Field.Mode.REQUIRED).setDescription("IntegerDescription").build();
        Schema tableSchema = Schema.of((Field[])new Field[]{intFieldSchema});
        TableId testTableId = TableId.of((String)DATASET, (String)"test_date_partitioned_table");
        bigquery.create(TableInfo.of((TableId)testTableId, (TableDefinition)StandardTableDefinition.newBuilder().setTimePartitioning(TimePartitioning.of((TimePartitioning.Type)TimePartitioning.Type.DAY)).setSchema(tableSchema).build()), new BigQuery.TableOption[0]);
        this.RunQueryAppendJobAndExpectSuccess(TableId.of((String)DATASET, (String)(testTableId.getTable() + "$20190101")), "SELECT 1 AS num_field");
        this.RunQueryAppendJobAndExpectSuccess(TableId.of((String)DATASET, (String)(testTableId.getTable() + "$20190102")), "SELECT 2 AS num_field");
        TableReferenceProto.TableReference tableReference = TableReferenceProto.TableReference.newBuilder().setTableId(testTableId.getTable()).setDatasetId(testTableId.getDataset()).setProjectId(ServiceOptions.getDefaultProjectId()).build();
        List<GenericData.Record> unfilteredRows = this.ReadAllRows(tableReference, null);
        Assert.assertEquals((String)("Actual rows read: " + unfilteredRows.toString()), (long)2L, (long)unfilteredRows.size());
        List<GenericData.Record> partitionFilteredRows = this.ReadAllRows(tableReference, "_PARTITIONDATE > \"2019-01-01\"");
        Assert.assertEquals((String)("Actual rows read: " + partitionFilteredRows.toString()), (long)1L, (long)partitionFilteredRows.size());
        Assert.assertEquals((Object)2L, (Object)partitionFilteredRows.get(0).get("num_field"));
    }

    @Test
    public void testBasicSqlTypes() throws InterruptedException, IOException {
        String table_name = "test_basic_sql_types";
        String createTableStatement = String.format(" CREATE TABLE %s.%s  (int_field INT64 NOT NULL,  num_field NUMERIC NOT NULL,  float_field FLOAT64 NOT NULL,  bool_field BOOL NOT NULL,  str_field STRING NOT NULL,  bytes_field BYTES NOT NULL)  OPTIONS(    description=\"a table with basic column types\"  ) AS    SELECT      17,     CAST(1234.56 AS NUMERIC),     6.547678,     TRUE,     \"String field value\",     b\"\u0430\u0431\u0432\u0433\u0434\"", DATASET, table_name);
        this.RunQueryJobAndExpectSuccess(QueryJobConfiguration.newBuilder((String)createTableStatement).build());
        TableReferenceProto.TableReference tableReference = TableReferenceProto.TableReference.newBuilder().setTableId(table_name).setDatasetId(DATASET).setProjectId(ServiceOptions.getDefaultProjectId()).build();
        List<GenericData.Record> rows = this.ReadAllRows(tableReference, null);
        Assert.assertEquals((String)("Actual rows read: " + rows.toString()), (long)1L, (long)rows.size());
        GenericData.Record record = rows.get(0);
        org.apache.avro.Schema avroSchema = record.getSchema();
        String actualSchemaMessage = String.format("Unexpected schema. Actual schema:%n%s", avroSchema.toString(true));
        String rowAssertMessage = String.format("Row not matching expectations: %s", record.toString());
        Assert.assertEquals((String)actualSchemaMessage, (Object)Schema.Type.RECORD, (Object)avroSchema.getType());
        Assert.assertEquals((String)actualSchemaMessage, (Object)"__root__", (Object)avroSchema.getName());
        Assert.assertEquals((String)actualSchemaMessage, (long)6L, (long)avroSchema.getFields().size());
        Assert.assertEquals((String)actualSchemaMessage, (Object)Schema.Type.LONG, (Object)avroSchema.getField("int_field").schema().getType());
        Assert.assertEquals((String)rowAssertMessage, (long)17L, (long)((Long)record.get("int_field")));
        Assert.assertEquals((String)actualSchemaMessage, (Object)Schema.Type.BYTES, (Object)avroSchema.getField("num_field").schema().getType());
        Assert.assertEquals((String)actualSchemaMessage, (Object)LogicalTypes.decimal((int)38, (int)9), (Object)avroSchema.getField("num_field").schema().getLogicalType());
        BigDecimal actual_num_field = new Conversions.DecimalConversion().fromBytes((ByteBuffer)record.get("num_field"), avroSchema, avroSchema.getField("num_field").schema().getLogicalType());
        Assert.assertEquals((String)rowAssertMessage, (Object)BigDecimal.valueOf(1234560000000L, 9), (Object)actual_num_field);
        Assert.assertEquals((String)actualSchemaMessage, (Object)Schema.Type.DOUBLE, (Object)avroSchema.getField("float_field").schema().getType());
        Assert.assertEquals((String)rowAssertMessage, (double)6.547678, (double)((Double)record.get("float_field")), (double)1.0E-4);
        Assert.assertEquals((String)actualSchemaMessage, (Object)Schema.Type.BOOLEAN, (Object)avroSchema.getField("bool_field").schema().getType());
        Assert.assertEquals((String)rowAssertMessage, (Object)true, (Object)record.get("bool_field"));
        Assert.assertEquals((String)actualSchemaMessage, (Object)Schema.Type.STRING, (Object)avroSchema.getField("str_field").schema().getType());
        Assert.assertEquals((String)rowAssertMessage, (Object)new Utf8("String field value"), (Object)record.get("str_field"));
        Assert.assertEquals((String)actualSchemaMessage, (Object)Schema.Type.BYTES, (Object)avroSchema.getField("bytes_field").schema().getType());
        Assert.assertArrayEquals((String)rowAssertMessage, (byte[])Utf8.getBytesFor((String)"\u0430\u0431\u0432\u0433\u0434"), (byte[])((ByteBuffer)record.get("bytes_field")).array());
    }

    @Test
    public void testDateAndTimeSqlTypes() throws InterruptedException, IOException {
        String table_name = "test_date_and_time_sql_types";
        String createTableStatement = String.format(" CREATE TABLE %s.%s  (date_field DATE NOT NULL,  datetime_field DATETIME NOT NULL,  time_field TIME NOT NULL,  timestamp_field TIMESTAMP NOT NULL) OPTIONS(    description=\"a table with date and time column types\"  ) AS    SELECT      CAST(\"2019-05-31\" AS DATE),     CAST(\"2019-04-30 21:47:59.999999\" AS DATETIME),     CAST(\"21:47:59.999999\" AS TIME),     CAST(\"2019-04-30 19:24:19.123456 UTC\" AS TIMESTAMP)", DATASET, table_name);
        this.RunQueryJobAndExpectSuccess(QueryJobConfiguration.newBuilder((String)createTableStatement).build());
        TableReferenceProto.TableReference tableReference = TableReferenceProto.TableReference.newBuilder().setTableId(table_name).setDatasetId(DATASET).setProjectId(ServiceOptions.getDefaultProjectId()).build();
        List<GenericData.Record> rows = this.ReadAllRows(tableReference, null);
        Assert.assertEquals((String)("Actual rows read: " + rows.toString()), (long)1L, (long)rows.size());
        GenericData.Record record = rows.get(0);
        org.apache.avro.Schema avroSchema = record.getSchema();
        String actualSchemaMessage = String.format("Unexpected schema. Actual schema:%n%s", avroSchema.toString(true));
        String rowAssertMessage = String.format("Row not matching expectations: %s", record.toString());
        Assert.assertEquals((String)actualSchemaMessage, (Object)Schema.Type.RECORD, (Object)avroSchema.getType());
        Assert.assertEquals((String)actualSchemaMessage, (Object)"__root__", (Object)avroSchema.getName());
        Assert.assertEquals((String)actualSchemaMessage, (long)4L, (long)avroSchema.getFields().size());
        Assert.assertEquals((String)actualSchemaMessage, (Object)Schema.Type.INT, (Object)avroSchema.getField("date_field").schema().getType());
        Assert.assertEquals((String)actualSchemaMessage, (Object)LogicalTypes.date(), (Object)avroSchema.getField("date_field").schema().getLogicalType());
        Assert.assertEquals((String)rowAssertMessage, (Object)LocalDate.of((int)2019, (int)5, (int)31), (Object)LocalDate.ofEpochDay((long)((Integer)record.get("date_field")).intValue()));
        Assert.assertEquals((String)actualSchemaMessage, (Object)Schema.Type.STRING, (Object)avroSchema.getField("datetime_field").schema().getType());
        Assert.assertEquals((String)actualSchemaMessage, (Object)"datetime", (Object)avroSchema.getField("datetime_field").schema().getObjectProp("logicalType"));
        Assert.assertEquals((String)rowAssertMessage, (Object)new Utf8("2019-04-30T21:47:59.999999"), (Object)((Utf8)record.get("datetime_field")));
        Assert.assertEquals((String)actualSchemaMessage, (Object)Schema.Type.LONG, (Object)avroSchema.getField("time_field").schema().getType());
        Assert.assertEquals((String)actualSchemaMessage, (Object)LogicalTypes.timeMicros(), (Object)avroSchema.getField("time_field").schema().getLogicalType());
        Assert.assertEquals((String)rowAssertMessage, (Object)LocalTime.of((int)21, (int)47, (int)59, (int)999999000), (Object)LocalTime.ofNanoOfDay((long)(1000L * (Long)record.get("time_field"))));
        Assert.assertEquals((String)actualSchemaMessage, (Object)Schema.Type.LONG, (Object)avroSchema.getField("timestamp_field").schema().getType());
        Assert.assertEquals((String)actualSchemaMessage, (Object)LogicalTypes.timestampMicros(), (Object)avroSchema.getField("timestamp_field").schema().getLogicalType());
        ZonedDateTime expected_timestamp = ZonedDateTime.parse((CharSequence)"2019-04-30T19:24:19Z", (DateTimeFormatter)DateTimeFormatter.ISO_INSTANT.withZone((ZoneId)ZoneOffset.UTC)).withNano(123456000);
        long actual_timestamp_micros = (Long)record.get("timestamp_field");
        ZonedDateTime actual_timestamp = ZonedDateTime.ofInstant((Instant)Instant.ofEpochSecond((long)(actual_timestamp_micros / 1000000L), (long)(actual_timestamp_micros % 1000000L * 1000L)), (ZoneId)ZoneOffset.UTC);
        Assert.assertEquals((String)rowAssertMessage, (Object)expected_timestamp, (Object)actual_timestamp);
    }

    @Test
    public void testGeographySqlType() throws InterruptedException, IOException {
        String table_name = "test_geography_sql_type";
        String createTableStatement = String.format(" CREATE TABLE %s.%s  (geo_field GEOGRAPHY NOT NULL) OPTIONS(    description=\"a table with a geography column type\"  ) AS    SELECT ST_GEOGPOINT(1.1, 2.2)", DATASET, table_name);
        this.RunQueryJobAndExpectSuccess(QueryJobConfiguration.newBuilder((String)createTableStatement).build());
        TableReferenceProto.TableReference tableReference = TableReferenceProto.TableReference.newBuilder().setTableId(table_name).setDatasetId(DATASET).setProjectId(ServiceOptions.getDefaultProjectId()).build();
        List<GenericData.Record> rows = this.ReadAllRows(tableReference, null);
        Assert.assertEquals((String)("Actual rows read: " + rows.toString()), (long)1L, (long)rows.size());
        GenericData.Record record = rows.get(0);
        org.apache.avro.Schema avroSchema = record.getSchema();
        String actualSchemaMessage = String.format("Unexpected schema. Actual schema:%n%s", avroSchema.toString(true));
        String rowAssertMessage = String.format("Row not matching expectations: %s", record.toString());
        Assert.assertEquals((String)actualSchemaMessage, (Object)Schema.Type.RECORD, (Object)avroSchema.getType());
        Assert.assertEquals((String)actualSchemaMessage, (Object)"__root__", (Object)avroSchema.getName());
        Assert.assertEquals((String)actualSchemaMessage, (long)1L, (long)avroSchema.getFields().size());
        Assert.assertEquals((String)actualSchemaMessage, (Object)Schema.Type.STRING, (Object)avroSchema.getField("geo_field").schema().getType());
        Assert.assertEquals((String)actualSchemaMessage, (Object)"GEOGRAPHY", (Object)avroSchema.getField("geo_field").schema().getObjectProp("sqlType"));
        Assert.assertEquals((String)rowAssertMessage, (Object)new Utf8("POINT(1.1 2.2)"), (Object)((Utf8)record.get("geo_field")));
    }

    @Test
    public void testStructAndArraySqlTypes() throws InterruptedException, IOException {
        String table_name = "test_struct_and_array_sql_types";
        String createTableStatement = String.format(" CREATE TABLE %s.%s  (array_field ARRAY<INT64>,  struct_field STRUCT<int_field INT64 NOT NULL, str_field STRING NOT NULL> NOT NULL) OPTIONS(    description=\"a table with array and time column types\"  ) AS    SELECT      [1, 2, 3],     (10, 'abc')", DATASET, table_name);
        this.RunQueryJobAndExpectSuccess(QueryJobConfiguration.newBuilder((String)createTableStatement).build());
        TableReferenceProto.TableReference tableReference = TableReferenceProto.TableReference.newBuilder().setTableId(table_name).setDatasetId(DATASET).setProjectId(ServiceOptions.getDefaultProjectId()).build();
        List<GenericData.Record> rows = this.ReadAllRows(tableReference, null);
        Assert.assertEquals((String)("Actual rows read: " + rows.toString()), (long)1L, (long)rows.size());
        GenericData.Record record = rows.get(0);
        org.apache.avro.Schema avroSchema = record.getSchema();
        String actualSchemaMessage = String.format("Unexpected schema. Actual schema:%n%s", avroSchema.toString(true));
        String rowAssertMessage = String.format("Row not matching expectations: %s", record.toString());
        Assert.assertEquals((String)actualSchemaMessage, (Object)Schema.Type.RECORD, (Object)avroSchema.getType());
        Assert.assertEquals((String)actualSchemaMessage, (Object)"__root__", (Object)avroSchema.getName());
        Assert.assertEquals((String)actualSchemaMessage, (long)2L, (long)avroSchema.getFields().size());
        Assert.assertEquals((String)actualSchemaMessage, (Object)Schema.Type.ARRAY, (Object)avroSchema.getField("array_field").schema().getType());
        Assert.assertEquals((String)actualSchemaMessage, (Object)Schema.Type.LONG, (Object)avroSchema.getField("array_field").schema().getElementType().getType());
        Assert.assertArrayEquals((String)rowAssertMessage, (Object[])new Long[]{1L, 2L, 3L}, (Object[])((GenericData.Array)record.get("array_field")).toArray((Object[])new Long[0]));
        org.apache.avro.Schema structSchema = avroSchema.getField("struct_field").schema();
        Assert.assertEquals((String)actualSchemaMessage, (Object)Schema.Type.RECORD, (Object)structSchema.getType());
        GenericData.Record structRecord = (GenericData.Record)record.get("struct_field");
        Assert.assertEquals((String)actualSchemaMessage, (Object)Schema.Type.LONG, (Object)structSchema.getField("int_field").schema().getType());
        Assert.assertEquals((String)rowAssertMessage, (long)10L, (long)((Long)structRecord.get("int_field")));
        Assert.assertEquals((String)actualSchemaMessage, (Object)Schema.Type.STRING, (Object)structSchema.getField("str_field").schema().getType());
        Assert.assertEquals((String)rowAssertMessage, (Object)new Utf8("abc"), (Object)structRecord.get("str_field"));
    }

    @Test
    public void testUniverseDomainWithInvalidUniverseDomain() throws IOException {
        BigQueryStorageSettings bigQueryStorageSettings = ((BigQueryStorageSettings.Builder)((BigQueryStorageSettings.Builder)BigQueryStorageSettings.newBuilder().setCredentialsProvider((CredentialsProvider)FixedCredentialsProvider.create((Credentials)ITBigQueryStorageTest.loadCredentials(FAKE_JSON_CRED_WITH_INVALID_DOMAIN)))).setUniverseDomain("invalid.domain")).build();
        BigQueryStorageClient localClient = BigQueryStorageClient.create((BigQueryStorageSettings)bigQueryStorageSettings);
        TableReferenceProto.TableReference tableReference = TableReferenceProto.TableReference.newBuilder().setProjectId("bigquery-public-data").setDatasetId("samples").setTableId("shakespeare").build();
        try {
            localClient.createReadSession(tableReference, parentProjectId, 1);
            Assert.fail((String)"RPCs to invalid universe domain should fail");
        }
        catch (UnauthenticatedException e) {
            Truth.assertThat((Boolean)e.getMessage().contains("does not match the universe domain found in the credentials")).isTrue();
        }
        localClient.close();
    }

    @Test
    public void testInvalidUniverseDomainWithMismatchCredentials() throws IOException {
        BigQueryStorageSettings bigQueryStorageSettings = ((BigQueryStorageSettings.Builder)((BigQueryStorageSettings.Builder)BigQueryStorageSettings.newBuilder().setCredentialsProvider((CredentialsProvider)FixedCredentialsProvider.create((Credentials)ITBigQueryStorageTest.loadCredentials(FAKE_JSON_CRED_WITH_INVALID_DOMAIN)))).setUniverseDomain("invalid.domain")).build();
        BigQueryStorageClient localClient = BigQueryStorageClient.create((BigQueryStorageSettings)bigQueryStorageSettings);
        TableReferenceProto.TableReference tableReference = TableReferenceProto.TableReference.newBuilder().setProjectId("bigquery-public-data").setDatasetId("samples").setTableId("shakespeare").build();
        try {
            localClient.createReadSession(tableReference, parentProjectId, 1);
            Assert.fail((String)"RPCs to invalid universe domain should fail");
        }
        catch (UnauthenticatedException e) {
            Truth.assertThat((Boolean)e.getMessage().contains("does not match the universe domain found in the credentials")).isTrue();
        }
        localClient.close();
    }

    @Test
    public void testUniverseDomainWithMatchingDomain() throws IOException {
        BigQueryStorageSettings bigQueryStorageSettings = ((BigQueryStorageSettings.Builder)BigQueryStorageSettings.newBuilder().setUniverseDomain("googleapis.com")).build();
        BigQueryStorageClient localClient = BigQueryStorageClient.create((BigQueryStorageSettings)bigQueryStorageSettings);
        TableReferenceProto.TableReference tableReference = TableReferenceProto.TableReference.newBuilder().setProjectId("bigquery-public-data").setDatasetId("samples").setTableId("shakespeare").build();
        Storage.ReadSession session = localClient.createReadSession(tableReference, parentProjectId, 1);
        Assert.assertEquals((String)String.format("Did not receive expected number of streams for table reference '%s' CreateReadSession response:%n%s", TextFormat.shortDebugString((MessageOrBuilder)tableReference), session.toString()), (long)1L, (long)session.getStreamsCount());
        Storage.StreamPosition readPosition = Storage.StreamPosition.newBuilder().setStream(session.getStreams(0)).build();
        Storage.ReadRowsRequest readRowsRequest = Storage.ReadRowsRequest.newBuilder().setReadPosition(readPosition).build();
        long rowCount = 0L;
        ServerStream stream = client.readRowsCallable().call((Object)readRowsRequest);
        for (Storage.ReadRowsResponse response : stream) {
            rowCount += response.getRowCount();
        }
        Assert.assertEquals((long)164656L, (long)rowCount);
        localClient.close();
    }

    public void testUniverseDomain() throws IOException {
        BigQueryStorageSettings bigQueryStorageSettings = ((BigQueryStorageSettings.Builder)BigQueryStorageSettings.newBuilder().setUniverseDomain("apis-tpclp.goog")).build();
        BigQueryStorageClient localClient = BigQueryStorageClient.create((BigQueryStorageSettings)bigQueryStorageSettings);
        TableReferenceProto.TableReference tableReference = TableReferenceProto.TableReference.newBuilder().setProjectId("google-tpc-testing-environment:cloudsdk-test-project").setDatasetId("tpc_demo_dataset").setTableId("new_table").build();
        Storage.ReadSession session = localClient.createReadSession(tableReference, parentProjectId, 1);
        Storage.StreamPosition readPosition = Storage.StreamPosition.newBuilder().setStream(session.getStreams(0)).build();
        Storage.ReadRowsRequest readRowsRequest = Storage.ReadRowsRequest.newBuilder().setReadPosition(readPosition).build();
        long rowCount = 0L;
        ServerStream stream = localClient.readRowsCallable().call((Object)readRowsRequest);
        for (Storage.ReadRowsResponse response : stream) {
            rowCount += response.getRowCount();
        }
        Assert.assertEquals((long)1L, (long)rowCount);
        localClient.close();
    }

    private long ReadStreamToOffset(Storage.Stream stream, long rowOffset) {
        Storage.StreamPosition readPosition = Storage.StreamPosition.newBuilder().setStream(stream).build();
        Storage.ReadRowsRequest readRowsRequest = Storage.ReadRowsRequest.newBuilder().setReadPosition(readPosition).build();
        long rowCount = 0L;
        ServerStream serverStream = client.readRowsCallable().call((Object)readRowsRequest);
        for (Storage.ReadRowsResponse response : serverStream) {
            if ((rowCount += response.getRowCount()) < rowOffset) continue;
            return rowOffset;
        }
        return rowCount;
    }

    private void ProcessRowsAtSnapshot(TableReferenceProto.TableReference tableReference, Long snapshotInMillis, String filter, SimpleRowReader.AvroRowConsumer consumer) throws IOException {
        Preconditions.checkNotNull((Object)tableReference);
        Preconditions.checkNotNull((Object)consumer);
        Storage.CreateReadSessionRequest.Builder createSessionRequestBuilder = Storage.CreateReadSessionRequest.newBuilder().setParent(parentProjectId).setRequestedStreams(1).setTableReference(tableReference).setFormat(Storage.DataFormat.AVRO);
        if (snapshotInMillis != null) {
            Timestamp snapshotTimestamp = Timestamp.newBuilder().setSeconds(snapshotInMillis / 1000L).setNanos((int)(snapshotInMillis % 1000L * 1000000L)).build();
            createSessionRequestBuilder.setTableModifiers(TableReferenceProto.TableModifiers.newBuilder().setSnapshotTime(snapshotTimestamp).build());
        }
        if (filter != null && !filter.isEmpty()) {
            createSessionRequestBuilder.setReadOptions(ReadOptions.TableReadOptions.newBuilder().setRowRestriction(filter).build());
        }
        Storage.ReadSession session = client.createReadSession(createSessionRequestBuilder.build());
        Assert.assertEquals((String)String.format("Did not receive expected number of streams for table reference '%s' CreateReadSession response:%n%s", TextFormat.shortDebugString((MessageOrBuilder)tableReference), session.toString()), (long)1L, (long)session.getStreamsCount());
        Storage.StreamPosition readPosition = Storage.StreamPosition.newBuilder().setStream(session.getStreams(0)).build();
        Storage.ReadRowsRequest readRowsRequest = Storage.ReadRowsRequest.newBuilder().setReadPosition(readPosition).build();
        SimpleRowReader reader = new SimpleRowReader(new Schema.Parser().parse(session.getAvroSchema().getSchema()));
        ServerStream stream = client.readRowsCallable().call((Object)readRowsRequest);
        for (Storage.ReadRowsResponse response : stream) {
            reader.processRows(response.getAvroRows(), consumer);
        }
    }

    List<GenericData.Record> ReadAllRows(TableReferenceProto.TableReference tableReference, String filter) throws IOException {
        final ArrayList<GenericData.Record> rows = new ArrayList<GenericData.Record>();
        this.ProcessRowsAtSnapshot(tableReference, null, filter, new SimpleRowReader.AvroRowConsumer(){

            @Override
            public void accept(GenericData.Record record) {
                rows.add(new GenericRecordBuilder(record).build());
            }
        });
        return rows;
    }

    private Job RunQueryAppendJobAndExpectSuccess(TableId destinationTableId, String query) throws InterruptedException {
        return this.RunQueryJobAndExpectSuccess(QueryJobConfiguration.newBuilder((String)query).setDestinationTable(destinationTableId).setUseQueryCache(Boolean.valueOf(false)).setUseLegacySql(Boolean.valueOf(false)).setWriteDisposition(JobInfo.WriteDisposition.WRITE_APPEND).build());
    }

    private Job RunQueryJobAndExpectSuccess(QueryJobConfiguration configuration) throws InterruptedException {
        Job job = bigquery.create(JobInfo.of((JobConfiguration)configuration), new BigQuery.JobOption[0]);
        Job completedJob = job.waitFor(new RetryOption[]{RetryOption.initialRetryDelay((Duration)Duration.ofSeconds((long)1L)), RetryOption.totalTimeout((Duration)Duration.ofMinutes((long)1L))});
        Assert.assertNotNull((Object)completedJob);
        Assert.assertNull((String)("Received a job status that is not a success: " + completedJob.getStatus().toString()), (Object)completedJob.getStatus().getError());
        return completedJob;
    }

    static GoogleCredentials loadCredentials(String credentialFile) {
        try {
            ByteArrayInputStream keyStream = new ByteArrayInputStream(credentialFile.getBytes());
            return GoogleCredentials.fromStream((InputStream)keyStream);
        }
        catch (IOException e) {
            Assert.fail((String)"Couldn't create fake JSON credentials.");
            return null;
        }
    }
}

